/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 *
 * Copyright 2003 The Nimbus Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus Project.
 */
package jp.ossc.nimbus.service.template;

import java.io.File;
import java.io.Writer;
import java.io.IOException;
import java.io.Serializable;
import java.util.Iterator;
import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.LinkedHashSet;
import java.util.Collections;
import java.util.Locale;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/* **Version Difference Comment Start**
import jakarta.servlet.ServletContext;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
**Version Difference Comment End** */

import org.thymeleaf.*;
import org.thymeleaf.dialect.IDialect;
import org.thymeleaf.linkbuilder.ILinkBuilder;
import org.thymeleaf.templateresolver.*;
import org.thymeleaf.templatemode.*;
import org.thymeleaf.templateresource.*;
import org.thymeleaf.messageresolver.*;
import org.thymeleaf.context.*;

import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.beans.*;
import jp.ossc.nimbus.service.message.*;

/**
 * <a href="https://www.thymeleaf.org">Thymeleaf</a>を使った{@link TemplateEngine}サービス。<p>
 *
 * @author M.Takata
 */
public class ThymeleafTemplateEngineService extends ServiceBase implements TemplateEngine, ThymeleafTemplateEngineServiceMBean{

    private File templateFileRootDirectory;
    private String characterEncoding;
    private Locale locale;
    private String prefix;
    private String suffix;
    private boolean isForceSuffix;
    private String templateMode;
    private boolean isForceTemplateMode;
    private boolean isCacheable = true;
    private long cacheTTL;
    private Map templateAliases;
    private Set xmlTemplateModePatterns;
    private Set htmlTemplateModePatterns;
    private Set textTemplateModePatterns;
    private Set javaScriptTemplateModePatterns;
    private Set cssTemplateModePatterns;
    private Set rawTemplateModePatterns;
    private Set cacheablePatterns;
    private Set nonCacheablePatterns;
    private Property servletContextProperty;
    private Property httpServletRequestProperty;
    private Property httpServletResponseProperty;
    private ServiceName messageRecordFactoryServiceName;
    private Set dialects;
    private Set linkBuilders;

    private Map templateMap;
    private File templateFileDirectory;
    private org.thymeleaf.TemplateEngine engine;
    private MessageRecordFactory messageRecordFactory;

    public void setTemplateFileRootDirectory(File dir){
        templateFileRootDirectory = dir;
    }
    public File getTemplateFileRootDirectory(){
        return templateFileRootDirectory;
    }

    public void setCharacterEncoding(String encoding){
        characterEncoding = encoding;
    }
    public String getCharacterEncoding(){
        return characterEncoding;
    }

    public void setLocale(Locale lo){
        locale = lo;
    }
    public Locale getLocale(){
        return locale;
    }

    public void setPrefix(String prefix){
        this.prefix = prefix;
    }
    public String getPrefix(){
        return prefix;
    }

    public void setSuffix(String suffix){
        this.suffix = suffix;
    }
    public String getSuffix(){
        return suffix;
    }

    public void setForceSuffix(boolean isForce){
        isForceSuffix = isForce;
    }
    public boolean isForceSuffix(){
        return isForceSuffix;
    }

    public void setTemplateMode(String mode){
        templateMode = mode;
    }
    public String getTemplateMode(){
        return templateMode;
    }

    public void setForceTemplateMode(boolean isForce){
        isForceTemplateMode = isForce;
    }
    public boolean isForceTemplateMode(){
        return isForceTemplateMode;
    }

    public void setCacheable(boolean isCacheable){
        this.isCacheable = isCacheable;
    }
    public boolean isCacheable(){
        return isCacheable;
    }

    public void setCacheTTL(long millis){
        cacheTTL = millis;
    }
    public long getCacheTTL(){
        return cacheTTL;
    }

    public void setTemplateAliases(Map aliases){
        templateAliases = aliases;
    }
    public Map getTemplateAliases(){
        return templateAliases;
    }

    public void setXmlTemplateModePatterns(Set patterns){
        xmlTemplateModePatterns = patterns;
    }
    public Set getXmlTemplateModePatterns(){
        return xmlTemplateModePatterns;
    }

    public void setHtmlTemplateModePatterns(Set patterns){
        htmlTemplateModePatterns = patterns;
    }
    public Set getHtmlTemplateModePatterns(){
        return htmlTemplateModePatterns;
    }

    public void setTextTemplateModePatterns(Set patterns){
        textTemplateModePatterns = patterns;
    }
    public Set getTextTemplateModePatterns(){
        return textTemplateModePatterns;
    }

    public void setJavaScriptTemplateModePatterns(Set patterns){
        javaScriptTemplateModePatterns = patterns;
    }
    public Set getJavaScriptTemplateModePatterns(){
        return javaScriptTemplateModePatterns;
    }

    public void setCssTemplateModePatterns(Set patterns){
        cssTemplateModePatterns = patterns;
    }
    public Set getCssTemplateModePatterns(){
        return cssTemplateModePatterns;
    }

    public void setRawTemplateModePatterns(Set patterns){
        rawTemplateModePatterns = patterns;
    }
    public Set getRawTemplateModePatterns(){
        return rawTemplateModePatterns;
    }

    public void setCacheablePatterns(Set patterns){
        cacheablePatterns = patterns;
    }
    public Set getCacheablePatterns(){
        return cacheablePatterns;
    }

    public void setNonCacheablePatterns(Set patterns){
        nonCacheablePatterns = patterns;
    }
    public Set getNonCacheablePatterns(){
        return nonCacheablePatterns;
    }

    public void setServletContextProperty(Property prop){
        servletContextProperty = prop;
    }
    public Property getServletContextProperty(){
        return servletContextProperty;
    }

    public void setHttpServletRequestProperty(Property prop){
        httpServletRequestProperty = prop;
    }
    public Property getHttpServletRequestProperty(){
        return httpServletRequestProperty;
    }

    public void setHttpServletResponseProperty(Property prop){
        httpServletResponseProperty = prop;
    }
    public Property getHttpServletResponseProperty(){
        return httpServletResponseProperty;
    }

    public void setMessageRecordFactoryServiceName(ServiceName name){
        messageRecordFactoryServiceName = name;
    }
    public ServiceName getMessageRecordFactoryServiceName(){
        return messageRecordFactoryServiceName;
    }

    public void setMessageRecordFactory(MessageRecordFactory factory){
        messageRecordFactory = factory;
    }

    public void addDialect(IDialect dialect){
        if(dialects == null){
            dialects = new LinkedHashSet();
        }
        dialects.add(dialect);
    }

    public void addLinkBuilder(ILinkBuilder linkBuilder){
        if(linkBuilders == null){
            linkBuilders = new LinkedHashSet();
        }
        linkBuilders.add(linkBuilder);
    }

    public void createService() throws Exception{
        templateMap = Collections.synchronizedMap(new HashMap());
    }

    public void startService() throws Exception{
        engine = new org.thymeleaf.TemplateEngine();

        if(templateFileRootDirectory != null){
            templateFileDirectory = templateFileRootDirectory;
            if(!templateFileDirectory.exists()){
                if(getServiceNameObject() != null){
                    ServiceMetaData metaData = ServiceManagerFactory.getServiceMetaData(getServiceNameObject());
                    if(metaData != null){
                        jp.ossc.nimbus.core.ServiceLoader loader = metaData.getServiceLoader();
                        if(loader != null){
                            String filePath = loader.getServiceURL().getFile();
                            if(filePath != null){
                                File serviceDefDir = new File(filePath).getParentFile();
                                File dir = new File(serviceDefDir, templateFileDirectory.getPath());
                                if(dir.exists()){
                                    templateFileDirectory = dir;
                                }
                            }
                        }
                    }
                }
            }
        }
        if(templateFileDirectory == null){
            templateFileDirectory = new File(".");
        }
        TemplateResolver templateResolver = new TemplateResolver();
        templateResolver.setPrefix(prefix);
        templateResolver.setSuffix(suffix);
        templateResolver.setForceSuffix(isForceSuffix);
        templateResolver.setCharacterEncoding(characterEncoding);
        if(templateMode != null){
            templateResolver.setTemplateMode(templateMode);
        }
        templateResolver.setForceTemplateMode(isForceTemplateMode);
        templateResolver.setCacheable(isCacheable);
        if(cacheTTL > 0){
            templateResolver.setCacheTTLMs(new Long(cacheTTL));
        }
        if(templateAliases != null){
            templateResolver.setTemplateAliases(templateAliases);
        }
        if(xmlTemplateModePatterns != null){
            templateResolver.setXmlTemplateModePatterns(xmlTemplateModePatterns);
        }
        if(htmlTemplateModePatterns != null){
            templateResolver.setHtmlTemplateModePatterns(htmlTemplateModePatterns);
        }
        if(textTemplateModePatterns != null){
            templateResolver.setTextTemplateModePatterns(textTemplateModePatterns);
        }
        if(javaScriptTemplateModePatterns != null){
            templateResolver.setJavaScriptTemplateModePatterns(javaScriptTemplateModePatterns);
        }
        if(cssTemplateModePatterns != null){
            templateResolver.setCSSTemplateModePatterns(cssTemplateModePatterns);
        }
        if(rawTemplateModePatterns != null){
            templateResolver.setRawTemplateModePatterns(rawTemplateModePatterns);
        }
        if(cacheablePatterns != null){
            templateResolver.setCacheablePatterns(cacheablePatterns);
        }
        if(nonCacheablePatterns != null){
            templateResolver.setNonCacheablePatterns(nonCacheablePatterns);
        }
        engine.setTemplateResolver(templateResolver);
        if(messageRecordFactoryServiceName != null){
            messageRecordFactory = (MessageRecordFactory)ServiceManagerFactory.getServiceObject(messageRecordFactoryServiceName);
        }

        if(messageRecordFactory != null || super.getMessageRecordFactory() != null){
            engine.addMessageResolver(new MessageResolver());
        }
        if(dialects != null){
            Iterator itr = dialects.iterator();
            while(itr.hasNext()){
                engine.addDialect((IDialect)itr.next());
            }
        }
        if(linkBuilders != null){
            Iterator itr = linkBuilders.iterator();
            while(itr.hasNext()){
                engine.addLinkBuilder((ILinkBuilder)itr.next());
            }
        }

        if(httpServletRequestProperty != null && servletContextProperty == null){
            servletContextProperty = new NestedProperty(httpServletRequestProperty, new SimpleProperty("ServletContext"));
        }
    }

    public void stopService() throws Exception{
        engine = null;
    }

    public void destroyService() throws Exception{
        templateMap = null;
    }

    public void setTemplate(String name, String template){
        setTemplate(name, template, null);
    }

    public void setTemplate(String name, String template, String encoding){
        templateMap.put(name, new StringTemplateResource(template));
    }

    public void setTemplateFile(String name, File templateFile){
        setTemplateFile(name, templateFile, null);
    }

    public void setTemplateFile(String name, File templateFile, String encoding){
        templateMap.put(name, new FileTemplateResource(templateFile.getPath(), encoding == null ? characterEncoding : encoding));
    }

    public String transform(String name, Map dataMap) throws TemplateTransformException{
        try{
            return engine.process(name, createContext(dataMap));
        }catch(Exception e){
            throw new TemplateTransformException(e);
        }
    }

    public void transform(String name, Map dataMap, Writer writer) throws TemplateTransformException, IOException{
        try{
            writer.write(transform(name, dataMap));
        }catch(IOException e){
            throw e;
        }
    }

    protected IContext createContext(Map dataMap) throws TemplateTransformException{
        IContext ctx = null;
        ServletContext servletContext = null;
        HttpServletRequest request = null;
        HttpServletResponse response = null;
        if(httpServletRequestProperty != null && httpServletResponseProperty != null && servletContextProperty != null){
            try{
                request = (HttpServletRequest)httpServletRequestProperty.getProperty(dataMap);
                response = (HttpServletResponse)httpServletResponseProperty.getProperty(dataMap);
                servletContext = (ServletContext)servletContextProperty.getProperty(dataMap);
            }catch(Exception e){
                throw new TemplateTransformException(e);
            }
        }
        if(request != null && response != null && servletContext != null){
            throw new UnsupportedOperationException("Servlet API 6 is no supported this method.");
/*
            ctx = new WebContext(
                request,
                response,
                servletContext,
                locale == null ? request.getLocale() : locale,
                dataMap
            );
*/
        }else{
            ctx = new Context(locale, dataMap);
        }
        return ctx;
    }

    public void clearTemplateCache(){
        if(engine != null){
            engine.clearTemplateCache();
        }
    }

    public void clearTemplateCacheFor(String name){
        if(engine != null){
            engine.clearTemplateCacheFor(name);
        }
    }

    protected class TemplateResolver extends AbstractConfigurableTemplateResolver{

        protected ITemplateResource computeTemplateResource(
            final IEngineConfiguration configuration,
            final String ownerTemplate,
            final String template,
            final String resourceName,
            final String characterEncoding,
            final Map<String, Object> templateResolutionAttributes
        ){
            ITemplateResource resource = (ITemplateResource)templateMap.get(template);
            if(resource != null){
                return resource;
            }
            resource = new ClassLoaderTemplateResource(Thread.currentThread().getContextClassLoader(), resourceName, characterEncoding);
            if(resource.exists()){
                return resource;
            }

            resource = new FileTemplateResource(new File(templateFileDirectory, resourceName).getPath(), characterEncoding);
            if(resource.exists()){
                return resource;
            }
            return new StringTemplateResource(template);
        }
    }

    protected class MessageResolver extends AbstractMessageResolver{
        public String resolveMessage(final ITemplateContext context, final Class<?> origin, final String key, final Object[] messageParameters){
            MessageRecordFactory factory = messageRecordFactory == null ? ThymeleafTemplateEngineService.super.getMessageRecordFactory() : messageRecordFactory;
            return factory.findEmbedMessage(context.getLocale(), key, messageParameters);
        }
        public String createAbsentMessageRepresentation(final ITemplateContext context, final Class<?> origin, final String key, final Object[] messageParameters){
            if (context.getLocale() != null){
                return "??" + key + "_" + context.getLocale().toString() + "??";
            }
            return "??" + key + "_" + "??";
        }
    }
}
