/*
 * Decompiled with CFR 0.152.
 */
package com.github.nosan.embedded.cassandra;

import com.github.nosan.embedded.cassandra.CassandraDirectoryProvider;
import com.github.nosan.embedded.cassandra.Version;
import com.github.nosan.embedded.cassandra.commons.FileLock;
import com.github.nosan.embedded.cassandra.commons.FileUtils;
import com.github.nosan.embedded.cassandra.commons.StreamUtils;
import com.github.nosan.embedded.cassandra.commons.StringUtils;
import com.github.nosan.embedded.cassandra.commons.logging.Logger;
import com.github.nosan.embedded.cassandra.commons.web.HttpClient;
import com.github.nosan.embedded.cassandra.commons.web.HttpRequest;
import com.github.nosan.embedded.cassandra.commons.web.HttpResponse;
import com.github.nosan.embedded.cassandra.commons.web.JdkHttpClient;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.URI;
import java.nio.charset.Charset;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.security.NoSuchAlgorithmException;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.ArchiveInputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream;

public class WebCassandraDirectoryProvider
implements CassandraDirectoryProvider {
    protected static final String[] ALGORITHMS = new String[]{"SHA-512", "SHA-256", "SHA-1", "MD5"};
    private static final Logger LOGGER = Logger.get(WebCassandraDirectoryProvider.class);
    private final HttpClient httpClient;
    private final Path downloadDirectory;

    public WebCassandraDirectoryProvider() {
        this(new JdkHttpClient(), Paths.get(System.getProperty("user.home"), new String[0]));
    }

    public WebCassandraDirectoryProvider(HttpClient httpClient) {
        this(httpClient, Paths.get(System.getProperty("user.home"), new String[0]));
    }

    public WebCassandraDirectoryProvider(Path downloadDirectory) {
        this(new JdkHttpClient(), downloadDirectory);
    }

    public WebCassandraDirectoryProvider(HttpClient httpClient, Path downloadDirectory) {
        Objects.requireNonNull(httpClient, "HTTP Client must not be null");
        Objects.requireNonNull(downloadDirectory, "Download Directory must not be null");
        this.httpClient = httpClient;
        this.downloadDirectory = downloadDirectory;
    }

    @Override
    public final Path getDirectory(Version version) throws IOException {
        Objects.requireNonNull(version, "Version must not be null");
        Path downloadDirectory = this.downloadDirectory.resolve(".embedded-cassandra").resolve("cassandra").resolve(version.toString());
        Path successFile = downloadDirectory.resolve(".success");
        Path cassandraDirectory = downloadDirectory.resolve(String.format("apache-cassandra-%s", version));
        if (Files.exists(successFile, new LinkOption[0]) && Files.exists(cassandraDirectory, new LinkOption[0])) {
            return cassandraDirectory;
        }
        LOGGER.info("Cassandra directory: ''{0}'' is not found. Initializing...", cassandraDirectory);
        Files.createDirectories(downloadDirectory, new FileAttribute[0]);
        Path lockFile = downloadDirectory.resolve(".lock");
        FileLock fileLock = FileLock.of(lockFile);
        LOGGER.info("Acquires a lock to the file ''{0}''...", lockFile);
        if (!this.tryLock(fileLock)) {
            throw new IOException(String.format("Unable to provide Cassandra Directory for a version: '%s'. File lock could not be acquired for a file: '%s'", version, lockFile));
        }
        if (Files.exists(successFile, new LinkOption[0]) && Files.exists(cassandraDirectory, new LinkOption[0])) {
            Path path = cassandraDirectory;
            return path;
        }
        List<CassandraPackage> cassandraPackages = this.getCassandraPackages(version);
        if (cassandraPackages.isEmpty()) {
            throw new FileNotFoundException(String.format("Unable to provide Cassandra Directory for a version: '%s'. No Packages!", version));
        }
        ArrayList<Exception> failures = new ArrayList<Exception>();
        for (CassandraPackage cassandraPackage : cassandraPackages) {
            try {
                this.downloadAndExtract(version, downloadDirectory, cassandraDirectory, cassandraPackage);
                if (!Thread.currentThread().isInterrupted()) {
                    Files.write(successFile, Collections.singleton(ZonedDateTime.now().toString()), new OpenOption[0]);
                }
                LOGGER.info("Cassandra directory: ''{0}'' is initialized.", cassandraDirectory);
                Path path = cassandraDirectory;
                return path;
            }
            catch (Exception ex) {
                failures.add(ex);
            }
        }
        StringBuilder builder = new StringBuilder("Unable to provide Cassandra Directory for a version: '").append(version).append("'").append(System.lineSeparator());
        for (Exception failure : failures) {
            StringWriter writer = new StringWriter();
            failure.printStackTrace(new PrintWriter(writer));
            builder.append(writer).append(System.lineSeparator());
        }
        throw new IOException(builder.substring(0, builder.length() - System.lineSeparator().length()));
        finally {
            if (fileLock != null) {
                fileLock.close();
            }
        }
    }

    protected List<CassandraPackage> getCassandraPackages(Version version) {
        ArrayList<CassandraPackage> packages = new ArrayList<CassandraPackage>();
        packages.add(WebCassandraDirectoryProvider.createPackage(String.format("apache-cassandra-%1$s-bin.tar.gz", version), String.format("https://downloads.apache.org/cassandra/%1$s/apache-cassandra-%1$s-bin.tar.gz", version)));
        packages.add(WebCassandraDirectoryProvider.createPackage(String.format("apache-cassandra-%1$s-bin.tar.gz", version), String.format("https://archive.apache.org/dist/cassandra/%1$s/apache-cassandra-%1$s-bin.tar.gz", version)));
        return packages;
    }

    protected boolean tryLock(FileLock fileLock) throws IOException {
        return fileLock.tryLock(5L, TimeUnit.MINUTES);
    }

    protected void download(HttpClient httpClient, Version version, URI uri, OutputStream os) throws IOException {
        block14: {
            try (HttpResponse response = httpClient.send(new HttpRequest(uri));){
                if (response.getStatus() == 200) {
                    LOGGER.info("Downloading Apache Cassandra: ''{0}'' from URI: ''{1}''. It takes a while...", version, response.getUri());
                    long totalBytes = response.getHeaders().getFirst("Content-Length").map(Long::parseLong).orElse(-1L);
                    long readBytes = 0L;
                    int lastPercent = 0;
                    byte[] buffer = new byte[8192];
                    try (InputStream is = response.getInputStream();){
                        int read;
                        while ((read = is.read(buffer)) != -1) {
                            int percent;
                            os.write(buffer, 0, read);
                            if (totalBytes <= 0L || (percent = (int)((readBytes += (long)read) * 100L / totalBytes)) - lastPercent < 10 && percent != 100) continue;
                            LOGGER.info("{0} / {1} {2}%", readBytes, totalBytes, percent);
                            lastPercent = percent;
                        }
                        break block14;
                    }
                }
                throw new FileNotFoundException(String.format("Could not download a file. Error: %s", response));
            }
        }
    }

    protected void extract(Path archiveFile, Path destination) throws IOException {
        try (ArchiveInputStream archiveInputStream = this.createArchiveInputStream(archiveFile);){
            ArchiveEntry entry;
            while ((entry = archiveInputStream.getNextEntry()) != null) {
                if (entry.isDirectory()) {
                    Files.createDirectories(destination.resolve(entry.getName()).normalize(), new FileAttribute[0]).toAbsolutePath();
                    continue;
                }
                Path file = destination.resolve(entry.getName()).normalize().toAbsolutePath();
                Path parent = file.getParent();
                if (!Files.exists(parent, new LinkOption[0])) {
                    Files.createDirectories(parent, new FileAttribute[0]);
                }
                Files.copy((InputStream)archiveInputStream, file, StandardCopyOption.REPLACE_EXISTING);
            }
        }
    }

    protected ArchiveInputStream createArchiveInputStream(Path archiveFile) throws IOException {
        return new TarArchiveInputStream((InputStream)new GzipCompressorInputStream(Files.newInputStream(archiveFile, new OpenOption[0])));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void downloadAndExtract(Version version, Path downloadDirectory, Path cassandraDirectory, CassandraPackage cassandraPackage) throws IOException, NoSuchAlgorithmException {
        Path downloadFile = Files.createTempFile(downloadDirectory, "", "-" + cassandraPackage.getName(), new FileAttribute[0]);
        try (OutputStream outputStream = Files.newOutputStream(downloadFile, StandardOpenOption.WRITE);){
            this.download(this.httpClient, version, cassandraPackage.getUri(), outputStream);
            this.verifyChecksums(this.httpClient, downloadFile, cassandraPackage);
            Path extractDirectory = Files.createTempDirectory(downloadDirectory, String.format("apache-cassandra-%s-", version), new FileAttribute[0]);
            try {
                LOGGER.info("Extracting...");
                this.extract(downloadFile, extractDirectory);
                Path cassandraHome = this.findCassandraHome(extractDirectory);
                FileUtils.copy(cassandraHome, cassandraDirectory, StandardCopyOption.REPLACE_EXISTING);
            }
            finally {
                WebCassandraDirectoryProvider.deleteSilently(extractDirectory);
            }
        }
        finally {
            WebCassandraDirectoryProvider.deleteSilently(downloadFile);
        }
    }

    private void verifyChecksums(HttpClient httpClient, Path archiveFile, CassandraPackage cassandraPackage) throws IOException, NoSuchAlgorithmException {
        LOGGER.info("Verifying checksum...");
        Map<String, URI> checksums = cassandraPackage.getChecksums();
        if (checksums.isEmpty()) {
            LOGGER.warn("No checksum defined for ''{0}'', skipping verification.", cassandraPackage.getName());
            return;
        }
        for (Map.Entry<String, URI> checksum : checksums.entrySet()) {
            String algo = checksum.getKey();
            URI uri = checksum.getValue();
            HttpResponse response = httpClient.send(new HttpRequest(uri));
            try {
                String expected;
                if (response.getStatus() != 200) continue;
                try (InputStream stream = response.getInputStream();){
                    expected = StreamUtils.toString(stream, Charset.defaultCharset()).trim();
                }
                String[] tokens = expected.split("\\s+");
                String actual = FileUtils.checksum(archiveFile, algo);
                if (tokens.length == 2) {
                    this.verify(actual + " " + cassandraPackage.getName(), tokens[0] + " " + tokens[1]);
                } else {
                    this.verify(actual, tokens[0]);
                }
                LOGGER.info("Checksums are identical");
                return;
            }
            finally {
                if (response == null) continue;
                response.close();
            }
        }
        LOGGER.warn("No checksum downloaded for ''{0}'', skipping verification.", cassandraPackage.getName());
    }

    private void verify(String actual, String expected) {
        if (!actual.equals(expected)) {
            throw new IllegalStateException(String.format("Checksum mismatch. Actual: '%s' Expected: '%s'", actual, expected));
        }
    }

    private Path findCassandraHome(Path directory) throws IOException {
        try (Stream<Path> stream = Files.find(directory, 5, this::isCassandraHome, new FileVisitOption[0]);){
            Path path = stream.findFirst().orElseThrow(() -> new IllegalStateException("Could not find Apache Cassandra directory in directory: '" + directory + "'"));
            return path;
        }
    }

    private boolean isCassandraHome(Path path, BasicFileAttributes attributes) {
        if (attributes.isDirectory()) {
            return Files.isDirectory(path.resolve("bin"), new LinkOption[0]) && Files.isDirectory(path.resolve("lib"), new LinkOption[0]) && Files.isDirectory(path.resolve("conf"), new LinkOption[0]);
        }
        return false;
    }

    private static CassandraPackage createPackage(String name, String uri) {
        LinkedHashMap<String, URI> checksums = new LinkedHashMap<String, URI>();
        for (String algo : ALGORITHMS) {
            checksums.put(algo, URI.create(String.format("%s.%s", uri, algo.toLowerCase(Locale.ENGLISH).replace("-", ""))));
        }
        return new CassandraPackage(name, URI.create(uri), checksums);
    }

    private static void deleteSilently(Path path) {
        try {
            FileUtils.delete(path);
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    protected static final class CassandraPackage {
        private final String name;
        private final URI uri;
        private final Map<String, URI> checksums;

        public CassandraPackage(String name, URI uri, Map<String, URI> checksums) {
            Objects.requireNonNull(name, "Name must not be null");
            Objects.requireNonNull(uri, "URI must not be null");
            Objects.requireNonNull(checksums, "Checksums must not be null");
            if (!StringUtils.hasText(name)) {
                throw new IllegalArgumentException("Name must not be empty");
            }
            this.name = name;
            this.uri = uri;
            this.checksums = Collections.unmodifiableMap(checksums);
        }

        public URI getUri() {
            return this.uri;
        }

        public String getName() {
            return this.name;
        }

        public Map<String, URI> getChecksums() {
            return this.checksums;
        }
    }
}

