/*
 * Decompiled with CFR 0.152.
 */
package io.omam.halo;

import io.omam.halo.DnsMessage;
import io.omam.halo.HaloProperties;
import io.omam.halo.HaloThreadFactory;
import java.io.IOException;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.NetworkInterface;
import java.net.ProtocolFamily;
import java.net.SocketOption;
import java.net.StandardProtocolFamily;
import java.net.StandardSocketOptions;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.DatagramChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.time.Clock;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;

final class HaloChannel
implements AutoCloseable {
    private static final Logger LOGGER = Logger.getLogger(HaloChannel.class.getName());
    private final Clock clock;
    private final ExecutorService executor;
    private final List<SelectionKey> ipv4;
    private final List<SelectionKey> ipv6;
    private final Consumer<DnsMessage> listener;
    private Future<?> receiver;
    private final Selector selector;
    private Future<?> sender;
    private final BlockingQueue<DnsMessage> sent;

    private HaloChannel(Consumer<DnsMessage> aListener, Clock aClock, Collection<NetworkInterface> nis) throws IOException {
        this.clock = aClock;
        this.executor = Executors.newFixedThreadPool(2, new HaloThreadFactory("channel"));
        this.listener = aListener;
        this.selector = Selector.open();
        this.sent = new LinkedBlockingQueue<DnsMessage>();
        this.ipv4 = new ArrayList<SelectionKey>();
        this.ipv6 = new ArrayList<SelectionKey>();
        for (NetworkInterface ni : nis) {
            this.openChannel(ni, StandardProtocolFamily.INET, false).map(this::register).ifPresent(this.ipv4::add);
            this.openChannel(ni, StandardProtocolFamily.INET6, false).map(this::register).ifPresent(this.ipv6::add);
        }
        if (this.ipv4.isEmpty() && this.ipv6.isEmpty()) {
            for (NetworkInterface ni : nis) {
                LOGGER.info(() -> "No Network Interface found, adding Loopback interface");
                this.openChannel(ni, StandardProtocolFamily.INET, true).map(this::register).ifPresent(this.ipv4::add);
                this.openChannel(ni, StandardProtocolFamily.INET6, true).map(this::register).ifPresent(this.ipv6::add);
            }
        }
        if (this.ipv4.isEmpty() && this.ipv6.isEmpty()) {
            throw new IOException("No network interface suitable for multicast");
        }
    }

    static HaloChannel allNetworkInterfaces(Consumer<DnsMessage> listener, Clock clock) throws IOException {
        Enumeration<NetworkInterface> nics = NetworkInterface.getNetworkInterfaces();
        ArrayList<NetworkInterface> allNics = new ArrayList<NetworkInterface>();
        while (nics.hasMoreElements()) {
            allNics.add(nics.nextElement());
        }
        return HaloChannel.networkInterfaces(listener, clock, allNics);
    }

    static HaloChannel networkInterfaces(Consumer<DnsMessage> listener, Clock clock, Collection<NetworkInterface> nics) throws IOException {
        return new HaloChannel(listener, clock, nics);
    }

    @Override
    public final synchronized void close() {
        LOGGER.fine("Closing channel");
        this.selector.wakeup();
        this.disable();
        this.executor.shutdownNow();
        this.close(this.ipv4);
        this.close(this.ipv6);
    }

    final synchronized void enable() {
        if (this.sender == null) {
            this.sender = this.executor.submit(new Sender());
        }
        if (this.receiver == null) {
            this.receiver = this.executor.submit(new Receiver());
        }
    }

    final void send(DnsMessage message) {
        this.sent.add(message);
    }

    private void close(List<SelectionKey> keys) {
        for (SelectionKey key : keys) {
            try {
                key.channel().close();
            }
            catch (IOException e) {
                LOGGER.log(Level.WARNING, "I/O error when closing channel", e);
            }
        }
    }

    private synchronized void disable() {
        if (this.sender != null) {
            this.sender.cancel(true);
        }
        if (this.receiver != null) {
            this.receiver.cancel(true);
        }
    }

    private boolean hasIpv(NetworkInterface iface, Class<? extends InetAddress> inetClass) {
        Enumeration<InetAddress> e = iface.getInetAddresses();
        while (e.hasMoreElements()) {
            if (!e.nextElement().getClass().isAssignableFrom(inetClass)) continue;
            return true;
        }
        return false;
    }

    private Optional<DatagramChannel> openChannel(NetworkInterface iface, ProtocolFamily family, boolean loopback) {
        boolean ipv4Protocol = family == StandardProtocolFamily.INET;
        InetAddress addr = ipv4Protocol ? HaloProperties.IPV4_ADDR : HaloProperties.IPV6_ADDR;
        try {
            Optional<DatagramChannel> channel;
            Class ipvClass;
            Class clazz = ipvClass = ipv4Protocol ? Inet4Address.class : Inet6Address.class;
            if (iface.supportsMulticast() && iface.isUp() && iface.isLoopback() == loopback && this.hasIpv(iface, ipvClass) && (channel = this.openChannel(family)).isPresent()) {
                channel.get().setOption((SocketOption)StandardSocketOptions.IP_MULTICAST_IF, iface);
                channel.get().join(addr, iface);
                LOGGER.info(() -> "Joined multicast address " + addr + " on " + iface);
                return channel;
            }
            LOGGER.fine(() -> "Ignored " + iface + " for " + addr);
            return Optional.empty();
        }
        catch (IOException e) {
            LOGGER.log(Level.WARNING, e, () -> "Ignored " + iface + " for " + addr);
            return Optional.empty();
        }
    }

    private Optional<DatagramChannel> openChannel(ProtocolFamily family) {
        try {
            DatagramChannel channel = DatagramChannel.open(family);
            channel.configureBlocking(false);
            channel.setOption((SocketOption)StandardSocketOptions.SO_REUSEADDR, (Object)true);
            channel.setOption((SocketOption)StandardSocketOptions.IP_MULTICAST_TTL, (Object)255);
            channel.bind(new InetSocketAddress(HaloProperties.MDNS_PORT));
            return Optional.of(channel);
        }
        catch (UnsupportedOperationException e) {
            LOGGER.log(Level.FINE, e, () -> "Protocol Family [" + family.name() + "] not supported on this machine.");
            return Optional.empty();
        }
        catch (IOException e) {
            LOGGER.log(Level.WARNING, "Fail to create channel", e);
            return Optional.empty();
        }
    }

    private SelectionKey register(DatagramChannel channel) {
        try {
            return channel.register(this.selector, 1);
        }
        catch (ClosedChannelException e) {
            LOGGER.severe(() -> "Could not register channel with selector");
            throw new IllegalStateException(e);
        }
    }

    private final class Sender
    implements Runnable {
        Sender() {
        }

        @Override
        public final void run() {
            ByteBuffer buf = ByteBuffer.allocate(65536);
            buf.order(ByteOrder.BIG_ENDIAN);
            while (!Thread.currentThread().isInterrupted()) {
                try {
                    DnsMessage msg = (DnsMessage)HaloChannel.this.sent.take();
                    LOGGER.fine(() -> "Sending " + msg);
                    byte[] packet = msg.encode();
                    buf.clear();
                    buf.put(packet);
                    buf.flip();
                    HaloChannel.this.ipv4.forEach(ni -> this.send((SelectionKey)ni, buf, HaloProperties.IPV4_SOA));
                    HaloChannel.this.ipv6.forEach(ni -> this.send((SelectionKey)ni, buf, HaloProperties.IPV6_SOA));
                }
                catch (InterruptedException e) {
                    LOGGER.log(Level.FINE, "Interrupted while waiting to send DNS message", e);
                    Thread.currentThread().interrupt();
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void send(SelectionKey key, ByteBuffer src, InetSocketAddress target) {
            int position = src.position();
            try {
                ((DatagramChannel)key.channel()).send(src, target);
                LOGGER.fine(() -> "Sent DNS message to " + target);
            }
            catch (IOException e) {
                LOGGER.log(Level.WARNING, e, () -> "I/O error while sending DNS message to " + target);
            }
            finally {
                src.position(position);
            }
        }
    }

    private final class Receiver
    implements Runnable {
        Receiver() {
        }

        @Override
        public final void run() {
            ByteBuffer buf = ByteBuffer.allocate(65536);
            buf.order(ByteOrder.BIG_ENDIAN);
            while (!Thread.currentThread().isInterrupted()) {
                try {
                    HaloChannel.this.selector.select();
                    LOGGER.fine("Channels ready for I/O operations");
                    Set<SelectionKey> selected = HaloChannel.this.selector.selectedKeys();
                    for (SelectionKey key : selected) {
                        DatagramChannel channel = (DatagramChannel)key.channel();
                        buf.clear();
                        InetSocketAddress address = (InetSocketAddress)channel.receive(buf);
                        if (address == null || buf.position() == 0) continue;
                        buf.flip();
                        byte[] bytes = new byte[buf.remaining()];
                        buf.get(bytes);
                        DnsMessage msg = DnsMessage.decode(bytes, HaloChannel.this.clock.instant());
                        LOGGER.fine(() -> "Received " + msg + " on " + address);
                        HaloChannel.this.listener.accept(msg);
                    }
                }
                catch (ClosedChannelException e) {
                    LOGGER.log(Level.FINE, "Channel closed while waiting to receive DNS message", e);
                    Thread.currentThread().interrupt();
                }
                catch (IOException e) {
                    LOGGER.log(Level.WARNING, "I/O error while receiving DNS message", e);
                }
            }
        }
    }
}

