/*
 * Decompiled with CFR 0.152.
 */
package io.omam.halo;

import io.omam.halo.AddressRecord;
import io.omam.halo.Announcer;
import io.omam.halo.Browser;
import io.omam.halo.Cache;
import io.omam.halo.Canceller;
import io.omam.halo.DnsMessage;
import io.omam.halo.DnsQuestion;
import io.omam.halo.DnsRecord;
import io.omam.halo.Halo;
import io.omam.halo.HaloChannel;
import io.omam.halo.HaloHelper;
import io.omam.halo.HaloProperties;
import io.omam.halo.HaloRegistrationTypeBrowser;
import io.omam.halo.HaloServiceBrowser;
import io.omam.halo.MulticastDnsSd;
import io.omam.halo.PtrRecord;
import io.omam.halo.Reaper;
import io.omam.halo.RegisterableService;
import io.omam.halo.RegisterableServiceImpl;
import io.omam.halo.RegisteredService;
import io.omam.halo.RegisteredServiceImpl;
import io.omam.halo.RegistrationTypeBrowserListener;
import io.omam.halo.ResolvableService;
import io.omam.halo.ResolvedService;
import io.omam.halo.ResponseListener;
import io.omam.halo.SequentialBatchExecutor;
import io.omam.halo.Service;
import io.omam.halo.ServiceBrowserListener;
import io.omam.halo.SrvRecord;
import io.omam.halo.TxtRecord;
import java.io.IOException;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

final class HaloImpl
extends HaloHelper
implements Halo,
Consumer<DnsMessage> {
    private static final Logger LOGGER = Logger.getLogger(HaloImpl.class.getName());
    private static final String ON_DOMAIN = " on the local domain";
    private static final Pattern INSTANCE_NAME_PATTERN = Pattern.compile("([\\s\\S]*?)( \\((?<i>\\d+)\\))$");
    private final Announcer announcer;
    private final Cache cache;
    private final Canceller canceller;
    private final HaloChannel channel;
    private final Clock clock;
    private final Reaper reaper;
    private final Collection<ResponseListener> rls;
    private final HaloRegistrationTypeBrowser rBrowser;
    private final HaloServiceBrowser sBrowser;
    private final Map<String, RegisterableService> announcing;
    private final Map<String, RegisteredService> registered;
    private final Set<String> registrationPointerNames;

    HaloImpl(Clock aClock, Collection<NetworkInterface> nics) throws IOException {
        SequentialBatchExecutor executor = new SequentialBatchExecutor("registration");
        this.announcer = new Announcer(this, executor);
        this.cache = new Cache();
        this.canceller = new Canceller(this, executor);
        this.channel = nics.isEmpty() ? HaloChannel.allNetworkInterfaces(this, aClock) : HaloChannel.networkInterfaces(this, aClock, nics);
        this.clock = aClock;
        this.reaper = new Reaper(this.cache, this.clock);
        this.rls = new ConcurrentLinkedQueue<ResponseListener>();
        this.rBrowser = new HaloRegistrationTypeBrowser(this);
        this.sBrowser = new HaloServiceBrowser(this);
        this.announcing = new ConcurrentHashMap<String, RegisterableService>();
        this.registered = new ConcurrentHashMap<String, RegisteredService>();
        this.registrationPointerNames = ConcurrentHashMap.newKeySet();
        this.channel.enable();
        this.reaper.start();
    }

    private static String changeInstanceName(String instanceName) {
        String result;
        Matcher matcher = INSTANCE_NAME_PATTERN.matcher(instanceName);
        if (matcher.matches()) {
            int next = Integer.parseInt(matcher.group("i")) + 1;
            int start = matcher.start("i");
            int end = matcher.end("i");
            result = instanceName.substring(0, start) + next + instanceName.substring(end);
        } else {
            result = instanceName + " (2)";
        }
        LOGGER.info(() -> "Change service instance name from: [" + instanceName + "] to [" + result + "]");
        return result;
    }

    @Override
    public final void accept(DnsMessage message) {
        if (message.isQuery()) {
            this.handleQuery(message);
        } else if (message.isResponse()) {
            this.handleResponse(message);
        } else {
            LOGGER.warning("Ignored received DNS message.");
        }
    }

    @Override
    public final Browser browse(RegistrationTypeBrowserListener listener) {
        this.rBrowser.addListener(listener);
        this.rBrowser.start();
        return () -> this.rBrowser.removeListener(listener);
    }

    @Override
    public final Browser browse(String registrationType, ServiceBrowserListener listener) {
        this.sBrowser.addListener(registrationType, listener);
        this.sBrowser.start();
        return () -> this.sBrowser.removeListener(registrationType, listener);
    }

    @Override
    public final void close() {
        this.sBrowser.close();
        this.rBrowser.close();
        this.reaper.close();
        try {
            this.deregisterAll();
        }
        catch (IOException e) {
            LOGGER.log(Level.WARNING, "I/O error when de-registering all services", e);
        }
        finally {
            this.announcer.close();
            this.canceller.close();
            this.channel.close();
            this.cache.clear();
            this.rls.clear();
        }
    }

    @Override
    public final void deregister(RegisteredService service) throws IOException {
        String serviceKey = MulticastDnsSd.toLowerCase(service.name());
        if (this.registered.containsKey(serviceKey)) {
            this.canceller.cancel(service);
            this.registrationPointerNames.remove(service.registrationPointerName());
            this.registered.remove(serviceKey);
            this.cache.removeAll(service.name());
        } else {
            LOGGER.info(() -> service + " is not registered");
        }
    }

    @Override
    public final void deregisterAll() throws IOException {
        ArrayList<String> messages = new ArrayList<String>();
        for (RegisteredService service : this.registered.values()) {
            try {
                this.deregister(service);
            }
            catch (IOException e) {
                messages.add(e.getMessage());
            }
        }
        if (!messages.isEmpty()) {
            throw new IOException(messages.stream().collect(Collectors.joining(",")));
        }
    }

    @Override
    public final RegisteredService register(RegisterableService registerable, Duration ttl, boolean allowNameChange) throws IOException {
        LOGGER.fine(() -> "Registering " + registerable + ON_DOMAIN);
        RegisterableService service = this.makeUnique(registerable, allowNameChange);
        String serviceKey = MulticastDnsSd.toLowerCase(service.name());
        this.announcing.put(serviceKey, service);
        String rpn = service.registrationPointerName();
        this.registrationPointerNames.add(rpn);
        boolean announced = this.announcer.announce(service, ttl);
        this.announcing.remove(serviceKey);
        if (!announced) {
            this.registrationPointerNames.remove(service.registrationPointerName());
            String msg = "Found conflicts while announcing " + service + " on network";
            LOGGER.warning(msg);
            throw new IOException(msg);
        }
        LOGGER.info(() -> "Registered " + service + ON_DOMAIN);
        RegisteredServiceImpl rservice = new RegisteredServiceImpl(service, this);
        this.registered.put(serviceKey, rservice);
        return rservice;
    }

    @Override
    public final void resetBrowsingInterval() {
        this.rBrowser.resetQueryInterval();
        this.sBrowser.resetQueryInterval();
    }

    @Override
    public final Optional<ResolvedService> resolve(String instanceName, String registrationType, Duration timeout) {
        this.cache.clean(this.now());
        ResolvableService service = new ResolvableService(instanceName, registrationType);
        LOGGER.fine(() -> "Resolving " + service.toString() + ON_DOMAIN);
        try {
            boolean resolved = service.resolve(this, timeout);
            if (resolved) {
                LOGGER.info(() -> "Resolved " + service.toString() + ON_DOMAIN);
                return Optional.of(service);
            }
        }
        catch (InterruptedException e) {
            LOGGER.log(Level.FINE, "Interrupted while waiting for response", e);
            Thread.currentThread().interrupt();
        }
        LOGGER.info(() -> "Could not resolve " + service.toString() + ON_DOMAIN);
        return Optional.empty();
    }

    @Override
    final void addResponseListener(ResponseListener listener) {
        Objects.requireNonNull(listener);
        LOGGER.fine(() -> "Adding response listener " + listener);
        this.rls.add(listener);
    }

    @Override
    final Optional<DnsRecord> cachedRecord(String name, short type, short clazz) {
        return this.cache.get(name, type, clazz);
    }

    @Override
    final Instant now() {
        return this.clock.instant();
    }

    @Override
    final void reannounce(RegisteredService service, Duration ttl) throws IOException {
        this.announcer.reannounce(service, ttl);
    }

    @Override
    final void removeResponseListener(ResponseListener listener) {
        Objects.requireNonNull(listener);
        LOGGER.fine(() -> "Removing response listener " + listener);
        this.rls.remove(listener);
    }

    @Override
    final void sendMessage(DnsMessage msg) {
        this.channel.send(msg);
    }

    private void addIpv4Address(DnsMessage query, DnsQuestion question, DnsMessage.Builder builder, Instant now) {
        this.announcingOrRegistered().filter(s -> s.hostname().equalsIgnoreCase(question.name())).filter(h -> h.ipv4Address().isPresent()).forEach(s -> {
            InetAddress addr = s.ipv4Address().get();
            builder.addAnswer(query, new AddressRecord(question.name(), MulticastDnsSd.uniqueClass((short)1), HaloProperties.TTL, now, addr));
        });
    }

    private void addIpv6Address(DnsMessage query, DnsQuestion question, DnsMessage.Builder builder, Instant now) {
        this.announcingOrRegistered().filter(s -> s.hostname().equalsIgnoreCase(question.name())).filter(h -> h.ipv6Address().isPresent()).forEach(s -> {
            InetAddress addr = s.ipv6Address().get();
            builder.addAnswer(query, new AddressRecord(question.name(), MulticastDnsSd.uniqueClass((short)1), HaloProperties.TTL, now, addr));
        });
    }

    private void addPtrAnswer(DnsMessage query, DnsQuestion question, DnsMessage.Builder builder, Instant now) {
        if (question.name().equals("_services._dns-sd._udp.local.")) {
            for (String rpn : this.registrationPointerNames) {
                builder.addAnswer(query, new PtrRecord("_services._dns-sd._udp.local.", 1, HaloProperties.TTL, now, rpn));
            }
        } else {
            this.announcingOrRegistered().forEach(service -> {
                if (question.name().equalsIgnoreCase(service.registrationPointerName())) {
                    builder.addAnswer(query, new PtrRecord(service.registrationPointerName(), 1, HaloProperties.TTL, now, service.name()));
                }
            });
        }
    }

    private void addServiceAnswer(DnsMessage query, DnsQuestion question, Service service, DnsMessage.Builder builder, Instant now) {
        short unique = MulticastDnsSd.uniqueClass((short)1);
        String hostname = service.hostname();
        if (question.type() == 33 || question.type() == 255) {
            builder.addAnswer(query, new SrvRecord(question.name(), unique, HaloProperties.TTL, now, service.port(), hostname));
        }
        if (question.type() == 16 || question.type() == 255) {
            builder.addAnswer(query, new TxtRecord(question.name(), unique, HaloProperties.TTL, now, service.attributes()));
        }
        if (question.type() == 33) {
            service.ipv4Address().ifPresent(a -> builder.addAnswer(query, new AddressRecord(hostname, unique, HaloProperties.TTL, now, (InetAddress)a)));
            service.ipv6Address().ifPresent(a -> builder.addAnswer(query, new AddressRecord(hostname, unique, HaloProperties.TTL, now, (InetAddress)a)));
        }
    }

    private Stream<Service> announcingOrRegistered() {
        return Stream.concat(this.announcing.values().stream(), this.registered.values().stream());
    }

    private Service announcingOrRegistered(String name) {
        String lcName = MulticastDnsSd.toLowerCase(name);
        RegisterableService aService = this.announcing.get(lcName);
        if (aService != null) {
            return aService;
        }
        RegisteredService rService = this.registered.get(lcName);
        if (rService != null) {
            return rService;
        }
        LOGGER.fine(() -> "No announcing or registered service matching: " + name);
        return null;
    }

    private DnsMessage buildResponse(DnsMessage query) {
        DnsMessage.Builder builder = DnsMessage.response(1024);
        Instant now = this.now();
        for (DnsQuestion question : query.questions()) {
            Service service;
            if (question.type() == 12) {
                this.addPtrAnswer(query, question, builder, now);
                continue;
            }
            if (question.type() == 1 || question.type() == 255) {
                this.addIpv4Address(query, question, builder, now);
            }
            if (question.type() == 28 || question.type() == 255) {
                this.addIpv6Address(query, question, builder, now);
            }
            if ((service = this.announcingOrRegistered(question.name())) == null) continue;
            this.addServiceAnswer(query, question, service, builder, now);
        }
        return builder.get();
    }

    private void handleQuery(DnsMessage query) {
        LOGGER.fine(() -> "Trying to respond to " + query);
        DnsMessage response = this.buildResponse(query);
        if (response.answers().isEmpty()) {
            LOGGER.fine(() -> "Ignoring query");
        } else {
            LOGGER.fine(() -> "Responding with " + response);
            this.channel.send(response);
        }
    }

    private void handleResponse(DnsMessage response) {
        LOGGER.fine(() -> "Handling response " + response);
        for (DnsRecord record : response.answers()) {
            if (record.ttl().isZero()) {
                this.cache.expire(record);
                continue;
            }
            this.cache.add(record);
        }
        if (this.rls.isEmpty()) {
            LOGGER.fine(() -> "No listener registered for " + response);
        } else {
            this.rls.forEach(l -> l.responseReceived(response, this));
        }
    }

    private RegisterableService makeUnique(RegisterableService service, boolean allowNameChange) throws IOException {
        String hostname = service.hostname();
        short port = service.port();
        boolean collision = false;
        RegisterableService result = service;
        do {
            Optional<SrvRecord> record;
            String msg;
            collision = false;
            Instant now = this.now();
            Service own = this.announcingOrRegistered(result.name());
            if (own != null) {
                String otherHostname = own.hostname();
                boolean bl = collision = own.port() != port || !otherHostname.equals(hostname);
                if (collision) {
                    msg = "Own registered service collision: " + own;
                    result = this.tryResolveCollision(result, allowNameChange, msg);
                }
            }
            if (collision || !(record = this.cache.entries(result.name()).stream().filter(e -> e instanceof SrvRecord).filter(e -> !e.isExpired(now)).map(e -> (SrvRecord)e).filter(e -> e.port() != port || !e.server().equals(hostname)).findFirst()).isPresent()) continue;
            collision = true;
            msg = "Cache collision: " + record.get();
            result = this.tryResolveCollision(result, allowNameChange, msg);
        } while (collision);
        return result;
    }

    private RegisterableService tryResolveCollision(RegisterableService service, boolean allowNameChange, String msg) throws IOException {
        if (!allowNameChange) {
            throw new IOException(msg);
        }
        LOGGER.info(msg);
        String instanceName = HaloImpl.changeInstanceName(service.instanceName());
        return new RegisterableServiceImpl(instanceName, service);
    }
}

