/*
 * Decompiled with CFR 0.152.
 */
package org.iban4j;

import org.iban4j.CountryCode;
import org.iban4j.Iban;
import org.iban4j.Iban4jException;
import org.iban4j.IbanFormat;
import org.iban4j.IbanFormatException;
import org.iban4j.InvalidCheckDigitException;
import org.iban4j.UnsupportedCountryException;
import org.iban4j.bban.BbanEntryType;
import org.iban4j.bban.BbanStructure;
import org.iban4j.bban.BbanStructureEntry;

public final class IbanUtil {
    private static final int MOD = 97;
    private static final long MAX = 999999999L;
    private static final int COUNTRY_CODE_INDEX = 0;
    private static final int COUNTRY_CODE_LENGTH = 2;
    private static final int CHECK_DIGIT_INDEX = 2;
    private static final int CHECK_DIGIT_LENGTH = 2;
    private static final int BBAN_INDEX = 4;
    private static final String ASSERT_UPPER_LETTERS = "[%s] must contain only upper case letters.";
    private static final String ASSERT_DIGITS_AND_LETTERS = "[%s] must contain only digits or letters.";
    private static final String ASSERT_DIGITS = "[%s] must contain only digits.";

    private IbanUtil() {
    }

    public static String calculateCheckDigit(String iban) throws IbanFormatException {
        String reformattedIban = IbanUtil.replaceCheckDigit(iban, "00");
        int modResult = IbanUtil.calculateMod(reformattedIban);
        int checkDigitIntValue = 98 - modResult;
        String checkDigit = Integer.toString(checkDigitIntValue);
        return checkDigitIntValue > 9 ? checkDigit : "0" + checkDigit;
    }

    public static void validate(String iban) throws IbanFormatException, InvalidCheckDigitException, UnsupportedCountryException {
        try {
            IbanUtil.validateEmpty(iban);
            IbanUtil.validateCountryCode(iban);
            IbanUtil.validateCheckDigitPresence(iban);
            BbanStructure structure = IbanUtil.getBbanStructure(iban);
            IbanUtil.validateBbanLength(iban, structure);
            IbanUtil.validateBbanEntries(iban, structure);
            IbanUtil.validateCheckDigit(iban);
        }
        catch (Iban4jException e) {
            throw e;
        }
        catch (RuntimeException e) {
            throw new IbanFormatException(IbanFormatException.IbanFormatViolation.UNKNOWN, e.getMessage());
        }
    }

    public static void validate(String iban, IbanFormat format) throws IbanFormatException, InvalidCheckDigitException, UnsupportedCountryException {
        switch (format) {
            case Default: {
                String ibanWithoutSpaces = iban.replace(" ", "");
                IbanUtil.validate(ibanWithoutSpaces);
                if (IbanUtil.toFormattedString(ibanWithoutSpaces).equals(iban)) break;
                throw new IbanFormatException(IbanFormatException.IbanFormatViolation.IBAN_FORMATTING, String.format("Iban must be formatted using 4 characters and space combination. Instead of [%s]", iban));
            }
            default: {
                IbanUtil.validate(iban);
            }
        }
    }

    public static boolean isSupportedCountry(CountryCode countryCode) {
        return BbanStructure.forCountry(countryCode) != null;
    }

    public static int getIbanLength(CountryCode countryCode) {
        BbanStructure structure = IbanUtil.getBbanStructure(countryCode);
        return 4 + structure.getBbanLength();
    }

    public static String getCheckDigit(String iban) {
        return iban.substring(2, 4);
    }

    public static String getCountryCode(String iban) {
        return iban.substring(0, 2);
    }

    public static String getCountryCodeAndCheckDigit(String iban) {
        return iban.substring(0, 4);
    }

    public static String getBban(String iban) {
        return iban.substring(4);
    }

    public static String getAccountNumber(String iban) {
        return IbanUtil.extractBbanEntry(iban, BbanEntryType.account_number);
    }

    public static String getBankCode(String iban) {
        return IbanUtil.extractBbanEntry(iban, BbanEntryType.bank_code);
    }

    static String getBranchCode(String iban) {
        return IbanUtil.extractBbanEntry(iban, BbanEntryType.branch_code);
    }

    static String getNationalCheckDigit(String iban) {
        return IbanUtil.extractBbanEntry(iban, BbanEntryType.national_check_digit);
    }

    static String getAccountType(String iban) {
        return IbanUtil.extractBbanEntry(iban, BbanEntryType.account_type);
    }

    static String getOwnerAccountType(String iban) {
        return IbanUtil.extractBbanEntry(iban, BbanEntryType.owner_account_number);
    }

    static String getIdentificationNumber(String iban) {
        return IbanUtil.extractBbanEntry(iban, BbanEntryType.identification_number);
    }

    static String calculateCheckDigit(Iban iban) {
        return IbanUtil.calculateCheckDigit(iban.toString());
    }

    static String replaceCheckDigit(String iban, String checkDigit) {
        return IbanUtil.getCountryCode(iban) + checkDigit + IbanUtil.getBban(iban);
    }

    static String toFormattedString(String iban) {
        StringBuilder ibanBuffer = new StringBuilder(iban);
        int length = ibanBuffer.length();
        for (int i = 0; i < length / 4; ++i) {
            ibanBuffer.insert((i + 1) * 4 + i, ' ');
        }
        return ibanBuffer.toString().trim();
    }

    private static void validateCheckDigit(String iban) {
        if (IbanUtil.calculateMod(iban) != 1) {
            String checkDigit = IbanUtil.getCheckDigit(iban);
            String expectedCheckDigit = IbanUtil.calculateCheckDigit(iban);
            throw new InvalidCheckDigitException(checkDigit, expectedCheckDigit, String.format("[%s] has invalid check digit: %s, expected check digit is: %s", iban, checkDigit, expectedCheckDigit));
        }
    }

    private static void validateEmpty(String iban) {
        if (iban == null) {
            throw new IbanFormatException(IbanFormatException.IbanFormatViolation.IBAN_NOT_NULL, "Null can't be a valid Iban.");
        }
        if (iban.length() == 0) {
            throw new IbanFormatException(IbanFormatException.IbanFormatViolation.IBAN_NOT_EMPTY, "Empty string can't be a valid Iban.");
        }
    }

    private static void validateCountryCode(String iban) {
        if (iban.length() < 2) {
            throw new IbanFormatException(IbanFormatException.IbanFormatViolation.COUNTRY_CODE_TWO_LETTERS, iban, "Iban must contain 2 char country code.");
        }
        String countryCode = IbanUtil.getCountryCode(iban);
        if (!(countryCode.equals(countryCode.toUpperCase()) && Character.isLetter(countryCode.charAt(0)) && Character.isLetter(countryCode.charAt(1)))) {
            throw new IbanFormatException(IbanFormatException.IbanFormatViolation.COUNTRY_CODE_UPPER_CASE_LETTERS, countryCode, "Iban country code must contain upper case letters.");
        }
        if (CountryCode.getByCode(countryCode) == null) {
            throw new IbanFormatException(IbanFormatException.IbanFormatViolation.COUNTRY_CODE_EXISTS, countryCode, "Iban contains non existing country code.");
        }
        BbanStructure structure = BbanStructure.forCountry(CountryCode.getByCode(countryCode));
        if (structure == null) {
            throw new UnsupportedCountryException(countryCode, "Country code is not supported.");
        }
    }

    private static void validateCheckDigitPresence(String iban) {
        if (iban.length() < 4) {
            throw new IbanFormatException(IbanFormatException.IbanFormatViolation.CHECK_DIGIT_TWO_DIGITS, iban.substring(2), "Iban must contain 2 digit check digit.");
        }
        String checkDigit = IbanUtil.getCheckDigit(iban);
        if (!Character.isDigit(checkDigit.charAt(0)) || !Character.isDigit(checkDigit.charAt(1))) {
            throw new IbanFormatException(IbanFormatException.IbanFormatViolation.CHECK_DIGIT_ONLY_DIGITS, checkDigit, "Iban's check digit should contain only digits.");
        }
    }

    private static void validateBbanLength(String iban, BbanStructure structure) {
        String bban;
        int bbanLength;
        int expectedBbanLength = structure.getBbanLength();
        if (expectedBbanLength != (bbanLength = (bban = IbanUtil.getBban(iban)).length())) {
            throw new IbanFormatException(IbanFormatException.IbanFormatViolation.BBAN_LENGTH, bbanLength, expectedBbanLength, String.format("[%s] length is %d, expected BBAN length is: %d", bban, bbanLength, expectedBbanLength));
        }
    }

    private static void validateBbanEntries(String iban, BbanStructure structure) {
        String bban = IbanUtil.getBban(iban);
        int bbanEntryOffset = 0;
        for (BbanStructureEntry entry : structure.getEntries()) {
            int entryLength = entry.getLength();
            String entryValue = bban.substring(bbanEntryOffset, bbanEntryOffset + entryLength);
            bbanEntryOffset += entryLength;
            IbanUtil.validateBbanEntryCharacterType(entry, entryValue);
        }
    }

    private static void validateBbanEntryCharacterType(BbanStructureEntry entry, String entryValue) {
        switch (entry.getCharacterType()) {
            case a: {
                for (char ch : entryValue.toCharArray()) {
                    if (Character.isUpperCase(ch)) continue;
                    throw new IbanFormatException(IbanFormatException.IbanFormatViolation.BBAN_ONLY_UPPER_CASE_LETTERS, entry.getEntryType(), entryValue, ch, String.format(ASSERT_UPPER_LETTERS, entryValue));
                }
                break;
            }
            case c: {
                for (char ch : entryValue.toCharArray()) {
                    if (Character.isLetterOrDigit(ch)) continue;
                    throw new IbanFormatException(IbanFormatException.IbanFormatViolation.BBAN_ONLY_DIGITS_OR_LETTERS, entry.getEntryType(), entryValue, ch, String.format(ASSERT_DIGITS_AND_LETTERS, entryValue));
                }
                break;
            }
            case n: {
                for (char ch : entryValue.toCharArray()) {
                    if (Character.isDigit(ch)) continue;
                    throw new IbanFormatException(IbanFormatException.IbanFormatViolation.BBAN_ONLY_DIGITS, entry.getEntryType(), entryValue, ch, String.format(ASSERT_DIGITS, entryValue));
                }
                break;
            }
        }
    }

    private static int calculateMod(String iban) {
        String reformattedIban = IbanUtil.getBban(iban) + IbanUtil.getCountryCodeAndCheckDigit(iban);
        long total = 0L;
        for (int i = 0; i < reformattedIban.length(); ++i) {
            int numericValue = Character.getNumericValue(reformattedIban.charAt(i));
            if (numericValue < 0 || numericValue > 35) {
                throw new IbanFormatException(IbanFormatException.IbanFormatViolation.IBAN_VALID_CHARACTERS, null, null, reformattedIban.charAt(i), String.format("Invalid Character[%d] = '%d'", i, numericValue));
            }
            total = (numericValue > 9 ? total * 100L : total * 10L) + (long)numericValue;
            if (total <= 999999999L) continue;
            total %= 97L;
        }
        return (int)(total % 97L);
    }

    private static BbanStructure getBbanStructure(String iban) {
        String countryCode = IbanUtil.getCountryCode(iban);
        return IbanUtil.getBbanStructure(CountryCode.getByCode(countryCode));
    }

    private static BbanStructure getBbanStructure(CountryCode countryCode) {
        return BbanStructure.forCountry(countryCode);
    }

    private static String extractBbanEntry(String iban, BbanEntryType entryType) {
        String bban = IbanUtil.getBban(iban);
        BbanStructure structure = IbanUtil.getBbanStructure(iban);
        int bbanEntryOffset = 0;
        for (BbanStructureEntry entry : structure.getEntries()) {
            int entryLength = entry.getLength();
            String entryValue = bban.substring(bbanEntryOffset, bbanEntryOffset + entryLength);
            bbanEntryOffset += entryLength;
            if (entry.getEntryType() != entryType) continue;
            return entryValue;
        }
        return null;
    }
}

