package better.files

import akka.actor._

/**
 * An actor that can watch a file or a directory
 * Instead of directly calling the constructor of this, call file.newWatcher to create the actor
 *
 * @param file watch this file (or directory)
 * @param maxDepth In case of directories, how much depth should we watch
 */
class FileWatcher(file: File, maxDepth: Int) extends Actor {
  import FileWatcher._
  def this(file: File, recursive: Boolean = true) = this(file, if (recursive) Int.MaxValue else 0)

  protected[this] val callbacks = newMultiMap[Event, Callback]

  private[this] val monitor: File.Monitor = new ThreadBackedFileMonitor(file, maxDepth) {
    override def onEvent(event: Event, file: File) = self ! Message.NewEvent(event, file)
    override def onException(exception: Throwable) = self ! Status.Failure(exception)
  }

  override def preStart() = monitor.start()

  override def receive = {
    case Message.NewEvent(event, target) if callbacks contains event => callbacks(event) foreach {f => f(event -> target)}
    case Message.RegisterCallback(events, callback) => events foreach {event => callbacks.addBinding(event, callback)}
    case Message.RemoveCallback(event, callback) => callbacks.removeBinding(event, callback)
  }

  override def postStop() = monitor.stop()
}

object FileWatcher {
  import java.nio.file.{Path, WatchEvent}
  type Event = WatchEvent.Kind[Path]
  type Callback = PartialFunction[(Event, File), Unit]

  sealed trait Message
  object Message {
    case class NewEvent(event: Event, file: File) extends Message
    case class RegisterCallback(events: Seq[Event], callback: Callback) extends Message
    case class RemoveCallback(event: Event, callback: Callback) extends Message
  }

  implicit class FileWatcherOps(file: File) {
    def watcherProps(recursive: Boolean): Props = Props(new FileWatcher(file, recursive))

    def newWatcher(recursive: Boolean = true)(implicit system: ActorSystem): ActorRef = system.actorOf(watcherProps(recursive))
  }

  def when(events: Event*)(callback: Callback): Message = Message.RegisterCallback(events.distinct, callback)

  def on(event: Event)(callback: File => Unit): Message = when(event){case (`event`, file) => callback(file)}

  def stop(event: Event, callback: Callback): Message = Message.RemoveCallback(event, callback)

  import scala.collection.mutable
  private[files] def newMultiMap[A, B]: mutable.MultiMap[A, B] = new mutable.HashMap[A, mutable.Set[B]] with mutable.MultiMap[A, B]
}
