package pureconfig.derivation

import scala.reflect.macros.whitebox

/**
  * Encapsulates the logic to parse Scala syntax trees genetrated by `Lazy`, providing a high-level way to navigate
  * through such a tree. The code on this class is heavily-dependent on the implementation details of `Lazy`.
  */
trait LazyContextParser {
  val c: whitebox.Context

  import c.universe._

  /**
    * A representation of all the relevant information of a `Lazy`-generated tree.
    *
    * @param anonClass the name of the anonymous class containing lazy value definitions
    * @param defs a map from lazy value names to their definition
    * @param entrypoint the tree serving as entrypoint to a lazy implicit
    */
  case class LazyContext(anonClass: TypeName, defs: Map[TermName, Tree], entrypoint: Tree) {

    /**
      * Checks if a `Tree` is a reference to a lazy value generated by `Lazy`. If it is a valid reference, it returns
      * a `Tree` with the corresponding definition and a new `LazyContext` to be used in inner searches. The new
      * `LazyContext` does not have the followed reference in order to prevent infinite loops when searching (the
      * original use case for `Lazy`).
      *
      * @param tree the tree to be checked
      * @return if `tree` is a reference to a `Lazy`-generated value, a `Some` with the respective definition and a new
      *         `LazyContext`; `None` otherwise.
      */
    def followRef(tree: Tree): Option[(LazyContext, Tree)] =
      tree match {
        case q"${`anonClass`}.this.$lazyRef" =>
          defs.get(lazyRef).map { lzyTree => (copy(defs = defs - lazyRef), lzyTree) }

        case _ => None
      }
  }

  /**
    * An object containing an extractor of `LazyContext` instances from Scala trees.
    */
  object LazyContextTree {

    /**
      * Extracts a `LazyContext` from a `Tree`.
      */
    def unapply(tree: Tree): Option[LazyContext] =
      tree match {
        case q"""{
              val $outer = {
                final class $anonClass extends ${_} with ${_} { ..$lazyDefs }
                new $anonClassRef().$entrypoint
              }
              shapeless.Lazy.apply[${_}]($outerRef)
            }""" if isRefTo(outerRef, outer) && isRefTo(anonClassRef, anonClass) =>
          val lazyDefMap: Map[TermName, Tree] = lazyDefs.collect { case q"lazy val $valName = $valBody" =>
            valName -> valBody
          }.toMap

          val entrypointDef = lazyDefMap(entrypoint)
          Some(LazyContext(anonClass, lazyDefMap - entrypoint, entrypointDef))

        case _ => None
      }

    private[this] def isRefTo(tree: Tree, name: Name): Boolean =
      tree match {
        case Ident(otherName) => name == otherName
        case _ => false
      }
  }
}
