package pureconfig

import scala.collection.JavaConverters._
import scala.collection.generic.CanBuildFrom
import scala.collection.mutable.Builder
import scala.language.higherKinds

import com.typesafe.config._
import pureconfig.ConvertHelpers._
import pureconfig.error._
import shapeless._
import shapeless.labelled._
import shapeless.ops.hlist.HKernelAux

/**
 * The default behavior of ConfigReaders that are implicitly derived in PureConfig is to raise a
 * KeyNotFoundException when a required key is missing. Mixing in this trait to a ConfigReader
 * allows customizing this behavior. When a key is missing, but the ConfigReader of the given
 * type extends this trait, the `from` method of the ConfigReader is called with null.
 */
trait AllowMissingKey { self: ConfigReader[_] => }

/**
 * Trait extending [[DerivedReaders1]] that contains `ConfigReader` instances for `AnyVal`.
 *
 * This trait exists to give priority to the `AnyVal` derivation over the generic product derivation.
 */
trait DerivedReaders extends DerivedReaders1 {
  implicit def deriveAnyVal[T, U](
    implicit
    ev: T <:< AnyVal,
    generic: Generic[T],
    unwrapped: Unwrapped.Aux[T, U],
    reader: ConfigReader[U]): ConfigReader[T] =
    new ConfigReader[T] {
      def from(value: ConfigValue): Either[ConfigReaderFailures, T] =
        reader.from(value).right.map(unwrapped.wrap)
    }

  // used for tuples
  implicit def deriveTupleInstance[F: IsTuple, Repr <: HList, LRepr <: HList, DefaultRepr <: HList](
    implicit
    g: Generic.Aux[F, Repr],
    gcr: ConfigReader[Repr],
    lg: LabelledGeneric.Aux[F, LRepr],
    default: Default.AsOptions.Aux[F, DefaultRepr],
    pr: WrappedDefaultValue[F, LRepr, DefaultRepr]): ConfigReader[F] = new ConfigReader[F] {
    override def from(value: ConfigValue) = {
      // Try to read first as the product representation (i.e.
      // ConfigObject with '_1', '_2', etc. keys) and afterwards as the Generic
      // representation (i.e. ConfigList).
      value.valueType match {
        case ConfigValueType.OBJECT => deriveTupleInstanceAsObject(value)
        case ConfigValueType.LIST => deriveTupleInstanceAsList(value)
        case other => fail(WrongType(other, Set(ConfigValueType.LIST, ConfigValueType.OBJECT), ConfigValueLocation(value), ""))
      }
    }
  }

  private[pureconfig] def deriveTupleInstanceAsList[F: IsTuple, Repr <: HList](value: ConfigValue)(
    implicit
    gen: Generic.Aux[F, Repr],
    cr: ConfigReader[Repr]): Either[ConfigReaderFailures, F] =
    cr.from(value).right.map(gen.from)

  private[pureconfig] def deriveTupleInstanceAsObject[F: IsTuple, Repr <: HList, DefaultRepr <: HList](value: ConfigValue)(
    implicit
    gen: LabelledGeneric.Aux[F, Repr],
    default: Default.AsOptions.Aux[F, DefaultRepr],
    cr: WrappedDefaultValue[F, Repr, DefaultRepr]): Either[ConfigReaderFailures, F] =
    cr.fromWithDefault(value, default()).right.map(gen.from)
}

/**
 * Trait containing `ConfigReader` instances for collection, product and coproduct types.
 */
trait DerivedReaders1 {

  private[pureconfig] trait WrappedConfigReader[Wrapped, SubRepr] extends ConfigReader[SubRepr]

  protected[pureconfig] trait WrappedDefaultValue[Wrapped, SubRepr <: HList, DefaultRepr <: HList] {
    def fromWithDefault(config: ConfigValue, default: DefaultRepr): Either[ConfigReaderFailures, SubRepr] = config match {
      case co: ConfigObject => fromConfigObject(co, default)
      case other => fail(WrongType(other.valueType, Set(ConfigValueType.OBJECT), ConfigValueLocation(other), ""))
    }
    def fromConfigObject(co: ConfigObject, default: DefaultRepr): Either[ConfigReaderFailures, SubRepr]
  }

  implicit final def labelledHNilConfigReader[Wrapped](
    implicit
    hint: ProductHint[Wrapped]): WrappedDefaultValue[Wrapped, HNil, HNil] = new WrappedDefaultValue[Wrapped, HNil, HNil] {

    override def fromConfigObject(config: ConfigObject, default: HNil): Either[ConfigReaderFailures, HNil] = {
      if (!hint.allowUnknownKeys && !config.isEmpty) {
        val keys = config.keySet().asScala.toList map {
          k => UnknownKey(k, ConfigValueLocation(config.get(k)))
        }
        Left(new ConfigReaderFailures(keys.head, keys.tail))
      } else {
        Right(HNil)
      }
    }
  }

  implicit final def labelledHConsConfigReader[Wrapped, K <: Symbol, V, T <: HList, U <: HList](
    implicit
    key: Witness.Aux[K],
    vFieldReader: Derivation[Lazy[ConfigReader[V]]],
    tConfigReader: Lazy[WrappedDefaultValue[Wrapped, T, U]],
    hint: ProductHint[Wrapped]): WrappedDefaultValue[Wrapped, FieldType[K, V] :: T, Option[V] :: U] = new WrappedDefaultValue[Wrapped, FieldType[K, V] :: T, Option[V] :: U] {

    override def fromConfigObject(co: ConfigObject, default: Option[V] :: U): Either[ConfigReaderFailures, FieldType[K, V] :: T] = {
      val fieldName = key.value.name
      val keyStr = hint.configKey(fieldName)
      val headResult = improveFailures[V](
        (co.get(keyStr), vFieldReader.value.value) match {
          case (null, reader) =>
            default.head match {
              case Some(defaultValue) if hint.useDefaultArgs => Right[Nothing, V](defaultValue)
              case _ if reader.isInstanceOf[AllowMissingKey] => reader.from(null)
              case _ => fail[V](CannotConvertNull(fieldName, co.keySet.asScala))
            }
          case (value, reader) => reader.from(value)
        },
        keyStr,
        ConfigValueLocation(co))
      // for performance reasons only, we shouldn't clone the config object unless necessary
      val tailCo = if (hint.allowUnknownKeys) co else co.withoutKey(keyStr)
      val tailResult = tConfigReader.value.fromWithDefault(tailCo, default.tail)
      combineResults(headResult, tailResult)((head, tail) => field[K](head) :: tail)
    }
  }

  implicit final def cNilConfigReader[Wrapped]: WrappedConfigReader[Wrapped, CNil] = new WrappedConfigReader[Wrapped, CNil] {
    override def from(config: ConfigValue): Either[ConfigReaderFailures, CNil] =
      fail(NoValidCoproductChoiceFound(config, ConfigValueLocation(config), ""))
  }

  implicit final def coproductConfigReader[Wrapped, Name <: Symbol, V, T <: Coproduct](
    implicit
    coproductHint: CoproductHint[Wrapped],
    vName: Witness.Aux[Name],
    vFieldConvert: Derivation[Lazy[ConfigReader[V]]],
    tConfigReader: Lazy[WrappedConfigReader[Wrapped, T]]): WrappedConfigReader[Wrapped, FieldType[Name, V] :+: T] =
    new WrappedConfigReader[Wrapped, FieldType[Name, V] :+: T] {

      override def from(config: ConfigValue): Either[ConfigReaderFailures, FieldType[Name, V] :+: T] =
        coproductHint.from(config, vName.value.name) match {
          case Right(Some(hintConfig)) =>
            vFieldConvert.value.value.from(hintConfig) match {
              case Left(_) if coproductHint.tryNextOnFail(vName.value.name) =>
                tConfigReader.value.from(config).right.map(s => Inr(s))

              case vTry => vTry.right.map(v => Inl(field[Name](v)))
            }

          case Right(None) => tConfigReader.value.from(config).right.map(s => Inr(s))
          case l: Left[_, _] => l.asInstanceOf[Either[ConfigReaderFailures, FieldType[Name, V] :+: T]]
        }
    }

  implicit def deriveOption[T](implicit conv: Derivation[Lazy[ConfigReader[T]]]) = new OptionConfigReader[T]

  class OptionConfigReader[T](implicit conv: Derivation[Lazy[ConfigReader[T]]]) extends ConfigReader[Option[T]] with AllowMissingKey {
    override def from(config: ConfigValue): Either[ConfigReaderFailures, Option[T]] = {
      if (config == null || config.unwrapped() == null)
        Right(None)
      else
        conv.value.value.from(config).right.map(Some(_))
    }
  }

  implicit def deriveTraversable[T, F[T] <: TraversableOnce[T]](
    implicit
    configConvert: Derivation[Lazy[ConfigReader[T]]],
    cbf: CanBuildFrom[F[T], T, F[T]]) = new ConfigReader[F[T]] {

    override def from(config: ConfigValue): Either[ConfigReaderFailures, F[T]] = {
      config match {
        case co: ConfigList =>
          val z: Either[ConfigReaderFailures, Builder[T, F[T]]] = Right(cbf())

          // we called all the failures in the list
          co.asScala.foldLeft(z) {
            case (acc, value) =>
              combineResults(acc, configConvert.value.value.from(value))(_ += _)
          }.right.map(_.result())
        case o: ConfigObject =>
          val z: Either[ConfigReaderFailures, List[(Int, T)]] = Right(List.empty[(Int, T)])
          def keyValueReader(key: String, value: ConfigValue): Either[ConfigReaderFailures, (Int, T)] = {
            val keyResult = catchReadError(_.toInt)(implicitly)(key)(ConfigValueLocation(value)).left.flatMap(t => fail(CannotConvert(key, "Int",
              s"To convert an object to a collection, its keys must be read as Int but key $key has value" +
                s"$value which cannot converted. Error: ${t.because}", ConfigValueLocation(value), key)))
            val valueResult = configConvert.value.value.from(value)
            combineResults(keyResult, valueResult)(_ -> _)
          }

          o.asScala.foldLeft(z) {
            case (acc, (str, v)) =>
              combineResults(acc, keyValueReader(str, v))(_ :+ _)
          }.right.map {
            l =>
              val r = cbf()
              r ++= l.sortBy(_._1).map(_._2)
              r.result()
          }
        case other =>
          fail(WrongType(other.valueType, Set(ConfigValueType.LIST, ConfigValueType.OBJECT), ConfigValueLocation(other), ""))
      }
    }
  }

  implicit def deriveMap[T](implicit configConvert: Derivation[Lazy[ConfigReader[T]]]) = new ConfigReader[Map[String, T]] {

    override def from(config: ConfigValue): Either[ConfigReaderFailures, Map[String, T]] = {
      config match {
        case co: ConfigObject =>
          val z: Either[ConfigReaderFailures, Map[String, T]] = Right(Map.empty[String, T])

          co.asScala.foldLeft(z) {
            case (acc, (key, value)) =>
              combineResults(
                acc,
                improveFailures(configConvert.value.value.from(value), key, ConfigValueLocation(value))) {
                  (map, valueConverted) => map + (key -> valueConverted)
                }
          }

        case other =>
          fail(WrongType(other.valueType, Set(ConfigValueType.OBJECT), ConfigValueLocation(other), ""))
      }
    }
  }

  implicit final lazy val hNilConfigReader: ConfigReader[HNil] =
    new ConfigReader[HNil] {
      def from(cv: ConfigValue): Either[ConfigReaderFailures, HNil] = {
        cv match {
          case cl: ConfigList if cl.size == 0 => Right(HNil)
          case cl: ConfigList => fail(WrongSizeList(0, cl.size, ConfigValueLocation(cv), ""))
          case other => fail(WrongType(other.valueType, Set(ConfigValueType.LIST), ConfigValueLocation(other), ""))
        }
      }
    }

  implicit final def hConsConfigReader[H, T <: HList](implicit hr: Derivation[Lazy[ConfigReader[H]]], tr: Lazy[ConfigReader[T]], tl: HKernelAux[T]): ConfigReader[H :: T] =
    new ConfigReader[H :: T] {
      def from(cv: ConfigValue): Either[ConfigReaderFailures, H :: T] = {
        cv match {
          case cl: ConfigList if cl.size != tl().length + 1 => fail(WrongSizeList(tl().length + 1, cl.size, ConfigValueLocation(cv), ""))
          case cl: ConfigList =>
            val sl = cl.asScala
            val hv = hr.value.value.from(sl.head)
            val tv = tr.value.from(ConfigValueFactory.fromAnyRef(sl.tail.asJava))
            combineResults(hv, tv)(_ :: _)
          case other => fail(WrongType(other.valueType, Set(ConfigValueType.LIST), ConfigValueLocation(other), ""))
        }
      }
    }

  implicit final def deriveProductInstance[F, Repr <: HList, DefaultRepr <: HList](
    implicit
    gen: LabelledGeneric.Aux[F, Repr],
    default: Default.AsOptions.Aux[F, DefaultRepr],
    cc: Lazy[WrappedDefaultValue[F, Repr, DefaultRepr]]): ConfigReader[F] = new ConfigReader[F] {

    override def from(config: ConfigValue): Either[ConfigReaderFailures, F] = {
      cc.value.fromWithDefault(config, default()).right.map(gen.from)
    }
  }

  implicit final def deriveCoproductInstance[F, Repr <: Coproduct](
    implicit
    gen: LabelledGeneric.Aux[F, Repr],
    cc: Lazy[WrappedConfigReader[F, Repr]]): ConfigReader[F] = new ConfigReader[F] {
    override def from(config: ConfigValue): Either[ConfigReaderFailures, F] = {
      cc.value.from(config).right.map(gen.from)
    }
  }

}

object DerivedReaders extends DerivedReaders
