/*
 * Decompiled with CFR 0.152.
 */
package com.github.robtimus.filesystems.sftp;

import com.github.robtimus.filesystems.sftp.FileSystemExceptionFactory;
import com.github.robtimus.filesystems.sftp.OpenOptions;
import com.github.robtimus.filesystems.sftp.SFTPEnvironment;
import com.github.robtimus.filesystems.sftp.SFTPLogger;
import com.github.robtimus.filesystems.sftp.SFTPMessages;
import com.jcraft.jsch.ChannelSftp;
import com.jcraft.jsch.JSch;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.SftpATTRS;
import com.jcraft.jsch.SftpException;
import com.jcraft.jsch.SftpStatVFS;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileSystemException;
import java.nio.file.OpenOption;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.slf4j.Logger;

final class SSHChannelPool {
    private static final Logger LOGGER = SFTPLogger.createLogger(SSHChannelPool.class);
    private static final AtomicLong CHANNEL_COUNTER = new AtomicLong();
    private final JSch jsch;
    private final String hostname;
    private final int port;
    private final SFTPEnvironment env;
    private final FileSystemExceptionFactory exceptionFactory;
    private final BlockingQueue<Channel> pool;
    private final long poolWaitTimeout;

    SSHChannelPool(String hostname, int port, SFTPEnvironment env) throws IOException {
        this.jsch = env.createJSch();
        this.hostname = hostname;
        this.port = port;
        this.env = env.clone();
        this.exceptionFactory = env.getExceptionFactory();
        int poolSize = env.getClientConnectionCount();
        this.pool = new ArrayBlockingQueue<Channel>(poolSize);
        this.poolWaitTimeout = env.getClientConnectionWaitTimeout();
        SFTPLogger.creatingPool(LOGGER, hostname, port, poolSize, this.poolWaitTimeout);
        this.fillPool(hostname, port, poolSize);
    }

    private void fillPool(String hostname, int port, int poolSize) throws IOException {
        try {
            for (int i = 0; i < poolSize; ++i) {
                this.pool.add(new Channel(true));
            }
            SFTPLogger.createdPool(LOGGER, hostname, port, poolSize);
        }
        catch (IOException e) {
            SFTPLogger.failedToCreatePool(LOGGER, e);
            for (Channel channel : this.pool) {
                try {
                    channel.disconnect();
                }
                catch (IOException e2) {
                    e.addSuppressed(e2);
                }
            }
            throw e;
        }
    }

    Channel get() throws IOException {
        try {
            Channel channel = this.getWithinTimeout();
            try {
                SFTPLogger.tookChannel(LOGGER, channel.channelId, this.pool.size());
                if (!channel.isConnected()) {
                    SFTPLogger.channelNotConnected(LOGGER, channel.channelId);
                    channel = new Channel(true);
                }
            }
            catch (Exception e) {
                this.pool.add(channel);
                SFTPLogger.returnedBrokenChannel(LOGGER, channel.channelId, this.pool.size());
                throw e;
            }
            channel.increaseRefCount();
            return channel;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            InterruptedIOException iioe = new InterruptedIOException(e.getMessage());
            iioe.initCause(e);
            throw iioe;
        }
    }

    private Channel getWithinTimeout() throws InterruptedException, IOException {
        if (this.poolWaitTimeout == 0L) {
            return this.pool.take();
        }
        Channel channel = this.pool.poll(this.poolWaitTimeout, TimeUnit.MILLISECONDS);
        if (channel == null) {
            throw new IOException(SFTPMessages.clientConnectionWaitTimeoutExpired());
        }
        return channel;
    }

    Channel getOrCreate() throws IOException {
        Channel channel = (Channel)this.pool.poll();
        if (channel == null) {
            return new Channel(false);
        }
        try {
            SFTPLogger.tookChannel(LOGGER, channel.channelId, this.pool.size());
            if (!channel.isConnected()) {
                SFTPLogger.channelNotConnected(LOGGER, channel.channelId);
                channel = new Channel(true);
            }
        }
        catch (Exception e) {
            this.pool.add(channel);
            SFTPLogger.returnedBrokenChannel(LOGGER, channel.channelId, this.pool.size());
            throw e;
        }
        channel.increaseRefCount();
        return channel;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void keepAlive() throws IOException {
        ArrayList channels = new ArrayList();
        this.pool.drainTo(channels);
        SFTPLogger.drainedPoolForKeepAlive(LOGGER);
        IOException exception = null;
        for (Channel channel : channels) {
            try {
                channel.keepAlive();
            }
            catch (IOException e) {
                exception = this.add(exception, e);
            }
            finally {
                this.returnToPool(channel);
            }
        }
        if (exception != null) {
            throw exception;
        }
    }

    void close() throws IOException {
        ArrayList channels = new ArrayList();
        this.pool.drainTo(channels);
        SFTPLogger.drainedPoolForClose(LOGGER);
        IOException exception = null;
        for (Channel channel : channels) {
            try {
                channel.disconnect();
            }
            catch (IOException e) {
                exception = this.add(exception, e);
            }
        }
        if (exception != null) {
            throw exception;
        }
    }

    private IOException add(IOException existing, IOException e) {
        if (existing == null) {
            return e;
        }
        existing.addSuppressed(e);
        return existing;
    }

    private void returnToPool(Channel channel) {
        assert (channel.refCount == 0);
        this.pool.add(channel);
        SFTPLogger.returnedChannel(LOGGER, channel.channelId, this.pool.size());
    }

    IOException asIOException(Exception e) throws IOException {
        if (e instanceof IOException) {
            throw (IOException)e;
        }
        FileSystemException exception = new FileSystemException(null, null, e.getMessage());
        exception.initCause(e);
        throw exception;
    }

    static /* synthetic */ AtomicLong access$700() {
        return CHANNEL_COUNTER;
    }

    final class Channel
    implements Closeable {
        private final String channelId = "channel-" + SSHChannelPool.access$700().incrementAndGet();
        private final ChannelSftp channel;
        private final boolean pooled;
        private int refCount = 0;

        private Channel(boolean pooled) throws IOException {
            this.channel = SSHChannelPool.this.env.openChannel(SSHChannelPool.this.jsch, SSHChannelPool.this.hostname, SSHChannelPool.this.port);
            this.pooled = pooled;
            SFTPLogger.createdChannel(LOGGER, this.channelId, pooled);
        }

        private void increaseRefCount() {
            ++this.refCount;
            SFTPLogger.increasedRefCount(LOGGER, this.channelId, this.refCount);
        }

        private int decreaseRefCount() {
            if (this.refCount > 0) {
                --this.refCount;
                SFTPLogger.decreasedRefCount(LOGGER, this.channelId, this.refCount);
            }
            return this.refCount;
        }

        private void keepAlive() throws IOException {
            try {
                this.channel.getSession().sendKeepAliveMsg();
            }
            catch (Exception e) {
                throw SSHChannelPool.this.asIOException(e);
            }
        }

        private boolean isConnected() {
            if (this.channel.isConnected()) {
                try {
                    this.channel.getSession().sendKeepAliveMsg();
                    return true;
                }
                catch (Exception e) {
                    this.disconnectQuietly();
                }
            }
            return false;
        }

        private void disconnect() throws IOException {
            this.channel.disconnect();
            try {
                this.channel.getSession().disconnect();
            }
            catch (JSchException e) {
                throw SSHChannelPool.this.asIOException((Exception)((Object)e));
            }
            SFTPLogger.disconnectedChannel(LOGGER, this.channelId);
        }

        private void disconnectQuietly() {
            this.channel.disconnect();
            try {
                this.channel.getSession().disconnect();
            }
            catch (JSchException jSchException) {
                // empty catch block
            }
            SFTPLogger.disconnectedChannel(LOGGER, this.channelId);
        }

        @Override
        public void close() throws IOException {
            if (this.decreaseRefCount() == 0) {
                if (this.pooled) {
                    SSHChannelPool.this.returnToPool(this);
                } else {
                    this.disconnect();
                }
            }
        }

        String pwd() throws IOException {
            try {
                return this.channel.pwd();
            }
            catch (SftpException e) {
                throw SSHChannelPool.this.asIOException((Exception)((Object)e));
            }
        }

        InputStream newInputStream(String path, OpenOptions options) throws IOException {
            assert (options.read);
            try {
                InputStream in = this.channel.get(path);
                this.increaseRefCount();
                return new SFTPInputStream(path, in, options.deleteOnClose);
            }
            catch (SftpException e) {
                throw SSHChannelPool.this.exceptionFactory.createNewInputStreamException(path, e);
            }
        }

        OutputStream newOutputStream(String path, OpenOptions options) throws IOException {
            assert (options.write);
            int mode = options.append ? 2 : 0;
            try {
                OutputStream out = this.channel.put(path, mode);
                this.increaseRefCount();
                return new SFTPOutputStream(path, out, options.deleteOnClose);
            }
            catch (SftpException e) {
                throw SSHChannelPool.this.exceptionFactory.createNewOutputStreamException(path, e, options.options);
            }
        }

        private void finalizeStream() throws IOException {
            assert (this.refCount > 0);
            this.close();
        }

        void storeFile(String path, InputStream local, Collection<? extends OpenOption> openOptions) throws IOException {
            try {
                this.channel.put(local, path);
            }
            catch (SftpException e) {
                throw SSHChannelPool.this.exceptionFactory.createNewOutputStreamException(path, e, openOptions);
            }
        }

        SftpATTRS readAttributes(String path, boolean followLinks) throws IOException {
            try {
                return followLinks ? this.channel.stat(path) : this.channel.lstat(path);
            }
            catch (SftpException e) {
                throw SSHChannelPool.this.exceptionFactory.createGetFileException(path, e);
            }
        }

        String readSymbolicLink(String path) throws IOException {
            try {
                return this.channel.readlink(path);
            }
            catch (SftpException e) {
                throw SSHChannelPool.this.exceptionFactory.createReadLinkException(path, e);
            }
        }

        List<ChannelSftp.LsEntry> listFiles(String path) throws IOException {
            ArrayList<ChannelSftp.LsEntry> entries = new ArrayList<ChannelSftp.LsEntry>();
            ChannelSftp.LsEntrySelector selector = entry -> {
                entries.add(entry);
                return 0;
            };
            try {
                this.channel.ls(path, selector);
            }
            catch (SftpException e) {
                throw SSHChannelPool.this.exceptionFactory.createListFilesException(path, e);
            }
            return entries;
        }

        void mkdir(String path) throws IOException {
            try {
                this.channel.mkdir(path);
            }
            catch (SftpException e) {
                if (this.fileExists(path)) {
                    throw new FileAlreadyExistsException(path);
                }
                throw SSHChannelPool.this.exceptionFactory.createCreateDirectoryException(path, e);
            }
        }

        private boolean fileExists(String path) {
            try {
                this.channel.stat(path);
                return true;
            }
            catch (SftpException e) {
                return false;
            }
        }

        void delete(String path, boolean isDirectory) throws IOException {
            try {
                if (isDirectory) {
                    this.channel.rmdir(path);
                } else {
                    this.channel.rm(path);
                }
            }
            catch (SftpException e) {
                throw SSHChannelPool.this.exceptionFactory.createDeleteException(path, e, isDirectory);
            }
        }

        void rename(String source, String target) throws IOException {
            try {
                this.channel.rename(source, target);
            }
            catch (SftpException e) {
                throw SSHChannelPool.this.exceptionFactory.createMoveException(source, target, e);
            }
        }

        void chown(String path, int uid) throws IOException {
            try {
                this.channel.chown(uid, path);
            }
            catch (SftpException e) {
                throw SSHChannelPool.this.exceptionFactory.createSetOwnerException(path, e);
            }
        }

        void chgrp(String path, int gid) throws IOException {
            try {
                this.channel.chgrp(gid, path);
            }
            catch (SftpException e) {
                throw SSHChannelPool.this.exceptionFactory.createSetGroupException(path, e);
            }
        }

        void chmod(String path, int permissions) throws IOException {
            try {
                this.channel.chmod(permissions, path);
            }
            catch (SftpException e) {
                throw SSHChannelPool.this.exceptionFactory.createSetPermissionsException(path, e);
            }
        }

        void setMtime(String path, long mtime) throws IOException {
            try {
                this.channel.setMtime(path, (int)mtime);
            }
            catch (SftpException e) {
                throw SSHChannelPool.this.exceptionFactory.createSetModificationTimeException(path, e);
            }
        }

        SftpStatVFS statVFS(String path) throws IOException {
            try {
                return this.channel.statVFS(path);
            }
            catch (SftpException e) {
                if (e.id == 8) {
                    throw new UnsupportedOperationException(e);
                }
                throw SSHChannelPool.this.exceptionFactory.createGetFileException(path, e);
            }
        }

        private final class SFTPOutputStream
        extends OutputStream {
            private final String path;
            private final OutputStream out;
            private final boolean deleteOnClose;
            private boolean open = true;

            private SFTPOutputStream(String path, OutputStream out, boolean deleteOnClose) {
                this.path = path;
                this.out = out;
                this.deleteOnClose = deleteOnClose;
                SFTPLogger.createdOutputStream(LOGGER, Channel.this.channelId, path);
            }

            @Override
            public void write(int b) throws IOException {
                this.out.write(b);
            }

            @Override
            public void write(byte[] b) throws IOException {
                this.out.write(b);
            }

            @Override
            public void write(byte[] b, int off, int len) throws IOException {
                this.out.write(b, off, len);
            }

            @Override
            public void flush() throws IOException {
                this.out.flush();
            }

            @Override
            public void close() throws IOException {
                if (this.open) {
                    try {
                        this.out.close();
                    }
                    finally {
                        this.open = false;
                        Channel.this.finalizeStream();
                    }
                    if (this.deleteOnClose) {
                        Channel.this.delete(this.path, false);
                    }
                    SFTPLogger.closedOutputStream(LOGGER, Channel.this.channelId, this.path);
                }
            }
        }

        private final class SFTPInputStream
        extends InputStream {
            private final String path;
            private final InputStream in;
            private final boolean deleteOnClose;
            private boolean open = true;

            private SFTPInputStream(String path, InputStream in, boolean deleteOnClose) {
                this.path = path;
                this.in = in;
                this.deleteOnClose = deleteOnClose;
                SFTPLogger.createdInputStream(LOGGER, Channel.this.channelId, path);
            }

            @Override
            public int read() throws IOException {
                return this.in.read();
            }

            @Override
            public int read(byte[] b) throws IOException {
                return this.in.read(b);
            }

            @Override
            public int read(byte[] b, int off, int len) throws IOException {
                return this.in.read(b, off, len);
            }

            @Override
            public long skip(long n) throws IOException {
                return this.in.skip(n);
            }

            @Override
            public int available() throws IOException {
                return this.in.available();
            }

            @Override
            public void close() throws IOException {
                if (this.open) {
                    try {
                        this.in.close();
                    }
                    finally {
                        this.open = false;
                        Channel.this.finalizeStream();
                    }
                    if (this.deleteOnClose) {
                        Channel.this.delete(this.path, false);
                    }
                    SFTPLogger.closedInputStream(LOGGER, Channel.this.channelId, this.path);
                }
            }

            @Override
            public synchronized void mark(int readlimit) {
                this.in.mark(readlimit);
            }

            @Override
            public synchronized void reset() throws IOException {
                this.in.reset();
            }

            @Override
            public boolean markSupported() {
                return this.in.markSupported();
            }
        }
    }
}

