/*
 * Decompiled with CFR 0.152.
 */
package com.hulaki.smtp.storage;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.hulaki.smtp.api.MailMessage;
import com.hulaki.smtp.exceptions.SmtpException;
import com.hulaki.smtp.storage.MailMessageDao;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import org.apache.commons.io.FileUtils;
import org.apache.log4j.Logger;

public class FileBasedMailMessageDao
implements MailMessageDao {
    private static final String EMAIL_FILE_PREFIX = "EML_";
    public static final String EMAIL_FILE_SUFFIX = ".xml";
    private static final Logger logger = Logger.getLogger(FileBasedMailMessageDao.class);
    private final File mailsFolder;
    private Map<String, List<String>> fileNamesByRecipients = Maps.newConcurrentMap();
    private int storedEmailsCount = 0;

    public FileBasedMailMessageDao(String mailsFolderPath) {
        this.mailsFolder = this.recreateFolder(mailsFolderPath);
    }

    @Override
    public synchronized void storeMessage(String recipient, MailMessage email) {
        this.ensureFolderExists(this.mailsFolder.getPath());
        try {
            File recipientFolder = this.getRecipientMailsFolder(recipient);
            this.ensureFolderExists(recipientFolder.getPath());
            File emailFile = File.createTempFile(EMAIL_FILE_PREFIX, EMAIL_FILE_SUFFIX, recipientFolder);
            FileBasedMailMessageDao.marshal(email, emailFile);
            if (!this.fileNamesByRecipients.containsKey(recipient)) {
                this.fileNamesByRecipients.put(recipient, new ArrayList());
            }
            this.fileNamesByRecipients.get(recipient).add(emailFile.getName());
            ++this.storedEmailsCount;
            logger.info((Object)("Stored message for " + recipient + ", sum total: " + this.storedEmailsCount));
        }
        catch (IOException ioe) {
            throw new SmtpException("Failed to create file to store mail for " + recipient, ioe);
        }
    }

    @Override
    public synchronized void clearMessagesForRecipient(String recipient) {
        if (!this.fileNamesByRecipients.containsKey(recipient)) {
            return;
        }
        this.ensureFolderExists(this.mailsFolder.getPath());
        this.storedEmailsCount -= this.fileNamesByRecipients.get(recipient).size();
        this.fileNamesByRecipients.remove(recipient);
        this.deleteRecipientFolder(recipient);
        logger.info((Object)("Deleted all messages for " + recipient + ", sum total: " + this.storedEmailsCount));
    }

    @Override
    public synchronized List<MailMessage> retrieveMessages(String recipient) {
        ArrayList mails = Lists.newArrayList();
        if (!this.fileNamesByRecipients.containsKey(recipient)) {
            return mails;
        }
        this.ensureFolderExists(this.mailsFolder.getPath());
        File recipientFolder = this.getRecipientMailsFolder(recipient);
        for (String file : this.fileNamesByRecipients.get(recipient)) {
            MailMessage mailMessage = FileBasedMailMessageDao.unmarshal(new File(recipientFolder.getPath(), file));
            mails.add(mailMessage);
        }
        logger.info((Object)("Retrieved " + mails.size() + " messages for " + recipient + ", sum total: " + this.storedEmailsCount));
        return mails;
    }

    @Override
    public synchronized void clearMessages() {
        try {
            FileUtils.deleteDirectory((File)this.mailsFolder);
            this.storedEmailsCount = 0;
            this.fileNamesByRecipients.clear();
            logger.info((Object)("Deleted ALL messages sum total: " + this.storedEmailsCount));
        }
        catch (IOException ex) {
            throw new SmtpException(ex);
        }
    }

    @Override
    public synchronized int countMessagesForRecipient(String recipient) {
        if (!this.fileNamesByRecipients.containsKey(recipient)) {
            return 0;
        }
        return this.fileNamesByRecipients.get(recipient).size();
    }

    @Override
    public synchronized int countAllMessagesReceived() {
        return this.storedEmailsCount;
    }

    private File recreateFolder(String folderPath) {
        try {
            FileUtils.forceDelete((File)new File(folderPath));
        }
        catch (FileNotFoundException ex) {
        }
        catch (IOException ioe) {
            throw new SmtpException("Could not clear the mails folder.", ioe);
        }
        return this.ensureFolderExists(folderPath);
    }

    private File ensureFolderExists(String folderPath) {
        try {
            File folder = new File(folderPath);
            FileUtils.forceMkdir((File)folder);
            return folder;
        }
        catch (IOException ex) {
            throw new SmtpException(ex);
        }
    }

    private File getRecipientMailsFolder(String recipient) {
        return new File(this.mailsFolder.getPath(), recipient);
    }

    private void deleteRecipientFolder(String recipient) {
        try {
            File recipientFolder = this.getRecipientMailsFolder(recipient);
            FileUtils.deleteDirectory((File)recipientFolder);
        }
        catch (IOException ex) {
            throw new SmtpException(ex);
        }
    }

    private static MailMessage unmarshal(File file) {
        try {
            JAXBContext context = JAXBContext.newInstance((Class[])new Class[]{MailMessage.class});
            Unmarshaller um = context.createUnmarshaller();
            return (MailMessage)um.unmarshal((Reader)new FileReader(file));
        }
        catch (Exception ex) {
            throw new SmtpException("Could not serialize the mail message.", ex);
        }
    }

    public static void marshal(MailMessage mail, File file) {
        try {
            JAXBContext context = JAXBContext.newInstance((Class[])new Class[]{MailMessage.class});
            Marshaller m = context.createMarshaller();
            m.marshal((Object)mail, file);
        }
        catch (Exception ex) {
            throw new SmtpException("Could not deserialize the saved mail message.", ex);
        }
    }
}

