/*
 * Decompiled with CFR 0.152.
 */
package com.github.signaflo.timeseries;

import com.github.signaflo.data.DataSet;
import com.github.signaflo.data.DoubleDataSet;
import com.github.signaflo.math.operations.DoubleFunctions;
import com.github.signaflo.math.operations.Operators;
import com.github.signaflo.timeseries.TimePeriod;
import com.github.signaflo.timeseries.TimeUnit;
import java.text.DecimalFormat;
import java.time.LocalDateTime;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import lombok.NonNull;

public final class TimeSeries
implements DataSet {
    private final TimePeriod timePeriod;
    private final int n;
    private final double mean;
    private final double[] series;
    private final List<OffsetDateTime> observationTimes;
    private final Map<OffsetDateTime, Integer> dateTimeIndex;
    private final DoubleDataSet dataSet;

    private TimeSeries(double ... series) {
        this(OffsetDateTime.of(1, 1, 1, 0, 0, 0, 0, ZoneOffset.ofHours(0)), series);
    }

    private TimeSeries(OffsetDateTime startTime, double ... series) {
        this(TimePeriod.oneYear(), startTime, series);
    }

    private TimeSeries(TimePeriod timePeriod, double ... series) {
        this(timePeriod, OffsetDateTime.of(1, 1, 1, 0, 0, 0, 0, ZoneOffset.UTC), series);
    }

    private TimeSeries(TimePeriod timePeriod, CharSequence startTime, double ... series) {
        OffsetDateTime dateTime;
        this.dataSet = new DoubleDataSet(series);
        this.series = (double[])series.clone();
        this.n = series.length;
        this.mean = this.dataSet.mean();
        this.timePeriod = timePeriod;
        HashMap<OffsetDateTime, Integer> dateTimeIndex = new HashMap<OffsetDateTime, Integer>(series.length);
        ArrayList<OffsetDateTime> dateTimes = new ArrayList<OffsetDateTime>(series.length);
        try {
            dateTime = OffsetDateTime.parse(startTime);
            dateTimes.add(dateTime);
            dateTimeIndex.put(dateTime, 0);
        }
        catch (DateTimeParseException e) {
            dateTime = OffsetDateTime.of(LocalDateTime.parse(startTime), ZoneOffset.ofHours(0));
            dateTimes.add(dateTime);
            dateTimeIndex.put(dateTime, 0);
        }
        for (int i = 1; i < series.length; ++i) {
            dateTime = ((OffsetDateTime)dateTimes.get(i - 1)).plus(timePeriod.unitLength(), timePeriod.timeUnit().temporalUnit());
            dateTimes.add(dateTime);
            dateTimeIndex.put(dateTime, i);
        }
        this.observationTimes = Collections.unmodifiableList(dateTimes);
        this.dateTimeIndex = Collections.unmodifiableMap(dateTimeIndex);
    }

    private TimeSeries(TimePeriod timePeriod, OffsetDateTime startTime, double ... series) {
        this.dataSet = new DoubleDataSet(series);
        this.series = (double[])series.clone();
        this.n = series.length;
        this.mean = this.dataSet.mean();
        this.timePeriod = timePeriod;
        ArrayList<OffsetDateTime> dateTimes = new ArrayList<OffsetDateTime>(series.length);
        HashMap<OffsetDateTime, Integer> dateTimeIndex = new HashMap<OffsetDateTime, Integer>(series.length);
        dateTimes.add(startTime);
        dateTimeIndex.put(startTime, 0);
        for (int i = 1; i < series.length; ++i) {
            OffsetDateTime dateTime = ((OffsetDateTime)dateTimes.get(i - 1)).plus(timePeriod.unitLength(), timePeriod.timeUnit().temporalUnit());
            dateTimes.add(dateTime);
            dateTimeIndex.put(dateTime, i);
        }
        this.observationTimes = Collections.unmodifiableList(dateTimes);
        this.dateTimeIndex = Collections.unmodifiableMap(dateTimeIndex);
    }

    private TimeSeries(TimeUnit timeUnit, CharSequence startTime, double ... series) {
        this(new TimePeriod(timeUnit, 1L), startTime, series);
    }

    private TimeSeries(TimePeriod timePeriod, List<OffsetDateTime> observationTimes, double ... series) {
        this.dataSet = new DoubleDataSet(series);
        this.series = (double[])series.clone();
        this.n = series.length;
        this.mean = this.dataSet.mean();
        this.timePeriod = timePeriod;
        this.observationTimes = Collections.unmodifiableList(observationTimes);
        HashMap<OffsetDateTime, Integer> dateTimeIndex = new HashMap<OffsetDateTime, Integer>(series.length);
        int i = 0;
        for (OffsetDateTime dt : observationTimes) {
            dateTimeIndex.put(dt, i);
            ++i;
        }
        this.dateTimeIndex = Collections.unmodifiableMap(dateTimeIndex);
    }

    public static TimeSeries from(double ... series) {
        if (series == null) {
            throw new NullPointerException("series");
        }
        return new TimeSeries(series);
    }

    public static TimeSeries from(@NonNull OffsetDateTime startTime, double ... series) {
        if (startTime == null) {
            throw new NullPointerException("startTime");
        }
        if (series == null) {
            throw new NullPointerException("series");
        }
        return new TimeSeries(startTime, series);
    }

    public static TimeSeries from(@NonNull TimePeriod timePeriod, double ... series) {
        if (timePeriod == null) {
            throw new NullPointerException("timePeriod");
        }
        if (series == null) {
            throw new NullPointerException("series");
        }
        return new TimeSeries(timePeriod, series);
    }

    public static TimeSeries from(@NonNull TimePeriod timePeriod, @NonNull CharSequence startTime, double ... series) {
        if (timePeriod == null) {
            throw new NullPointerException("timePeriod");
        }
        if (startTime == null) {
            throw new NullPointerException("startTime");
        }
        if (series == null) {
            throw new NullPointerException("series");
        }
        return new TimeSeries(timePeriod, startTime, series);
    }

    public static TimeSeries from(@NonNull TimePeriod timePeriod, @NonNull List<OffsetDateTime> observationTimes, double ... series) {
        if (timePeriod == null) {
            throw new NullPointerException("timePeriod");
        }
        if (observationTimes == null) {
            throw new NullPointerException("observationTimes");
        }
        if (series == null) {
            throw new NullPointerException("series");
        }
        return new TimeSeries(timePeriod, observationTimes, series);
    }

    public static TimeSeries from(@NonNull TimeUnit timeUnit, @NonNull CharSequence startTime, double ... series) {
        if (timeUnit == null) {
            throw new NullPointerException("timeUnit");
        }
        if (startTime == null) {
            throw new NullPointerException("startTime");
        }
        if (series == null) {
            throw new NullPointerException("series");
        }
        return new TimeSeries(timeUnit, startTime, series);
    }

    public static TimeSeries from(@NonNull TimePeriod timePeriod, @NonNull OffsetDateTime startTime, double ... series) {
        if (timePeriod == null) {
            throw new NullPointerException("timePeriod");
        }
        if (startTime == null) {
            throw new NullPointerException("startTime");
        }
        if (series == null) {
            throw new NullPointerException("series");
        }
        return new TimeSeries(timePeriod, startTime, series);
    }

    public static TimeSeries from(@NonNull TimeUnit timeUnit, @NonNull OffsetDateTime startTime, double ... series) {
        if (timeUnit == null) {
            throw new NullPointerException("timeUnit");
        }
        if (startTime == null) {
            throw new NullPointerException("startTime");
        }
        if (series == null) {
            throw new NullPointerException("series");
        }
        return new TimeSeries(new TimePeriod(timeUnit, 1L), startTime, series);
    }

    public static double[] difference(@NonNull double[] series, int lag, int times) {
        if (series == null) {
            throw new NullPointerException("series");
        }
        TimeSeries.validate(lag);
        TimeSeries.validate(series, lag, times);
        if (times == 0) {
            return (double[])series.clone();
        }
        double[] diffed = TimeSeries.differenceArray(series, lag);
        for (int i = 1; i < times; ++i) {
            diffed = TimeSeries.differenceArray(diffed, lag);
        }
        return diffed;
    }

    public static double[] difference(double[] series, int times) {
        return TimeSeries.difference(series, 1, times);
    }

    private static double[] differenceArray(double[] series, int lag) {
        double[] differenced = new double[series.length - lag];
        for (int i = 0; i < differenced.length; ++i) {
            differenced[i] = series[i + lag] - series[i];
        }
        return differenced;
    }

    private static void validate(double[] series, int lag, int times) {
        if (times < 0) {
            throw new IllegalArgumentException("The value of times must be non-negative but was " + times);
        }
        if (times * lag > series.length) {
            throw new IllegalArgumentException("The product of lag and times must be less than or equal to the length of the series, but " + times + " * " + lag + " = " + times * lag + " is greater than " + series.length);
        }
    }

    private static void validate(double[] series, int lag) {
        if (lag < 1) {
            throw new IllegalArgumentException("The lag must be positive, but was " + lag);
        }
        if (lag > series.length) {
            throw new IllegalArgumentException("The lag must be less than or equal to the length of the series, but " + lag + " is greater than " + series.length);
        }
    }

    private static void validate(int lag) {
        if (lag < 1) {
            throw new IllegalArgumentException("The lag must be positive, but was " + lag);
        }
    }

    public final TimeSeries aggregateToYears() {
        return this.aggregate(TimePeriod.oneYear());
    }

    public final TimeSeries aggregate(TimeUnit timeUnit) {
        return this.aggregate(new TimePeriod(timeUnit, 1L));
    }

    public final TimeSeries aggregate(@NonNull TimePeriod timePeriod) {
        if (timePeriod == null) {
            throw new NullPointerException("timePeriod");
        }
        int period = (int)this.timePeriod.frequencyPer(timePeriod);
        if (period == 0) {
            throw new IllegalArgumentException("The given time period was of a smaller magnitude than the original time period. To aggregate a series, the time period argument must be of a larger magnitude than the original.");
        }
        ArrayList<OffsetDateTime> obsTimes = new ArrayList<OffsetDateTime>();
        double[] aggregated = new double[this.series.length / period];
        for (int i = 0; i < aggregated.length; ++i) {
            double sum = 0.0;
            for (int j = 0; j < period; ++j) {
                sum += this.series[j + period * i];
            }
            aggregated[i] = sum;
            obsTimes.add(this.observationTimes.get(i * period));
        }
        return new TimeSeries(timePeriod, obsTimes, aggregated);
    }

    public final double at(int index) {
        if (index < 0 || index >= this.series.length) {
            throw new IndexOutOfBoundsException("No observation available at index: " + index);
        }
        return this.series[index];
    }

    public final double at(@NonNull OffsetDateTime dateTime) {
        if (dateTime == null) {
            throw new NullPointerException("dateTime");
        }
        if (!this.dateTimeIndex.containsKey(dateTime)) {
            throw new IllegalArgumentException("No observation available at date-time: " + dateTime);
        }
        return this.series[this.dateTimeIndex.get(dateTime)];
    }

    public final double autoCorrelationAtLag(int k) {
        this.validateLag(k);
        double variance = this.autoCovarianceAtLag(0);
        return this.autoCovarianceAtLag(k) / variance;
    }

    public final double[] autoCorrelationUpToLag(int k) {
        this.validateLag(k);
        double[] autoCorrelation = new double[Math.min(k + 1, this.n)];
        for (int i = 0; i < Math.min(k + 1, this.n); ++i) {
            autoCorrelation[i] = this.autoCorrelationAtLag(i);
        }
        return autoCorrelation;
    }

    public final double autoCovarianceAtLag(int k) {
        this.validateLag(k);
        double sumOfProductOfDeviations = 0.0;
        for (int t = 0; t < this.n - k; ++t) {
            sumOfProductOfDeviations += (this.series[t] - this.mean) * (this.series[t + k] - this.mean);
        }
        return sumOfProductOfDeviations / (double)this.n;
    }

    public final double[] autoCovarianceUpToLag(int k) {
        this.validateLag(k);
        double[] acv = new double[Math.min(k + 1, this.n)];
        for (int i = 0; i < Math.min(k + 1, this.n); ++i) {
            acv[i] = this.autoCovarianceAtLag(i);
        }
        return acv;
    }

    private void validateLag(int k) {
        if (k < 0) {
            throw new IllegalArgumentException("The lag, k, must be non-negative, but was " + k);
        }
    }

    public final TimeSeries transform(double boxCoxLambda) {
        if (boxCoxLambda > 2.0 || boxCoxLambda < -1.0) {
            throw new IllegalArgumentException("The BoxCox parameter must lie between -1 and 2, but the provided parameter was equal to " + boxCoxLambda);
        }
        double[] boxCoxed = DoubleFunctions.boxCox((double[])this.series, (double)boxCoxLambda);
        return new TimeSeries(this.timePeriod, this.observationTimes, boxCoxed);
    }

    public final TimeSeries backTransform(double boxCoxLambda) {
        if (boxCoxLambda > 2.0 || boxCoxLambda < -1.0) {
            throw new IllegalArgumentException("The BoxCox parameter must lie between -1 and 2, but the provided parameter was equal to " + boxCoxLambda);
        }
        double[] invBoxCoxed = DoubleFunctions.inverseBoxCox((double[])this.series, (double)boxCoxLambda);
        return new TimeSeries(this.timePeriod, this.observationTimes, invBoxCoxed);
    }

    public final TimeSeries movingAverage(int m) {
        int c = m % 2;
        int k = (m - c) / 2;
        double[] average = new double[this.n - m + 1];
        for (int t = 0; t < average.length; ++t) {
            double sum = 0.0;
            for (int j = -k; j < k + c; ++j) {
                sum += this.series[t + k + j];
            }
            average[t] = sum / (double)m;
        }
        List<OffsetDateTime> times = this.observationTimes.subList(k + c - 1, this.n - k);
        return new TimeSeries(this.timePeriod, times, average);
    }

    public final TimeSeries centeredMovingAverage(int m) {
        if (m % 2 == 1) {
            return this.movingAverage(m);
        }
        TimeSeries firstAverage = this.movingAverage(m);
        int k = m / 2;
        List<OffsetDateTime> times = this.observationTimes.subList(k, this.n - k);
        return new TimeSeries(this.timePeriod, times, firstAverage.movingAverage((int)2).series);
    }

    public final TimeSeries demean() {
        double[] demeaned = new double[this.series.length];
        for (int t = 0; t < demeaned.length; ++t) {
            demeaned[t] = this.series[t] - this.mean;
        }
        return new TimeSeries(this.timePeriod, this.observationTimes, demeaned);
    }

    public final TimeSeries difference(int lag, int times) {
        TimeSeries.validate(this.series, lag, times);
        if (times > 0) {
            TimeSeries diffed = this.difference(lag);
            for (int i = 1; i < times; ++i) {
                diffed = diffed.difference(lag);
            }
            return diffed;
        }
        return this;
    }

    public final TimeSeries difference(int lag) {
        TimeSeries.validate(this.series, lag);
        double[] diffed = TimeSeries.differenceArray(this.asArray(), lag);
        List<OffsetDateTime> obsTimes = this.observationTimes.subList(lag, this.n);
        return new TimeSeries(this.timePeriod, obsTimes, diffed);
    }

    public final TimeSeries difference() {
        return this.difference(1);
    }

    public final TimeSeries minus(@NonNull TimeSeries otherSeries) {
        if (otherSeries == null) {
            throw new NullPointerException("otherSeries");
        }
        if (otherSeries.size() == 0) {
            return this;
        }
        if (otherSeries.size() != this.series.length) {
            throw new IllegalArgumentException("The two series must have the same length.");
        }
        double[] subtracted = new double[this.series.length];
        for (int t = 0; t < subtracted.length; ++t) {
            subtracted[t] = this.series[t] - otherSeries.series[t];
        }
        return new TimeSeries(this.timePeriod, this.observationTimes, subtracted);
    }

    public final TimeSeries minus(@NonNull double[] otherSeries) {
        if (otherSeries == null) {
            throw new NullPointerException("otherSeries");
        }
        if (otherSeries.length == 0) {
            return this;
        }
        if (otherSeries.length != this.series.length) {
            throw new IllegalArgumentException("The two series must have the same length.");
        }
        double[] subtracted = new double[this.series.length];
        for (int t = 0; t < subtracted.length; ++t) {
            subtracted[t] = this.series[t] - otherSeries[t];
        }
        return new TimeSeries(this.timePeriod, this.observationTimes, subtracted);
    }

    public final TimeSeries slice(int start, int end) {
        double[] sliced = new double[end - start + 1];
        System.arraycopy(this.series, start, sliced, 0, end - start + 1);
        List<OffsetDateTime> obsTimes = this.observationTimes.subList(start, end + 1);
        return new TimeSeries(this.timePeriod, obsTimes, sliced);
    }

    public final TimeSeries slice(@NonNull OffsetDateTime start, @NonNull OffsetDateTime end) {
        if (start == null) {
            throw new NullPointerException("start");
        }
        if (end == null) {
            throw new NullPointerException("end");
        }
        int startIdx = this.dateTimeIndex.get(start);
        int endIdx = this.dateTimeIndex.get(end);
        double[] sliced = new double[endIdx - startIdx + 1];
        System.arraycopy(this.series, startIdx, sliced, 0, endIdx - startIdx + 1);
        List<OffsetDateTime> obsTimes = this.observationTimes.subList(startIdx, endIdx + 1);
        return new TimeSeries(this.timePeriod, obsTimes, sliced);
    }

    public final TimeSeries timeSlice(int start, int end) {
        double[] sliced = new double[end - start + 1];
        System.arraycopy(this.series, start - 1, sliced, 0, end - start + 1);
        List<OffsetDateTime> obsTimes = this.observationTimes.subList(start - 1, end);
        return new TimeSeries(this.timePeriod, obsTimes, sliced);
    }

    public final void print() {
        System.out.println(this.toString());
    }

    public final List<Double> asList() {
        return DoubleFunctions.listFrom((double[])((double[])this.series.clone()));
    }

    public final TimePeriod timePeriod() {
        return this.timePeriod;
    }

    public final OffsetDateTime startTime() {
        return this.observationTimes.get(0);
    }

    public final List<OffsetDateTime> observationTimes() {
        return this.observationTimes;
    }

    public final Map<OffsetDateTime, Integer> dateTimeIndex() {
        return this.dateTimeIndex;
    }

    @Override
    public final double[] asArray() {
        return (double[])this.series.clone();
    }

    @Override
    public double sum() {
        return this.dataSet.sum();
    }

    @Override
    public double sumOfSquares() {
        return this.dataSet.sumOfSquares();
    }

    @Override
    public double mean() {
        return this.dataSet.mean();
    }

    @Override
    public double median() {
        return this.dataSet.median();
    }

    @Override
    public int size() {
        return this.dataSet.size();
    }

    @Override
    public TimeSeries times(@NonNull DataSet otherData) {
        if (otherData == null) {
            throw new NullPointerException("otherData");
        }
        return new TimeSeries(this.timePeriod(), this.observationTimes(), Operators.productOf((double[])this.asArray(), (double[])otherData.asArray()));
    }

    @Override
    public TimeSeries plus(@NonNull DataSet otherData) {
        if (otherData == null) {
            throw new NullPointerException("otherData");
        }
        return new TimeSeries(this.timePeriod(), this.observationTimes(), Operators.sumOf((double[])this.asArray(), (double[])otherData.asArray()));
    }

    @Override
    public double variance() {
        return this.dataSet.variance();
    }

    @Override
    public double stdDeviation() {
        return this.dataSet.stdDeviation();
    }

    @Override
    public double covariance(DataSet otherData) {
        return this.dataSet.covariance(otherData);
    }

    @Override
    public double correlation(DataSet otherData) {
        return this.dataSet.correlation(otherData);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        TimeSeries that = (TimeSeries)o;
        if (this.timePeriod != null ? !this.timePeriod.equals(that.timePeriod) : that.timePeriod != null) {
            return false;
        }
        if (!Arrays.equals(this.series, that.series)) {
            return false;
        }
        return this.observationTimes.equals(that.observationTimes);
    }

    public int hashCode() {
        int result = this.timePeriod != null ? this.timePeriod.hashCode() : 0;
        result = 31 * result + Arrays.hashCode(this.series);
        result = 31 * result + this.observationTimes.hashCode();
        return result;
    }

    public String toString() {
        String newLine = System.lineSeparator();
        DecimalFormat numFormatter = new DecimalFormat("#0.00");
        return newLine + "Time Series: " + newLine + "number of observations: " + this.n + newLine + "mean: " + numFormatter.format(this.mean) + newLine + "std: " + numFormatter.format(this.stdDeviation()) + newLine + "period: " + this.timePeriod;
    }
}

