/*
 * Decompiled with CFR 0.152.
 */
package com.github.signaflo.timeseries.model.arima;

import com.github.signaflo.data.Range;
import com.github.signaflo.data.regression.LinearRegression;
import com.github.signaflo.math.function.AbstractMultivariateFunction;
import com.github.signaflo.math.linear.doubles.Matrix;
import com.github.signaflo.math.linear.doubles.MatrixBuilder;
import com.github.signaflo.math.linear.doubles.Vector;
import com.github.signaflo.math.operations.DoubleFunctions;
import com.github.signaflo.math.operations.Operators;
import com.github.signaflo.math.optim.BFGS;
import com.github.signaflo.math.stats.Statistics;
import com.github.signaflo.timeseries.TimePeriod;
import com.github.signaflo.timeseries.TimeSeries;
import com.github.signaflo.timeseries.forecast.Forecast;
import com.github.signaflo.timeseries.model.arima.Arima;
import com.github.signaflo.timeseries.model.arima.ArimaCoefficients;
import com.github.signaflo.timeseries.model.arima.ArimaForecaster;
import com.github.signaflo.timeseries.model.arima.ArimaKalmanFilter;
import com.github.signaflo.timeseries.model.arima.ArimaOrder;
import com.github.signaflo.timeseries.model.arima.ArimaParameters;
import com.github.signaflo.timeseries.model.arima.ArimaStateSpace;
import com.github.signaflo.timeseries.model.regression.TimeSeriesLinearRegression;
import com.github.signaflo.timeseries.model.regression.TimeSeriesLinearRegressionBuilder;
import com.github.signaflo.timeseries.operators.LagPolynomial;
import java.text.DecimalFormat;
import java.util.Arrays;

final class ArimaModel
implements Arima {
    private static final double EPSILON = Math.ulp(1.0);
    private static final double DEFAULT_TOLERANCE = Math.sqrt(EPSILON);
    private final TimeSeries observations;
    private final TimeSeries differencedSeries;
    private final TimeSeries fittedSeries;
    private final TimeSeries residuals;
    private final ArimaOrder order;
    private final ModelInformation modelInfo;
    private final ArimaCoefficients coefficients;
    private final Arima.FittingStrategy fittingStrategy;
    private final int seasonalFrequency;
    private final double[] arSarCoeffs;
    private final double[] maSmaCoeffs;
    private final double[] stdErrors;

    ArimaModel(TimeSeries observations, ArimaOrder order, TimePeriod seasonalCycle, Arima.FittingStrategy fittingStrategy) {
        this(observations, order, seasonalCycle, fittingStrategy, null);
    }

    private ArimaModel(TimeSeries observations, ArimaOrder order, TimePeriod seasonalCycle, Arima.FittingStrategy fittingStrategy, LinearRegression regression) {
        Matrix initHessian;
        Vector initParams;
        this.observations = observations;
        this.order = order;
        this.fittingStrategy = fittingStrategy;
        this.seasonalFrequency = (int)observations.timePeriod().frequencyPer(seasonalCycle);
        this.differencedSeries = observations.difference(1, order.d()).difference(this.seasonalFrequency, order.D());
        ArimaParameters parameters = ArimaParameters.initializePars(order.p(), order.q(), order.P(), order.Q());
        Matrix regressionMatrix = ArimaModel.getRegressionMatrix(observations.size(), order);
        if (regression == null) {
            regression = this.getLinearRegression(this.differencedSeries, regressionMatrix);
        }
        if (order.constant().include()) {
            parameters.setMean(regression.beta()[0]);
            parameters.setMeanParScale(10.0 * regression.standardErrors()[0]);
        }
        if (order.drift().include()) {
            parameters.setDrift(regression.beta()[order.constant().asInt()]);
            parameters.setDriftParScale(10.0 * regression.standardErrors()[order.constant().asInt()]);
        }
        if (fittingStrategy == Arima.FittingStrategy.CSSML) {
            Arima.FittingStrategy subStrategy = Arima.FittingStrategy.CSS;
            ArimaModel firstModel = new ArimaModel(observations, order, seasonalCycle, subStrategy, regression);
            double meanParScale = parameters.getMeanParScale();
            double driftParScale = parameters.getDriftParScale();
            parameters = ArimaParameters.fromCoefficients(firstModel.coefficients());
            parameters.setMeanParScale(meanParScale);
            parameters.setDriftParScale(driftParScale);
            initParams = Vector.from((double[])parameters.getAllScaled(order));
            initHessian = this.getInitialHessian(firstModel);
        } else {
            initParams = Vector.from((double[])parameters.getAllScaled(order));
            initHessian = this.getInitialHessian(initParams.size());
        }
        OptimFunction function = new OptimFunction(observations, order, parameters, fittingStrategy, regressionMatrix, this.seasonalFrequency);
        BFGS optimizer = new BFGS((AbstractMultivariateFunction)function, initParams, DEFAULT_TOLERANCE, DEFAULT_TOLERANCE, initHessian);
        Vector optimizedParams = optimizer.parameters();
        Matrix inverseHessian = optimizer.inverseHessian();
        this.stdErrors = DoubleFunctions.sqrt((double[])Operators.scale((double[])inverseHessian.diagonal(), (double)(1.0 / (double)this.differencedSeries.size())));
        if (order.constant().include()) {
            int n = order.sumARMA();
            this.stdErrors[n] = this.stdErrors[n] * parameters.getMeanParScale();
        }
        if (order.drift().include()) {
            int n = order.sumARMA() + order.constant().asInt();
            this.stdErrors[n] = this.stdErrors[n] * parameters.getDriftParScale();
        }
        double[] arCoeffs = this.getArCoeffs(optimizedParams);
        double[] maCoeffs = this.getMaCoeffs(optimizedParams);
        double[] sarCoeffs = this.getSarCoeffs(optimizedParams);
        double[] smaCoeffs = this.getSmaCoeffs(optimizedParams);
        if (order.constant().include()) {
            parameters.setAndScaleMean(optimizedParams.at(order.sumARMA()));
        }
        if (order.drift().include()) {
            parameters.setAndScaleDrift(optimizedParams.at(order.sumARMA() + order.constant().asInt()));
        }
        this.coefficients = new ArimaCoefficients(arCoeffs, maCoeffs, sarCoeffs, smaCoeffs, order.d(), order.D(), parameters.getMean(), parameters.getDrift(), this.seasonalFrequency);
        this.arSarCoeffs = this.coefficients.getAllAutoRegressiveCoefficients();
        this.maSmaCoeffs = this.coefficients.getAllMovingAverageCoefficients();
        Vector regressionParameters = Vector.from((double[])parameters.getRegressors(order));
        Vector regressionEffects = regressionMatrix.times(regressionParameters);
        TimeSeries armaSeries = this.observations.minus(regressionEffects.elements());
        TimeSeries differencedSeries = armaSeries.difference(1, order.d()).difference(this.seasonalFrequency, order.D());
        if (fittingStrategy == Arima.FittingStrategy.CSS) {
            this.modelInfo = ArimaModel.fitCSS(differencedSeries, this.arSarCoeffs, this.maSmaCoeffs, order.npar());
            double[] residuals = DoubleFunctions.combine((double[][])new double[][]{new double[order.d() + order.D() * this.seasonalFrequency], this.modelInfo.residuals});
            this.fittedSeries = observations.minus(TimeSeries.from(residuals));
            this.residuals = observations.minus(this.fittedSeries);
        } else {
            double[] delta = ArimaModel.getDelta(this.order, this.seasonalFrequency);
            this.modelInfo = ArimaModel.fitML(armaSeries, this.arSarCoeffs, this.maSmaCoeffs, delta, order.npar());
            double[] residuals = this.modelInfo.residuals;
            this.fittedSeries = observations.minus(TimeSeries.from(residuals));
            this.residuals = observations.minus(this.fittedSeries);
        }
    }

    ArimaModel(TimeSeries observations, ArimaCoefficients coeffs, TimePeriod seasonalCycle, Arima.FittingStrategy fittingStrategy) {
        this.observations = observations;
        this.coefficients = coeffs;
        this.fittingStrategy = fittingStrategy;
        this.order = coeffs.extractModelOrder();
        this.seasonalFrequency = (int)observations.timePeriod().frequencyPer(seasonalCycle);
        this.differencedSeries = observations.difference(1, this.order.d()).difference(this.seasonalFrequency, this.order.D());
        this.arSarCoeffs = ArimaCoefficients.expandArCoefficients(coeffs.arCoeffs(), coeffs.seasonalARCoeffs(), this.seasonalFrequency);
        this.maSmaCoeffs = ArimaCoefficients.expandMaCoefficients(coeffs.maCoeffs(), coeffs.seasonalMACoeffs(), this.seasonalFrequency);
        this.stdErrors = DoubleFunctions.fill((int)(this.order.sumARMA() + this.order.constant().asInt() + this.order.drift().asInt()), (double)0.0);
        ArimaParameters parameters = ArimaParameters.fromCoefficients(coeffs);
        Matrix regressionMatrix = ArimaModel.getRegressionMatrix(observations.size(), this.order);
        Vector regressionParameters = Vector.from((double[])parameters.getRegressors(this.order));
        Vector regressionEffects = regressionMatrix.times(regressionParameters);
        TimeSeries armaSeries = this.observations.minus(regressionEffects.elements());
        TimeSeries differencedSeries = armaSeries.difference(1, this.order.d()).difference(this.seasonalFrequency, this.order.D());
        if (fittingStrategy == Arima.FittingStrategy.CSS) {
            this.modelInfo = ArimaModel.fitCSS(differencedSeries, this.arSarCoeffs, this.maSmaCoeffs, this.order.npar());
            double[] residuals = DoubleFunctions.combine((double[][])new double[][]{new double[this.arSarCoeffs.length], this.modelInfo.residuals});
            this.fittedSeries = observations.minus(TimeSeries.from(residuals));
            this.residuals = observations.minus(this.fittedSeries);
        } else {
            double[] delta = ArimaModel.getDelta(this.order, this.seasonalFrequency);
            this.modelInfo = ArimaModel.fitML(armaSeries, this.arSarCoeffs, this.maSmaCoeffs, delta, this.order.npar());
            double[] residuals = this.modelInfo.residuals;
            this.fittedSeries = observations.minus(TimeSeries.from(residuals));
            this.residuals = observations.minus(this.fittedSeries);
        }
    }

    private static Matrix getRegressionMatrix(int size, ArimaOrder order) {
        double[][] matrix = new double[order.numRegressors()][size];
        if (order.constant().include()) {
            matrix[0] = DoubleFunctions.fill((int)size, (double)1.0);
        }
        if (order.drift().include()) {
            matrix[order.constant().asInt()] = Range.inclusiveRange(1.0, size).asArray();
        }
        return Matrix.create((Matrix.Layout)Matrix.Layout.BY_COLUMN, (double[][])matrix);
    }

    private LinearRegression getLinearRegression(TimeSeries differencedSeries, Matrix designMatrix) {
        int i;
        double[][] diffedMatrix = new double[designMatrix.ncol()][];
        double[][] designMatrixTwoD = designMatrix.data2D(Matrix.Layout.BY_COLUMN);
        for (i = 0; i < diffedMatrix.length; ++i) {
            diffedMatrix[i] = TimeSeries.difference(designMatrixTwoD[i], this.order.d());
        }
        for (i = 0; i < diffedMatrix.length; ++i) {
            diffedMatrix[i] = TimeSeries.difference(diffedMatrix[i], this.seasonalFrequency, this.order.D());
        }
        TimeSeriesLinearRegressionBuilder regressionBuilder = TimeSeriesLinearRegression.builder();
        regressionBuilder.response(differencedSeries);
        regressionBuilder.hasIntercept(TimeSeriesLinearRegression.Intercept.EXCLUDE);
        regressionBuilder.timeTrend(TimeSeriesLinearRegression.TimeTrend.EXCLUDE);
        regressionBuilder.externalRegressors(Matrix.create((Matrix.Layout)Matrix.Layout.BY_COLUMN, (double[][])diffedMatrix));
        return regressionBuilder.build();
    }

    private static ModelInformation fitCSS(TimeSeries differencedSeries, double[] arCoeffs, double[] maCoeffs, int npar) {
        int offset = arCoeffs.length;
        int n = differencedSeries.size();
        double[] fitted = new double[n];
        double[] residuals = new double[n];
        for (int t = offset; t < fitted.length; ++t) {
            for (int i = 0; i < arCoeffs.length; ++i) {
                if (!(Math.abs(arCoeffs[i]) > 0.0)) continue;
                int n2 = t;
                fitted[n2] = fitted[n2] + arCoeffs[i] * differencedSeries.at(t - i - 1);
            }
            for (int j = 0; j < Math.min(t, maCoeffs.length); ++j) {
                if (!(Math.abs(maCoeffs[j]) > 0.0)) continue;
                int n3 = t;
                fitted[n3] = fitted[n3] + maCoeffs[j] * residuals[t - j - 1];
            }
            residuals[t] = differencedSeries.at(t) - fitted[t];
        }
        int m = differencedSeries.size() - arCoeffs.length;
        double sigma2 = Statistics.sumOfSquared((double[])residuals) / (double)m;
        double logLikelihood = (double)(-n) / 2.0 * (Math.log(Math.PI * 2 * sigma2) + 1.0);
        return new ModelInformation(npar, sigma2, logLikelihood, residuals, fitted);
    }

    private static ModelInformation fitML(TimeSeries observations, double[] arCoeffs, double[] maCoeffs, double[] delta, int npar) {
        double[] series = observations.asArray();
        ArimaKalmanFilter.KalmanOutput output = ArimaModel.kalmanFit(observations, arCoeffs, maCoeffs, delta);
        double sigma2 = output.sigma2();
        double logLikelihood = output.logLikelihood();
        double[] residuals = output.residuals();
        double[] fitted = Operators.differenceOf((double[])series, (double[])residuals);
        return new ModelInformation(++npar, sigma2, logLikelihood, residuals, fitted);
    }

    private static ArimaKalmanFilter.KalmanOutput kalmanFit(TimeSeries observations, double[] arCoeffs, double[] maCoeffs, double[] delta) {
        double[] series = observations.asArray();
        ArimaStateSpace ss = new ArimaStateSpace(series, arCoeffs, maCoeffs, delta);
        ArimaKalmanFilter kalmanFilter = new ArimaKalmanFilter(ss);
        return kalmanFilter.output();
    }

    static double[] getDelta(ArimaOrder order, int observationFrequency) {
        LagPolynomial differencesPolynomial = LagPolynomial.differences(order.d());
        LagPolynomial seasonalDifferencesPolynomial = LagPolynomial.seasonalDifferences(observationFrequency, order.D());
        LagPolynomial finalPolynomial = differencesPolynomial.times(seasonalDifferencesPolynomial);
        return Operators.scale((double[])finalPolynomial.parameters(), (double)-1.0);
    }

    @Override
    public Forecast forecast(int steps, double alpha) {
        Matrix regressionMatrix = ArimaModel.getRegressionMatrix(this.observations.size(), this.order);
        ArimaForecaster forecaster = new ArimaForecaster.Builder().setObservations(this.observations).setCoefficients(this.coefficients).setOrder(this.order).setDifferencedSeries(this.differencedSeries).setResiduals(this.residuals).setRegressionMatrix(regressionMatrix).setSigma2(this.modelInfo.sigma2).build();
        return forecaster.forecast(steps, alpha);
    }

    private Matrix getInitialHessian(int n) {
        return Matrix.identity((int)n);
    }

    private Matrix getInitialHessian(ArimaModel model) {
        double[] stdErrors = model.stdErrors;
        MatrixBuilder builder = Matrix.identityBuilder((int)stdErrors.length);
        for (int i = 0; i < stdErrors.length; ++i) {
            builder.set(i, i, stdErrors[i] * stdErrors[i] * (double)this.observations.size());
        }
        return builder.build();
    }

    private double[] getSarCoeffs(Vector optimizedParams) {
        double[] sarCoeffs = new double[this.order.P()];
        for (int i = 0; i < this.order.P(); ++i) {
            sarCoeffs[i] = optimizedParams.at(i + this.order.p() + this.order.q());
        }
        return sarCoeffs;
    }

    private double[] getSmaCoeffs(Vector optimizedParams) {
        double[] smaCoeffs = new double[this.order.Q()];
        for (int i = 0; i < this.order.Q(); ++i) {
            smaCoeffs[i] = optimizedParams.at(i + this.order.p() + this.order.q() + this.order.P());
        }
        return smaCoeffs;
    }

    private double[] getArCoeffs(Vector optimizedParams) {
        double[] arCoeffs = new double[this.order.p()];
        for (int i = 0; i < this.order.p(); ++i) {
            arCoeffs[i] = optimizedParams.at(i);
        }
        return arCoeffs;
    }

    private double[] getMaCoeffs(Vector optimizedParams) {
        double[] arCoeffs = new double[this.order.q()];
        for (int i = 0; i < this.order.q(); ++i) {
            arCoeffs[i] = optimizedParams.at(i + this.order.p());
        }
        return arCoeffs;
    }

    @Override
    public TimeSeries observations() {
        return this.observations;
    }

    @Override
    public TimeSeries fittedSeries() {
        return this.fittedSeries;
    }

    @Override
    public TimeSeries predictionErrors() {
        return this.residuals;
    }

    @Override
    public double sigma2() {
        return this.modelInfo.sigma2;
    }

    @Override
    public int seasonalFrequency() {
        return this.seasonalFrequency;
    }

    @Override
    public double[] stdErrors() {
        return (double[])this.stdErrors.clone();
    }

    @Override
    public ArimaCoefficients coefficients() {
        return this.coefficients;
    }

    @Override
    public ArimaOrder order() {
        return this.order;
    }

    @Override
    public double logLikelihood() {
        return this.modelInfo.logLikelihood;
    }

    @Override
    public double aic() {
        return this.modelInfo.aic;
    }

    public String toString() {
        String newLine = System.lineSeparator();
        return newLine + this.order + newLine + this.modelInfo + newLine + this.coefficients + newLine + newLine + "fit using " + (Object)((Object)this.fittingStrategy);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        ArimaModel that = (ArimaModel)o;
        if (this.seasonalFrequency != that.seasonalFrequency) {
            return false;
        }
        if (!this.observations.equals(that.observations)) {
            return false;
        }
        if (!this.order.equals(that.order)) {
            return false;
        }
        if (!this.modelInfo.equals(that.modelInfo)) {
            return false;
        }
        if (!this.coefficients.equals(that.coefficients)) {
            return false;
        }
        return this.fittingStrategy == that.fittingStrategy;
    }

    public int hashCode() {
        int result = this.observations.hashCode();
        result = 31 * result + this.order.hashCode();
        result = 31 * result + this.modelInfo.hashCode();
        result = 31 * result + this.coefficients.hashCode();
        result = 31 * result + this.fittingStrategy.hashCode();
        result = 31 * result + this.seasonalFrequency;
        return result;
    }

    private static class OptimFunction
    extends AbstractMultivariateFunction {
        private final TimeSeries observations;
        private final ArimaOrder order;
        private final ArimaParameters parameters;
        private final Arima.FittingStrategy fittingStrategy;
        private final int seasonalFrequency;
        private final Matrix externalRegressors;

        private OptimFunction(TimeSeries observations, ArimaOrder order, ArimaParameters parameters, Arima.FittingStrategy fittingStrategy, Matrix externalRegressors, int seasonalFrequency) {
            this.observations = observations;
            this.order = order;
            this.parameters = parameters;
            this.fittingStrategy = fittingStrategy;
            this.externalRegressors = externalRegressors;
            this.seasonalFrequency = seasonalFrequency;
        }

        public final double at(Vector point) {
            ++this.functionEvaluations;
            double[] params = point.elements();
            this.parameters.setAutoRegressivePars(DoubleFunctions.slice((double[])params, (int)0, (int)this.order.p()));
            this.parameters.setMovingAveragePars(DoubleFunctions.slice((double[])params, (int)this.order.p(), (int)(this.order.p() + this.order.q())));
            this.parameters.setSeasonalAutoRegressivePars(DoubleFunctions.slice((double[])params, (int)(this.order.p() + this.order.q()), (int)(this.order.p() + this.order.q() + this.order.P())));
            this.parameters.setSeasonalMovingAveragePars(DoubleFunctions.slice((double[])params, (int)(this.order.p() + this.order.q() + this.order.P()), (int)(this.order.p() + this.order.q() + this.order.P() + this.order.Q())));
            if (this.order.constant().include()) {
                this.parameters.setAndScaleMean(params[this.order.sumARMA()]);
            }
            if (this.order.drift().include()) {
                this.parameters.setAndScaleDrift(params[this.order.sumARMA() + this.order.constant().asInt()]);
            }
            double[] arCoeffs = ArimaCoefficients.expandArCoefficients(this.parameters.getAutoRegressivePars(), this.parameters.getSeasonalAutoRegressivePars(), this.seasonalFrequency);
            double[] maCoeffs = ArimaCoefficients.expandMaCoefficients(this.parameters.getMovingAveragePars(), this.parameters.getSeasonalMovingAveragePars(), this.seasonalFrequency);
            Vector regressionParameters = Vector.from((double[])this.parameters.getRegressors(this.order));
            Vector regressionEffects = this.externalRegressors.times(regressionParameters);
            TimeSeries armaSeries = this.observations.minus(regressionEffects.elements());
            if (this.fittingStrategy == Arima.FittingStrategy.ML || this.fittingStrategy == Arima.FittingStrategy.CSSML) {
                double[] delta = ArimaModel.getDelta(this.order, this.seasonalFrequency);
                ArimaKalmanFilter.KalmanOutput output = ArimaModel.kalmanFit(armaSeries, arCoeffs, maCoeffs, delta);
                return 0.5 * (Math.log(output.sigma2()) + output.sumLog() / (double)output.n());
            }
            TimeSeries differencedSeries = armaSeries.difference(1, this.order.d()).difference(this.seasonalFrequency, this.order.D());
            ModelInformation info = ArimaModel.fitCSS(differencedSeries, arCoeffs, maCoeffs, this.order.npar());
            return 0.5 * Math.log(info.sigma2);
        }

        public String toString() {
            String newLine = System.lineSeparator();
            return this.order + newLine + "fittingStrategy: " + (Object)((Object)this.fittingStrategy) + newLine + "seasonalFrequency: " + this.seasonalFrequency + newLine + "parameters" + this.parameters;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || ((Object)((Object)this)).getClass() != o.getClass()) {
                return false;
            }
            OptimFunction that = (OptimFunction)((Object)o);
            if (this.seasonalFrequency != that.seasonalFrequency) {
                return false;
            }
            if (!this.observations.equals(that.observations)) {
                return false;
            }
            if (!this.order.equals(that.order)) {
                return false;
            }
            if (this.fittingStrategy != that.fittingStrategy) {
                return false;
            }
            return this.parameters == that.parameters;
        }

        public int hashCode() {
            int result = this.observations.hashCode();
            result = 31 * result + this.order.hashCode();
            result = 31 * result + this.fittingStrategy.hashCode();
            result = 31 * result + this.seasonalFrequency;
            result = 31 * result + this.parameters.hashCode();
            return result;
        }
    }

    static class ModelInformation {
        private final double sigma2;
        private final double logLikelihood;
        private final double aic;
        private final double[] residuals;
        private final double[] fitted;

        ModelInformation(int npar, double sigma2, double logLikelihood, double[] residuals, double[] fitted) {
            this.sigma2 = sigma2;
            this.logLikelihood = logLikelihood;
            this.aic = (double)(2 * npar) - 2.0 * logLikelihood;
            this.residuals = (double[])residuals.clone();
            this.fitted = (double[])fitted.clone();
        }

        public String toString() {
            String newLine = System.lineSeparator();
            DecimalFormat numFormatter = new DecimalFormat("#0.0000");
            return newLine + "sigma2: " + numFormatter.format(this.sigma2) + newLine + "logLikelihood: " + numFormatter.format(this.logLikelihood) + newLine + "AIC: " + numFormatter.format(this.aic);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            ModelInformation that = (ModelInformation)o;
            if (Double.compare(that.sigma2, this.sigma2) != 0) {
                return false;
            }
            if (Double.compare(that.logLikelihood, this.logLikelihood) != 0) {
                return false;
            }
            if (Double.compare(that.aic, this.aic) != 0) {
                return false;
            }
            if (!Arrays.equals(this.residuals, that.residuals)) {
                return false;
            }
            return Arrays.equals(this.fitted, that.fitted);
        }

        public int hashCode() {
            long temp = Double.doubleToLongBits(this.sigma2);
            int result = (int)(temp ^ temp >>> 32);
            temp = Double.doubleToLongBits(this.logLikelihood);
            result = 31 * result + (int)(temp ^ temp >>> 32);
            temp = Double.doubleToLongBits(this.aic);
            result = 31 * result + (int)(temp ^ temp >>> 32);
            result = 31 * result + Arrays.hashCode(this.residuals);
            result = 31 * result + Arrays.hashCode(this.fitted);
            return result;
        }
    }
}

