/*
 * Decompiled with CFR 0.152.
 */
package com.github.silaev.mongodb.replicaset;

import com.github.dockerjava.api.DockerClient;
import com.github.dockerjava.api.model.Capability;
import com.github.dockerjava.api.model.HostConfig;
import com.github.dockerjava.api.model.Info;
import com.github.silaev.mongodb.replicaset.converter.impl.MongoNodeToMongoSocketAddressConverter;
import com.github.silaev.mongodb.replicaset.converter.impl.StringToMongoRsStatusConverter;
import com.github.silaev.mongodb.replicaset.converter.impl.UserInputToApplicationPropertiesConverter;
import com.github.silaev.mongodb.replicaset.core.Generated;
import com.github.silaev.mongodb.replicaset.exception.IncorrectUserInputException;
import com.github.silaev.mongodb.replicaset.exception.MongoNodeInitializationException;
import com.github.silaev.mongodb.replicaset.model.ApplicationProperties;
import com.github.silaev.mongodb.replicaset.model.MongoDbVersion;
import com.github.silaev.mongodb.replicaset.model.MongoNode;
import com.github.silaev.mongodb.replicaset.model.MongoRsStatus;
import com.github.silaev.mongodb.replicaset.model.MongoSocketAddress;
import com.github.silaev.mongodb.replicaset.model.Pair;
import com.github.silaev.mongodb.replicaset.model.ReplicaSetMemberState;
import com.github.silaev.mongodb.replicaset.model.UserInputProperties;
import com.github.silaev.mongodb.replicaset.util.StringUtils;
import eu.rekawek.toxiproxy.model.ToxicDirection;
import java.io.IOException;
import java.time.Duration;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import lombok.NonNull;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.testcontainers.DockerClientFactory;
import org.testcontainers.containers.Container;
import org.testcontainers.containers.GenericContainer;
import org.testcontainers.containers.Network;
import org.testcontainers.containers.ToxiproxyContainer;
import org.testcontainers.containers.wait.strategy.Wait;
import org.testcontainers.containers.wait.strategy.WaitStrategy;
import org.testcontainers.lifecycle.Startable;

public class MongoDbReplicaSet
implements Startable,
AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(MongoDbReplicaSet.class);
    public static final int MAX_VOTING_MEMBERS = 7;
    public static final Comparator<MongoSocketAddress> COMPARATOR_MAPPED_PORT = Comparator.comparing(MongoSocketAddress::getMappedPort);
    public static final String RECONFIG_RS_MSG = "Reconfiguring a replica set";
    static final String STATUS_COMMAND = "rs.status()";
    static final int CONTAINER_EXIT_CODE_OK = 0;
    private static final String SHOPIFY_TOXIPROXY_IMAGE = "shopify/toxiproxy:2.1.3";
    private static final String DEAD_LETTER_DB_NAME = "dead_letter";
    private static final String CLASS_NAME = MongoDbReplicaSet.class.getCanonicalName();
    private static final String LOCALHOST = "localhost";
    private static final String DOCKER_HOST_WORKAROUND = "dockerhost";
    private static final String DOCKER_HOST_INTERNAL = "host.docker.internal";
    private static final int MONGO_DB_INTERNAL_PORT = 27017;
    private static final String MONGO_ARBITER_NODE_NAME = "mongo-arbiter";
    private static final String DOCKER_HOST_CONTAINER_NAME = "qoomon/docker-host:2.4.0";
    private static final String TOXIPROXY_CONTAINER_NAME = "toxiproxy";
    private static final MongoDbVersion FIRST_SUPPORTED_MONGODB_VERSION = MongoDbVersion.of(3, 6, 14);
    private static final boolean MOVE_FORWARD = true;
    private static final boolean STOP_PIPELINE = false;
    private static final String READ_PREFERENCE_PRIMARY = "primary";
    private static final String RS_STATUS_MEMBERS_DEFINED_CONDITION = "rs.status().ok === 1 && rs.status().members !== undefined && ";
    private static final String RS_EXCEPTION = "throw new Error('Replica set status is not ok, errmsg: ' + rs.status().errmsg + ', codeName: ' + rs.status().codeName);";
    private static final String MONGODB_DATABASE_NAME_DEFAULT = "test";
    private static final int RECONFIG_MAX_TIME_MS = 10000;
    private final StringToMongoRsStatusConverter statusConverter;
    private final MongoNodeToMongoSocketAddressConverter socketAddressConverter;
    private final ApplicationProperties properties;
    private final NavigableMap<MongoSocketAddress, GenericContainer> workingNodeStore;
    private final Map<MongoSocketAddress, ToxiproxyContainer.ContainerProxy> toxyNodeStore;
    private final Map<String, Pair<GenericContainer, MongoSocketAddress>> supplementaryNodeStore;
    private final Map<MongoSocketAddress, Pair<Boolean, GenericContainer>> disconnectedNodeStore;
    private final Network network;

    private MongoDbReplicaSet(Integer replicaSetNumber, Integer awaitNodeInitAttempts, String propertyFileName, String mongoDockerImageName, Boolean addArbiter, Boolean addToxiproxy, Integer slaveDelayTimeout, Integer slaveDelayNumber, Boolean useHostDockerInternal, List<String> commandLineOptions) {
        UserInputToApplicationPropertiesConverter propertyConverter = new UserInputToApplicationPropertiesConverter();
        this.properties = propertyConverter.convert(UserInputProperties.builder().replicaSetNumber(replicaSetNumber).awaitNodeInitAttempts(awaitNodeInitAttempts).propertyFileName(propertyFileName).mongoDockerImageName(mongoDockerImageName).addArbiter(addArbiter).addToxiproxy(addToxiproxy).slaveDelayTimeout(slaveDelayTimeout).slaveDelayNumber(slaveDelayNumber).useHostDockerInternal(useHostDockerInternal).commandLineOptions(commandLineOptions).build());
        this.statusConverter = new StringToMongoRsStatusConverter();
        this.socketAddressConverter = new MongoNodeToMongoSocketAddressConverter();
        this.workingNodeStore = new ConcurrentSkipListMap<MongoSocketAddress, GenericContainer>(COMPARATOR_MAPPED_PORT);
        this.supplementaryNodeStore = new ConcurrentHashMap<String, Pair<GenericContainer, MongoSocketAddress>>();
        this.disconnectedNodeStore = new ConcurrentHashMap<MongoSocketAddress, Pair<Boolean, GenericContainer>>();
        this.toxyNodeStore = new ConcurrentHashMap<MongoSocketAddress, ToxiproxyContainer.ContainerProxy>();
        this.network = Network.newNetwork();
    }

    MongoDbReplicaSet(StringToMongoRsStatusConverter statusConverter, NavigableMap<MongoSocketAddress, GenericContainer> workingNodeStore, Map<String, Pair<GenericContainer, MongoSocketAddress>> supplementaryNodeStore, Map<MongoSocketAddress, Pair<Boolean, GenericContainer>> disconnectedNodeStore, Map<MongoSocketAddress, ToxiproxyContainer.ContainerProxy> toxyNodeStore, Network network) {
        UserInputToApplicationPropertiesConverter propertyConverter = new UserInputToApplicationPropertiesConverter();
        this.properties = propertyConverter.convert(UserInputProperties.builder().build());
        this.statusConverter = statusConverter;
        this.socketAddressConverter = new MongoNodeToMongoSocketAddressConverter();
        this.workingNodeStore = workingNodeStore;
        this.supplementaryNodeStore = supplementaryNodeStore;
        this.disconnectedNodeStore = disconnectedNodeStore;
        this.toxyNodeStore = toxyNodeStore;
        this.network = network;
    }

    @Override
    public void close() {
        this.stop();
    }

    public String getReplicaSetUrl() {
        this.verifyWorkingNodeStoreIsNotEmpty();
        return this.buildMongoRsUrl(READ_PREFERENCE_PRIMARY);
    }

    public String getReplicaSetUrl(String readPreference) {
        this.verifyWorkingNodeStoreIsNotEmpty();
        return this.buildMongoRsUrl(readPreference);
    }

    public MongoRsStatus getMongoRsStatus() {
        this.verifyWorkingNodeStoreIsNotEmpty();
        return this.statusConverter.convert(this.execMongoDbCommandInContainer((GenericContainer)this.workingNodeStore.entrySet().iterator().next().getValue(), STATUS_COMMAND).getStdout());
    }

    public synchronized void stop() {
        Stream.concat(this.disconnectedNodeStore.values().stream().map(Pair::getRight), Stream.concat(this.supplementaryNodeStore.values().stream().map(Pair::getLeft), this.workingNodeStore.values().stream())).forEach(Startable::stop);
        this.disconnectedNodeStore.clear();
        this.supplementaryNodeStore.clear();
        this.workingNodeStore.clear();
        this.toxyNodeStore.clear();
        this.network.close();
    }

    public boolean isEnabled() {
        return this.properties.isEnabled();
    }

    public String mongoDockerImageName() {
        return this.properties.getMongoDockerImageName();
    }

    public int getReplicaSetNumber() {
        return this.properties.getReplicaSetNumber();
    }

    public int getAwaitNodeInitAttempts() {
        return this.properties.getAwaitNodeInitAttempts();
    }

    public String getMongoDockerImageName() {
        return this.properties.getMongoDockerImageName();
    }

    public boolean getAddArbiter() {
        return this.properties.isAddArbiter();
    }

    public boolean getAddToxiproxy() {
        return this.properties.isAddToxiproxy();
    }

    public int getSlaveDelayTimeout() {
        return this.properties.getSlaveDelayTimeout();
    }

    public int getSlaveDelayNumber() {
        return this.properties.getSlaveDelayNumber();
    }

    public boolean getUseHostDockerInternal() {
        return this.properties.isUseHostDockerInternal();
    }

    private String getDockerHostName() {
        return this.getUseHostDockerInternal() ? DOCKER_HOST_INTERNAL : DOCKER_HOST_WORKAROUND;
    }

    private String[] buildMongoEvalCommand(String command) {
        return new String[]{"mongo", "--eval", command};
    }

    public synchronized void start() {
        if (this.properties.isEnabled()) {
            int attempt = 0;
            Exception lastException = null;
            boolean doContinue = true;
            int maxAttempts = 3;
            while (doContinue && attempt < 3) {
                log.debug("Provisioning a replica set, attempt: {} out of {}. Please, wait.", (Object)(attempt + 1), (Object)3);
                try {
                    this.startInternal();
                    doContinue = false;
                }
                catch (IncorrectUserInputException e) {
                    throw e;
                }
                catch (Exception e) {
                    this.stop();
                    lastException = e;
                    ++attempt;
                }
            }
            if (doContinue) {
                throw new MongoNodeInitializationException("Retry limit hit with exception", lastException);
            }
        } else {
            log.info("{} is disabled", (Object)CLASS_NAME);
        }
    }

    public void startInternal() {
        this.decideOnDockerHost();
        boolean addExtraHost = this.shouldAddExtraHost();
        ToxiproxyContainer toxiproxyContainer = null;
        if (this.getAddToxiproxy()) {
            toxiproxyContainer = this.getAndStartToxiproxyContainer();
            this.supplementaryNodeStore.put(TOXIPROXY_CONTAINER_NAME, Pair.of(toxiproxyContainer, null));
        }
        int replicaSetNumber = this.properties.getReplicaSetNumber();
        for (int i = 0; i < replicaSetNumber; ++i) {
            GenericContainer mongoContainer = this.getAndStartMongoDbContainer(this.network, addExtraHost);
            Pair<ToxiproxyContainer.ContainerProxy, Integer> pair = this.getContainerProxyAndPort(mongoContainer, toxiproxyContainer);
            MongoSocketAddress mongoSocketAddress = this.getMongoSocketAddress(mongoContainer.getContainerIpAddress(), pair.getRight());
            this.workingNodeStore.put(mongoSocketAddress, mongoContainer);
            if (!this.getAddToxiproxy()) continue;
            this.toxyNodeStore.put(mongoSocketAddress, pair.getLeft());
        }
        GenericContainer mongoContainer = this.workingNodeStore.firstEntry().getValue();
        if (Objects.isNull(mongoContainer)) {
            throw new IllegalStateException("MongoDb container is not supposed to be null");
        }
        int awaitNodeInitAttempts = this.getAwaitNodeInitAttempts();
        GenericContainer masterNode = this.initMasterNode(mongoContainer, awaitNodeInitAttempts);
        if (this.getAddArbiter()) {
            this.addArbiterNode(this.network, toxiproxyContainer, masterNode, awaitNodeInitAttempts, addExtraHost);
        }
        log.debug("REPLICA SET STATUS:\n{}", (Object)this.execMongoDbCommandInContainer(mongoContainer, STATUS_COMMAND).getStdout());
    }

    private void decideOnDockerHost() {
        if (!this.getUseHostDockerInternal() && this.getReplicaSetNumber() > 1 && LOCALHOST.equals(this.getHostIpAddress())) {
            this.warnAboutTheNeedToModifyHostFile();
            this.supplementaryNodeStore.put(DOCKER_HOST_WORKAROUND, Pair.of(this.getAndRunDockerHostContainer(this.network, this.getDockerHostName()), null));
        }
    }

    private boolean shouldAddExtraHost() {
        boolean addExtraHost = false;
        if (this.getUseHostDockerInternal() && this.getReplicaSetNumber() > 1) {
            boolean isLinux;
            Pair<String, String> ipAddressAndOS = this.getHostIpAddressAndOS();
            boolean bl = isLinux = !ipAddressAndOS.getRight().toLowerCase(Locale.ENGLISH).contains("docker desktop");
            if (isLinux && LOCALHOST.equals(ipAddressAndOS.getLeft())) {
                addExtraHost = true;
            }
        }
        return addExtraHost;
    }

    private Pair<ToxiproxyContainer.ContainerProxy, Integer> getContainerProxyAndPort(GenericContainer mongoContainer, ToxiproxyContainer toxiproxyContainer) {
        int port;
        ToxiproxyContainer.ContainerProxy containerProxy = null;
        if (this.getAddToxiproxy()) {
            Objects.requireNonNull(toxiproxyContainer, "toxiproxyContainer is not supposed to be null");
            containerProxy = toxiproxyContainer.getProxy(mongoContainer, 27017);
            port = containerProxy.getProxyPort();
            log.debug("Real port: {}, proxy port: {}", (Object)mongoContainer.getMappedPort(27017), (Object)port);
        } else {
            Objects.requireNonNull(mongoContainer, "mongoContainer is not supposed to be null");
            port = mongoContainer.getMappedPort(27017);
        }
        return Pair.of(containerProxy, port);
    }

    Container.ExecResult execMongoDbCommandInContainer(GenericContainer mongoContainer, String command) {
        return mongoContainer.execInContainer(this.buildMongoEvalCommand(command));
    }

    private void warnAboutTheNeedToModifyHostFile() {
        log.warn("Please, check that the host file of your OS has 127.0.0.1 dockerhost. If you don't want to modify it, then consider the following:\n1) set replicaSetNumber to 1;\n2) use remote docker daemon;\n3) use local docker host running tests from inside a container with mapping the Docker socket.");
    }

    private String getHostIpAddress() {
        return Optional.ofNullable(DockerClientFactory.instance()).map(DockerClientFactory::dockerHostIpAddress).orElseThrow(() -> new IllegalStateException("The instance of the DockerClientFactory is not initialized"));
    }

    @NonNull
    private Pair<String, String> getHostIpAddressAndOS() {
        try {
            DockerClientFactory clientFactory = DockerClientFactory.instance();
            DockerClient client = clientFactory.client();
            Info dockerInfo = (Info)client.infoCmd().exec();
            String hostIp = clientFactory.dockerHostIpAddress();
            Objects.requireNonNull(hostIp, "DockerClient: dockerHostIpAddress is not supposed to be null");
            String os = dockerInfo.getOperatingSystem();
            Objects.requireNonNull(os, "DockerClient: operatingSystem is not supposed to be null");
            return Pair.of(hostIp, os);
        }
        catch (Exception e) {
            throw new IllegalStateException("Cannot getHostIpAddressAndOS", e);
        }
    }

    private GenericContainer initMasterNode(GenericContainer mongoContainer, int awaitNodeInitAttempts) {
        log.debug("Initializing a {} node replica set...", (Object)this.getReplicaSetNumber());
        Container.ExecResult execResultInitRs = this.execMongoDbCommandInContainer(mongoContainer, this.getMongoReplicaSetInitializer());
        String stdoutInitRs = execResultInitRs.getStdout();
        log.debug("initMasterNode => execResultInitRs: {}", (Object)stdoutInitRs);
        this.checkMongoNodeExitCodeAndStatus(execResultInitRs, "initializing a master node");
        this.verifyVersion(stdoutInitRs);
        return this.checkAndGetMasterNode(mongoContainer, awaitNodeInitAttempts);
    }

    private GenericContainer checkAndGetMasterNode(GenericContainer mongoContainer, int awaitNodeInitAttempts) {
        return this.getReplicaSetNumber() == 1 ? this.checkAndGetMasterNodeInSingleNodeReplicaSet(mongoContainer, awaitNodeInitAttempts) : this.checkAndGetMasterNodeInMultiNodeReplicaSet(mongoContainer, awaitNodeInitAttempts);
    }

    void verifyVersion(String stdoutInitRs) {
        MongoDbVersion inputVersion = this.statusConverter.convert(stdoutInitRs).getVersion();
        if (this.checkVersionPart(inputVersion.getMajor(), FIRST_SUPPORTED_MONGODB_VERSION.getMajor()) && this.checkVersionPart(inputVersion.getMinor(), FIRST_SUPPORTED_MONGODB_VERSION.getMinor())) {
            this.checkVersionPart(inputVersion.getPatch(), FIRST_SUPPORTED_MONGODB_VERSION.getPatch());
        }
    }

    private boolean checkVersionPart(int inputVersion, int supportedVersion) {
        if (inputVersion == supportedVersion) {
            return true;
        }
        if (inputVersion > supportedVersion) {
            return false;
        }
        throw new IncorrectUserInputException(String.format("Please, use a MongoDB version that is more or equal to: %s", FIRST_SUPPORTED_MONGODB_VERSION));
    }

    private String buildWaitStopCondition(String condition) {
        return "!(rs.status().ok === 1 && rs.status().members !== undefined && " + condition + ")";
    }

    private GenericContainer checkAndGetMasterNodeInMultiNodeReplicaSet(GenericContainer mongoContainer, int awaitNodeInitAttempts) {
        log.debug("Searching for a master node in a replica set, up to {} attempts", (Object)awaitNodeInitAttempts);
        Container.ExecResult execResultWaitForAnyMaster = this.waitForCondition(mongoContainer, this.buildWaitStopCondition("rs.status().members.filter(o => o.state === 1).length === 1"), awaitNodeInitAttempts, "Searching for a master node");
        log.debug(execResultWaitForAnyMaster.getStdout());
        this.checkMongoNodeExitCodeAfterWaiting(mongoContainer, execResultWaitForAnyMaster, "master candidate", awaitNodeInitAttempts);
        GenericContainer masterNode = this.findMasterElected(mongoContainer);
        log.debug("Verifying that a node is a master one, up to {} attempts", (Object)awaitNodeInitAttempts);
        Container.ExecResult execResultWaitForMaster = this.waitForCondition(masterNode, "db.runCommand( { isMaster: 1 } ).ismaster==false", awaitNodeInitAttempts, "verifying that a node is a master one");
        this.checkMongoNodeExitCodeAfterWaiting(masterNode, execResultWaitForMaster, "master", awaitNodeInitAttempts);
        return masterNode;
    }

    private GenericContainer findMasterElected(GenericContainer mongoContainer) {
        log.debug("Waiting for a single master node up to {} attempts", (Object)this.getAwaitNodeInitAttempts());
        Container.ExecResult execResultMasterAddress = this.execMongoDbCommandInContainer(mongoContainer, String.format("var attempt = 0; while (attempt <= %d) { print('Waiting for a single master node up to ' + attempt); sleep(1000); attempt++;if (%s rs.status().members.filter(o => o.state === 1).length === 1) { rs.status().members.find(o => o.state === 1).name; break; }}; if(attempt > %d) {quit(1)};", this.getAwaitNodeInitAttempts(), RS_STATUS_MEMBERS_DEFINED_CONDITION, this.getAwaitNodeInitAttempts()));
        this.checkMongoNodeExitCode(execResultMasterAddress, "finding a master node");
        String stdout = execResultMasterAddress.getStdout();
        MongoSocketAddress mongoSocketAddress = Optional.ofNullable(this.statusConverter.extractRawPayloadFromMongoDBShell(stdout)).map(StringUtils::getArrayByDelimiter).filter(a -> ((String[])a).length == 2).map(a -> {
            int port = Integer.parseInt(a[1]);
            return MongoSocketAddress.builder().ip(a[0]).replSetPort(port).mappedPort(port).build();
        }).orElseThrow(() -> new IllegalArgumentException(String.format("Cannot find an address in a MongoDb reply:%n %s", stdout)));
        log.debug("Found the master elected: {}", (Object)mongoSocketAddress);
        return this.extractGenericContainer(mongoSocketAddress, this.workingNodeStore);
    }

    @NotNull
    private String buildJsIfStatement(String condition, String thenClause) {
        return String.format("if (%s) {%s} else {%s}", condition, thenClause, RS_EXCEPTION);
    }

    private <T> T extractGenericContainer(MongoSocketAddress mongoSocketAddress, Map<MongoSocketAddress, T> nodeStore) {
        return Optional.ofNullable(nodeStore.get(mongoSocketAddress)).orElseThrow(() -> new IllegalStateException(String.format("Cannot find a node in a node store by %s", mongoSocketAddress)));
    }

    private Pair<Boolean, GenericContainer> extractWorkingOrArbiterGenericContainer(@NotNull MongoSocketAddress mongoSocketAddress) {
        return Optional.ofNullable(this.workingNodeStore.get(mongoSocketAddress)).map(n -> Pair.of(Boolean.TRUE, n)).orElseGet(() -> Optional.ofNullable(this.supplementaryNodeStore.get(MONGO_ARBITER_NODE_NAME)).filter(arbiter -> mongoSocketAddress.equals(arbiter.getRight())).map(n -> Pair.of(Boolean.FALSE, n.getLeft())).orElseThrow(() -> new IllegalStateException(String.format("Cannot find a node in a working and arbiter node store by %s", mongoSocketAddress))));
    }

    private GenericContainer checkAndGetMasterNodeInSingleNodeReplicaSet(GenericContainer mongoContainer, int awaitNodeInitAttempts) {
        log.debug("Awaiting a master node, up to {} attempts", (Object)awaitNodeInitAttempts);
        Container.ExecResult execResultWaitForMaster = this.waitForCondition(mongoContainer, "db.runCommand( { isMaster: 1 } ).ismaster==false", awaitNodeInitAttempts, "awaiting for a node to be a master one");
        log.debug(execResultWaitForMaster.getStdout());
        this.checkMongoNodeExitCodeAfterWaiting(mongoContainer, execResultWaitForMaster, "master", awaitNodeInitAttempts);
        return mongoContainer;
    }

    private MongoSocketAddress getMongoSocketAddress(String containerIpAddress, int port) {
        if (LOCALHOST.equals(containerIpAddress)) {
            return this.getReplicaSetNumber() == 1 ? MongoSocketAddress.builder().ip(LOCALHOST).replSetPort(27017).mappedPort(port).build() : MongoSocketAddress.builder().ip(this.getDockerHostName()).replSetPort(port).mappedPort(port).build();
        }
        return MongoSocketAddress.builder().ip(containerIpAddress).replSetPort(port).mappedPort(port).build();
    }

    private void addArbiterNode(Network network, ToxiproxyContainer toxiproxyContainer, GenericContainer masterNode, int awaitNodeInitAttempts, boolean addExtraHost) {
        log.debug("Awaiting an arbiter node to be available, up to {} attempts", (Object)this.properties.getAwaitNodeInitAttempts());
        GenericContainer mongoContainerArbiter = this.getAndStartMongoDbContainer(network, addExtraHost);
        Pair<ToxiproxyContainer.ContainerProxy, Integer> pair = this.getContainerProxyAndPort(mongoContainerArbiter, toxiproxyContainer);
        MongoSocketAddress mongoSocketAddress = this.getMongoSocketAddress(mongoContainerArbiter.getContainerIpAddress(), pair.getRight());
        this.supplementaryNodeStore.put(MONGO_ARBITER_NODE_NAME, Pair.of(mongoContainerArbiter, mongoSocketAddress));
        if (this.getAddToxiproxy()) {
            this.toxyNodeStore.put(mongoSocketAddress, pair.getLeft());
        }
        Container.ExecResult execResultAddArbiter = this.execMongoDbCommandInContainer(masterNode, String.format("rs.addArb(\"%s:%d\")", mongoSocketAddress.getIp(), mongoSocketAddress.getReplSetPort()));
        log.debug("Add an arbiter node result: {}", (Object)execResultAddArbiter.getStdout());
        this.checkMongoNodeExitCodeAndStatus(execResultAddArbiter, "initializing an arbiter node");
        Container.ExecResult execResultWaitArbiter = this.waitForCondition(masterNode, this.buildWaitStopCondition("rs.status().members.find(o => o.state === 7) !== undefined"), awaitNodeInitAttempts, "awaiting an arbiter node to be up");
        log.debug("Wait for an arbiter node result: {}", (Object)execResultWaitArbiter.getStdout());
        this.checkMongoNodeExitCodeAfterWaiting(masterNode, execResultWaitArbiter, "arbiter", awaitNodeInitAttempts);
    }

    void checkMongoNodeExitCodeAfterWaiting(GenericContainer mongoContainer, Container.ExecResult execResultWaitForMaster, String nodeName, int awaitNodeInitAttempts) {
        if (execResultWaitForMaster.getExitCode() != 0) {
            String errorMessage = String.format("The %s node was not initialized in a set timeout: %d attempts. Replica set status: %s", nodeName, awaitNodeInitAttempts, this.execMongoDbCommandInContainer(mongoContainer, STATUS_COMMAND).getStdout());
            log.error(errorMessage);
            throw new MongoNodeInitializationException(errorMessage);
        }
    }

    void checkMongoNodeExitCode(Container.ExecResult execResult, String commandDescription) {
        Objects.requireNonNull(execResult);
        String stdout = execResult.getStdout();
        Objects.requireNonNull(stdout);
        if (execResult.getExitCode() != 0) {
            String errorMessage = String.format("Error occurred while %s: %s", commandDescription, execResult.getStdout());
            log.error(errorMessage);
            throw new MongoNodeInitializationException(errorMessage);
        }
    }

    void checkMongoNodeExitCodeAndStatus(Container.ExecResult execResult, String commandDescription) {
        Objects.requireNonNull(execResult);
        String stdout = execResult.getStdout();
        Objects.requireNonNull(stdout);
        if (execResult.getExitCode() != 0 || this.statusConverter.convert(stdout).getStatus() != 1) {
            String errorMessage = String.format("Error occurred while %s: %s", commandDescription, execResult.getStdout());
            log.error(errorMessage);
            throw new MongoNodeInitializationException(errorMessage);
        }
    }

    private Container.ExecResult waitForCondition(GenericContainer mongoContainer, String condition, int awaitNodeInitAttempts, String waitingMessage) {
        return this.execMongoDbCommandInContainer(mongoContainer, this.buildMongoWaitCommand(condition, awaitNodeInitAttempts, waitingMessage));
    }

    private String buildMongoWaitCommand(String condition, int attempts, String waitingMessage) {
        return String.format("var attempt = 0; while(%s) { if (attempt > %d) {quit(1);} print('%s ' + attempt); sleep(1000); attempt++;  }", condition, attempts, waitingMessage);
    }

    private String getMongoReplicaSetInitializer() {
        MongoSocketAddress[] addresses = this.workingNodeStore.keySet().toArray(new MongoSocketAddress[0]);
        int length = addresses.length;
        int slaveDelayTimeout = this.getSlaveDelayTimeout();
        int workingNodeNumber = this.getReplicaSetNumber() + (this.getAddArbiter() ? 1 : 0) - this.getSlaveDelayNumber();
        String replicaSetInitializer = IntStream.range(0, length).mapToObj(i -> {
            MongoSocketAddress address = addresses[i];
            if (slaveDelayTimeout > 0 && i > workingNodeNumber - 1) {
                return String.format("        {\"_id\": %d, \"host\": \"%s:%d\", \"slaveDelay\":%d, \"priority\": 0, \"hidden\": true}", i, address.getIp(), address.getReplSetPort(), slaveDelayTimeout);
            }
            return String.format("        {\"_id\": %d, \"host\": \"%s:%d\"}", i, address.getIp(), address.getReplSetPort());
        }).collect(Collectors.joining(",\n", "rs.initiate({\n    \"_id\": \"docker-rs\",\n    \"members\": [\n", "\n    ]\n});"));
        log.debug("replicaSetInitializer: {}", (Object)replicaSetInitializer);
        return "cfg = " + replicaSetInitializer + this.buildJsIfStatement("cfg.ok===1", "cfg");
    }

    private String buildMongoRsUrl(String readPreference) {
        return this.workingNodeStore.keySet().stream().map(a -> String.format("%s:%d", a.getIp(), a.getMappedPort())).collect(Collectors.joining(",", "mongodb://", String.format("/%s%s&readPreference=%s", MONGODB_DATABASE_NAME_DEFAULT, this.getReplicaSetNumber() == 1 ? "" : "?replicaSet=docker-rs", readPreference)));
    }

    @NonNull
    private GenericContainer getAndRunDockerHostContainer(Network network, String dockerHostName) {
        GenericContainer dockerHostContainer = new GenericContainer(DOCKER_HOST_CONTAINER_NAME).withCreateContainerCmdModifier(it -> it.withHostConfig(HostConfig.newHostConfig().withCapAdd(new Capability[]{Capability.NET_ADMIN, Capability.NET_RAW}).withNetworkMode(network.getId()))).withNetwork(network).withNetworkAliases(new String[]{dockerHostName}).waitingFor((WaitStrategy)Wait.forLogMessage((String)".*Forwarding ports.*", (int)1));
        dockerHostContainer.start();
        return dockerHostContainer;
    }

    @NonNull
    private GenericContainer getAndStartMongoDbContainer(Network network, boolean addExtraHost) {
        String[] commands = (String[])Stream.concat(Stream.of("--bind_ip", "0.0.0.0", "--replSet", "docker-rs"), this.properties.getCommandLineOptions().stream()).toArray(String[]::new);
        GenericContainer mongoDbContainer = new GenericContainer(this.properties.getMongoDockerImageName()).withNetwork(this.getReplicaSetNumber() == 1 ? null : network).withExposedPorts(new Integer[]{27017}).withCommand(commands).waitingFor((WaitStrategy)Wait.forListeningPort()).withStartupTimeout(Duration.ofSeconds(60L)).withStartupAttempts(3);
        if (addExtraHost) {
            mongoDbContainer.withExtraHost(DOCKER_HOST_INTERNAL, "host-gateway");
        }
        mongoDbContainer.start();
        return mongoDbContainer;
    }

    @NonNull
    private ToxiproxyContainer getAndStartToxiproxyContainer() {
        ToxiproxyContainer toxiproxy = (ToxiproxyContainer)((ToxiproxyContainer)((ToxiproxyContainer)new ToxiproxyContainer(SHOPIFY_TOXIPROXY_IMAGE).withNetwork(this.network)).withStartupTimeout(Duration.ofSeconds(60L))).withStartupAttempts(3);
        toxiproxy.start();
        return toxiproxy;
    }

    public void stopNode(MongoNode mongoNode) {
        this.validateFaultToleranceTestSupportAvailability();
        MongoSocketAddress mongoSocketAddress = this.socketAddressConverter.convert(mongoNode);
        Pair<Boolean, GenericContainer> pair = this.extractWorkingOrArbiterGenericContainer(mongoSocketAddress);
        Boolean isWorkingNode = pair.getLeft();
        GenericContainer genericContainer = pair.getRight();
        genericContainer.stop();
        this.removeNodeFromInternalStore(isWorkingNode, mongoSocketAddress);
        if (this.getAddToxiproxy()) {
            this.toxyNodeStore.remove(mongoSocketAddress);
        }
    }

    public void killNode(MongoNode mongoNode) {
        this.validateFaultToleranceTestSupportAvailability();
        MongoSocketAddress mongoSocketAddress = this.socketAddressConverter.convert(mongoNode);
        Pair<Boolean, GenericContainer> pair = this.extractWorkingOrArbiterGenericContainer(mongoSocketAddress);
        Boolean isWorkingNode = pair.getLeft();
        GenericContainer genericContainer = pair.getRight();
        DockerClientFactory.instance().client().killContainerCmd(genericContainer.getContainerId()).exec();
        this.removeNodeFromInternalStore(isWorkingNode, mongoSocketAddress);
        if (this.getAddToxiproxy()) {
            this.toxyNodeStore.remove(mongoSocketAddress);
        }
    }

    public synchronized void disconnectNodeFromNetwork(MongoNode mongoNode) {
        this.validateFaultToleranceTestSupportAvailability();
        MongoSocketAddress mongoSocketAddress = this.socketAddressConverter.convert(mongoNode);
        Pair<Boolean, GenericContainer> pair = this.extractWorkingOrArbiterGenericContainer(mongoSocketAddress);
        Boolean isWorkingNode = pair.getLeft();
        GenericContainer genericContainer = pair.getRight();
        if (this.getAddToxiproxy()) {
            this.extractGenericContainer(mongoSocketAddress, this.toxyNodeStore).setConnectionCut(true);
        } else {
            DockerClientFactory.instance().client().disconnectFromNetworkCmd().withContainerId(genericContainer.getContainerId()).withNetworkId(this.network.getId()).exec();
        }
        this.removeNodeFromInternalStore(isWorkingNode, mongoSocketAddress);
        this.disconnectedNodeStore.put(mongoSocketAddress, Pair.of(isWorkingNode, genericContainer));
    }

    @Generated
    void addLatencyToDownstream(MongoNode mongoNode, long latency) {
        this.validateFaultToleranceTestSupportAvailability();
        MongoSocketAddress mongoSocketAddress = this.socketAddressConverter.convert(mongoNode);
        try {
            this.extractGenericContainer(mongoSocketAddress, this.toxyNodeStore).toxics().latency(String.format("ADD_LATENCY_DOWNSTREAM_%d", mongoSocketAddress.getMappedPort()), ToxicDirection.DOWNSTREAM, latency).setLatency(latency);
        }
        catch (IOException e) {
            throw new RuntimeException("Could not control proxy", e);
        }
    }

    @Generated
    void removeLatencyFromDownstream(MongoNode mongoNode) {
        this.validateFaultToleranceTestSupportAvailability();
        MongoSocketAddress mongoSocketAddress = this.socketAddressConverter.convert(mongoNode);
        try {
            this.extractGenericContainer(mongoSocketAddress, this.toxyNodeStore).toxics().get(String.format("ADD_LATENCY_DOWNSTREAM_%d", mongoSocketAddress.getMappedPort())).remove();
        }
        catch (IOException e) {
            throw new RuntimeException("Could not control proxy", e);
        }
    }

    public synchronized void connectNodeToNetworkWithReconfiguration(MongoNode mongoNode) {
        if (this.getAddToxiproxy()) {
            throw new UnsupportedOperationException("Please, use connectNodeToNetwork with Toxiproxy");
        }
        this.validateFaultToleranceTestSupportAvailability();
        this.verifyWorkingNodeStoreIsNotEmpty();
        MongoSocketAddress disconnectedMongoSocketAddress = this.socketAddressConverter.convert(mongoNode);
        Pair<Boolean, GenericContainer> pair = this.extractGenericContainer(disconnectedMongoSocketAddress, this.disconnectedNodeStore);
        Boolean isWorkingNode = pair.getLeft();
        GenericContainer disconnectedNode = pair.getRight();
        DockerClientFactory.instance().client().connectToNetworkCmd().withContainerId(disconnectedNode.getContainerId()).withNetworkId(this.network.getId()).exec();
        this.restartGenericContainer(disconnectedNode);
        this.reconfigureReplSetRemoveDownAndUnknownNodes();
        this.waitForMaster();
        GenericContainer masterNode = this.findMasterElected((GenericContainer)this.workingNodeStore.values().iterator().next());
        MongoSocketAddress newMongoSocketAddress = this.getMongoSocketAddress(mongoNode.getIp(), disconnectedNode.getMappedPort(27017));
        this.addNodeToReplSetConfig(isWorkingNode, masterNode, newMongoSocketAddress);
        this.addNodeToInternalStore(isWorkingNode, newMongoSocketAddress, disconnectedNode);
        this.disconnectedNodeStore.remove(disconnectedMongoSocketAddress);
    }

    private void removeNodeFromInternalStore(Boolean isWorkingNode, MongoSocketAddress disconnectedMongoSocketAddress) {
        if (Boolean.TRUE.equals(isWorkingNode)) {
            this.workingNodeStore.remove(disconnectedMongoSocketAddress);
        } else {
            this.supplementaryNodeStore.remove(MONGO_ARBITER_NODE_NAME);
        }
    }

    private void addNodeToInternalStore(Boolean isWorkingNode, MongoSocketAddress disconnectedMongoSocketAddress, GenericContainer disconnectedNode) {
        if (Boolean.TRUE.equals(isWorkingNode)) {
            this.workingNodeStore.put(disconnectedMongoSocketAddress, disconnectedNode);
        } else {
            this.supplementaryNodeStore.put(MONGO_ARBITER_NODE_NAME, Pair.of(disconnectedNode, disconnectedMongoSocketAddress));
        }
    }

    public synchronized void connectNodeToNetwork(MongoNode mongoNode) {
        this.connectNodeToNetwork(mongoNode, true, false);
    }

    public synchronized void connectNodeToNetworkWithForceRemoval(MongoNode mongoNode) {
        this.connectNodeToNetwork(mongoNode, true, true);
    }

    public synchronized void connectNodeToNetworkWithoutRemoval(MongoNode mongoNode) {
        this.connectNodeToNetwork(mongoNode, false, false);
    }

    private synchronized void connectNodeToNetwork(MongoNode mongoNode, boolean remove, boolean force) {
        this.validateFaultToleranceTestSupportAvailability();
        this.verifyWorkingNodeStoreIsNotEmpty();
        MongoSocketAddress disconnectedMongoSocketAddress = this.socketAddressConverter.convert(mongoNode);
        Pair<Boolean, GenericContainer> pair = this.extractGenericContainer(disconnectedMongoSocketAddress, this.disconnectedNodeStore);
        Boolean isWorkingNode = pair.getLeft();
        GenericContainer disconnectedNode = pair.getRight();
        if (this.getAddToxiproxy()) {
            if (force) {
                throw new IllegalArgumentException("addToxiproxy does not work with force");
            }
            this.extractGenericContainer(disconnectedMongoSocketAddress, this.toxyNodeStore).setConnectionCut(false);
            this.addNodeToInternalStore(isWorkingNode, disconnectedMongoSocketAddress, disconnectedNode);
        } else {
            DockerClientFactory.instance().client().connectToNetworkCmd().withContainerId(disconnectedNode.getContainerId()).withNetworkId(this.network.getId()).exec();
            this.restartGenericContainer(disconnectedNode);
            this.waitForMaster();
            GenericContainer masterNode = this.findMasterElected((GenericContainer)this.workingNodeStore.values().iterator().next());
            if (remove) {
                if (force) {
                    this.removeNodeFromReplSetConfigWithForce(disconnectedMongoSocketAddress, masterNode);
                } else {
                    this.removeNodeFromReplSetConfig(disconnectedMongoSocketAddress, masterNode);
                }
            }
            MongoSocketAddress newMongoSocketAddress = this.getMongoSocketAddress(mongoNode.getIp(), disconnectedNode.getMappedPort(27017));
            this.addNodeToReplSetConfig(isWorkingNode, masterNode, newMongoSocketAddress);
            this.addNodeToInternalStore(isWorkingNode, newMongoSocketAddress, disconnectedNode);
        }
        this.disconnectedNodeStore.remove(disconnectedMongoSocketAddress);
    }

    private void addNodeToReplSetConfig(Boolean isWorkingNode, GenericContainer masterNode, MongoSocketAddress mongoSocketAddress) {
        if (Boolean.TRUE.equals(isWorkingNode)) {
            this.addWorkingNodeToReplSetConfig(masterNode, mongoSocketAddress);
        } else {
            this.addArbiterNodeToReplSetConfig(masterNode, mongoSocketAddress);
        }
    }

    public void reconfigureReplSetRemoveDownAndUnknownNodes() {
        List<MongoNode> members = this.getMongoRsStatus().getMembers();
        String replicaSetReConfig = this.getReplicaSetReConfigRemoveDownAndUnknownNodes(members);
        log.debug("Reconfiguring a node replica set as per: {}", (Object)replicaSetReConfig);
        Container.ExecResult execResult = this.execMongoDbCommandInContainer((GenericContainer)this.workingNodeStore.values().iterator().next(), replicaSetReConfig);
        log.debug(execResult.getStdout());
        this.checkMongoNodeExitCode(execResult, RECONFIG_RS_MSG);
    }

    public void reconfigureReplSetToDefaults() {
        CompletableFuture<Void> cf = CompletableFuture.runAsync(this::reconfigureReplSetToDefaultsInternal);
        try {
            cf.get(20000L, TimeUnit.MILLISECONDS);
        }
        catch (TimeoutException e) {
            throw new MongoNodeInitializationException("Timeout exceeded for reconfigureReplSetToDefaults", e);
        }
    }

    private void reconfigureReplSetToDefaultsInternal() {
        this.verifyWorkingNodeStoreIsNotEmpty();
        String replicaSetReConfig = this.getReplicaSetReConfigUnsetSlaveDelay();
        log.debug("Reconfiguring a replica set as per: {}", (Object)replicaSetReConfig);
        Container.ExecResult execResult = this.execMongoDbCommandInContainer(this.findMasterElected((GenericContainer)this.workingNodeStore.values().iterator().next()), replicaSetReConfig);
        log.debug(execResult.getStdout());
        this.checkMongoNodeExitCodeAndStatus(execResult, RECONFIG_RS_MSG);
    }

    @Generated
    void dropConnections(List<MongoNode> members) {
        this.operateOnConnections(members, 1);
    }

    @Generated
    void enableConnections(List<MongoNode> members) {
        this.operateOnConnections(members, 0);
    }

    @Generated
    private void operateOnConnections(List<MongoNode> members, int drop) {
        this.verifyWorkingNodeStoreIsNotEmpty();
        String replicaSetReConfig = this.getDropConnectionsCommand(members, drop);
        log.debug("Dropping connections: {}", (Object)replicaSetReConfig);
        Container.ExecResult execResult = this.execMongoDbCommandInContainer((GenericContainer)this.workingNodeStore.values().iterator().next(), replicaSetReConfig);
        log.debug(execResult.getStdout());
        this.checkMongoNodeExitCodeAndStatus(execResult, "Dropping connections");
    }

    @Generated
    private String getDropConnectionsCommand(List<MongoNode> members, int drop) {
        return members.stream().map(member -> String.format("\"%s:%d\"", member.getIp(), member.getPort())).collect(Collectors.joining(",", String.format("db.adminCommand({\"dropConnections\" : %d, \"hostAndPort\":[", drop), "]});"));
    }

    private String getReplicaSetReConfigUnsetSlaveDelay() {
        int workingNodeNumber = this.getReplicaSetNumber() + (this.getAddArbiter() ? 1 : 0) - this.getSlaveDelayNumber();
        return IntStream.rangeClosed(workingNodeNumber + 1, this.getMongoRsStatus().getMembers().size()).mapToObj(i -> String.format("cfg.members[%d].slaveDelay=0;cfg.members[%d].priority=1;cfg.members[%d].hidden=false", i - 1, i - 1, i - 1)).collect(Collectors.joining(";\n", "cfg = rs.conf();\n", String.format(";%nrs.reconfig(cfg, {force : true, maxTimeMS: %d})", 10000)));
    }

    private String getReplicaSetReConfigRemoveDownAndUnknownNodes(List<MongoNode> members) {
        return IntStream.range(0, members.size()).filter(i -> {
            ReplicaSetMemberState memberState = ((MongoNode)members.get(i)).getState();
            return memberState != ReplicaSetMemberState.DOWN && memberState != ReplicaSetMemberState.UNKNOWN;
        }).mapToObj(i -> String.format("cfg.members[%d]", i)).collect(Collectors.joining(",", "cfg = rs.conf();\ncfg.members = [", String.format("];%nrs.reconfig(cfg, {force : true, maxTimeMS: %d})", 10000)));
    }

    private String getReplicaSetReConfigRemoveNode(MongoSocketAddress mongoSocketAddress) {
        List<MongoNode> members = this.getMongoRsStatus().getMembers();
        return IntStream.range(0, members.size()).filter(i -> {
            MongoNode mongoNode = (MongoNode)members.get(i);
            return !Objects.equals(mongoSocketAddress.getIp(), mongoNode.getIp()) || !Objects.equals(mongoSocketAddress.getMappedPort(), mongoNode.getPort());
        }).mapToObj(i -> String.format("cfg.members[%d]", i)).collect(Collectors.joining(",", "cfg = rs.conf();\ncfg.members = [", String.format("];%nrs.reconfig(cfg, {force : true, maxTimeMS: %d})", 10000)));
    }

    private void validateFaultToleranceTestSupportAvailability() {
        if (this.getReplicaSetNumber() == 1) {
            throw new IllegalStateException("This operation is not supported for a single node replica set. Please, construct at least a Primary with Two Secondary Members(P-S-S) or Primary with a Secondary and an Arbiter (PSA) replica set");
        }
    }

    private void addWorkingNodeToReplSetConfig(GenericContainer masterNode, MongoSocketAddress newMongoSocketAddress) {
        Container.ExecResult execResultAddNode = this.execMongoDbCommandInContainer(masterNode, String.format("rs.add(\"%s:%d\")", newMongoSocketAddress.getIp(), newMongoSocketAddress.getMappedPort()));
        log.debug("Add a node: {} to a replica set, stdout: {}", (Object)newMongoSocketAddress, (Object)execResultAddNode.getStdout());
        this.checkMongoNodeExitCodeAndStatus(execResultAddNode, "Adding a node");
    }

    private void addArbiterNodeToReplSetConfig(GenericContainer masterNode, MongoSocketAddress newMongoSocketAddress) {
        Container.ExecResult execResultAddNode = this.execMongoDbCommandInContainer(masterNode, String.format("rs.addArb(\"%s:%d\")", newMongoSocketAddress.getIp(), newMongoSocketAddress.getMappedPort()));
        log.debug("Add a node: {} to a replica set, stdout: {}", (Object)newMongoSocketAddress, (Object)execResultAddNode.getStdout());
        this.checkMongoNodeExitCode(execResultAddNode, "Adding a node");
    }

    public void removeNodeFromReplSetConfig(MongoNode mongoNodeToRemove) {
        this.validateFaultToleranceTestSupportAvailability();
        this.verifyWorkingNodeStoreIsNotEmpty();
        this.removeNodeFromReplSetConfig(this.socketAddressConverter.convert(mongoNodeToRemove), this.findMasterElected((GenericContainer)this.workingNodeStore.values().iterator().next()));
    }

    public void removeNodeFromReplSetConfigWithForce(MongoNode mongoNodeToRemove) {
        this.validateFaultToleranceTestSupportAvailability();
        this.verifyWorkingNodeStoreIsNotEmpty();
        this.removeNodeFromReplSetConfigWithForce(this.socketAddressConverter.convert(mongoNodeToRemove), this.findMasterElected((GenericContainer)this.workingNodeStore.values().iterator().next()));
    }

    private void removeNodeFromReplSetConfigWithForce(MongoSocketAddress mongoSocketAddressToRemove, GenericContainer masterNode) {
        String replicaSetReConfig = this.getReplicaSetReConfigRemoveNode(mongoSocketAddressToRemove);
        log.debug("Reconfiguring a node replica set as per: {}", (Object)replicaSetReConfig);
        Container.ExecResult execResult = this.execMongoDbCommandInContainer(masterNode, replicaSetReConfig);
        log.debug(execResult.getStdout());
        this.checkMongoNodeExitCodeAndStatus(execResult, RECONFIG_RS_MSG);
    }

    private void removeNodeFromReplSetConfig(MongoSocketAddress mongoSocketAddressToRemove, GenericContainer masterNode) {
        Container.ExecResult execResultRemoveNode = this.execMongoDbCommandInContainer(masterNode, String.format("rs.remove(\"%s:%d\")", mongoSocketAddressToRemove.getIp(), mongoSocketAddressToRemove.getMappedPort()));
        log.debug("Remove a node: {} from a replica set, stdout {}", (Object)mongoSocketAddressToRemove, (Object)execResultRemoveNode.getStdout());
        this.checkMongoNodeExitCodeAndStatus(execResultRemoveNode, "Removing a node");
    }

    private void restartGenericContainer(GenericContainer genericContainer) {
        genericContainer.stop();
        genericContainer.start();
    }

    public void waitForMasterReelection(MongoNode previousMasterMongoNode) {
        this.validateFaultToleranceTestSupportAvailability();
        this.verifyWorkingNodeStoreIsNotEmpty();
        String prevMasterName = String.format("%s:%s", previousMasterMongoNode.getIp(), previousMasterMongoNode.getPort());
        String reelectionMessage = String.format("Waiting for the reelection of %s", prevMasterName);
        Container.ExecResult execResultMasterReelection = this.waitForCondition((GenericContainer)this.workingNodeStore.values().iterator().next(), String.format(this.buildWaitStopCondition("rs.status().members.filter(o => o.state === 1).length === 1 && rs.status().members.find(o => o.state === 1 && o.name === '%s') === undefined"), prevMasterName), this.getAwaitNodeInitAttempts() * 2, reelectionMessage);
        this.checkMongoNodeExitCode(execResultMasterReelection, reelectionMessage);
    }

    public void waitForMaster() {
        this.verifyWorkingNodeStoreIsNotEmpty();
        String message = "Waiting for a master node to be present in a cluster";
        Container.ExecResult execResult = this.waitForCondition((GenericContainer)this.workingNodeStore.values().iterator().next(), this.buildWaitStopCondition("rs.status().members.filter(o => o.state === 1).length === 1"), this.getAwaitNodeInitAttempts(), "Waiting for a master node to be present in a cluster");
        this.checkMongoNodeExitCode(execResult, "Waiting for a master node to be present in a cluster");
    }

    public void waitForAllMongoNodesUp() {
        this.validateFaultToleranceTestSupportAvailability();
        this.verifyWorkingNodeStoreIsNotEmpty();
        String waitingMessage = "Waiting for all nodes are up and running";
        Container.ExecResult execResultWaitForNodesUp = this.waitForCondition((GenericContainer)this.workingNodeStore.values().iterator().next(), this.buildWaitStopCondition("rs.status().members.filter(o => o.state === 0 || o.state === 3 || o.state === 5 || o.state === 6 || o.state === 8 || o.state === 9).length === 0"), this.getAwaitNodeInitAttempts(), "Waiting for all nodes are up and running");
        this.checkMongoNodeExitCode(execResultWaitForNodesUp, "Waiting for all nodes are up and running");
    }

    public void waitForMongoNodesDown(int nodeNumber) {
        this.validateFaultToleranceTestSupportAvailability();
        this.verifyWorkingNodeStoreIsNotEmpty();
        String waitingMessage = String.format("Waiting for %d node(s) is(are) down", nodeNumber);
        Container.ExecResult execResultWaitForNodesUp = this.waitForCondition((GenericContainer)this.workingNodeStore.values().iterator().next(), String.format(this.buildWaitStopCondition("rs.status().members.filter(o => o.state === 8).length === %d"), nodeNumber), this.getAwaitNodeInitAttempts(), waitingMessage);
        this.checkMongoNodeExitCode(execResultWaitForNodesUp, waitingMessage);
    }

    private void verifyWorkingNodeStoreIsNotEmpty() {
        if (this.workingNodeStore.isEmpty()) {
            throw new IllegalStateException("There is no any working Mongo DB node. Please, consider starting one.");
        }
    }

    public MongoNode getArbiterMongoNode(List<MongoNode> mongoNodes) {
        return this.getMongoNode(mongoNodes, ReplicaSetMemberState.ARBITER);
    }

    public MongoNode getMasterMongoNode(List<MongoNode> mongoNodes) {
        return this.getMongoNode(mongoNodes, ReplicaSetMemberState.PRIMARY);
    }

    public MongoNode getSecondaryMongoNode(List<MongoNode> mongoNodes) {
        return this.getMongoNode(mongoNodes, ReplicaSetMemberState.SECONDARY);
    }

    private MongoNode getMongoNode(List<MongoNode> mongoNodes, ReplicaSetMemberState memberState) {
        return this.mongoNodes(mongoNodes, memberState).findAny().orElseThrow(() -> new IllegalStateException(String.format("Cannot find a node in a cluster via a memberState: %s", new Object[]{memberState})));
    }

    public Stream<MongoNode> mongoNodes(List<MongoNode> mongoNodes, ReplicaSetMemberState memberState) {
        this.validateFaultToleranceTestSupportAvailability();
        Objects.requireNonNull(memberState, "mongoNodes is not supposed to be null");
        Objects.requireNonNull(memberState, "memberState is not supposed to be null");
        return mongoNodes.stream().filter(n -> memberState.equals((Object)n.getState()));
    }

    public List<ReplicaSetMemberState> nodeStates(List<MongoNode> mongoNodes) {
        return mongoNodes.stream().map(MongoNode::getState).collect(Collectors.toList());
    }

    @Generated
    boolean loadCollectionToDeadLetterDb(MongoNode mongoNode, String collectionFullName) {
        this.validateFaultToleranceTestSupportAvailability();
        this.verifyWorkingNodeStoreIsNotEmpty();
        String path = "/data/db/rollback/" + collectionFullName;
        MongoSocketAddress mongoSocketAddress = this.socketAddressConverter.convert(mongoNode);
        GenericContainer genericContainer = this.extractGenericContainer(mongoSocketAddress, this.workingNodeStore);
        Container.ExecResult waitForRollbackFile = genericContainer.execInContainer(new String[]{"sh", "-c", String.format("COUNTER=1; while [ $COUNTER != %d ] && [ ! -d %s ]; do sleep 1; COUNTER=$((COUNTER+1)); echo waiting for a rollback directory: $COUNTER up to %d; done", this.getAwaitNodeInitAttempts(), path, this.getAwaitNodeInitAttempts())});
        log.debug("waitForRollbackFile: stdout: {}, stderr: {}", (Object)waitForRollbackFile.getStdout(), (Object)waitForRollbackFile.getStderr());
        boolean waitFiled = waitForRollbackFile.getStdout().contains(String.format("%d up to", this.getAwaitNodeInitAttempts() - 1));
        if (waitForRollbackFile.getExitCode() != 0 || waitFiled) {
            log.debug("Cannot find any rollback file");
            return false;
        }
        Container.ExecResult execResultMongorestore = genericContainer.execInContainer(new String[]{"mongorestore", "--uri=" + this.getReplicaSetUrl(), "--db", DEAD_LETTER_DB_NAME, path});
        log.debug("mongorestore: stdout: {}, stderr: {}", (Object)execResultMongorestore.getStdout(), (Object)execResultMongorestore.getStderr());
        if (execResultMongorestore.getExitCode() != 0) {
            throw new IllegalStateException("Cannot execute mongorestore to extract rollback files");
        }
        return true;
    }

    public static MongoDbReplicaSetBuilder builder() {
        return new MongoDbReplicaSetBuilder();
    }

    public static class MongoDbReplicaSetBuilder {
        private Integer replicaSetNumber;
        private Integer awaitNodeInitAttempts;
        private String propertyFileName;
        private String mongoDockerImageName;
        private Boolean addArbiter;
        private Boolean addToxiproxy;
        private Integer slaveDelayTimeout;
        private Integer slaveDelayNumber;
        private Boolean useHostDockerInternal;
        private List<String> commandLineOptions;

        MongoDbReplicaSetBuilder() {
        }

        public MongoDbReplicaSetBuilder replicaSetNumber(Integer replicaSetNumber) {
            this.replicaSetNumber = replicaSetNumber;
            return this;
        }

        public MongoDbReplicaSetBuilder awaitNodeInitAttempts(Integer awaitNodeInitAttempts) {
            this.awaitNodeInitAttempts = awaitNodeInitAttempts;
            return this;
        }

        public MongoDbReplicaSetBuilder propertyFileName(String propertyFileName) {
            this.propertyFileName = propertyFileName;
            return this;
        }

        public MongoDbReplicaSetBuilder mongoDockerImageName(String mongoDockerImageName) {
            this.mongoDockerImageName = mongoDockerImageName;
            return this;
        }

        public MongoDbReplicaSetBuilder addArbiter(Boolean addArbiter) {
            this.addArbiter = addArbiter;
            return this;
        }

        public MongoDbReplicaSetBuilder addToxiproxy(Boolean addToxiproxy) {
            this.addToxiproxy = addToxiproxy;
            return this;
        }

        public MongoDbReplicaSetBuilder slaveDelayTimeout(Integer slaveDelayTimeout) {
            this.slaveDelayTimeout = slaveDelayTimeout;
            return this;
        }

        public MongoDbReplicaSetBuilder slaveDelayNumber(Integer slaveDelayNumber) {
            this.slaveDelayNumber = slaveDelayNumber;
            return this;
        }

        public MongoDbReplicaSetBuilder useHostDockerInternal(Boolean useHostDockerInternal) {
            this.useHostDockerInternal = useHostDockerInternal;
            return this;
        }

        public MongoDbReplicaSetBuilder commandLineOptions(List<String> commandLineOptions) {
            this.commandLineOptions = commandLineOptions;
            return this;
        }

        public MongoDbReplicaSet build() {
            return new MongoDbReplicaSet(this.replicaSetNumber, this.awaitNodeInitAttempts, this.propertyFileName, this.mongoDockerImageName, this.addArbiter, this.addToxiproxy, this.slaveDelayTimeout, this.slaveDelayNumber, this.useHostDockerInternal, this.commandLineOptions);
        }

        public String toString() {
            return "MongoDbReplicaSet.MongoDbReplicaSetBuilder(replicaSetNumber=" + this.replicaSetNumber + ", awaitNodeInitAttempts=" + this.awaitNodeInitAttempts + ", propertyFileName=" + this.propertyFileName + ", mongoDockerImageName=" + this.mongoDockerImageName + ", addArbiter=" + this.addArbiter + ", addToxiproxy=" + this.addToxiproxy + ", slaveDelayTimeout=" + this.slaveDelayTimeout + ", slaveDelayNumber=" + this.slaveDelayNumber + ", useHostDockerInternal=" + this.useHostDockerInternal + ", commandLineOptions=" + this.commandLineOptions + ")";
        }
    }
}

