/*
 * Decompiled with CFR 0.152.
 */
package jme3utilities.math;

import com.jme3.math.FastMath;
import com.jme3.math.Line;
import com.jme3.math.Matrix3f;
import com.jme3.math.Quaternion;
import com.jme3.math.Vector3f;
import com.jme3.scene.Spatial;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import jme3utilities.MySpatial;
import jme3utilities.MyString;
import jme3utilities.Validate;
import jme3utilities.math.MyMath;
import jme3utilities.math.ReadXZ;
import jme3utilities.math.VectorXZ;

public class MyVector3f {
    public static final int firstAxis = 0;
    public static final int numAxes = 3;
    public static final int xAxis = 0;
    public static final int yAxis = 1;
    public static final int zAxis = 2;
    public static final int lastAxis = 2;
    public static final Logger logger = Logger.getLogger(MyVector3f.class.getName());
    private static final Pattern pattern = Pattern.compile("\\(\\s*([^,]+),\\s+([^,]+),\\s+(\\S+)\\s*\\)");
    private static final Vector3f unitX = new Vector3f(1.0f, 0.0f, 0.0f);
    private static final Vector3f[] cardinalAxes = new Vector3f[]{new Vector3f(1.0f, 0.0f, 0.0f), new Vector3f(0.0f, 1.0f, 0.0f), new Vector3f(0.0f, 0.0f, 1.0f), new Vector3f(-1.0f, 0.0f, 0.0f), new Vector3f(0.0f, -1.0f, 0.0f), new Vector3f(0.0f, 0.0f, -1.0f)};

    private MyVector3f() {
    }

    public static void accumulateMaxima(Vector3f maxima, Vector3f input) {
        if (input.x > maxima.x) {
            maxima.x = input.x;
        }
        if (input.y > maxima.y) {
            maxima.y = input.y;
        }
        if (input.z > maxima.z) {
            maxima.z = input.z;
        }
    }

    public static void accumulateMinima(Vector3f minima, Vector3f input) {
        if (input.x < minima.x) {
            minima.x = input.x;
        }
        if (input.y < minima.y) {
            minima.y = input.y;
        }
        if (input.z < minima.z) {
            minima.z = input.z;
        }
    }

    public static void accumulateScaled(Vector3f total, Vector3f input, float scale) {
        assert (Validate.nonNull(total, "total"));
        assert (Validate.nonNull(input, "input"));
        total.x += input.x * scale;
        total.y += input.y * scale;
        total.z += input.z * scale;
    }

    public static float altitude(Vector3f offset) {
        assert (Validate.nonZero(offset, "offset"));
        float xzRange = MyMath.hypotenuse(offset.x, offset.z);
        float result = (float)Math.atan2(offset.y, xzRange);
        assert (result <= 1.5707964f) : result;
        assert (result >= -1.5707964f) : result;
        return result;
    }

    public static boolean areCollinear(Vector3f point1, Vector3f point2, Vector3f point3, float tolerance2) {
        assert (Validate.nonNull(point1, "first location"));
        assert (Validate.nonNull(point2, "2nd location"));
        assert (Validate.nonNegative(tolerance2, "tolerance"));
        Vector3f offset3 = point3.subtract(point1);
        double normSquared3 = MyVector3f.lengthSquared(offset3);
        if (normSquared3 <= (double)tolerance2) {
            return true;
        }
        Vector3f offset2 = point2.subtract(point1);
        double scaleFactor = MyVector3f.dot(offset2, offset3) / normSquared3;
        Vector3f projection = offset3.mult((float)scaleFactor);
        boolean result = MyVector3f.doCoincide(projection, offset2, tolerance2);
        return result;
    }

    public static boolean areWithinTolerance(Vector3f a, Vector3f b, float relativeTolerance) {
        float ax = a.x;
        float bx = b.x;
        boolean result = MyMath.areWithinTolerance(ax, bx, relativeTolerance) && MyMath.areWithinTolerance(a.y, b.y, relativeTolerance) && MyMath.areWithinTolerance(a.z, b.z, relativeTolerance);
        return result;
    }

    public static Vector3f axisVector(int axisIndex, float length, Vector3f storeResult) {
        assert (Validate.axisIndex(axisIndex, "axis index"));
        assert (Validate.nonNegative(length, "length"));
        Vector3f result = storeResult == null ? new Vector3f() : storeResult;
        result.zero();
        result.set(axisIndex, length);
        return result;
    }

    public static float azimuth(Vector3f offset) {
        float result = (float)Math.atan2(offset.z, offset.x);
        return result;
    }

    public static void cardinalizeLocal(Vector3f input) {
        MyVector3f.normalizeLocal(input);
        Vector3f bestCardinalDirection = new Vector3f();
        float bestDot = -2.0f;
        for (Vector3f x : cardinalAxes) {
            float dot = x.dot(input);
            if (!(dot > bestDot)) continue;
            bestDot = dot;
            bestCardinalDirection.set(x);
        }
        input.set(bestCardinalDirection);
    }

    public static int compare(Vector3f v1, Vector3f v2) {
        assert (Validate.nonNull(v1, "v1"));
        assert (Validate.nonNull(v2, "v2"));
        int result = Float.compare(v1.x, v2.x);
        if (result == 0 && (result = Float.compare(v1.y, v2.y)) == 0) {
            result = Float.compare(v1.z, v2.z);
        }
        return result;
    }

    public static Matrix3f covariance(Collection<Vector3f> collection, Matrix3f storeResult) {
        assert (Validate.nonEmpty(collection, "collection"));
        int numSamples = collection.size();
        assert (numSamples > 1) : numSamples;
        Matrix3f result = storeResult == null ? new Matrix3f() : storeResult;
        Vector3f sampleMean = MyVector3f.mean(collection, null);
        result.zero();
        float[] aboveMean = new float[3];
        for (Vector3f location : collection) {
            aboveMean[0] = location.x - sampleMean.x;
            aboveMean[1] = location.y - sampleMean.y;
            aboveMean[2] = location.z - sampleMean.z;
            for (int rowIndex = 0; rowIndex < 3; ++rowIndex) {
                for (int colIndex = rowIndex; colIndex < 3; ++colIndex) {
                    float sum = result.get(rowIndex, colIndex);
                    result.set(rowIndex, colIndex, sum += aboveMean[rowIndex] * aboveMean[colIndex]);
                }
            }
        }
        float nMinus1 = numSamples - 1;
        for (int rowIndex = 0; rowIndex < 3; ++rowIndex) {
            for (int columnIndex = rowIndex; columnIndex < 3; ++columnIndex) {
                float sum = result.get(rowIndex, columnIndex);
                float element = sum / nMinus1;
                result.set(rowIndex, columnIndex, element);
                result.set(columnIndex, rowIndex, element);
            }
        }
        return result;
    }

    public static String describe(Vector3f vector) {
        String result;
        if (vector == null) {
            result = "null";
        } else if (MyVector3f.isScaleUniform(vector)) {
            result = "xyz=" + MyString.describe(vector.x);
        } else {
            StringBuilder builder = new StringBuilder(40);
            if (vector.x != 0.0f) {
                builder.append("x=");
                String x = MyString.describe(vector.x);
                builder.append(x);
            }
            if (vector.y != 0.0f) {
                if (builder.length() > 0) {
                    builder.append(' ');
                }
                builder.append("y=");
                String y = MyString.describe(vector.y);
                builder.append(y);
            }
            if (vector.z != 0.0f) {
                if (builder.length() > 0) {
                    builder.append(' ');
                }
                builder.append("z=");
                String z = MyString.describe(vector.z);
                builder.append(z);
            }
            result = builder.toString();
        }
        assert (result != null);
        assert (!result.isEmpty());
        return result;
    }

    public static String describeDirection(Vector3f v) {
        String result;
        if (v == null) {
            result = "null";
        } else {
            StringBuilder builder = new StringBuilder(40);
            builder.append("dx=");
            String x = MyString.describeFraction(v.x);
            builder.append(x);
            builder.append(" dy=");
            String y = MyString.describeFraction(v.y);
            builder.append(y);
            builder.append(" dz=");
            String z = MyString.describeFraction(v.z);
            builder.append(z);
            result = builder.toString();
        }
        assert (result != null);
        assert (!result.isEmpty());
        return result;
    }

    public static double distanceSquared(Vector3f vector1, Vector3f vector2) {
        double dx = vector1.x - vector2.x;
        double dy = vector1.y - vector2.y;
        double dz = vector1.z - vector2.z;
        double result = dx * dx + dy * dy + dz * dz;
        return result;
    }

    public static double distanceSquaredToSegment(Vector3f location, Vector3f segStart, Vector3f segEnd, Vector3f storeClosest) {
        Vector3f segOffset = segEnd.subtract(segStart);
        double segLengthSquared = MyVector3f.lengthSquared(segOffset);
        if (segLengthSquared == 0.0) {
            if (storeClosest != null) {
                storeClosest.set(segStart);
            }
            double result = MyVector3f.distanceSquared(segStart, location);
            return result;
        }
        Vector3f locOffset = location.subtract(segStart);
        double dot = MyVector3f.dot(locOffset, segOffset);
        double t = dot / segLengthSquared;
        float tPrime = FastMath.saturate((float)((float)t));
        Vector3f closestOffset = segOffset.mult(tPrime);
        if (storeClosest != null) {
            storeClosest.set(segStart);
            storeClosest.addLocal(closestOffset);
        }
        double result = MyVector3f.distanceSquared(closestOffset, locOffset);
        assert (result >= 0.0) : result;
        return result;
    }

    public static boolean doCoincide(Vector3f point1, Vector3f point2, double tolerance2) {
        assert (Validate.nonNull(point1, "first point"));
        assert (Validate.nonNull(point2, "second point"));
        assert (Validate.nonNegative(tolerance2, "tolerance"));
        double d2 = MyVector3f.distanceSquared(point1, point2);
        return !(d2 > tolerance2);
    }

    public static double dot(Vector3f vector1, Vector3f vector2) {
        double x1 = vector1.x;
        double x2 = vector2.x;
        double y1 = vector1.y;
        double y2 = vector2.y;
        double z1 = vector1.z;
        double z2 = vector2.z;
        double product = x1 * x2 + y1 * y2 + z1 * z2;
        return product;
    }

    public static boolean eq(Vector3f v1, Vector3f v2) {
        assert (Validate.nonNull(v1, "first input vector"));
        assert (Validate.nonNull(v2, "2nd input vector"));
        boolean result = v1.x == v2.x && v1.y == v2.y && v1.z == v2.z;
        return result;
    }

    public static Vector3f fromAltAz(float altitude, float azimuth) {
        Quaternion elevate = new Quaternion().fromAngles(0.0f, 0.0f, altitude);
        Vector3f elevation = elevate.mult(unitX);
        Vector3f direction = MyVector3f.yRotate(elevation, azimuth);
        assert (direction.isUnitVector()) : direction;
        return direction;
    }

    public static void generateBasis(Vector3f in1, Vector3f store2, Vector3f store3) {
        assert (Validate.nonZero(in1, "starting direction"));
        assert (Validate.nonNull(store2, "2nd basis vector"));
        assert (Validate.nonNull(store3, "3nd basis vector"));
        MyVector3f.normalizeLocal(in1);
        float x = Math.abs(in1.x);
        float y = Math.abs(in1.y);
        float z = Math.abs(in1.z);
        if (x <= y && x <= z) {
            store3.set(1.0f, 0.0f, 0.0f);
        } else if (y <= z) {
            store3.set(0.0f, 1.0f, 0.0f);
        } else {
            store3.set(0.0f, 0.0f, 1.0f);
        }
        in1.cross(store3, store2);
        MyVector3f.normalizeLocal(store2);
        in1.cross(store2, store3);
        MyVector3f.normalizeLocal(store3);
    }

    public static ReadXZ horizontalDirection(Vector3f offset) {
        assert (Validate.nonNull(offset, "offset"));
        VectorXZ horizontalOffset = new VectorXZ(offset);
        ReadXZ result = horizontalOffset.normalize();
        return result;
    }

    public static Vector3f intersectSegments(Vector3f start1, Vector3f end1, Vector3f start2, Vector3f end2, float tolerance2) {
        Vector3f c2;
        assert (Validate.nonNull(start1, "start of first segment"));
        assert (Validate.nonNull(start2, "start of 2nd segment"));
        assert (Validate.nonNull(end2, "end of 2nd segment"));
        assert (Validate.nonNegative(tolerance2, "tolerance2"));
        Vector3f offset1 = end1.subtract(start1);
        double ls1 = MyVector3f.lengthSquared(offset1);
        if (ls1 == 0.0) {
            Vector3f closest = new Vector3f();
            double ds = MyVector3f.distanceSquaredToSegment(start1, start2, end2, closest);
            if (ds > (double)tolerance2) {
                return null;
            }
            return closest;
        }
        Vector3f offset2 = end2.subtract(start2);
        double ls2 = MyVector3f.lengthSquared(offset1);
        if (ls2 == 0.0) {
            Vector3f closest = new Vector3f();
            double ds = MyVector3f.distanceSquaredToSegment(start2, start1, end1, closest);
            if (ds > (double)tolerance2) {
                return null;
            }
            return closest;
        }
        Vector3f n = offset2.cross(offset1);
        if (MyVector3f.lengthSquared(n) <= (double)tolerance2) {
            Vector3f result = MyVector3f.intersectParallelSegments(start1, end1, start2, end2, tolerance2);
            return result;
        }
        Vector3f n1 = offset2.cross(n);
        Vector3f n2 = offset1.cross(n);
        float t1 = start1.subtract(start2).dot(n2) / offset2.dot(n2);
        float t2 = start2.subtract(start1).dot(n1) / offset1.dot(n1);
        Vector3f c1 = offset2.mult(t1).add(start2);
        if (!MyVector3f.doCoincide(c1, c2 = offset1.mult(t2).add(start1), tolerance2)) {
            return null;
        }
        double fuzz1 = (double)tolerance2 / ls2;
        if (t1 < 0.0f && (double)(t1 * t1) > fuzz1) {
            return null;
        }
        float ct1 = 1.0f - t1;
        if (ct1 < 0.0f && (double)(ct1 * ct1) > fuzz1) {
            return null;
        }
        double fuzz2 = (double)tolerance2 / ls1;
        if (t2 < 0.0f && (double)(t2 * t2) > fuzz2) {
            return null;
        }
        float ct2 = 1.0f - t2;
        if (ct2 < 0.0f && (double)(ct2 * ct2) > fuzz2) {
            return null;
        }
        return c1;
    }

    public static boolean isAllNonNegative(Vector3f vector) {
        return vector.x >= 0.0f && vector.y >= 0.0f && vector.z >= 0.0f;
    }

    public static boolean isAllPositive(Vector3f vector) {
        return vector.x > 0.0f && vector.y > 0.0f && vector.z > 0.0f;
    }

    public static boolean isBetween(Vector3f a, Vector3f b, Vector3f c) {
        boolean result = MyMath.isBetween(a.x, b.x, c.x) && MyMath.isBetween(a.y, b.y, c.y) && MyMath.isBetween(a.z, b.z, c.z);
        return result;
    }

    public static boolean isScaleIdentity(Vector3f vector) {
        return vector.x == 1.0f && vector.y == 1.0f && vector.z == 1.0f;
    }

    public static boolean isScaleUniform(Vector3f vector) {
        return vector.x == vector.y && vector.y == vector.z;
    }

    public static boolean isZero(Vector3f vector) {
        return vector.x == 0.0f && vector.y == 0.0f && vector.z == 0.0f;
    }

    public static double lengthSquared(Vector3f vector) {
        double result = MyMath.sumOfSquares(vector.x, vector.y, vector.z);
        return result;
    }

    public static Vector3f lerp(float t, Vector3f v0, Vector3f v1, Vector3f storeResult) {
        assert (Validate.nonNull(v0, "v0"));
        assert (Validate.nonNull(v1, "v1"));
        Vector3f result = storeResult == null ? new Vector3f() : storeResult;
        result.x = MyMath.lerp(t, v0.x, v1.x);
        result.y = MyMath.lerp(t, v0.y, v1.y);
        result.z = MyMath.lerp(t, v0.z, v1.z);
        return result;
    }

    public static Vector3f lineMeetsLine(Line line1, Line line2) {
        Vector3f d1 = line1.getDirection();
        Vector3f d2 = line2.getDirection();
        Vector3f p1 = line1.getOrigin();
        Vector3f p2 = line2.getOrigin();
        Vector3f n = d1.cross(d2);
        Vector3f n2 = d2.cross(n);
        Vector3f result = null;
        float denominator = d1.dot(n2);
        if (denominator != 0.0f) {
            float numerator = p2.dot(n2) - p1.dot(n2);
            result = d1.mult(numerator / denominator);
            result.addLocal(p1);
        }
        return result;
    }

    public static Vector3f lineMeetsSphere(Line line, Vector3f center, float radius, boolean farSide) {
        Vector3f result;
        float normV2;
        Vector3f lineOrigin;
        Vector3f offset;
        assert (Validate.nonNull(center, "center"));
        assert (Validate.nonNegative(radius, "radius"));
        Vector3f direction = line.getDirection().normalize();
        float dDotV = direction.dot(offset = (lineOrigin = line.getOrigin()).subtract(center));
        float discriminant = dDotV * dDotV - (normV2 = offset.lengthSquared()) + radius * radius;
        if (discriminant >= 0.0f) {
            float t = -dDotV;
            t = farSide ? (t += FastMath.sqrt((float)discriminant)) : (t -= FastMath.sqrt((float)discriminant));
            result = direction.mult(t);
            result.addLocal(lineOrigin);
        } else {
            Vector3f projection = direction.mult(dDotV);
            result = offset.subtract(projection);
            float factor = radius / result.length();
            assert (factor <= 1.0f) : factor;
            result.multLocal(factor);
            result.addLocal(center);
        }
        return result;
    }

    public static Vector3f localizeDirection(Vector3f worldDirection, Spatial spatial, Vector3f storeResult) {
        Vector3f result;
        assert (Validate.nonZero(worldDirection, "direction"));
        Vector3f vector3f = result = storeResult == null ? new Vector3f() : storeResult;
        if (MySpatial.isIgnoringTransforms(spatial)) {
            result.set(worldDirection);
        } else {
            spatial.worldToLocal(worldDirection, result);
        }
        double lengthSquared = MyVector3f.lengthSquared(result);
        double scaleFactor = 1.0 / Math.sqrt(lengthSquared);
        result.multLocal((float)scaleFactor);
        assert (result.isUnitVector());
        return result;
    }

    public static Vector3f maxAbs(Vector3f vector1, Vector3f vector2, Vector3f storeResult) {
        assert (Validate.nonNull(vector1, "vector1"));
        assert (Validate.nonNull(vector2, "vector2"));
        Vector3f result = storeResult == null ? new Vector3f() : storeResult;
        result.x = Math.max(Math.abs(vector1.x), Math.abs(vector2.x));
        result.y = Math.max(Math.abs(vector1.y), Math.abs(vector2.y));
        result.z = Math.max(Math.abs(vector1.z), Math.abs(vector2.z));
        return result;
    }

    public static Vector3f mean(Collection<Vector3f> collection, Vector3f storeResult) {
        assert (Validate.nonEmpty(collection, "collection"));
        Vector3f result = storeResult == null ? new Vector3f() : storeResult;
        result.zero();
        for (Vector3f location : collection) {
            result.addLocal(location);
        }
        int count = collection.size();
        result.divideLocal((float)count);
        return result;
    }

    public static Vector3f midpoint(Vector3f vector1, Vector3f vector2, Vector3f storeResult) {
        assert (Validate.finite(vector1, "first location"));
        assert (Validate.finite(vector2, "2nd location"));
        float x = (vector1.x + vector2.x) / 2.0f;
        float y = (vector1.y + vector2.y) / 2.0f;
        float z = (vector1.z + vector2.z) / 2.0f;
        if (storeResult == null) {
            return new Vector3f(x, y, z);
        }
        return storeResult.set(x, y, z);
    }

    public static Vector3f[] mostRemote(List<Vector3f> locations) {
        assert (Validate.nonNull(locations, "locations"));
        double largestSD = -1.0;
        Vector3f[] result = null;
        for (int i = 0; i < locations.size(); ++i) {
            Vector3f iVector = locations.get(i);
            for (int j = i + 1; j < locations.size(); ++j) {
                Vector3f jVector = locations.get(j);
                double squaredDistance = MyVector3f.distanceSquared(iVector, jVector);
                if (!(squaredDistance > largestSD)) continue;
                largestSD = squaredDistance;
                if (result != null) continue;
                result = new Vector3f[]{iVector, jVector};
            }
        }
        return result;
    }

    public static boolean ne(Vector3f v1, Vector3f v2) {
        assert (Validate.nonNull(v1, "first input vector"));
        assert (Validate.nonNull(v2, "2nd input vector"));
        boolean result = v1.x != v2.x || v1.y != v2.y || v1.z != v2.z;
        return result;
    }

    public static void normalizeLocal(Vector3f input) {
        assert (Validate.nonNull(input, "input"));
        double lengthSquared = MyVector3f.lengthSquared(input);
        double dScale = Math.sqrt(lengthSquared);
        float fScale = (float)dScale;
        if (fScale != 0.0f && fScale != 1.0f) {
            input.divideLocal(fScale);
        }
    }

    public static Vector3f parse(String textString) {
        assert (Validate.nonEmpty(textString, "text string"));
        Vector3f result = null;
        Matcher matcher = pattern.matcher(textString);
        boolean valid = matcher.matches();
        if (valid) {
            String xText = matcher.group(1);
            float x = Float.parseFloat(xText);
            String yText = matcher.group(2);
            float y = Float.parseFloat(yText);
            String zText = matcher.group(3);
            float z = Float.parseFloat(zText);
            result = new Vector3f(x, y, z);
        }
        return result;
    }

    public static Vector3f projection(Vector3f vector1, Vector3f vector2, Vector3f storeResult) {
        assert (Validate.nonNull(vector1, "vector1"));
        assert (Validate.nonZero(vector2, "vector2"));
        double dot = MyVector3f.dot(vector1, vector2);
        double lengthSquared = MyVector3f.lengthSquared(vector2);
        double scaleFactor = dot / lengthSquared;
        Vector3f projection = vector2.mult((float)scaleFactor, storeResult);
        return projection;
    }

    public static float scalarProjection(Vector3f vector1, Vector3f vector2) {
        assert (Validate.nonNull(vector1, "vector1"));
        assert (Validate.nonZero(vector2, "vector2"));
        double dot = MyVector3f.dot(vector1, vector2);
        double lengthSquared = MyVector3f.lengthSquared(vector2);
        double length = Math.sqrt(lengthSquared);
        double projection = dot / length;
        return (float)projection;
    }

    public static Vector3f rejection(Vector3f vector1, Vector3f vector2, Vector3f storeResult) {
        assert (Validate.nonNull(vector1, "vector1"));
        assert (Validate.nonZero(vector2, "vector2"));
        float x = vector1.x;
        float y = vector1.y;
        float z = vector1.z;
        double dot = MyVector3f.dot(vector1, vector2);
        double lengthSquared = MyVector3f.lengthSquared(vector2);
        double scaleFactor = -dot / lengthSquared;
        Vector3f result = vector2.mult((float)scaleFactor, storeResult);
        result.addLocal(x, y, z);
        return result;
    }

    public static Vector3f standardize(Vector3f input, Vector3f storeResult) {
        assert (Validate.nonNull(input, "input vector"));
        Vector3f result = storeResult == null ? new Vector3f() : storeResult;
        result.x = MyMath.standardize(input.x);
        result.y = MyMath.standardize(input.y);
        result.z = MyMath.standardize(input.z);
        return result;
    }

    public static Vector3f velocity(float timeInterval, Vector3f startValue, Vector3f endValue, Vector3f storeResult) {
        Validate.positive(timeInterval, "time interval");
        Vector3f result = endValue.subtract(startValue, storeResult);
        result.divideLocal(timeInterval);
        return result;
    }

    public static Vector3f yRotate(Vector3f input, float angle) {
        float cosine = FastMath.cos((float)angle);
        float sine = FastMath.sin((float)angle);
        float x = cosine * input.x - sine * input.z;
        float y = input.y;
        float z = cosine * input.z + sine * input.x;
        Vector3f result = new Vector3f(x, y, z);
        return result;
    }

    private static boolean allCollinear(Vector3f first, Vector3f last, Iterable<Vector3f> testLocations, float tolerance2) {
        assert (Validate.nonNull(first, "first location"));
        assert (Validate.nonNull(testLocations, "testLocations"));
        assert (Validate.nonNegative(tolerance2, "tolerance2"));
        Vector3f fl = last.subtract(first);
        double normSquaredFL = MyVector3f.lengthSquared(fl);
        if (normSquaredFL <= (double)tolerance2) {
            return true;
        }
        for (Vector3f middle : testLocations) {
            assert (middle != null);
            Vector3f fm = middle.subtract(first);
            double fmDotFl = MyVector3f.dot(fm, fl);
            double fraction = fmDotFl / normSquaredFL;
            Vector3f projection = fl.mult((float)fraction);
            boolean collin = MyVector3f.doCoincide(projection, fm, tolerance2);
            if (collin) continue;
            return false;
        }
        return true;
    }

    private static Vector3f intersectCollinearSegments(Vector3f ext, Vector3f start1, Vector3f end1, Vector3f start2, Vector3f end2) {
        Vector3f partner;
        Vector3f otherEnd;
        Vector3f otherStart;
        assert (ext != null);
        assert (ext == start1 || ext == end1 || ext == start2 || ext == end2);
        assert (start1 != null);
        assert (end1 != null);
        assert (start2 != null);
        assert (end2 != null);
        if (ext == start1 || ext == end1) {
            otherStart = start2;
            otherEnd = end2;
        } else {
            assert (ext == start2 || ext == end2);
            otherStart = start1;
            otherEnd = end1;
        }
        if (ext == start1) {
            partner = end1;
        } else if (ext == end1) {
            partner = start1;
        } else if (ext == start2) {
            partner = end2;
        } else {
            assert (ext == end2);
            partner = start2;
        }
        double sdPartner = MyVector3f.distanceSquared(ext, partner);
        double sdOtherStart = MyVector3f.distanceSquared(ext, otherStart);
        if (sdPartner >= sdOtherStart) {
            return otherStart.clone();
        }
        double sdOtherEnd = MyVector3f.distanceSquared(ext, otherEnd);
        if (sdPartner >= sdOtherEnd) {
            return otherEnd.clone();
        }
        return null;
    }

    private static Vector3f intersectParallelSegments(Vector3f start1, Vector3f end1, Vector3f start2, Vector3f end2, float tolerance2) {
        assert (start1 != null);
        assert (end1 != null);
        assert (start2 != null);
        assert (end2 != null);
        ArrayList<Vector3f> locations = new ArrayList<Vector3f>(4);
        locations.add(start1);
        locations.add(end1);
        locations.add(start2);
        locations.add(end2);
        Vector3f[] md = MyVector3f.mostRemote(locations);
        Vector3f first = md[0];
        Vector3f last = md[1];
        if (MyVector3f.doCoincide(first, last, tolerance2)) {
            return start2.clone();
        }
        boolean success = locations.remove(first);
        assert (success);
        success = locations.remove(last);
        assert (success);
        boolean collin = MyVector3f.allCollinear(first, last, locations, tolerance2);
        if (!collin) {
            return null;
        }
        Vector3f result = MyVector3f.intersectCollinearSegments(first, start2, end2, start1, end1);
        return result;
    }
}

