/*
 * Decompiled with CFR 0.152.
 */
package jme3utilities.math;

import com.jme3.math.FastMath;
import com.jme3.math.Quaternion;
import com.jme3.math.Vector3f;
import java.util.HashSet;
import java.util.logging.Level;
import java.util.logging.Logger;
import jme3utilities.MyString;
import jme3utilities.Validate;
import jme3utilities.math.MyMath;

public class MyQuaternion {
    public static final Logger logger = Logger.getLogger(MyQuaternion.class.getName());
    private static final Vector3f[] cardinalAxes = new Vector3f[]{new Vector3f(1.0f, 0.0f, 0.0f), new Vector3f(0.0f, 1.0f, 0.0f), new Vector3f(0.0f, 0.0f, 1.0f), new Vector3f(-1.0f, 0.0f, 0.0f), new Vector3f(0.0f, -1.0f, 0.0f), new Vector3f(0.0f, 0.0f, -1.0f)};

    private MyQuaternion() {
    }

    public static void accumulateScaled(Quaternion total, Quaternion input, float scale) {
        assert (Validate.nonNull(total, "total"));
        assert (Validate.nonNull(input, "input"));
        float x = total.getX() + input.getX() * scale;
        float y = total.getY() + input.getY() * scale;
        float z = total.getZ() + input.getZ() * scale;
        float w = total.getW() + input.getW() * scale;
        total.set(x, y, z, w);
    }

    public static float angleBetween(Quaternion q1, Quaternion q2) {
        assert (MyQuaternion.validateUnit(q1, "q1", 1.0E-4f));
        assert (MyQuaternion.validateUnit(q2, "q2", 1.0E-4f));
        double dotProduct = MyQuaternion.dot(q1, q2);
        double absDot = Math.abs(dotProduct);
        float result = absDot > 1.0 ? 0.0f : 2.0f * (float)Math.acos(absDot);
        return result;
    }

    public static void cardinalizeLocal(Quaternion input) {
        assert (Validate.nonNull(input, "input"));
        MyQuaternion.normalizeLocal(input);
        Quaternion cardinalRotation = new Quaternion();
        Quaternion bestCardinalRotation = new Quaternion();
        Vector3f z = new Vector3f();
        float bestAbsDot = -1.0f;
        for (Vector3f x : cardinalAxes) {
            for (Vector3f y : cardinalAxes) {
                x.cross(y, z);
                if (!z.isUnitVector()) continue;
                cardinalRotation.fromAxes(x, y, z);
                float dot = cardinalRotation.dot(input);
                float absDot = FastMath.abs((float)dot);
                if (!(absDot > bestAbsDot)) continue;
                bestAbsDot = absDot;
                bestCardinalRotation.set(cardinalRotation);
            }
        }
        input.set(bestCardinalRotation);
    }

    public static Quaternion conjugate(Quaternion q, Quaternion storeResult) {
        Quaternion result = storeResult == null ? new Quaternion() : storeResult;
        float qx = q.getX();
        float qy = q.getY();
        float qz = q.getZ();
        float qw = q.getW();
        result.set(-qx, -qy, -qz, qw);
        return result;
    }

    public static int countNe(Quaternion[] quaternions) {
        if (quaternions == null) {
            return 0;
        }
        HashSet<Quaternion> distinct = new HashSet<Quaternion>(quaternions.length);
        for (Quaternion rot : quaternions) {
            Quaternion standard = MyQuaternion.standardize(rot, null);
            distinct.add(standard);
        }
        int result = distinct.size();
        return result;
    }

    public static String describe(Quaternion q) {
        String result;
        if (q == null) {
            result = "null";
        } else {
            StringBuilder builder = new StringBuilder(40);
            builder.append("x=");
            String x = MyString.describeFraction(q.getX());
            builder.append(x);
            builder.append(" y=");
            String y = MyString.describeFraction(q.getY());
            builder.append(y);
            builder.append(" z=");
            String z = MyString.describeFraction(q.getZ());
            builder.append(z);
            builder.append(" w=");
            String w = MyString.describeFraction(q.getW());
            builder.append(w);
            result = builder.toString();
        }
        assert (result != null);
        assert (!result.isEmpty());
        return result;
    }

    public static double dot(Quaternion q1, Quaternion q2) {
        double w1 = q1.getW();
        double w2 = q2.getW();
        double x1 = q1.getX();
        double x2 = q2.getX();
        double y1 = q1.getY();
        double y2 = q2.getY();
        double z1 = q1.getZ();
        double z2 = q2.getZ();
        double result = w1 * w2 + x1 * x2 + y1 * y2 + z1 * z2;
        return result;
    }

    public static Quaternion exp(Quaternion q, Quaternion storeResult) {
        double qz;
        double qy;
        assert (Validate.require(MyQuaternion.isPure(q), "a pure quaternion"));
        Quaternion result = storeResult == null ? new Quaternion() : storeResult;
        double qx = q.getX();
        double theta = MyMath.hypotenuseDouble(qx, qy = (double)q.getY(), qz = (double)q.getZ());
        if (theta == 0.0) {
            result.loadIdentity();
        } else {
            float w = (float)Math.cos(theta);
            double scale = Math.sin(theta) / theta;
            float x = (float)(scale * qx);
            float y = (float)(scale * qy);
            float z = (float)(scale * qz);
            result.set(x, y, z, w);
        }
        return result;
    }

    public static boolean isPure(Quaternion q) {
        float qw = q.getW();
        return qw == 0.0f;
    }

    public static boolean isRotationIdentity(Quaternion q) {
        float qx = q.getX();
        float qy = q.getY();
        float qz = q.getZ();
        float qw = q.getW();
        return qx == 0.0f && qy == 0.0f && qz == 0.0f && qw != 0.0f && !Float.isNaN(qw);
    }

    public static boolean isZero(Quaternion q) {
        float qx = q.getX();
        float qy = q.getY();
        float qz = q.getZ();
        float qw = q.getW();
        return qx == 0.0f && qy == 0.0f && qz == 0.0f && qw == 0.0f;
    }

    public static double lengthSquared(Quaternion q) {
        double xx = q.getX();
        double yy = q.getY();
        double zz = q.getZ();
        double ww = q.getW();
        double result = xx * xx + yy * yy + zz * zz + ww * ww;
        return result;
    }

    public static Quaternion log(Quaternion q, Quaternion storeResult) {
        Quaternion result = storeResult == null ? new Quaternion() : storeResult;
        float qw = q.getW();
        if (qw >= 1.0f || qw <= -1.0f) {
            result.set(0.0f, 0.0f, 0.0f, 0.0f);
        } else {
            double qx = q.getX();
            double qy = q.getY();
            double qz = q.getZ();
            double sineTheta = MyMath.hypotenuseDouble(qx, qy, qz);
            if ((sineTheta = MyMath.clamp(sineTheta, 0.0, 1.0)) == 0.0) {
                result.set(0.0f, 0.0f, 0.0f, 0.0f);
            } else {
                double theta = Math.asin(sineTheta);
                double scale = theta / sineTheta;
                float x = (float)(scale * qx);
                float y = (float)(scale * qy);
                float z = (float)(scale * qz);
                result.set(x, y, z, 0.0f);
            }
        }
        return result;
    }

    public static Quaternion mirrorAxis(Quaternion input, int axisIndex, Quaternion storeResult) {
        Quaternion result = storeResult == null ? new Quaternion() : storeResult;
        float x = input.getX();
        float y = input.getY();
        float z = input.getZ();
        float w = input.getW();
        switch (axisIndex) {
            case 0: {
                y = -y;
                z = -z;
                break;
            }
            case 1: {
                x = -x;
                z = -z;
                break;
            }
            case 2: {
                x = -x;
                y = -y;
                break;
            }
            default: {
                String message = "axisIndex = " + axisIndex;
                throw new IllegalArgumentException(message);
            }
        }
        result.set(x, y, z, w);
        return result;
    }

    public static boolean ne(Quaternion a, Quaternion b) {
        assert (Validate.nonNull(a, "first input quaternion"));
        assert (Validate.nonNull(b, "2nd input quaternion"));
        boolean result = a.getW() != b.getW() || a.getX() != b.getX() || a.getY() != b.getY() || a.getZ() != b.getZ();
        return result;
    }

    public static void normalizeLocal(Quaternion input) {
        assert (Validate.nonNull(input, "input"));
        double lengthSquared = MyQuaternion.lengthSquared(input);
        double dScale = Math.sqrt(lengthSquared);
        float fScale = (float)dScale;
        if (fScale != 0.0f && fScale != 1.0f) {
            input.multLocal(1.0f / fScale);
        }
    }

    public static Quaternion pow(Quaternion base, float exponent, Quaternion storeResult) {
        Quaternion result = storeResult == null ? new Quaternion() : storeResult;
        float baseW = base.getW();
        if (baseW >= 1.0f || baseW <= -1.0f || exponent == 0.0f) {
            result.loadIdentity();
        } else {
            double baseX = base.getX();
            double baseY = base.getY();
            double baseZ = base.getZ();
            double sineTheta = MyMath.hypotenuseDouble(baseX, baseY, baseZ);
            if ((sineTheta = MyMath.clamp(sineTheta, 0.0, 1.0)) == 0.0) {
                result.loadIdentity();
            } else {
                double theta = Math.asin(sineTheta);
                float w = (float)Math.cos((double)exponent * theta);
                double scale = Math.sin((double)exponent * theta) / sineTheta;
                float x = (float)(scale * baseX);
                float y = (float)(scale * baseY);
                float z = (float)(scale * baseZ);
                result.set(x, y, z, w);
            }
        }
        return result;
    }

    public static Quaternion slerp(float t, Quaternion q0, Quaternion q1, Quaternion storeResult) {
        assert (Validate.inRange(t, "t", 0.0f, 1.0f));
        assert (MyQuaternion.validateUnit(q0, "q0", 1.0E-4f));
        assert (MyQuaternion.validateUnit(q1, "q1", 1.0E-4f));
        Quaternion result = storeResult == null ? new Quaternion() : storeResult;
        Quaternion q0inverse = MyQuaternion.conjugate(q0, null);
        Quaternion ratio = q0inverse.multLocal(q1);
        Quaternion power = MyQuaternion.pow(ratio, t, ratio);
        result.set(q0);
        result.multLocal(power);
        return result;
    }

    public static void snapLocal(Quaternion input, int axisIndex) {
        assert (Validate.axisIndex(axisIndex, "axis index"));
        float[] angles = new float[3];
        input.toAngles(angles);
        double angle = angles[axisIndex];
        angle = 1.5707963267948966 * (double)Math.round(angle / 1.5707963267948966);
        angles[axisIndex] = (float)angle;
        input.fromAngles(angles);
    }

    public static Quaternion squad(float t, Quaternion p, Quaternion a, Quaternion b, Quaternion q, Quaternion storeResult) {
        assert (Validate.inRange(t, "t", 0.0f, 1.0f));
        assert (MyQuaternion.validateUnit(p, "p", 1.0E-4f));
        assert (MyQuaternion.validateUnit(a, "a", 1.0E-4f));
        assert (MyQuaternion.validateUnit(b, "b", 1.0E-4f));
        assert (MyQuaternion.validateUnit(q, "q", 1.0E-4f));
        Quaternion result = storeResult == null ? new Quaternion() : storeResult;
        Quaternion qSlerp = MyQuaternion.slerp(t, p, q, null);
        Quaternion aSlerp = MyQuaternion.slerp(t, a, b, null);
        MyQuaternion.slerp(2.0f * t * (1.0f - t), qSlerp, aSlerp, result);
        return result;
    }

    public static Quaternion squadA(Quaternion q0, Quaternion q1, Quaternion q2, Quaternion storeResult) {
        assert (MyQuaternion.validateUnit(q0, "q0", 1.0E-4f));
        assert (MyQuaternion.validateUnit(q1, "q1", 1.0E-4f));
        assert (MyQuaternion.validateUnit(q2, "q2", 1.0E-4f));
        Quaternion result = storeResult == null ? new Quaternion() : storeResult;
        Quaternion q1c = MyQuaternion.conjugate(q1, null);
        Quaternion turn0 = q1c.mult(q0);
        Quaternion logTurn0 = MyQuaternion.log(turn0, turn0);
        Quaternion turn2 = q1c.mult(q2);
        Quaternion logTurn2 = MyQuaternion.log(turn2, turn2);
        Quaternion sum = logTurn2.addLocal(logTurn0);
        sum.multLocal(-0.25f);
        Quaternion exp = MyQuaternion.exp(sum, sum);
        result.set(q1);
        result.multLocal(exp);
        return result;
    }

    public static Quaternion standardize(Quaternion input, Quaternion storeResult) {
        assert (Validate.nonNull(input, "input quaternion"));
        Quaternion result = storeResult == null ? new Quaternion() : storeResult;
        float w = input.getW();
        float x = input.getX();
        float y = input.getY();
        float z = input.getZ();
        w = MyMath.standardize(w);
        x = MyMath.standardize(x);
        y = MyMath.standardize(y);
        z = MyMath.standardize(z);
        result.set(x, y, z, w);
        return result;
    }

    public static boolean validateUnit(Quaternion q, String description, float tolerance) {
        assert (Validate.nonNull(q, description));
        double norm = MyQuaternion.lengthSquared(q);
        double delta = Math.abs(1.0 - norm);
        if (!(delta <= (double)tolerance)) {
            String what = description == null ? "quaternion argument" : description;
            logger.log(Level.SEVERE, "norm({0})={1}", new Object[]{what, norm});
            String message = String.format("norm(%s) must be within %f of 1.", what, Float.valueOf(tolerance));
            throw new IllegalArgumentException(message);
        }
        return true;
    }
}

