/*
 * Decompiled with CFR 0.152.
 */
package com.github.susom.database;

import com.github.susom.database.Config;
import com.github.susom.database.Database;
import com.github.susom.database.DatabaseException;
import com.github.susom.database.DatabaseImpl;
import com.github.susom.database.DbCode;
import com.github.susom.database.DbCodeTx;
import com.github.susom.database.DbRun;
import com.github.susom.database.Flavor;
import com.github.susom.database.Metric;
import com.github.susom.database.Options;
import com.github.susom.database.OptionsDefault;
import com.github.susom.database.OptionsOverride;
import com.github.susom.database.TransactionImpl;
import java.io.Closeable;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.Properties;
import java.util.function.Supplier;
import javax.annotation.CheckReturnValue;
import javax.inject.Provider;
import javax.naming.Context;
import javax.sql.DataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import shaded.com.github.susom.database.shaded.com.zaxxer.hikari.HikariDataSource;

public final class DatabaseProvider
implements Provider<Database>,
Supplier<Database> {
    private static final Logger log = LoggerFactory.getLogger(DatabaseProvider.class);
    private DatabaseProvider delegateTo = null;
    private Provider<Connection> connectionProvider;
    private boolean txStarted = false;
    private Connection connection = null;
    private Database database = null;
    private final Options options;

    public DatabaseProvider(Provider<Connection> connectionProvider, Options options) {
        if (connectionProvider == null) {
            throw new IllegalArgumentException("Connection provider cannot be null");
        }
        this.connectionProvider = connectionProvider;
        this.options = options;
    }

    private DatabaseProvider(DatabaseProvider delegateTo) {
        this.delegateTo = delegateTo;
        this.options = delegateTo.options;
    }

    @CheckReturnValue
    public static Builder pooledBuilder(Config config) {
        return DatabaseProvider.fromPool(DatabaseProvider.createPool(config));
    }

    @CheckReturnValue
    public static Builder fromPool(Pool pool) {
        return new BuilderImpl(pool.poolShutdown, () -> {
            try {
                return pool.dataSource.getConnection();
            }
            catch (Exception e) {
                throw new DatabaseException("Unable to obtain a connection from the DataSource", e);
            }
        }, new OptionsDefault(pool.flavor));
    }

    @CheckReturnValue
    public static Builder fromDriverManager(String url) {
        return DatabaseProvider.fromDriverManager(url, Flavor.fromJdbcUrl(url), null, null, null);
    }

    @CheckReturnValue
    public static Builder fromDriverManager(String url, Flavor flavor) {
        return DatabaseProvider.fromDriverManager(url, flavor, null, null, null);
    }

    public static Builder fromDriverManager(Config config) {
        return DatabaseProvider.fromDriverManager(config.getString("database.url"), config.getString("database.user"), config.getString("database.password"));
    }

    @CheckReturnValue
    public static Builder fromDriverManager(String url, Properties info) {
        return DatabaseProvider.fromDriverManager(url, Flavor.fromJdbcUrl(url), info, null, null);
    }

    @CheckReturnValue
    public static Builder fromDriverManager(String url, Flavor flavor, Properties info) {
        return DatabaseProvider.fromDriverManager(url, flavor, info, null, null);
    }

    @CheckReturnValue
    public static Builder fromDriverManager(String url, String user, String password) {
        return DatabaseProvider.fromDriverManager(url, Flavor.fromJdbcUrl(url), null, user, password);
    }

    @CheckReturnValue
    public static Builder fromDriverManager(String url, Flavor flavor, String user, String password) {
        return DatabaseProvider.fromDriverManager(url, flavor, null, user, password);
    }

    private static Builder fromDriverManager(final String url, Flavor flavor, final Properties info, final String user, final String password) {
        OptionsDefault options = new OptionsDefault(flavor);
        return new BuilderImpl(null, (Provider)new Provider<Connection>(){

            public Connection get() {
                try {
                    if (info != null) {
                        return DriverManager.getConnection(url, info);
                    }
                    if (user != null) {
                        return DriverManager.getConnection(url, user, password);
                    }
                    return DriverManager.getConnection(url);
                }
                catch (Exception e) {
                    throw new DatabaseException("Unable to obtain a connection from DriverManager", e);
                }
            }
        }, options);
    }

    @CheckReturnValue
    public static Builder fromJndi(final Context context, final String lookupKey, Flavor flavor) {
        OptionsDefault options = new OptionsDefault(flavor);
        return new BuilderImpl(null, (Provider)new Provider<Connection>(){

            public Connection get() {
                DataSource ds;
                try {
                    ds = (DataSource)context.lookup(lookupKey);
                }
                catch (Exception e) {
                    throw new DatabaseException("Unable to locate the DataSource in JNDI using key " + lookupKey, e);
                }
                try {
                    return ds.getConnection();
                }
                catch (Exception e) {
                    throw new DatabaseException("Unable to obtain a connection from JNDI DataSource " + lookupKey, e);
                }
            }
        }, options);
    }

    public static Builder fromPropertyFile(String filename) {
        return DatabaseProvider.fromPropertyFile(filename, Charset.defaultCharset().newDecoder());
    }

    public static Builder fromPropertyFile(String filename, CharsetDecoder decoder) {
        Properties properties = new Properties();
        if (filename != null && filename.length() > 0) {
            try {
                properties.load(new InputStreamReader((InputStream)new FileInputStream(filename), decoder));
            }
            catch (Exception e) {
                throw new DatabaseException("Unable to read properties file: " + filename, e);
            }
        }
        return DatabaseProvider.fromProperties(properties, "", true);
    }

    public static Builder fromPropertyFile(String filename, String propertyPrefix) {
        return DatabaseProvider.fromPropertyFile(filename, propertyPrefix, Charset.defaultCharset().newDecoder());
    }

    public static Builder fromPropertyFile(String filename, String propertyPrefix, CharsetDecoder decoder) {
        Properties properties = new Properties();
        if (filename != null && filename.length() > 0) {
            try {
                properties.load(new InputStreamReader((InputStream)new FileInputStream(filename), decoder));
            }
            catch (Exception e) {
                throw new DatabaseException("Unable to read properties file: " + filename, e);
            }
        }
        return DatabaseProvider.fromProperties(properties, propertyPrefix, true);
    }

    public static Builder fromProperties(Properties properties) {
        return DatabaseProvider.fromProperties(properties, "", false);
    }

    public static Builder fromProperties(Properties properties, String propertyPrefix) {
        return DatabaseProvider.fromProperties(properties, propertyPrefix, false);
    }

    public static Builder fromPropertyFileOrSystemProperties(String filename) {
        return DatabaseProvider.fromPropertyFileOrSystemProperties(filename, Charset.defaultCharset().newDecoder());
    }

    public static Builder fromPropertyFileOrSystemProperties(String filename, CharsetDecoder decoder) {
        Properties properties = new Properties();
        if (filename != null && filename.length() > 0) {
            try {
                properties.load(new InputStreamReader((InputStream)new FileInputStream(filename), decoder));
            }
            catch (Exception e) {
                log.debug("Trying system properties - unable to read properties file: " + filename);
            }
        }
        return DatabaseProvider.fromProperties(properties, "", true);
    }

    public static Builder fromPropertyFileOrSystemProperties(String filename, String propertyPrefix) {
        return DatabaseProvider.fromPropertyFileOrSystemProperties(filename, propertyPrefix, Charset.defaultCharset().newDecoder());
    }

    public static Builder fromPropertyFileOrSystemProperties(String filename, String propertyPrefix, CharsetDecoder decoder) {
        Properties properties = new Properties();
        if (filename != null && filename.length() > 0) {
            try {
                properties.load(new InputStreamReader((InputStream)new FileInputStream(filename), decoder));
            }
            catch (Exception e) {
                log.debug("Trying system properties - unable to read properties file: " + filename);
            }
        }
        return DatabaseProvider.fromProperties(properties, propertyPrefix, true);
    }

    @CheckReturnValue
    public static Builder fromSystemProperties() {
        return DatabaseProvider.fromProperties(null, "", true);
    }

    @CheckReturnValue
    public static Builder fromSystemProperties(String propertyPrefix) {
        return DatabaseProvider.fromProperties(null, propertyPrefix, true);
    }

    private static Builder fromProperties(Properties properties, String propertyPrefix, boolean useSystemProperties) {
        String password;
        String user;
        String url;
        String flavorStr;
        String driver;
        if (propertyPrefix == null) {
            propertyPrefix = "";
        }
        if (useSystemProperties) {
            if (properties == null) {
                properties = new Properties();
            }
            driver = System.getProperty(propertyPrefix + "database.driver", properties.getProperty(propertyPrefix + "database.driver"));
            flavorStr = System.getProperty(propertyPrefix + "database.flavor", properties.getProperty(propertyPrefix + "database.flavor"));
            url = System.getProperty(propertyPrefix + "database.url", properties.getProperty(propertyPrefix + "database.url"));
            user = System.getProperty(propertyPrefix + "database.user", properties.getProperty(propertyPrefix + "database.user"));
            password = System.getProperty(propertyPrefix + "database.password", properties.getProperty(propertyPrefix + "database.password"));
        } else {
            if (properties == null) {
                throw new DatabaseException("No properties were provided");
            }
            driver = properties.getProperty(propertyPrefix + "database.driver");
            flavorStr = properties.getProperty(propertyPrefix + "database.flavor");
            url = properties.getProperty(propertyPrefix + "database.url");
            user = properties.getProperty(propertyPrefix + "database.user");
            password = properties.getProperty(propertyPrefix + "database.password");
        }
        if (url == null) {
            throw new DatabaseException("You must use -D" + propertyPrefix + "database.url=...");
        }
        if (user != null && password == null) {
            System.out.println("Enter database password for user " + user + ":");
            byte[] input = new byte[256];
            try {
                int bytesRead = System.in.read(input);
                password = new String(input, 0, bytesRead - 1, Charset.defaultCharset());
            }
            catch (IOException e) {
                throw new DatabaseException("Error reading password from standard input", e);
            }
        }
        Flavor flavor = flavorStr != null ? Flavor.valueOf(flavorStr) : Flavor.fromJdbcUrl(url);
        if (driver == null) {
            if (flavor == Flavor.oracle) {
                driver = "oracle.jdbc.OracleDriver";
            } else if (flavor == Flavor.postgresql) {
                driver = "org.postgresql.Driver";
            } else if (flavor == Flavor.derby) {
                driver = "org.apache.derby.jdbc.EmbeddedDriver";
            }
        }
        if (driver != null) {
            try {
                Class.forName(driver).newInstance();
            }
            catch (Exception e) {
                throw new DatabaseException("Unable to load JDBC driver: " + driver, e);
            }
        }
        if (user == null) {
            return DatabaseProvider.fromDriverManager(url, flavor);
        }
        return DatabaseProvider.fromDriverManager(url, flavor, user, password);
    }

    @Deprecated
    public void transact(DbRun run) {
        boolean complete = false;
        try {
            run.run(this);
            complete = true;
        }
        catch (DatabaseException | ThreadDeath t) {
            throw t;
        }
        catch (Throwable t) {
            throw new DatabaseException("Error during transaction", t);
        }
        finally {
            if (run.isRollbackOnly() || run.isRollbackOnError() && !complete) {
                this.rollbackAndClose();
            } else {
                this.commitAndClose();
            }
        }
    }

    public void transact(DbCode code) {
        boolean complete = false;
        try {
            code.run(this);
            complete = true;
        }
        catch (DatabaseException | ThreadDeath t) {
            throw t;
        }
        catch (Throwable t) {
            throw new DatabaseException("Error during transaction", t);
        }
        finally {
            if (!complete) {
                this.rollbackAndClose();
            } else {
                this.commitAndClose();
            }
        }
    }

    public void transact(DbCodeTx code) {
        TransactionImpl tx = new TransactionImpl();
        tx.setRollbackOnError(true);
        tx.setRollbackOnly(false);
        boolean complete = false;
        try {
            code.run(this, tx);
            complete = true;
        }
        catch (DatabaseException | ThreadDeath t) {
            throw t;
        }
        catch (Throwable t) {
            throw new DatabaseException("Error during transaction", t);
        }
        finally {
            if (!complete && tx.isRollbackOnError() || tx.isRollbackOnly()) {
                this.rollbackAndClose();
            } else {
                this.commitAndClose();
            }
        }
    }

    @Override
    public Database get() {
        if (this.delegateTo != null) {
            return this.delegateTo.get();
        }
        if (this.database != null) {
            return this.database;
        }
        if (this.connectionProvider == null) {
            throw new DatabaseException("Called get() on a DatabaseProvider after close()");
        }
        Metric metric = new Metric(log.isDebugEnabled());
        try {
            this.connection = (Connection)this.connectionProvider.get();
            this.txStarted = true;
            metric.checkpoint("getConn", new Object[0]);
            try {
                this.connection.setAutoCommit(false);
                metric.checkpoint("setAutoCommit", new Object[0]);
            }
            catch (SQLException e) {
                throw new DatabaseException("Unable to check/set autoCommit for the connection", e);
            }
            this.database = new DatabaseImpl(this.connection, this.options);
            metric.checkpoint("dbInit", new Object[0]);
        }
        catch (RuntimeException e) {
            metric.checkpoint("fail", new Object[0]);
            throw e;
        }
        finally {
            metric.done();
            if (log.isDebugEnabled()) {
                StringBuilder buf = new StringBuilder("Get ").append((Object)this.options.flavor()).append(" database: ");
                metric.printMessage(buf);
                log.debug(buf.toString());
            }
        }
        return this.database;
    }

    public Builder fakeBuilder() {
        return new Builder(){

            @Override
            public Builder withOptions(OptionsOverride optionsOverride) {
                return this;
            }

            @Override
            public Builder withSqlParameterLogging() {
                return this;
            }

            @Override
            public Builder withSqlInExceptionMessages() {
                return this;
            }

            @Override
            public Builder withDatePerAppOnly() {
                return this;
            }

            @Override
            public Builder withTransactionControl() {
                return this;
            }

            @Override
            public Builder withTransactionControlSilentlyIgnored() {
                return this;
            }

            @Override
            public Builder withConnectionAccess() {
                return this;
            }

            @Override
            public DatabaseProvider create() {
                return new DatabaseProvider(DatabaseProvider.this);
            }

            @Override
            public void transact(DbRun dbRun) {
                this.create().transact(dbRun);
            }

            @Override
            public void transact(DbCode tx) {
                this.create().transact(tx);
            }

            @Override
            public void transact(DbCodeTx tx) {
                this.create().transact(tx);
            }
        };
    }

    public void commitAndClose() {
        if (this.delegateTo != null) {
            log.debug("Ignoring commitAndClose() because this is a fake provider");
            return;
        }
        if (this.txStarted) {
            try {
                this.connection.commit();
            }
            catch (Exception e) {
                throw new DatabaseException("Unable to commit the transaction", e);
            }
            this.close();
        }
    }

    public void rollbackAndClose() {
        if (this.delegateTo != null) {
            log.debug("Ignoring rollbackAndClose() because this is a fake provider");
            return;
        }
        if (this.txStarted) {
            try {
                this.connection.rollback();
            }
            catch (Exception e) {
                log.error("Unable to rollback the transaction", (Throwable)e);
            }
            this.close();
        }
    }

    private void close() {
        try {
            this.connection.close();
        }
        catch (Exception e) {
            log.error("Unable to close the database connection", (Throwable)e);
        }
        this.connection = null;
        this.database = null;
        this.txStarted = false;
        this.connectionProvider = null;
    }

    public static Pool createPool(Config config) {
        String url = config.getString("database.url");
        if (url == null) {
            throw new DatabaseException("You must provide database.url");
        }
        HikariDataSource ds = new HikariDataSource();
        ds.setJdbcUrl(url);
        String driverClassName = config.getString("database.driver.class", Flavor.driverForJdbcUrl(url));
        ds.setDriverClassName(driverClassName);
        ds.setUsername(config.getString("database.user"));
        ds.setPassword(config.getString("database.password"));
        int poolSize = config.getInteger("database.pool.size", 10);
        ds.setMaximumPoolSize(poolSize);
        ds.setAutoCommit(false);
        String flavorString = config.getString("database.flavor");
        Flavor flavor = flavorString != null ? Flavor.valueOf(flavorString) : Flavor.fromJdbcUrl(url);
        log.debug("Created '" + (Object)((Object)flavor) + "' connection pool of size " + poolSize + " using driver " + driverClassName);
        return new Pool(ds, flavor, ds);
    }

    public static class Pool {
        public DataSource dataSource;
        public Flavor flavor;
        public Closeable poolShutdown;

        public Pool(DataSource dataSource, Flavor flavor, Closeable poolShutdown) {
            this.dataSource = dataSource;
            this.flavor = flavor;
            this.poolShutdown = poolShutdown;
        }
    }

    private static class BuilderImpl
    implements Builder {
        private Closeable pool;
        private final Provider<Connection> connectionProvider;
        private final Options options;

        private BuilderImpl(Closeable pool, Provider<Connection> connectionProvider, Options options) {
            this.pool = pool;
            this.connectionProvider = connectionProvider;
            this.options = options;
        }

        @Override
        public Builder withOptions(OptionsOverride options) {
            return new BuilderImpl(this.pool, this.connectionProvider, options.withParent(this.options));
        }

        @Override
        public Builder withSqlParameterLogging() {
            return new BuilderImpl(this.pool, this.connectionProvider, new OptionsOverride(){

                @Override
                public boolean isLogParameters() {
                    return true;
                }
            }.withParent(this.options));
        }

        @Override
        public Builder withSqlInExceptionMessages() {
            return new BuilderImpl(this.pool, this.connectionProvider, new OptionsOverride(){

                @Override
                public boolean isDetailedExceptions() {
                    return true;
                }
            }.withParent(this.options));
        }

        @Override
        public Builder withDatePerAppOnly() {
            return new BuilderImpl(this.pool, this.connectionProvider, new OptionsOverride(){

                @Override
                public boolean useDatePerAppOnly() {
                    return true;
                }
            }.withParent(this.options));
        }

        @Override
        public Builder withTransactionControl() {
            return new BuilderImpl(this.pool, this.connectionProvider, new OptionsOverride(){

                @Override
                public boolean allowTransactionControl() {
                    return true;
                }
            }.withParent(this.options));
        }

        @Override
        public Builder withTransactionControlSilentlyIgnored() {
            return new BuilderImpl(this.pool, this.connectionProvider, new OptionsOverride(){

                @Override
                public boolean ignoreTransactionControl() {
                    return true;
                }
            }.withParent(this.options));
        }

        @Override
        public Builder withConnectionAccess() {
            return new BuilderImpl(this.pool, this.connectionProvider, new OptionsOverride(){

                @Override
                public boolean allowConnectionAccess() {
                    return true;
                }
            }.withParent(this.options));
        }

        @Override
        public DatabaseProvider create() {
            return new DatabaseProvider(this.connectionProvider, this.options);
        }

        @Override
        public void transact(DbRun run) {
            this.create().transact(run);
        }

        @Override
        public void transact(DbCode tx) {
            this.create().transact(tx);
        }

        @Override
        public void transact(DbCodeTx tx) {
            this.create().transact(tx);
        }

        public void close() {
            if (this.pool != null) {
                try {
                    this.pool.close();
                }
                catch (IOException e) {
                    log.warn("Unable to close connection pool", (Throwable)e);
                }
                this.pool = null;
            }
        }
    }

    public static interface Builder {
        @CheckReturnValue
        public Builder withOptions(OptionsOverride var1);

        @CheckReturnValue
        public Builder withSqlParameterLogging();

        @CheckReturnValue
        public Builder withSqlInExceptionMessages();

        @CheckReturnValue
        public Builder withDatePerAppOnly();

        @CheckReturnValue
        public Builder withTransactionControl();

        @CheckReturnValue
        public Builder withTransactionControlSilentlyIgnored();

        @CheckReturnValue
        public Builder withConnectionAccess();

        @CheckReturnValue
        public DatabaseProvider create();

        @Deprecated
        public void transact(DbRun var1);

        public void transact(DbCode var1);

        public void transact(DbCodeTx var1);
    }
}

