/**
 * Logback: the reliable, generic, fast and flexible logging framework.
 * Copyright (C) 1999-2022, QOS.ch. All rights reserved.
 *
 * This program and the accompanying materials are dual-licensed under
 * either the terms of the Eclipse Public License v1.0 as published by
 * the Eclipse Foundation
 *
 *   or (per the licensee's choosing)
 *
 * under the terms of the GNU Lesser General Public License version 2.1
 * as published by the Free Software Foundation.
 */
package org.apache.juli.logging.ch.qos.logback.core.model.processor;

import org.apache.juli.logging.ch.qos.logback.core.Context;
import org.apache.juli.logging.ch.qos.logback.core.CoreConstants;
import org.apache.juli.logging.ch.qos.logback.core.hook.DefaultShutdownHook;
import org.apache.juli.logging.ch.qos.logback.core.hook.ShutdownHook;
import org.apache.juli.logging.ch.qos.logback.core.hook.ShutdownHookBase;
import org.apache.juli.logging.ch.qos.logback.core.model.Model;
import org.apache.juli.logging.ch.qos.logback.core.model.ShutdownHookModel;
import org.apache.juli.logging.ch.qos.logback.core.util.ContextUtil;
import org.apache.juli.logging.ch.qos.logback.core.util.DynamicClassLoadingException;
import org.apache.juli.logging.ch.qos.logback.core.util.IncompatibleClassException;
import org.apache.juli.logging.ch.qos.logback.core.util.OptionHelper;

public class ShutdownHookModelHandler extends ModelHandlerBase {

    static final String OLD_SHUTDOWN_HOOK_CLASSNAME = "org.apache.juli.logging.ch.qos.logback.core.hook.DelayingShutdownHook";
    static final String DEFAULT_SHUTDOWN_HOOK_CLASSNAME = DefaultShutdownHook.class.getName();
    static public final String RENAME_WARNING = OLD_SHUTDOWN_HOOK_CLASSNAME + " was renamed as "+ DEFAULT_SHUTDOWN_HOOK_CLASSNAME;

    public ShutdownHookModelHandler(Context context) {
        super(context);
    }
    boolean inError = false;
    ShutdownHook  hook = null;

    static public ModelHandlerBase makeInstance(Context context, ModelInterpretationContext mic) {
        return new ShutdownHookModelHandler(context);
    }

    @Override
    protected Class<ShutdownHookModel> getSupportedModelClass() {
        return ShutdownHookModel.class;
    }

    @Override
    public void handle(ModelInterpretationContext mic, Model model) {

        ShutdownHookModel shutdownHookModel = (ShutdownHookModel) model;

        String className = shutdownHookModel.getClassName();
        if (OptionHelper.isNullOrEmptyOrAllSpaces(className)) {
            className = DEFAULT_SHUTDOWN_HOOK_CLASSNAME;
            addInfo("Assuming className [" + className + "]");
        } else {
            className = mic.getImport(className);
            if(className.equals(OLD_SHUTDOWN_HOOK_CLASSNAME)) {
                className = DEFAULT_SHUTDOWN_HOOK_CLASSNAME;
                addWarn(RENAME_WARNING);
                addWarn("Please use the new class name");
            }
        }

        addInfo("About to instantiate shutdown hook of type [" + className + "]");

        try {
            hook = (ShutdownHookBase) OptionHelper.instantiateByClassName(className, ShutdownHookBase.class, context);
            hook.setContext(context);
        } catch (IncompatibleClassException | DynamicClassLoadingException e) {
            addError("Could not create a shutdown hook of type [" + className + "].", e);
            inError = true;
            return;
        }

        mic.pushObject(hook);
    }

    @Override
    public void postHandle(ModelInterpretationContext mic, Model model) throws ModelHandlerException {
        if (inError) {
            return;
        }
        Object o = mic.peekObject();

        if (o != hook) {
            addWarn("The object on the top the of the stack is not the hook object pushed earlier.");
        } else {
            ContextUtil contextUtil = new ContextUtil(context);
            contextUtil.addOrReplaceShutdownHook(hook);
            mic.popObject();
        }
    }
}