/*
 * Decompiled with CFR 0.152.
 */
package com.github.tommyettinger.digital;

import com.github.tommyettinger.digital.BitConversion;

public final class MathTools {
    public static final float FLOAT_ROUNDING_ERROR = 9.536743E-7f;
    public static final float EPSILON = 5.9604645E-8f;
    public static final double EPSILON_D = (double)1.110223E-16f;
    public static final float E = (float)Math.E;
    public static final float ROOT2 = 1.4142135f;
    public static final double ROOT2_D = 1.4142135623730951;
    public static final float ROOT2_INVERSE = 0.70710677f;
    public static final double ROOT2_INVERSE_D = 0.7071067811865475;
    public static final float ROOT3 = 1.7320508f;
    public static final double ROOT3_D = 1.7320508075688772;
    public static final float ROOT5 = 2.236068f;
    public static final double ROOT5_D = 2.23606797749979;
    public static final float GOLDEN_RATIO = 1.618034f;
    public static final float PHI = 1.618034f;
    public static final double GOLDEN_RATIO_D = 1.618033988749895;
    public static final double PHI_D = 1.618033988749895;
    public static final float GOLDEN_RATIO_INVERSE = 0.618034f;
    public static final double GOLDEN_RATIO_INVERSE_D = 0.6180339887498949;
    public static final float PSI = -0.618034f;
    public static final double PSI_D = -0.6180339887498949;
    private static final int BIG_ENOUGH_INT = 16384;
    private static final double BIG_ENOUGH_FLOOR = 16384.0;
    private static final double CEIL = 0.9999999403953552;
    private static final double BIG_ENOUGH_ROUND = 16384.5;
    public static final long[] GOLDEN_LONGS = new long[]{-7046029254386353131L, -4521708957497675121L, -7935046062780286179L, -3335678366873096957L, -6068174398545744893L, -8306560040656073257L, -2643881736870682267L, -4908828816729415329L, -6849152093303990923L, -8511378310903703927L, -2190154961491147213L, -4120276025608464847L, -5821236636912874053L, -7320244681018240793L, -8641277691047455699L, -1869509025959006485L, -3549550252929931025L, -5059325538916627611L, -6416090704903628737L, -7635352758483140741L, -8731047129599661285L, -1630833229475512307L, -3117488331269834705L, -4472711754330624987L, -5708123064052617379L, -6834314567400436299L, -7860942130408768077L, -8796807966721079869L, -1446237285916620779L, -2779088584646676641L, -4007443434729566763L, -5139494428835061493L, -6182791854793659543L, -7144294052732989265L, -8030413824180454797L, -8847061202660659549L, -1299201819648055891L, -2506901021737576647L, -3629542123889644151L, -4673115757175681021L, -5643190633288940319L, -6544943260272206457L, -7383185565370604643L, -8162390572410429077L, -8886716270723472955L, -1179316746353344257L, -2283238724791433465L, -3316585989847175981L, -4283870445682537651L, -5189315546415677883L, -6036874736988659851L, -6830248715094095491L, -7572901589531475569L, -8268076005545416047L, -8918807303187585107L, -1079694989804077987L, -2096195018679010113L, -3053198911871323597L, -3954189000970406023L, -4802443795365105159L, -5601049912035124909L, -6352913307087165557L, -7060769849904955699L, -7727195278390236269L, -8354614571519680137L, -8945310773322468517L, -995597925996739729L, -1937461966691644673L, -2828492218955386887L, -3671432257162413973L, -4468877580940426719L, -5223283606982384757L, -5936973229528656621L, -6612143972798821275L, -7250874757396199009L, -7855132301519571547L, -8426777176692084901L, -8967569536653551015L, -923657991232249891L, -1801066948957112275L, -2634542636282438197L, -3426284862351963387L, -4178383288345822065L, -4892822942765325219L, -5571489460558605383L, -6216174059914857571L, -6828578269862529541L, -7410318421149106321L, -7962929912255358371L, -8487871261803430539L, -8986527958054374029L, -861416141443125113L, -1682606334839607113L, -2465449029893836721L, -3211734957475391011L, -3923171225866041299L, -4601385225722419577L, -5247928352686836881L, -5864279556161622207L, -6451848722364704461L, -7011979899405082907L, -7545954371755456447L, -8054993591154782305L, -8540261970645122071L, -9002869548134056653L, -807034766619522789L, -1578762212232685821L, -2316727012444813851L, -3022406264164868471L, -3697212442135729557L, -4342496226117925563L, -4959549204385652171L, -5549606458946347691L, -6113849037658353647L, -6653406318194806283L, -7169358268585426019L, -7662737608860864215L, -8134531878126315147L, -8585685411201810167L, -9017101228785601225L, -759113014837439223L, -1486987315298347031L, -2184908424429785775L, -2854108963949889555L, -3495770831222138445L, -4111027286643660369L, -4700964955134166629L, -5266625745260417379L, -5809008689385646669L, -6329071708093896279L, -6827733302005466891L, -7305874173971458273L, -7764338784512410237L, -8203936843248156395L, -8625444738952954643L, -9029606910761562389L, -716564290116671515L, -1405293620317451803L, -2067269240399750001L, -2703530398982316105L, -3315075975042202165L, -3902866046074624541L, -4467823395337612377L, -5010834960547695515L, -5532753226300967831L, -6034397562405515385L, -6516555510226283907L, -6979984019061843897L, -7425410634494065651L, -7853534640576317891L, -8265028157653333825L, -8660537197536233785L, -9040682677689244533L, -678532797564381177L, -1332106894800354757L, -1961640355984724025L, -2568017475926922083L, -3152090022122698991L, -3714678431217251661L, -4256572961458447617L, -4778534802758973865L, -5281297145926696011L, -5765566212565157125L, -6232022247090899801L, -6681320472261081119L, -7114092009553593641L, -7530944765692534549L, -7932464286564310021L, -8319214579723856027L, -8691738906646335929L, -9050560545837177153L, -644335345744767269L, -1266164387064368819L, -1866273257673528539L, -2445420632037963863L, -3004338684511924189L, -3543734014973545095L, -4064288542128232575L, -4566660365609416457L, -5051484597966564967L, -5519374167592283027L, -5970920593603577229L, -6406694733656914131L, -6827247505642480629L, -7233110584170032557L, -7624797072726848419L, -8002802152357549229L, -8367603707685863665L, -8719662931069772803L, -9059424905653824327L, -613419915454104031L, -1206441434791681291L, -1779742877306976909L, -2333980005762716733L, -2869786776475087219L, -3387776064455719237L, -3888540364440119255L, -4372652468604409321L, -4840666121745572615L, -5293116654674623239L, -5730521596547198053L, -6153381266831976307L, -6562179347594041745L, -6957383436747785455L, -7339445582912179993L, -7708802802480211401L, -8065877579493911733L, -8411078348896767065L, -8744799963716262439L, -9067424146710943893L, -585335617120635399L, -1152097886917701825L, -1700876162269636593L, -2232241095235403209L, -2746745230452202299L, -3244923579704016045L, -3727294178258456443L, -4194358623550429279L, -4646602596772768915L, -5084496367916224639L, -5508495284783968971L, -5919040246489134091L, -6316558161927747091L, -6701462393703811289L, -7074153187968153113L, -7435018090618006381L, -7784432350290122867L, -8122759308566465211L, -8450350777798241089L, -8767547406941162361L, -9074679035782346345L, -559710544201952807L, -1102438368199241991L, -1628698761168635009L, -2138991377398466029L, -2633800710681942335L, -3113596554316423547L, -3578834447145416781L, -4029956106066779253L, -4467389845417770221L, -4891550983635134525L, -5302842237576318313L, -5701654104876202501L, -6088365234702379933L, -6463342787260987269L, -6826942782394421911L, -7179510437601917687L, -7521380495803910547L, -7852877543161387935L, -8174316317251973163L, -8486002005895340471L, -8788230536911678493L, -9081288859088311289L, -536235202809451485L, -1056882386856182153L, -1562394686003231723L, -2053212061798353495L, -2529761686384902613L, -2992458314281746601L, -3441704643355768635L, -3877891665301128065L, -4301399005930310133L, -4712595255573129737L, -5111838289871244763L, -5499475581247375487L, -5875844501320310411L, -6241272614528898839L, -6596077963220579383L, -6940569344452564951L, -7275046578746592061L, -7599800771031139287L, -7915114563998220179L, -8221262384095254181L, -8518510680366109179L, -8807118156349185755L, -9087335995234370551L, -514649945370257159L, -1014941553832317289L, -1501275411921987701L, -1974040930119401841L, -2433616654652360793L, -2880370570600601819L, -3314660396543691355L, -3736833870988468847L, -4147229030805385613L, -4546174481896684391L, -4933989662313145253L, -5310985098030077057L, -5677462651587355815L, -6033715763792597593L, -6380029688680999135L, -6716681721919979973L, -7043941422841511027L, -7362070830279912331L, -7671324672387942511L, -7971950570599181039L, -8264189237900017203L, -8548274671570003343L, -8824434340544900767L, -9092889367552441047L, -494735323191866831L, -976202015397379859L, -1444755936730015861L, -1900743403257562661L, -2344501442970137621L, -2776358044883232065L, -3196632401459897813L, -3605635144531254167L, -4003668574889688153L, -4391026885724444193L, -4767996380064748223L, -5134855682391182099L, -5491875944571713813L, -5839321046274594319L, -6177447790006249459L, -6506506090918322831L, -6826739161524159101L, -7138383691461254869L, -7441670022432542505L, -7736822318455808993L, -8024058731547084007L, -8303591562960456571L, -8575627420103495363L, -8840367369244251491L, -9098007084122712047L, -476304595659548751L, -940310757137981629L, -1392336036448622791L, -1832689786245456731L, -2261673371534881809L, -2679580375920955415L, -3086696802525278843L, -3483301269719025489L, -3869665201801065537L, -4246053014752681581L, -4612722297196000183L, -4969923986679981913L, -5317902541414614787L, -5656896107570840871L, -5987136682260711121L, -6308850272309307257L, -6622257048927089427L, -6927571498388522843L, -7225002568820103323L, -7514753813198239133L, -7797023528654852591L, -8072004892186038069L, -8339886092856651359L, -8600850460591307267L, -8855076591639926187L, -9102738470803694543L, -459197856978340503L, -906964825610697847L, -1343585456894844861L, -1769337218453414777L, -2184490670861532337L, -2589309639585091967L, -2984051382638947901L, -3368966754071561195L, -3744300363379996337L, -4110290730956573923L, -4467170439665964055L, -4815166282649046033L, -5154499407447462077L, -5485385456540454643L, -5808034704383296981L, -6122652191034403307L, -6429437852456037193L, -6728586647571422831L, -7020288682159002567L, -7304729329662574155L, -7582089348994081247L, -7852544999403919497L, -8116268152491757093L, -8373426401429051389L, -8624183167462671319L, -8868697803767307487L, -9107125696712667019L, -443277353706131783L, -875902702292901765L, -1298132014652278501L, -1710215108714705323L, -2112395799257403905L, -2504912042160827497L, -2887996075198615243L, -3261874555444347885L, -3626768693376403555L, -3982894383760258657L, -4330462333385672201L, -4669678185734331099L, -5000742642651717805L, -5323851583095189151L, -5639196179028525387L, -5946963008531520045L, -6247334166191533523L, -6540487370842325101L, -6826596070713908615L, -7105829546055645145L, -7378353009293291141L, -7644327702779261335L, -7903910994193941709L, -8157256469654498119L, -8404514024586269651L, -8645829952410512095L, -8881347031100964885L, -9111204607660453893L, -428423774188374677L, -846897449348086403L, -1255652115527458845L, -1654913495731705809L, -2044902070571967193L, -2425833200019383643L, -2797917242331445059L, -3161359670216286651L, -3516361184299080899L, -3863117823953183981L, -4201821075557239997L, -4532657978238024823L, -4855811227157423075L, -5171459274400575437L, -5479776427520908897L, -5780932945796468607L, -6075095134250706105L, -6362425435489644197L, -6643082519406132939L, -6917221370800733291L, -7184993374967614583L, -7446546401292728113L, -7702024884910421591L, -7951569906463586917L, -8195319270011386579L, -8433407579127580925L, -8665966311231479455L, -8893123890192563257L, -9115005757248872377L, -414532997051093231L, -819750658834481089L, -1215862318435266713L, -1603072604830836289L, -1981581548600997525L, -2351584685262612699L, -2713273156282108359L, -3066833807818044151L, -3412449287244750605L, -3750298137506899439L, -4080554889353749379L, -4403390151500715197L, -4718970698764836903L, -5027459558219679345L, -5329016093414169347L, -5623796086698877355L, -5911951819702272851L, -6193632151998527115L, -6468982598007502657L, -6738145402166655403L, -7001259612413683011L, -7258461152017880019L, -7509882889797307485L, -7755654708758050895L, -7995903573191025005L, -8230753594260987399L, -8460326094121643683L, -8684739668589965829L, -8904110248412100797L, -9118551159152519123L, -401517742898554689L, -794295921935966159L, -1178524765259891381L, -1554390360451165463L, -1922074744648705973L, -2281755992712729701L, -2633608303468980605L, -2977802084075738251L, -3314504032554466901L, -3643877218524076067L, -3966081162177893259L, -4281271911541598637L, -4589602118049538707L, -4891221110476021761L, -5186274967257401021L, -5474906587239972175L, -5757255758887949491L, -6033459227985038937L, -6303650763862397187L, -6567961224185051621L, -6826518618328158365L, -7079448169373792383L, -7326872374758295569L, -7568911065599555229L, -7805681464732945997L, -8037298243484042853L, -8263873577205601071L, -8485517199605700467L, -8702336455893365843L, -8914436354767402833L, -9121919619273628083L, -389297556979196597L, -770379431707672061L, -1143419006816873065L, -1508586005894971485L, -1866046570706811835L, -2215963336784219529L, -2558495507421061635L, -2893798926106726023L, -3222026147430974297L, -3543326506492428445L, -3857846186842270289L, -4165728286994066439L, -4467112885529979015L, -4762137104832983821L, -5050935173474092537L, -5333638487282963525L, -5610375669129686843L, -5881272627444942669L, -6146452613505158315L, -6406036277508727163L, -6660141723468802755L, -6908884562947642885L, -7152377967656951469L, -7390732720948150009L, -7624057268216005433L, -7852457766238546685L, -8076038131475718481L, -8294900087348746901L, -8509143210521727735L, -8718864976206494533L, -8924160802511378943L, -9125124093854040869L, -377770870813703545L, -747805371985962531L, -1110261936630214603L, -1465295753308176519L, -1813058832475018661L, -2153700071563808803L, -2487365318737091579L, -2814197435332899347L, -3134336357031931195L, -3447919153772089309L, -3755080088436025541L, -4055950674336825715L, -4350659731526444613L, -4639333441951000515L, -4922095403476544485L, -5199066682808435899L, -5470365867326982119L, -5736109115861536079L, -5996410208424791147L, -6251380594928567353L, -6501129442901947037L, -6745763684232190809L, -6985388060948446279L, -7220105170067852229L, -7450015507523239419L, -7675217511191236007L, -7895807603039200399L, -8111880230409027073L, -8323527906455501337L, -8530841249756517029L, -8733909023112116557L, -8932818171548965423L, -9127653859546533101L, -366969762528547673L, -726639222842668957L, -1079153609290483717L, -1424655261124225351L, -1763283685974385133L, -2095175616180496333L, -2420465064001303551L, -2739283375726610149L, -3051759284712653245L, -3358018963362421019L, -3658186074071901113L, -3952381819162831313L, -4240724989822114543L, -4523332014067659019L, -4800317003760011381L, -5071791800678765273L, -5337866021682350245L, -5598647102969435711L, -5854240343459821975L, -6104748947312334763L, -6350274065596891269L, -6590914837137564177L, -6826768428543135393L, -7057930073441303119L, -7284493110932384379L, -7506549023278039957L, -7724187472840239765L, -7937496338285384007L, -8146561750068198709L, -8351468125209733421L, -8552298201383503833L, -8749133070323542711L, -8942052210567848767L, -9131133519550454669L, -356751606092980565L, -706603799046093819L, -1049690010358628277L, -1386141091025611961L, -1716085361443586317L, -2039648660351225273L, -2356954392823465063L, -2668123577337449513L, -2973274891928241421L, -3272524719451903573L, -3565987191973212435L, -3853774234294933731L, -4135995606645261725L, -4412758946539702897L, -4684169809833369931L, -4950331710979343071L, -5211346162508453159L, -5467312713745543689L, -5718328988776978057L, -5964490723683872527L, -6205891803055255511L, -6442624295795078987L, -6674778490236738703L, -6902442928578495569L, -7125704440652931743L, -7344648177043320845L, -7559357641559542715L, -7769914723085928781L, -7976399726813184665L, -8178891404866301669L, -8377466986340138421L, -8572202206754128109L, -8763171336937345113L, -8950447211354947661L, -9134101255886799985L, -347215943435425085L, -687896375189546267L, -1022164310676096709L, -1350140449837159635L, -1671943220726397433L, -1987688822271930487L, -2297491266234306829L, -2601462418374713125L, -2899712038848292325L, -3192347821837153469L, -3479475434437384677L, -3761198554814110869L, -4037618909638373581L, -4308836310819350849L, -4574948691545180669L, -4836052141645401981L, -5092240942287782051L, -5343607600022058879L, -5590242880182891375L, -5832235839664078699L, -6069673859075883133L, -6302642674297068049L, -6531226407433044079L, -6755507597191302025L, -6975567228685100729L, -7191484762676171625L, -7403338164266999161L, -7611203931053037497L, -7815157120745028879L, -8015271378271397807L, -8211618962370507269L, -8404270771682379219L, -8593296370349300661L, -8778764013134559421L, -8960740670068379627L, -9139292050629956283L, -337540979504908275L, -668905589218493189L, -994206845177667133L, -1313555695436849401L, -1627061057908185215L, -1934829857509359417L, -2236967062631674485L, -2533575720940830823L, -2824756994522619601L, -3110610194385515019L, -3391232814331933563L, -3666720564209712417L, -3937167402555147933L, -4202665568638727399L, -4463305613924483775L, -4719176432953702935L, -4970365293663516745L, -5216957867150722477L, -5459038256890979889L, -5696689027423351551L, -5929991232509969637L, -6159024442780433371L, -6383866772870365623L, -6604594908063384567L, -6821284130445577007L, -7034008344581393653L, -7242840102719723471L, -7447850629538743889L, -7649109846437986485L, -7846686395385903181L, -8040647662331066523L, -8231059800184988683L, -8417987751384397793L, -8601495270040666731L, -8781644943683948717L, -8958498214609435813L, -9132115400833020747L, -329513078923119663L, -653140085553614985L, -970986162446516633L, -1283154574001724189L, -1589746740013378181L, -1890862268619941721L, -2186598988665696811L, -2477052981484169189L, -2762318612113807775L, -3042488559956060297L, -3317653848885805587L, -3587903876823925011L, -3853326444781620847L, -4114007785385917755L, -4370032590895614921L, -4621484040716790947L, -4868443828426800925L, -5110992188315545479L, -5349207921452634719L, -5583168421288916013L, -5812949698800683211L, -6038626407184736367L, -6260271866112315101L, -6477958085549785415L, -6691755789153819001L, -6901734437248665883L, -7107962249392985407L, -7310506226543567305L, -7509432172823143521L, -7704804716899362951L, -7896687332981874823L, -8085142361444342451L, -8270231029078087193L, -8452013468983942793L, -8630548740108782727L, -8805894846433067719L, -8978108755815647279L, -9147246418501937667L, -319884274012047485L, -634221447226721713L, -943107711703952019L, -1246637591440884775L, -1544903971297695455L, -1837998125421799321L, -2126009745179158965L, -2409026966601236133L, -2687136397355987025L, -2960423143251154601L, -3228970834277968339L, -3492861650203221185L, -3752176345717555303L, -4006994275147652369L, -4257393416739890741L, -4503450396522900651L, -4745240511756319777L, -4982837753972924835L, -5216314831621190471L, -5445743192315204411L, -5671193044698747691L, -5892733379930230735L, -6110431992795059969L, -6324355502451895723L, -6534569372819150057L, -6741137932607963145L, -6944124395007788567L, -7143590877030611641L, -7339598418519720425L, -7532207000828846329L, -7721475565177390507L, -7907462030687352953L, -8090223312107483927L, -8269815337230081547L, -8446293064005765369L, -8619710497361463353L, -8790120705726758755L, -8957575837273654289L, -9122127135874723159L, -311436742236626423L, -617615491841740281L, -918625019608134017L, -1214552597609563119L, -1505484024503649039L, -1791503650407592703L, -2072694401353910859L, -2349137803333285747L, -2620914005931498869L, -2888101805567301945L, -3150778668337962433L, -3409020752479107255L, -3662902930445376543L, -3912498810618289279L, -4157880758647614599L, -4399119918432436329L, -4636286232747993765L, -4869448463524279135L, -5098674211782271063L, -5324029937233584241L, -5545580977549217813L, -5763391567302989115L, -5977524856595145039L, -6188042929361550613L, -6395006821373763179L, -6598476537935210973L, -6798511071278606779L, -6995168417669640689L, -7188505594221910903L, -7378578655427967715L, -7565442709411263547L, -7749151933903721013L, -7929759591953551371L, -8107318047367877577L, -8281878779894639221L, -8453492400148181081L, -8622208664282852653L, -8788076488418873023L, -8951143962824643581L, -9111458365859620465L, -307475288446703603L, -609825496140532001L, -907136049305023949L, -1199490950256462893L, -1486972801137945813L, -1769662827257846261L, -2047640900039265587L, -2320985559586956571L, -2589774036878095499L, -2854082275583172475L, -3113984953523165279L, -3369555503769059287L, -3620866135389674909L, -3867987853853664671L, -4110990481091444325L, -4349942675222726295L, -4584911949955229303L, -4815964693660045075L, -5043166188129051717L, -5266580627019673481L, -5486271133992198315L, -5702299780544777737L, -5914727603551148123L, -6123614622506028557L, -6329019856483067747L, -6531001340810131327L, -6729616143466640979L, -6924920381207598317L, -7116969235418849189L, -7305816967708068181L, -7491516935235868405L, -7674121605791368207L, -7853682572616474315L, -8030250568983069805L, -8203875482527225619L, -8374606369344485571L, -8542491467850207363L, -8707578212408875697L, -8869913246736238363L, -9029542437078051879L, -9186510885169160301L, -295059398302388969L, -585399261917318025L, -871095080755612885L, -1152221137251724189L, -1428850525677556065L, -1701055171147366109L, -1968905848318678153L, -2232472199794070143L, -2491822754228621723L, -2747024944147729511L, -2998145123479922841L, -3245248584809238561L, -3488399576351640627L, -3727661318659898411L, -3963096021061267133L, -4194764897832244249L, -4422728184114607365L, -4647045151576871889L, -4867774123825240517L, -5084972491568051467L, -5298696727537668291L, -5509002401173691053L, -5715944193071306579L, -5919575909198534421L, -6119950494886065093L, -6317120048593328037L, -6511135835454368537L, -6702048300607055589L, -6889907082309086415L, -7074761024844197829L, -7256658191221940129L, -7435645875674315571L, -7611770615952530553L, -7785078205427058783L, -7955613704994161497L, -8123421454791960455L, -8288545085729110011L, -8451027530829065727L, -8610911036392899119L, -8768237172983560943L, -8923046846234448943L, -9075380307485090395L, -291824837332796573L, -579033047723514713L, -861697665546188933L, -1139890569781563899L, -1413682502295250045L, -1683143085826721407L, -1948340841693730087L, -2209343207216639443L, -2466216552867106815L, -2719026199145476575L, -2967836433191175233L, -3212710525130332501L, -3453710744164785317L, -3690898374406556165L, -3924333730461832213L, -4154076172768408147L, -4380184122690492859L, -4602715077374718479L, -4821725624371129471L, -5037271456022869779L, -5249407383628227217L, -5458187351378636321L, -5663664450076184027L, -5865890930634106375L, -6064918217363709293L, -6260796921051092207L, -6453576851826999771L, -6643307031833074393L, -6830035707687730453L, -7013810362754820213L, -7194677729218211165L, -7372683799965345333L, -7547873840282802353L, -7720292399366840439L, -7889983321651842243L, -8056989757959546341L, -8221354176471899461L, -8383118373530319791L, -8542323484264117475L, -8699009993050775037L, -8853217743810747673L, -9004985950139401279L, -9154353205278664681L, -290372100001259721L, -576173422844109719L, -857475917602967779L, -1134350400794454587L, -1406866574205102759L, -1675093042438438059L, -1939097330185850223L, -2198945899225601201L, -2454704165154250273L, -2706436513854708077L, -2954206317705065293L, -3198075951532276491L, -3438106808314715397L, -3674359314637554611L, -3906892945904860603L, -4135766241312233563L, -4361036818583761367L, -4582761388476997643L, -4800995769059615481L, -5015794899761330859L, -5227212855204633303L, -5435302858817805583L, -5640117296233659459L, -5841707728477360565L, -6040124904946662359L, -6235418776187816883L, -6427638506470378599L, -6616832486164066937L, -6803048343920803369L, -6986332958664989791L, -7166732471395046691L, -7344292296799182089L, -7519057134688315459L, -7691070981249034827L, -7860377140119419895L, -8027018233290519505L, -8191036211836227817L, -8352472366474260401L, -8511367337960888911L, -8667761127322051183L, -8821693105923412355L, -8973202025381912131L, -9122326027321293349L, 9177641420735483833L, -272717998259932667L, -541404113913497709L, -806117954604690939L, -1066918246733789145L, -1323862848485720947L, -1577008762665825281L, -1826412149345845119L, -2072128338322961869L, -2314211841394634549L, -2552716364451966823L, -2787694819394284831L, -3019199335867569047L, -3247281272829344299L, -3471991229942593635L, -3693379058801223723L, -3911493873989572171L, -4126384063978410297L, -4338097301859858585L, -4546680555923596397L, -4752180100076712223L, -4954641524109506113L, -5154109743809521751L, -5350629010926051939L, -5544242922987328109L, -5734994432972571809L, -5922925858841053869L, -6108078892920275269L, -6290494611155352465L, -6470213482221659133L, -6647275376502745951L, -6821719574935530185L, -6993584777724717327L, -7162909112928388113L, -7329730144916655597L, -7494084882705268819L, -7656009788166011883L, -7815540784115719901L, -7972713262285706331L, -8127562091173369737L, -8280121623777721825L, -8430425705220552857L, -8578507680254925233L, -8724400400662660929L, -8868136232542463977L, -9009747063490294789L, -263187768843543679L, -522620522634454647L, -778351835815449053L, -1030434518459227973L, -1278920627174078871L, -1523861475853882051L, -1765307646274742065L, -2003308998540432355L, -2237914681378810159L, -2469173142291327961L, -2697132137557737413L, -2921838742098051791L, -3143339359193803489L, -3361679730070604101L, -3576904943343985913L, -3789059444330475447L, -3998187044225821849L, -4204330929152275463L, -4407533669076784937L, -4607837226601954501L, -4805282965631576813L, -4999911659912530865L, -5191763501454808871L, -5380878108831410975L, -5567294535359821715L, -5751051277166757781L, -5932186281137852489L, -6110736952753918643L, -6286740163815407985L, -6460232260056662407L, -6631249068651529329L, -6799825905611891149L, -6965997583080636659L, -7129798416520580433L, -7291262231800814767L, -7450422372181957527L, -7607311705201738413L, -7761962629462345555L, -7914407081320934053L, -8064676541484678127L, -8212802041511728741L, -8358814170219419253L, -8502743080001042365L, -8644618493052502865L, -8784469707510131995L, -8922325603500930917L, -260461966854381461L, -517246296294454149L, -770404915352461833L, -1019989017868010221L, -1266049074840506161L, -1508634844635423809L, -1747795383046461669L, -1983579053215625281L, -2216033535413241601L, -2445205836679882795L, -2671142300332149219L, -2893888615334233889L, -3113489825537163525L, -3329990338787584551L, -3543433935907936019L, -3753863779549825459L, -3961322422922397961L, -4165851818397463543L, -4367493325993122967L, -4566287721737607525L, -4762275205915024177L, -4955495411194673449L, -5145987410645584057L, -5333789725637884923L, -5518940333632612417L, -5701476675861528073L, -5881435664898499811L, -6058853692123977737L, -6233766635084074005L, -6406209864745734871L, -6576218252649472123L, -6743826177961100271L, -6909067534423905549L, -7071975737212652549L, -7232583729690814547L, -7390923990072393959L, -7547028537989680067L, -7700928940968272177L, -7852656320810677575L, -8002241359889775163L, -8149714307353417473L, -8295104985241425733L, -8438442794516214963L, -8579756721008268643L, -8719075341277665217L, -8856426828392841771L, -8991838957627763471L, -248789026652138041L, -494222665516859825L, -736346170307142349L, -975204184404558921L, -1210840749090745665L, -1443299311667851217L, -1672622733469466883L, -1898853297763514357L, -2122032717548548125L, -2342202143244910085L, -2559402170282154475L, -2773672846584142119L, -2985053679953184099L, -3193583645354596369L, -3399301192103008455L, -3602244250951751259L, -3802450241086631163L, -3999956077025379915L, -4194798175424052473L, -4387012461791627781L, -4576634377114050521L, -4763698884388935209L, -4948240475072137521L, -5130293175437381487L, -5309890552850115157L, -5487065721956751511L, -5661851350790435843L, -5834279666794465375L, -6004382462764471735L, -6172191102710461937L, -6337736527639798715L, -6501049261262186495L, -6662159415617714899L, -6821096696628997503L, -6977890409578429577L, -7132569464511574687L, -7285162381567676485L, -7435697296238278513L, -7584201964554921641L, -7730703768206875653L, -7875229719589848587L, -8017806466786604749L, -8158460298480409731L, -8297217148802208371L, -8434102602112429421L, -8569141897718298575L, -8702359934527529667L, -8833781275639252101L, -253720923760588425L, -503952109337750045L, -750741555423939405L, -994136600526131149L, -1234183932046168791L, -1470929595236220193L, -1704419002031057579L, -1934696939758856249L, -2161807579732182899L, -2385794485720821421L, -2606700622308061445L, -2824568363132052495L, -3039439499013804573L, -3251355245973394309L, -3460356253135914271L, -3666482610528681981L, -3869773856771204235L, -4070268986659371839L, -4268006458645339503L, -4463024202214525689L, -4655359625161147437L, -4845049620763685747L, -5032130574861657903L, -5216638372835054167L, -5398608406487777655L, -5578075580836407707L, -5755074320805589017L, -5929638577831330759L, -6101801836373482363L, -6271597120338635153L, -6439056999414681861L, -6604213595318249075L, -6767098587956201043L, -6927743221502396659L, -7086178310390865285L, -7242434245226551007L, -7396540998614759097L, -7548528130910422887L, -7698424795888293837L, -7846259746335142465L, -7992061339565042813L, -8135857542858798361L, -8277675938828552789L, -8417543730708614593L, -8555487747573510445L, -8691534449484268195L, -8825709932563916663L, -8958039934003175803L, -9088549836997297405L, -249509850527558039L, -495644841770381331L, -738450621926998623L, -977972221760409325L, -1214254062949489369L, -1447339966327436323L, -1677273160008781669L, -1904096287406477449L, -2127851415140544163L, -2348580040839746649L, -2566323100837744813L, -2781120977765146587L, -2993013508038871105L, -3202039989250211071L, -3408239187452964529L, -3611649344352987687L, -3812308184400502155L, -4010252921786471949L, -4205520267344347815L, -4398146435358458851L, -4588167150280314131L, -4775617653354059917L, -4960532709152321251L, -5142946612023640011L, -5322893192452705221L, -5500405823334555165L, -5675517426163914911L, -5848260477140817129L, -6018667013193638559L, -6186768637920669137L, -6352596527451315733L, -6516181436228027493L, -6677553702710015115L, -6836743254999821879L, -6993779616393789905L, -7148691910857451069L, -7301508868426858007L, -7452258830536856953L, -7600969755277290607L, -7747669222578105817L, -7892384439324327729L, -8035142244401849025L, -8175969113674970035L, -8314891164896612847L, -8451934162552120105L, -8587123522637536765L, -8720484317373261047L, -8852041279853938733L, -8981818808635463213L, -9109840972259931963L};

    private MathTools() {
    }

    public static long raiseToPower(int value, int power) {
        if (power < 0) {
            throw new IllegalArgumentException("This method does not support negative powers.");
        }
        long result = 1L;
        for (int i = 0; i < power; ++i) {
            result *= (long)value;
        }
        return result;
    }

    public static double log(double base, double arg) {
        return Math.log(arg) / Math.log(base);
    }

    public static float log(float base, float value) {
        return (float)(Math.log(value) / Math.log(base));
    }

    public static float log2(float value) {
        return (float)(Math.log(value) * 1.4426950408889634);
    }

    public static float expHasty(float power) {
        return BitConversion.intBitsToFloat(1065353216 + (int)((1.2102203E7f - 8.392334E-5f * (float)BitConversion.floatToIntBits(power)) * power));
    }

    public static float exp(float power) {
        power *= 0.5f;
        power = (12.0f + power * (6.0f + power)) / (12.0f + power * (-6.0f + power));
        return power * power;
    }

    public static float gaussian2D(float x, float y) {
        x = Math.max(-3.5f, (x * x + y * y) * -0.5f);
        x = (12.0f + x * (6.0f + x)) / (12.0f + x * (-6.0f + x));
        return x * x;
    }

    public static float gaussianFinite2D(float x, float y) {
        x = Math.max(-3.4060497f, (x * x + y * y) * -0.5f);
        x = (12.0f + x * (6.0f + x)) / (12.0f + x * (-6.0f + x));
        return (x * x - 0.00516498f) * 1.0051918f;
    }

    public static float approach(float a, float b, float delta, float halfLife) {
        float x = -delta / halfLife;
        return b + (a - b) * (-275.988f + x * (-90.6997f + (-11.6318f - 0.594604f * x) * x)) / (-275.988f + x * (100.601f + (-15.0623f + x) * x));
    }

    public static double approach(double a, double b, double delta, double halfLife) {
        double x = -delta / halfLife;
        return b + (a - b) * (-275.988 + x * (-90.6997 + (-11.6318 - 0.594604 * x) * x)) / (-275.988 + x * (100.601 + (-15.0623 + x) * x));
    }

    public static float redistributeNormal(float x) {
        float xx = Math.min(x * x * 6.03435f, 6.03435f);
        float axx = 0.1400123f * xx;
        return Math.copySign((float)Math.sqrt(1.0051551f - MathTools.exp(xx * (-1.2732395f - axx) / (0.9952389f + axx))), x);
    }

    public static double redistributeNormal(double x) {
        double xx = x * x * 6.03435;
        double axx = 0.1400122886866665 * xx;
        return Math.copySign(Math.sqrt(1.0 - Math.exp(xx * (-1.2732395447351628 - axx) / (1.0 + axx))), x);
    }

    public static float redistribute(float n, float mul, float mix) {
        float x = (n - 0.5f) * 2.0f;
        float xx = x * x * mul;
        float axx = 0.1400123f * xx;
        float denormal = Math.copySign((float)Math.sqrt(1.0 - Math.exp(xx * (-1.2732395f - axx) / (1.0f + axx))), x);
        return MathTools.lerp(n, denormal * 0.5f + 0.5f, mix);
    }

    public static float redistribute(float n, float mul, float mix, float bias) {
        float x = (n - 0.5f) * 2.0f;
        float xx = x * x * mul;
        float axx = 0.1400123f * xx;
        float denormal = Math.copySign((float)Math.sqrt(1.0 - Math.exp(xx * (-1.2732395f - axx) / (1.0f + axx))), x);
        return (float)Math.pow(MathTools.lerp(n, denormal * 0.5f + 0.5f, mix), bias);
    }

    public static boolean isEqual(double a, double b, double tolerance) {
        return Math.abs(a - b) <= tolerance;
    }

    public static boolean isEqual(float a, float b) {
        return Math.abs(a - b) <= 9.536743E-7f;
    }

    public static boolean isEqual(float a, float b, float tolerance) {
        return Math.abs(a - b) <= tolerance;
    }

    public static int clamp(int value, int min, int max) {
        return Math.min(Math.max(value, min), max);
    }

    public static long clamp(long value, long min, long max) {
        return Math.min(Math.max(value, min), max);
    }

    public static double clamp(double value, double min, double max) {
        return Math.min(Math.max(value, min), max);
    }

    public static float clamp(float value, float min, float max) {
        return Math.min(Math.max(value, min), max);
    }

    public static float remainder(float op, float d) {
        return (op % d + d) % d;
    }

    public static double remainder(double op, double d) {
        return (op % d + d) % d;
    }

    public static int greatestCommonDivisor(int a, int b) {
        a = Math.abs(a);
        b = Math.abs(b);
        while (b != 0) {
            int temp = b;
            b = a % b;
            a = temp;
        }
        return a;
    }

    public static long greatestCommonDivisor(long a, long b) {
        a = Math.abs(a);
        b = Math.abs(b);
        while (b != 0L) {
            long temp = b;
            b = a % b;
            a = temp;
        }
        return a;
    }

    public static int modularMultiplicativeInverse(int a) {
        int x = 2 ^ a * 3;
        x = BitConversion.imul(x, 2 - BitConversion.imul(a, x));
        x = BitConversion.imul(x, 2 - BitConversion.imul(a, x));
        x = BitConversion.imul(x, 2 - BitConversion.imul(a, x));
        return x;
    }

    public static long modularMultiplicativeInverse(long a) {
        long x = 2L ^ a * 3L;
        x *= 2L - a * x;
        x *= 2L - a * x;
        x *= 2L - a * x;
        x *= 2L - a * x;
        return x;
    }

    public static long isqrt(long n) {
        int c = 63 - BitConversion.countLeadingZeros(n) >> 1;
        long a = 1L;
        long d = 0L;
        int s = 0x1F & 32 - BitConversion.countLeadingZeros(c);
        while (s > 0) {
            long e = d;
            d = c >>> --s;
            a = (a << (int)(d - e - 1L)) + (n >>> (int)((long)(c + c) - e - d + 1L)) / a;
        }
        return a - (n - a * a >>> 63);
    }

    public static float cbrt(float x) {
        int ix = BitConversion.floatToIntBits(x);
        int sign = ix & Integer.MIN_VALUE;
        ix &= Integer.MAX_VALUE;
        float x0 = x;
        ix = (ix >>> 2) + (ix >>> 4);
        ix += ix >>> 4;
        ix = ix + (ix >>> 8) + 709965728 | sign;
        x = BitConversion.intBitsToFloat(ix);
        x = 0.33333334f * (2.0f * x + x0 / (x * x));
        x = 0.33333334f * (1.9999999f * x + x0 / (x * x));
        return x;
    }

    public static double cbrt(double x) {
        double r;
        double a = Math.abs(x);
        boolean small = a < 0.015625;
        a = small ? a * 1.645504557321206E63 : a * 0.125;
        long ai = BitConversion.doubleToLongBits(a);
        if (ai >= 0x7FF0000000000000L || x == 0.0) {
            r = x + x;
        } else {
            long ai23 = 2L * (ai / 3L);
            long aim23 = 7678273666836070400L - ai23;
            double y = BitConversion.longBitsToDouble(aim23);
            double ayy = a * y * y;
            double y_a2y4 = y - ayy * ayy;
            y = y_a2y4 * 0.3333333333333333 + y;
            ayy = a * y * y;
            y_a2y4 = y - ayy * ayy;
            y = y_a2y4 * 0.33523333333 + y;
            ayy = a * y * y;
            y_a2y4 = y - ayy * ayy;
            y = y_a2y4 * 0.3333333333333333 + y;
            r = y * a;
            double r2_h = r * r;
            double diff = a - r2_h * r;
            r = (diff *= 0.3333333333333333) * y + r;
            r = small ? r * 8.470329472543003E-22 : r * 2.0;
            r = Math.copySign(r, x);
        }
        return r;
    }

    public static float nthrt(float x, float n) {
        float f = (float)Math.pow(x, 1.0f / n);
        if (Float.isNaN(f) || Float.isInfinite(f)) {
            return f;
        }
        int i = MathTools.round(f);
        return MathTools.isEqual(i, f) ? (float)i : f;
    }

    public static float invSqrt(float x) {
        int i = 1595932665 - (BitConversion.floatToIntBits(x) >> 1);
        float y = BitConversion.intBitsToFloat(i);
        return y * (0.70395225f * (2.3892446f - x * y * y));
    }

    public static double invSqrt(double x) {
        long i = 6910469410427058089L - (BitConversion.doubleToLongBits(x) >> 1);
        double y = BitConversion.longBitsToDouble(i);
        return y * (1.5 - 0.5 * x * y * y);
    }

    public static float signPreservingSqrt(float x) {
        return Math.copySign((float)Math.sqrt(Math.abs(x)), x);
    }

    public static float signPreservingPow(float x, float power) {
        return Math.copySign((float)Math.pow(Math.abs(x), power), x);
    }

    public static double signPreservingSqrt(double x) {
        return Math.copySign(Math.sqrt(Math.abs(x)), x);
    }

    public static double signPreservingPow(double x, double power) {
        return Math.copySign(Math.pow(Math.abs(x), power), x);
    }

    public static float barronSpline(float x, float shape, float turning) {
        float d = turning - x;
        int f = BitConversion.floatToIntBits(d) >> 31;
        int n = f | 1;
        return (turning * (float)n - (float)f) * (x + (float)f) / (Float.MIN_NORMAL - (float)f + (x + shape * d) * (float)n) - (float)f;
    }

    public static double barronSpline(double x, double shape, double turning) {
        double d = turning - x;
        int f = BitConversion.doubleToHighIntBits(d) >> 31;
        int n = f | 1;
        return (turning * (double)n - (double)f) * (x + (double)f) / (Double.MIN_NORMAL - (double)f + (x + shape * d) * (double)n) - (double)f;
    }

    public static float noiseSpline(float x, float shape, float turning) {
        turning = turning * 0.5f + 0.5f;
        x = x * 0.5f + 0.5f;
        float d = turning - x;
        int f = BitConversion.floatToIntBits(d) >> 31;
        int n = f | 1;
        return ((turning * (float)n - (float)f) * (x + (float)f) / (Float.MIN_NORMAL - (float)f + (x + shape * d) * (float)n) - (float)f - 0.5f) * 2.0f;
    }

    public static double noiseSpline(double x, double shape, double turning) {
        turning = turning * 0.5 + 0.5;
        x = x * 0.5 + 0.5;
        double d = turning - x;
        int f = BitConversion.doubleToHighIntBits(d) >> 31;
        int n = f | 1;
        return ((turning * (double)n - (double)f) * (x + (double)f) / (Double.MIN_NORMAL - (double)f + (x + shape * d) * (double)n) - (double)f - 0.5) * 2.0;
    }

    public static double probit(double d) {
        if (d <= 0.0 || d >= 1.0) {
            return Math.copySign(38.5, d - 0.5);
        }
        if (d < 0.02425) {
            double q = Math.sqrt(-2.0 * Math.log(d));
            return (((((-0.007784894002430293 * q - 0.3223964580411365) * q - 2.400758277161838) * q - 2.549732539343734) * q + 4.374664141464968) * q + 2.938163982698783) / ((((0.007784695709041462 * q + 0.3224671290700398) * q + 2.445134137142996) * q + 3.754408661907416) * q + 1.0);
        }
        if (0.97575 < d) {
            double q = Math.sqrt(-2.0 * Math.log(1.0 - d));
            return -(((((-0.007784894002430293 * q - 0.3223964580411365) * q - 2.400758277161838) * q - 2.549732539343734) * q + 4.374664141464968) * q + 2.938163982698783) / ((((0.007784695709041462 * q + 0.3224671290700398) * q + 2.445134137142996) * q + 3.754408661907416) * q + 1.0);
        }
        double q = d - 0.5;
        double r = q * q;
        return (((((-39.69683028665376 * r + 220.9460984245205) * r - 275.9285104469687) * r + 138.357751867269) * r - 30.66479806614716) * r + 2.506628277459239) * q / (((((-54.47609879822406 * r + 161.5858368580409) * r - 155.6989798598866) * r + 66.80131188771972) * r - 13.28068155288572) * r + 1.0);
    }

    public double probitInverse(double x) {
        double a1 = 0.0705230784;
        double a2 = 0.0422820123;
        double a3 = 0.0092705272;
        double a4 = 1.520143E-4;
        double a5 = 2.765672E-4;
        double a6 = 4.30638E-5;
        double sign = Math.signum(x);
        double y1 = sign * x * 0.7071067811865475;
        double y2 = y1 * y1;
        double y3 = y1 * y2;
        double y4 = y2 * y2;
        double y5 = y2 * y3;
        double y6 = y3 * y3;
        double n = 1.0 + 0.0705230784 * y1 + 0.0422820123 * y2 + 0.0092705272 * y3 + 1.520143E-4 * y4 + 2.765672E-4 * y5 + 4.30638E-5 * y6;
        n *= n;
        n *= n;
        n *= n;
        return sign * (0.5 - 0.5 / (n * n)) + 0.5;
    }

    public float probitInverse(float x) {
        float a1 = 0.070523076f;
        float a2 = 0.04228201f;
        float a3 = 0.009270527f;
        float a4 = 1.520143E-4f;
        float a5 = 2.765672E-4f;
        float a6 = 4.30638E-5f;
        float sign = Math.signum(x);
        float y1 = sign * x * 0.70710677f;
        float y2 = y1 * y1;
        float y3 = y1 * y2;
        float y4 = y2 * y2;
        float y5 = y2 * y3;
        float y6 = y3 * y3;
        float n = 1.0f + 0.070523076f * y1 + 0.04228201f * y2 + 0.009270527f * y3 + 1.520143E-4f * y4 + 2.765672E-4f * y5 + 4.30638E-5f * y6;
        n *= n;
        n *= n;
        n *= n;
        return sign * (0.5f - 0.5f / (n * n)) + 0.5f;
    }

    public double erf(double x) {
        double a1 = 0.0705230784;
        double a2 = 0.0422820123;
        double a3 = 0.0092705272;
        double a4 = 1.520143E-4;
        double a5 = 2.765672E-4;
        double a6 = 4.30638E-5;
        double sign = Math.signum(x);
        double y1 = sign * x;
        double y2 = y1 * y1;
        double y3 = y1 * y2;
        double y4 = y2 * y2;
        double y5 = y2 * y3;
        double y6 = y3 * y3;
        double n = 1.0 + 0.0705230784 * y1 + 0.0422820123 * y2 + 0.0092705272 * y3 + 1.520143E-4 * y4 + 2.765672E-4 * y5 + 4.30638E-5 * y6;
        n *= n;
        n *= n;
        n *= n;
        return sign * (1.0 - 1.0 / (n * n));
    }

    public float erf(float x) {
        float a1 = 0.070523076f;
        float a2 = 0.04228201f;
        float a3 = 0.009270527f;
        float a4 = 1.520143E-4f;
        float a5 = 2.765672E-4f;
        float a6 = 4.30638E-5f;
        float sign = Math.signum(x);
        float y1 = sign * x;
        float y2 = y1 * y1;
        float y3 = y1 * y2;
        float y4 = y2 * y2;
        float y5 = y2 * y3;
        float y6 = y3 * y3;
        float n = 1.0f + 0.070523076f * y1 + 0.04228201f * y2 + 0.009270527f * y3 + 1.520143E-4f * y4 + 2.765672E-4f * y5 + 4.30638E-5f * y6;
        n *= n;
        n *= n;
        n *= n;
        return sign * (1.0f - 1.0f / (n * n));
    }

    private static double erfcBase(double x) {
        return 0.5641895835477563 / (x + 2.0695502313291416) * ((x * (x + 2.710785400451478) + 5.807556131303016) / (x * (x + 3.4795405709951894) + 12.061668872862395)) * ((x * (x + 3.474695137774396) + 12.074020364063815) / (x * (x + 3.720684439602251) + 8.443197810039685)) * ((x * (x + 4.0056150920225955) + 9.305966594858878) / (x * (x + 3.902257040299241) + 6.361616309538805)) * ((x * (x + 5.167227058178126) + 9.126616176736732) / (x * (x + 4.032968931092625) + 5.135785305856816)) * ((x * (x + 5.959087954466333) + 9.194356128869693) / (x * (x + 4.112409429574509) + 4.486403295234087)) * Math.exp(-x * x);
    }

    public static double erfc(double x) {
        return x >= 0.0 ? MathTools.erfcBase(x) : 2.0 - MathTools.erfcBase(-x);
    }

    public static double probitHighPrecision(double d) {
        double x = MathTools.probit(d);
        if (d > 0.0 && d < 1.0) {
            double e = 0.5 * MathTools.erfc(-x / Math.sqrt(2.0)) - d;
            double u = e * Math.sqrt(Math.PI * 2) * Math.exp(x * x / 2.0);
            x -= u / (1.0 + x * u / 2.0);
        }
        return x;
    }

    public static int nextPowerOfTwo(int n) {
        return 1 << -BitConversion.countLeadingZeros(Math.max(2, n) - 1);
    }

    public static boolean isPowerOfTwo(int value) {
        return value != 0 && (value & value - 1) == 0;
    }

    public static double gamma(double x) {
        return MathTools.factorial(x - 1.0);
    }

    public static double factorial(double x) {
        double y;
        double p = 1.0;
        for (y = x + 1.0; y < 7.0; y += 1.0) {
            p *= y;
        }
        double r = Math.exp(y * Math.log(y) - y + 1.0 / (12.0 * y + 2.0 / (5.0 * y + 53.0 / (42.0 * y))));
        if (x < 7.0) {
            r /= p;
        }
        return r * Math.sqrt(Math.PI * 2 / y);
    }

    public static float gamma(float x) {
        return (float)MathTools.factorial((double)x - 1.0);
    }

    public static float factorial(float x) {
        double y;
        double p = 1.0;
        for (y = (double)x + 1.0; y < 7.0; y += 1.0) {
            p *= y;
        }
        double r = Math.exp(y * Math.log(y) - y + 1.0 / (12.0 * y + 2.0 / (5.0 * y + 53.0 / (42.0 * y))));
        if ((double)x < 7.0) {
            r /= p;
        }
        return (float)(r * Math.sqrt(Math.PI * 2 / y));
    }

    public static int fibonacci(int n) {
        return (int)(Math.pow(1.618033988749895, n) / 2.236067977499795 + 0.49999999999999917);
    }

    public static long fibonacci(long n) {
        return (long)(Math.pow(1.618033988749895, n) / 2.236067977499795 + 0.49999999999999917);
    }

    public static float square(float n) {
        return n * n;
    }

    public static double square(double n) {
        return n * n;
    }

    public static float cube(float n) {
        return n * n * n;
    }

    public static double cube(double n) {
        return n * n * n;
    }

    public static int fastFloor(float t) {
        return (int)((double)t + 16384.0) - 16384;
    }

    public static int fastCeil(float t) {
        return 16384 - (int)(16384.0 - (double)t);
    }

    public static float fastFract(float t) {
        t -= (float)((int)t - 1);
        return t - (float)((int)t);
    }

    public static long longFloor(double t) {
        long z = (long)t;
        return t < (double)z ? z - 1L : z;
    }

    public static long longFloor(float t) {
        long z = (long)t;
        return t < (float)z ? z - 1L : z;
    }

    public static int floor(double t) {
        int z = (int)t;
        return t < (double)z ? z - 1 : z;
    }

    public static int ceil(double t) {
        int z = (int)t;
        return t > (double)z ? z + 1 : z;
    }

    public static int floor(float value) {
        int z = (int)value;
        return value < (float)z ? z - 1 : z;
    }

    public static int floorPositive(float value) {
        return (int)value;
    }

    public static int ceil(float value) {
        int z = (int)value;
        return value > (float)z ? z + 1 : z;
    }

    public static int ceilPositive(float value) {
        return (int)((double)value + 0.9999999403953552);
    }

    public static int round(float value) {
        return (int)((double)value + 16384.5) - 16384;
    }

    public static int roundPositive(float value) {
        return (int)(value + 0.5f);
    }

    public static float fract(float t) {
        t -= (float)((int)t - 1);
        return t - (float)((int)t);
    }

    public static double fract(double t) {
        t -= (double)((long)t - 1L);
        return t - (double)((long)t);
    }

    public static float truncate(float n) {
        long i = (long)(n * 8192.0f);
        return (float)i * 1.2207031E-4f;
    }

    public static double truncate(double n) {
        long i = (long)(n * 4.398046511104E12);
        return (double)i * 2.2737367544323206E-13;
    }

    public static float lerp(float fromValue, float toValue, float progress) {
        return fromValue + (toValue - fromValue) * progress;
    }

    public static float norm(float rangeStart, float rangeEnd, float value) {
        return (value - rangeStart) / (rangeEnd - rangeStart);
    }

    public static float map(float inRangeStart, float inRangeEnd, float outRangeStart, float outRangeEnd, float value) {
        return outRangeStart + (value - inRangeStart) * (outRangeEnd - outRangeStart) / (inRangeEnd - inRangeStart);
    }

    public static double lerp(double fromValue, double toValue, double progress) {
        return fromValue + (toValue - fromValue) * progress;
    }

    public static double norm(double rangeStart, double rangeEnd, double value) {
        return (value - rangeStart) / (rangeEnd - rangeStart);
    }

    public static double map(double inRangeStart, double inRangeEnd, double outRangeStart, double outRangeEnd, double value) {
        return outRangeStart + (value - inRangeStart) * (outRangeEnd - outRangeStart) / (inRangeEnd - inRangeStart);
    }

    public static float lerpAngle(float fromRadians, float toRadians, float progress) {
        float delta = ((toRadians - fromRadians) % ((float)Math.PI * 2) + (float)Math.PI * 2 + (float)Math.PI) % ((float)Math.PI * 2) - (float)Math.PI;
        return ((fromRadians + delta * progress) % ((float)Math.PI * 2) + (float)Math.PI * 2) % ((float)Math.PI * 2);
    }

    public static float lerpAngleDeg(float fromDegrees, float toDegrees, float progress) {
        float delta = ((toDegrees - fromDegrees) % 360.0f + 360.0f + 180.0f) % 360.0f - 180.0f;
        return ((fromDegrees + delta * progress) % 360.0f + 360.0f) % 360.0f;
    }

    public static float lerpAngleTurns(float fromTurns, float toTurns, float progress) {
        return MathTools.fract(fromTurns + (((toTurns - fromTurns) % 1.0f + 1.5f) % 1.0f - 0.5f) * progress);
    }

    public static double lerpAngle(double fromRadians, double toRadians, double progress) {
        double delta = ((toRadians - fromRadians) % (Math.PI * 2) + Math.PI * 2 + Math.PI) % (Math.PI * 2) - Math.PI;
        return ((fromRadians + delta * progress) % (Math.PI * 2) + Math.PI * 2) % (Math.PI * 2);
    }

    public static double lerpAngleDeg(double fromDegrees, double toDegrees, double progress) {
        double delta = ((toDegrees - fromDegrees) % 360.0 + 360.0 + 180.0) % 360.0 - 180.0;
        return ((fromDegrees + delta * progress) % 360.0 + 360.0) % 360.0;
    }

    public static double lerpAngleTurns(double fromTurns, double toTurns, double progress) {
        return MathTools.fract(fromTurns + (((toTurns - fromTurns) % 1.0 + 1.5) % 1.0 - 0.5) * progress);
    }

    public static boolean isZero(float value) {
        return Math.abs(value) <= 9.536743E-7f;
    }

    public static boolean isZero(float value, float tolerance) {
        return Math.abs(value) <= tolerance;
    }

    public static boolean isZero(double value, double tolerance) {
        return Math.abs(value) <= tolerance;
    }

    public static float towardsZero(float x) {
        int bits = BitConversion.floatToIntBits(x);
        int sign = bits & Integer.MIN_VALUE;
        return BitConversion.intBitsToFloat(Math.max((bits ^ sign) - 1, 0) | sign);
    }

    public static float towardsZero(float x, int steps) {
        int bits = BitConversion.floatToIntBits(x);
        int sign = bits & Integer.MIN_VALUE;
        return BitConversion.intBitsToFloat(Math.max((bits ^ sign) - steps, 0) | sign);
    }

    public static double towardsZero(double x) {
        long bits = BitConversion.doubleToLongBits(x);
        long sign = bits & Long.MIN_VALUE;
        return BitConversion.longBitsToDouble(Math.max((bits ^ sign) - 1L, 0L) | sign);
    }

    public static double towardsZero(double x, long steps) {
        long bits = BitConversion.doubleToLongBits(x);
        long sign = bits & Long.MIN_VALUE;
        return BitConversion.longBitsToDouble(Math.max((bits ^ sign) - steps, 0L) | sign);
    }

    public static long interleaveBits(int x, int y) {
        long n = (long)x & 0xFFFFFFFFL | (long)y << 32;
        n = (n & 0xFFFF0000L) << 16 | n >>> 16 & 0xFFFF0000L | n & 0xFFFF00000000FFFFL;
        n = (n & 0xFF000000FF00L) << 8 | n >>> 8 & 0xFF000000FF00L | n & 0xFF0000FFFF0000FFL;
        n = (n & 0xF000F000F000F0L) << 4 | n >>> 4 & 0xF000F000F000F0L | n & 0xF00FF00FF00FF00FL;
        n = (n & 0xC0C0C0C0C0C0C0CL) << 2 | n >>> 2 & 0xC0C0C0C0C0C0C0CL | n & 0xC3C3C3C3C3C3C3C3L;
        return (n & 0x2222222222222222L) << 1 | n >>> 1 & 0x2222222222222222L | n & 0x9999999999999999L;
    }

    public static int disperseBits(int n) {
        n = (n & 0x22222222) << 1 | n >>> 1 & 0x22222222 | n & 0x99999999;
        n = (n & 0xC0C0C0C) << 2 | n >>> 2 & 0xC0C0C0C | n & 0xC3C3C3C3;
        n = (n & 0xF000F0) << 4 | n >>> 4 & 0xF000F0 | n & 0xF00FF00F;
        return (n & 0xFF00) << 8 | n >>> 8 & 0xFF00 | n & 0xFF0000FF;
    }

    public static long disperseBits(long n) {
        n = (n & 0x2222222222222222L) << 1 | n >>> 1 & 0x2222222222222222L | n & 0x9999999999999999L;
        n = (n & 0xC0C0C0C0C0C0C0CL) << 2 | n >>> 2 & 0xC0C0C0C0C0C0C0CL | n & 0xC3C3C3C3C3C3C3C3L;
        n = (n & 0xF000F000F000F0L) << 4 | n >>> 4 & 0xF000F000F000F0L | n & 0xF00FF00FF00FF00FL;
        n = (n & 0xFF000000FF00L) << 8 | n >>> 8 & 0xFF000000FF00L | n & 0xFF0000FFFF0000FFL;
        return (n & 0xFFFF0000L) << 16 | n >>> 16 & 0xFFFF0000L | n & 0xFFFF00000000FFFFL;
    }

    public static float zigzag(float value) {
        int floor = (int)((double)value + 16384.0) - 16384;
        value -= (float)floor;
        floor = -(floor & 1) | 1;
        return value * (float)(floor << 1) - (float)floor;
    }

    public static float triangleWave(float t) {
        return Math.abs(t - (float)((int)((double)t + 16384.5) - 16384)) * 4.0f - 1.0f;
    }

    public static double triangleWave(double t) {
        return Math.abs(t - (double)Math.round(t)) * 4.0 - 1.0;
    }

    public static float squareWave(float t) {
        return Math.signum(Math.abs(t - (float)((int)((double)t + 16384.5) - 16384)) - 0.25f);
    }

    public static double squareWave(double t) {
        return Math.signum(Math.abs(t - (double)Math.round(t)) - 0.25);
    }

    public static float sawtoothWave(float t) {
        return (t - (float)((int)((double)t + 16384.5) - 16384)) * 2.0f;
    }

    public static double sawtoothWave(double t) {
        return (t - (double)Math.round(t)) * 2.0;
    }

    public static float sway(float value) {
        int floor = (int)((double)value + 16384.0) - 16384;
        value -= (float)floor;
        floor = -(floor & 1) | 1;
        return value * value * value * (value * (value * 6.0f - 15.0f) + 10.0f) * (float)(floor << 1) - (float)floor;
    }

    public static float swayCubic(float value) {
        int floor = (int)((double)value + 16384.0) - 16384;
        value -= (float)floor;
        floor = -(floor & 1) | 1;
        return value * value * (3.0f - value * 2.0f) * (float)(floor << 1) - (float)floor;
    }

    public static float swayTight(float value) {
        int floor = (int)((double)value + 16384.0) - 16384;
        value -= (float)floor;
        return value * value * value * (value * (value * 6.0f - 15.0f) + 10.0f) * (float)(-(floor &= 1) | 1) + (float)floor;
    }

    public static double zigzag(double value) {
        int floor = (int)Math.floor(value);
        value -= (double)floor;
        floor = -(floor & 1) | 1;
        return value * (double)(floor << 1) - (double)floor;
    }

    public static double sway(double value) {
        int floor = (int)Math.floor(value);
        value -= (double)floor;
        floor = -(floor & 1) | 1;
        return value * value * value * (value * (value * 6.0 - 15.0) + 10.0) * (double)(floor << 1) - (double)floor;
    }

    public static double swayCubic(double value) {
        int floor = (int)Math.floor(value);
        value -= (double)floor;
        floor = -(floor & 1) | 1;
        return value * value * (3.0 - value * 2.0) * (double)(floor << 1) - (double)floor;
    }

    public static double swayTight(double value) {
        int floor = (int)Math.floor(value);
        value -= (double)floor;
        return value * value * value * (value * (value * 6.0 - 15.0) + 10.0) * (double)(-(floor &= 1) | 1) + (double)floor;
    }

    public static float signedGain(float x, float bias) {
        return x / ((bias - 1.0f) * (1.0f - Math.abs(x)) + 1.0f);
    }

    public static int boundedInt(long state, int bound) {
        int res = (int)((long)bound * (state & 0xFFFFFFFFL) >> 32);
        return res + (res >>> 31);
    }

    public static long boundedLong(long state, long innerBound, long outerBound) {
        if (outerBound < innerBound) {
            long t = outerBound;
            outerBound = innerBound + 1L;
            innerBound = t + 1L;
        }
        long bound = outerBound - innerBound;
        long randLow = state & 0xFFFFFFFFL;
        long boundLow = bound & 0xFFFFFFFFL;
        long randHigh = state >>> 32;
        long boundHigh = bound >>> 32;
        return innerBound + (randHigh * boundLow >>> 32) + (randLow * boundHigh >>> 32) + randHigh * boundHigh;
    }

    public static float exclusiveFloat(long bits) {
        return BitConversion.intBitsToFloat(126 - BitConversion.countLeadingZeros(bits) << 23 | (int)bits & 0x7FFFFF);
    }

    public static double exclusiveDouble(long bits) {
        return BitConversion.longBitsToDouble(1022L - (long)BitConversion.countLeadingZeros(bits) << 52 | bits & 0xFFFFFFFFFFFFFL);
    }

    public static int triangularNumber(int index) {
        return (index + 1) * index >>> 1;
    }

    public static int goldenLongsOffset(int dimension) {
        return (dimension - 1) * dimension >>> 1;
    }
}

