/*
 * Decompiled with CFR 0.152.
 */
package com.github.ulisesbocchio.jar.resources;

import com.google.common.base.Preconditions;
import java.io.File;
import java.io.InputStream;
import java.net.URL;
import java.util.Arrays;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.IntStream;
import org.apache.commons.io.FileUtils;
import org.apache.commons.vfs2.FileObject;
import org.apache.commons.vfs2.VFS;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;
import org.springframework.util.ResourceUtils;

public class JarUtils {
    private static final Logger log = LoggerFactory.getLogger(JarUtils.class);
    private static Pattern EXCLAMATION_PATH = Pattern.compile("/([^/!]*!)/");

    public static File getFile(Resource resource, String extractPath) {
        return ResourceUtils.isJarURL((URL)resource.getURL()) ? JarUtils.getFromJar(resource, extractPath) : resource.getFile();
    }

    private static File getFromJar(Resource resource, String extractPath) {
        Preconditions.checkArgument((extractPath != null ? 1 : 0) != 0, (Object)"Extract Path cannot be null");
        FileObject file = VFS.getManager().resolveFile(JarUtils.maybeFixUri(resource));
        File extractDir = new File(extractPath);
        if (!extractDir.exists() || !extractDir.isDirectory()) {
            FileUtils.forceMkdir((File)extractDir);
            log.debug("TEMP EXTRACT DIR CREATED {}", (Object)extractDir.getAbsolutePath());
        }
        return JarUtils.copyToDir(file, extractDir);
    }

    private static String maybeFixUri(Resource resource) {
        String uri = resource.getURI().toString();
        uri = JarUtils.maybeFixUriPrefix(uri);
        uri = JarUtils.maybeFixExclamationPath(uri);
        return uri;
    }

    private static String maybeFixExclamationPath(String uri) {
        String fixedUri = uri;
        Matcher matcher = EXCLAMATION_PATH.matcher(uri);
        while (matcher.find()) {
            String match = matcher.group(1);
            if (match.endsWith(".jar!")) continue;
            fixedUri = fixedUri.replaceFirst(match, match.substring(0, match.length() - 1));
        }
        return fixedUri;
    }

    private static String maybeFixUriPrefix(String uri) {
        int numOfJarsInResource = JarUtils.numbOfJars(uri);
        String jarPrefix = JarUtils.jarPrefix(numOfJarsInResource);
        String fixedUri = jarPrefix + uri.substring(4);
        return fixedUri;
    }

    private static String jarPrefix(int n) {
        return IntStream.range(0, n).mapToObj(num -> "jar:").reduce((r, l) -> r + l).orElse("jar:");
    }

    private static int numbOfJars(String uri) {
        Matcher matcher = Pattern.compile("\\.jar!").matcher(uri);
        int matches = 0;
        while (matcher.find()) {
            ++matches;
        }
        return matches;
    }

    private static File copyToDir(FileObject jarredFile, File destination) {
        return JarUtils.copyToDir(jarredFile, destination, true);
    }

    private static File copyToDir(FileObject jarredFile, File destination, boolean retryIfImaginary) {
        switch (jarredFile.getType()) {
            case FILE: {
                return JarUtils.copyFileToDir(jarredFile, destination);
            }
            case FOLDER: {
                return JarUtils.copyDirToDir(jarredFile, destination);
            }
            case IMAGINARY: {
                if (retryIfImaginary) {
                    log.debug("Imaginary file found, retrying extraction");
                    VFS.getManager().getFilesCache().removeFile(jarredFile.getFileSystem(), jarredFile.getName());
                    FileObject newJarredFile = VFS.getManager().resolveFile(jarredFile.getName().getURI());
                    return JarUtils.copyToDir(newJarredFile, destination, false);
                }
                log.debug("Imaginary file found after retry, abandoning retry");
            }
        }
        throw new IllegalStateException("File Type not supported: " + jarredFile.getType());
    }

    private static File copyDirToDir(FileObject jarredDir, File destination) {
        File tempDir = new File(destination, jarredDir.getName().getBaseName());
        JarUtils.createDir(tempDir);
        Arrays.stream(jarredDir.getChildren()).forEach(fileObject -> JarUtils.copyToDir(fileObject, tempDir));
        return tempDir;
    }

    private static File copyFileToDir(FileObject jarredFile, File destination) {
        File tempFile = new File(destination, jarredFile.getName().getBaseName());
        JarUtils.createFile(tempFile);
        log.info("TEMP FILE CREATED {}", (Object)tempFile.getAbsolutePath());
        FileUtils.copyInputStreamToFile((InputStream)jarredFile.getContent().getInputStream(), (File)tempFile);
        return tempFile;
    }

    private static void createDir(File file) {
        if (!file.exists() && !file.mkdir()) {
            throw new IllegalStateException(String.format("Could not create temp directory: %s", file.getAbsolutePath()));
        }
    }

    private static void createFile(File file) {
        if (file.exists()) {
            FileUtils.forceDelete((File)file);
        }
        if (!file.createNewFile()) {
            throw new IllegalStateException(String.format("Could not create temp jarredFile: %s", file.getAbsolutePath()));
        }
    }
}

