package mslinks.data;

/*-
 * #%L
 * https://github.com/vatbub/mslinks
 * %%
 * Copyright (C) 2015 - 2020 Dmitrii Shamrikov, Frederik Kammel
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import io.ByteReader;
import io.ByteWriter;
import io.Bytes;
import mslinks.Serializable;

import java.io.IOException;
import java.util.Random;

public class GUID implements Serializable {
	private static Random r = new Random();
	
	private int d1;
	private short d2;
	private short d3;
	private short d4;
	private long d5;
	
	public GUID() {
		d1 = r.nextInt();
		d2 = (short)r.nextInt();
		d3 = (short)r.nextInt();
		d4 = (short)r.nextInt();
		d5 = r.nextLong() & 0xffffffffffffL;
	}
	
	public GUID(byte[] d) {
		d1 = Bytes.makeIntL(d[0], d[1], d[2], d[3]);
		d2 = Bytes.makeShortL(d[4], d[5]);
		d3 = Bytes.makeShortL(d[6], d[7]);
		d4 = Bytes.makeShortB(d[8], d[9]);
		d5 = Bytes.makeLongB((byte)0, (byte)0, d[10], d[11], d[12], d[13], d[14], d[15]);
	}
	
	public GUID(ByteReader data) throws IOException {
		d1 = (int)data.read4bytes();
		d2 = (short)data.read2bytes();
		d3 = (short)data.read2bytes();
		data.changeEndiannes();
		d4 = (short)data.read2bytes();
		d5 = data.read6bytes();
		data.changeEndiannes();
	}
	
	public GUID(String s) {
		if (s.charAt(0) == '{' && s.charAt(s.length()-1) == '}')
			s = s.substring(1, s.length() - 1);
		String[] p = s.split("-");
		
		byte[] b = parse(p[0]);
		d1 = Bytes.makeIntB(b[0], b[1], b[2], b[3]);
		b = parse(p[1]);
		d2 = Bytes.makeShortB(b[0], b[1]);
		b = parse(p[2]);
		d3 = Bytes.makeShortB(b[0], b[1]);
		d4 = (short)Long.parseLong(p[3], 16);
		d5 = Long.parseLong(p[4], 16);
	}
	
	private byte[] parse(String s) {
		byte[] b = new byte[s.length() >> 1];
		for (int i=0, j=0; j<s.length(); i++, j+=2)
			b[i] = (byte)Long.parseLong(s.substring(j, j+2), 16);
		return b;
	}
	
	public String toString() {
		return String.format("%08X-%04X-%04X-%04X-%012X", d1, d2, d3, d4, d5);
	}
	
	public boolean equals(Object o) {
		if (o == this)
			return true;

		if (o == null || getClass() != o.getClass())
			return false;

		GUID g = (GUID)o;
		return d1 == g.d1 && d2 == g.d2 && d3 == g.d3 && d4 == g.d4 && d5 == g.d5;
	}

	@Override
	public int hashCode()
	{
		return (int)(d1 ^ d2 ^ d3 ^ d4 ^ ((d5 & 0xffffffff00000000l) >> 32) ^ (d5 & 0xffffffffl));
	}

	public void serialize(ByteWriter bw) throws IOException {
		bw.write4bytes(d1);
		bw.write2bytes(d2);
		bw.write2bytes(d3);
		bw.changeEndiannes();
		bw.write2bytes(d4);
		bw.write6bytes(d5);
		bw.changeEndiannes();
	}
}
