/*
 * Decompiled with CFR 0.152.
 */
package com.github.rollingmetrics.histogram;

import com.codahale.metrics.Histogram;
import com.codahale.metrics.Metric;
import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Reservoir;
import com.codahale.metrics.Timer;
import com.github.rollingmetrics.histogram.HdrReservoir;
import com.github.rollingmetrics.histogram.OverflowResolver;
import com.github.rollingmetrics.histogram.SnapshotCachingReservoir;
import com.github.rollingmetrics.histogram.accumulator.Accumulator;
import com.github.rollingmetrics.histogram.accumulator.ResetByChunksAccumulator;
import com.github.rollingmetrics.histogram.accumulator.ResetOnSnapshotAccumulator;
import com.github.rollingmetrics.histogram.accumulator.UniformAccumulator;
import com.github.rollingmetrics.util.Clock;
import com.github.rollingmetrics.util.ResilientExecutionUtil;
import java.time.Duration;
import java.util.Arrays;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Executor;
import java.util.function.Supplier;
import org.HdrHistogram.Recorder;

public class HdrBuilder {
    static final int MAX_CHUNKS = 60;
    static final long MIN_CHUNK_RESETTING_INTERVAL_MILLIS = 1000L;
    static int DEFAULT_NUMBER_OF_SIGNIFICANT_DIGITS = 2;
    static AccumulationFactory DEFAULT_ACCUMULATION_STRATEGY = AccumulationFactory.UNIFORM;
    static double[] DEFAULT_PERCENTILES = new double[]{0.5, 0.75, 0.9, 0.95, 0.98, 0.99, 0.999};
    private AccumulationFactory accumulationFactory;
    private int numberOfSignificantValueDigits;
    private Optional<Long> lowestDiscernibleValue;
    private Optional<Long> highestTrackableValue;
    private Optional<OverflowResolver> overflowResolver;
    private Optional<Long> snapshotCachingDurationMillis;
    private Optional<double[]> predefinedPercentiles;
    private Optional<Long> expectedIntervalBetweenValueSamples;
    private Optional<Executor> backgroundExecutor;
    private Clock clock;

    public HdrBuilder() {
        this(Clock.defaultClock());
    }

    public HdrBuilder resetReservoirOnSnapshot() {
        this.accumulationFactory = AccumulationFactory.RESET_ON_SNAPSHOT;
        return this;
    }

    public HdrBuilder resetReservoirPeriodically(Duration resettingPeriod) {
        int numberOfHistoryChunks = 0;
        return this.resetReservoirPeriodicallyByChunks(resettingPeriod.toMillis(), numberOfHistoryChunks);
    }

    public HdrBuilder resetReservoirPeriodicallyByChunks(Duration rollingTimeWindow, int numberChunks) {
        if (numberChunks < 2) {
            throw new IllegalArgumentException("numberHistoryChunks should be >= 2");
        }
        if (numberChunks > 60) {
            throw new IllegalArgumentException("numberHistoryChunks should be <= 60");
        }
        long resettingPeriodMillis = rollingTimeWindow.toMillis() / (long)numberChunks;
        return this.resetReservoirPeriodicallyByChunks(resettingPeriodMillis, numberChunks);
    }

    public HdrBuilder neverResetReservoir() {
        this.accumulationFactory = AccumulationFactory.UNIFORM;
        return this;
    }

    public HdrBuilder withSignificantDigits(int numberOfSignificantValueDigits) {
        if (numberOfSignificantValueDigits < 0 || numberOfSignificantValueDigits > 5) {
            throw new IllegalArgumentException("numberOfSignificantValueDigits must be between 0 and 5");
        }
        this.numberOfSignificantValueDigits = numberOfSignificantValueDigits;
        return this;
    }

    public HdrBuilder withLowestDiscernibleValue(long lowestDiscernibleValue) {
        if (lowestDiscernibleValue < 1L) {
            throw new IllegalArgumentException("lowestDiscernibleValue must be >= 1");
        }
        this.lowestDiscernibleValue = Optional.of(lowestDiscernibleValue);
        return this;
    }

    public HdrBuilder withHighestTrackableValue(long highestTrackableValue, OverflowResolver overflowResolver) {
        if (highestTrackableValue < 2L) {
            throw new IllegalArgumentException("highestTrackableValue must be >= 2");
        }
        this.highestTrackableValue = Optional.of(highestTrackableValue);
        this.overflowResolver = Optional.of(overflowResolver);
        return this;
    }

    public HdrBuilder withExpectedIntervalBetweenValueSamples(long expectedIntervalBetweenValueSamples) {
        this.expectedIntervalBetweenValueSamples = Optional.of(expectedIntervalBetweenValueSamples);
        return this;
    }

    public HdrBuilder withSnapshotCachingDuration(Duration duration) {
        if (duration.isNegative()) {
            throw new IllegalArgumentException(duration + " is negative");
        }
        this.snapshotCachingDurationMillis = duration.isZero() ? Optional.empty() : Optional.of(duration.toMillis());
        return this;
    }

    public HdrBuilder withPredefinedPercentiles(double[] predefinedPercentiles) {
        if ((predefinedPercentiles = Objects.requireNonNull(predefinedPercentiles, "predefinedPercentiles array should not be null")).length == 0) {
            String msg = "predefinedPercentiles.length is zero. Use withoutSnapshotOptimization() instead of passing empty array.";
            throw new IllegalArgumentException(msg);
        }
        for (double percentile : predefinedPercentiles) {
            if (!(percentile < 0.0) && !(percentile > 1.0)) continue;
            String msg = "Illegal percentiles " + Arrays.toString(predefinedPercentiles) + " - all values must be between 0 and 1";
            throw new IllegalArgumentException(msg);
        }
        double[] sortedPercentiles = HdrBuilder.copyAndSort(predefinedPercentiles);
        this.predefinedPercentiles = Optional.of(sortedPercentiles);
        return this;
    }

    public HdrBuilder withoutSnapshotOptimization() {
        this.predefinedPercentiles = Optional.empty();
        return this;
    }

    public HdrBuilder withBackgroundExecutor(Executor backgroundExecutor) {
        if (backgroundExecutor == null) {
            throw new IllegalArgumentException("backgroundExecutor must not be null");
        }
        this.backgroundExecutor = Optional.of(backgroundExecutor);
        return this;
    }

    public Reservoir buildReservoir() {
        HdrReservoir reservoir = this.buildHdrReservoir();
        reservoir = this.wrapAroundByDecorators(reservoir);
        return reservoir;
    }

    public Histogram buildHistogram() {
        return new Histogram(this.buildReservoir());
    }

    public Histogram buildAndRegisterHistogram(MetricRegistry registry, String name) {
        Histogram histogram = this.buildHistogram();
        registry.register(name, (Metric)histogram);
        return histogram;
    }

    public Timer buildTimer() {
        return new Timer(this.buildReservoir());
    }

    public Timer buildAndRegisterTimer(MetricRegistry registry, String name) {
        Timer timer = this.buildTimer();
        registry.register(name, (Metric)timer);
        return timer;
    }

    public int getEstimatedFootprintInBytes() {
        HdrReservoir hdrReservoir = this.buildHdrReservoir();
        return hdrReservoir.getEstimatedFootprintInBytes();
    }

    public HdrBuilder deepCopy() {
        return new HdrBuilder(this.clock, this.accumulationFactory, this.numberOfSignificantValueDigits, this.predefinedPercentiles, this.lowestDiscernibleValue, this.highestTrackableValue, this.overflowResolver, this.snapshotCachingDurationMillis, this.expectedIntervalBetweenValueSamples, this.backgroundExecutor);
    }

    public String toString() {
        return "HdrBuilder{accumulationStrategy=" + this.accumulationFactory + ", numberOfSignificantValueDigits=" + this.numberOfSignificantValueDigits + ", lowestDiscernibleValue=" + this.lowestDiscernibleValue + ", highestTrackableValue=" + this.highestTrackableValue + ", overflowResolver=" + this.overflowResolver + ", snapshotCachingDurationMillis=" + this.snapshotCachingDurationMillis + ", predefinedPercentiles=" + Arrays.toString(this.predefinedPercentiles.orElse(new double[0])) + '}';
    }

    public HdrBuilder(Clock clock) {
        this(clock, DEFAULT_ACCUMULATION_STRATEGY, DEFAULT_NUMBER_OF_SIGNIFICANT_DIGITS, Optional.of(DEFAULT_PERCENTILES), Optional.empty(), Optional.empty(), Optional.empty(), Optional.empty(), Optional.empty(), Optional.empty());
    }

    private HdrBuilder(Clock clock, AccumulationFactory accumulationFactory, int numberOfSignificantValueDigits, Optional<double[]> predefinedPercentiles, Optional<Long> lowestDiscernibleValue, Optional<Long> highestTrackableValue, Optional<OverflowResolver> overflowResolver, Optional<Long> snapshotCachingDurationMillis, Optional<Long> expectedIntervalBetweenValueSamples, Optional<Executor> backgroundExecutor) {
        this.clock = clock;
        this.accumulationFactory = accumulationFactory;
        this.numberOfSignificantValueDigits = numberOfSignificantValueDigits;
        this.lowestDiscernibleValue = lowestDiscernibleValue;
        this.highestTrackableValue = highestTrackableValue;
        this.overflowResolver = overflowResolver;
        this.snapshotCachingDurationMillis = snapshotCachingDurationMillis;
        this.predefinedPercentiles = predefinedPercentiles;
        this.expectedIntervalBetweenValueSamples = expectedIntervalBetweenValueSamples;
        this.backgroundExecutor = backgroundExecutor;
    }

    private HdrBuilder resetReservoirPeriodicallyByChunks(long resettingPeriodMillis, int numberHistoryChunks) {
        if (resettingPeriodMillis <= 0L) {
            throw new IllegalArgumentException("resettingPeriod must be a positive duration");
        }
        if (resettingPeriodMillis < 1000L) {
            throw new IllegalArgumentException("Interval between resetting must be >= 1000 millis");
        }
        this.accumulationFactory = (recorder, clock) -> new ResetByChunksAccumulator(recorder, numberHistoryChunks, resettingPeriodMillis, clock, this.getExecutor());
        return this;
    }

    private Executor getExecutor() {
        return this.backgroundExecutor.orElseGet(ResilientExecutionUtil.getInstance()::getBackgroundExecutor);
    }

    private HdrReservoir buildHdrReservoir() {
        this.validateParameters();
        Accumulator accumulator = this.accumulationFactory.createAccumulator(this::buildRecorder, this.clock);
        return new HdrReservoir(accumulator, this.predefinedPercentiles, this.highestTrackableValue, this.overflowResolver, this.expectedIntervalBetweenValueSamples);
    }

    private void validateParameters() {
        if (this.highestTrackableValue.isPresent() && this.lowestDiscernibleValue.isPresent() && this.highestTrackableValue.get() < 2L * this.lowestDiscernibleValue.get()) {
            throw new IllegalStateException("highestTrackableValue must be >= 2 * lowestDiscernibleValue");
        }
        if (this.lowestDiscernibleValue.isPresent() && !this.highestTrackableValue.isPresent()) {
            throw new IllegalStateException("lowestDiscernibleValue is specified but highestTrackableValue undefined");
        }
    }

    private Recorder buildRecorder() {
        if (this.lowestDiscernibleValue.isPresent()) {
            return new Recorder(this.lowestDiscernibleValue.get().longValue(), this.highestTrackableValue.get().longValue(), this.numberOfSignificantValueDigits);
        }
        if (this.highestTrackableValue.isPresent()) {
            return new Recorder(this.highestTrackableValue.get().longValue(), this.numberOfSignificantValueDigits);
        }
        return new Recorder(this.numberOfSignificantValueDigits);
    }

    private Reservoir wrapAroundByDecorators(Reservoir reservoir) {
        if (this.snapshotCachingDurationMillis.isPresent()) {
            reservoir = new SnapshotCachingReservoir(reservoir, this.snapshotCachingDurationMillis.get(), this.clock);
        }
        return reservoir;
    }

    private static double[] copyAndSort(double[] predefinedPercentiles) {
        double[] sortedPercentiles = Arrays.copyOf(predefinedPercentiles, predefinedPercentiles.length);
        Arrays.sort(sortedPercentiles);
        return sortedPercentiles;
    }

    static interface AccumulationFactory {
        public static final AccumulationFactory UNIFORM = (recorderSupplier, clock) -> new UniformAccumulator((Recorder)recorderSupplier.get());
        public static final AccumulationFactory RESET_ON_SNAPSHOT = (recorderSupplier, clock) -> new ResetOnSnapshotAccumulator((Recorder)recorderSupplier.get());

        public Accumulator createAccumulator(Supplier<Recorder> var1, Clock var2);
    }
}

