package com.github.webdriverextensions.newversion;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.Paths;
import java.util.regex.Pattern;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveInputStream;
import org.apache.commons.compress.compressors.bzip2.BZip2CompressorInputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream;
import org.codehaus.plexus.util.FileUtils;

public class FileExtractorImpl implements FileExtractor {

    private final Pattern extractPattern;

    private final PathMatcher TAR_BZ2 = FileSystems.getDefault().getPathMatcher("glob:**.tar.bz2");
    private final PathMatcher TAR_GZ = FileSystems.getDefault().getPathMatcher("glob:**.tar.gz");
    private final PathMatcher BZ2 = FileSystems.getDefault().getPathMatcher("glob:**.bz2");
    private final PathMatcher GZ = FileSystems.getDefault().getPathMatcher("glob:**.gz");
    private final PathMatcher TAR = FileSystems.getDefault().getPathMatcher("glob:**.tar");
    private final PathMatcher ZIP = FileSystems.getDefault().getPathMatcher("glob:**.zip");

    public FileExtractorImpl(String extractPattern) {
        this.extractPattern = extractPattern == null ? null : Pattern.compile(extractPattern);
    }

    @Override
    public boolean isExtractable(Path file) {
        return TAR_BZ2.matches(file) ||
               TAR_GZ.matches(file) ||
               BZ2.matches(file) ||
               GZ.matches(file) ||
               TAR.matches(file) ||
               ZIP.matches(file);
    }

    @Override
    public void extractFile(Path file, Path toDirectory) {
        try {
            if (TAR_BZ2.matches(file)) {
                extractTarBz2File(file, toDirectory);
            } else if (TAR_GZ.matches(file)) {
                extractTarGzFile(file, toDirectory);
            } else if (BZ2.matches(file)) {
                extractBz2File(file, toDirectory);
            } else if (GZ.matches(file)) {
                extractGzFile(file, toDirectory);
            } else if (TAR.matches(file)) {
                extractTarFile(file, toDirectory);
            } else if (ZIP.matches(file)) {
                extractZipFile(file, toDirectory);
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static String basename(Path file) {
        final String name = file.getFileName().toString();
        final String extension = FileUtils.extension(name);
        return FileUtils.basename(name, extension.isEmpty() ? extension : "." + extension);
    }

    private void extractBz2File(Path file, Path toDirectory) throws IOException {
        String extractedFilename = basename(file);
        Path fileToExtract = toDirectory.resolve(extractedFilename);
        try (FileInputStream fin = new FileInputStream(file.toFile())) {
            try (BufferedInputStream bin = new BufferedInputStream(fin)) {
                try (BZip2CompressorInputStream bzip2Archive = new BZip2CompressorInputStream(bin)) {
                    Files.copy(bzip2Archive, fileToExtract);
                }
            }
        }
    }

    private void extractGzFile(Path file, Path toDirectory) throws IOException {
        String extractedFilename = basename(file);
        Path fileToExtract = toDirectory.resolve(extractedFilename);
        try (FileInputStream fin = new FileInputStream(file.toFile())) {
            try (BufferedInputStream bin = new BufferedInputStream(fin)) {
                try (GzipCompressorInputStream gzipArchive = new GzipCompressorInputStream(bin)) {
                    Files.copy(gzipArchive, fileToExtract);
                }
            }
        }
    }

    private void extractTarFile(Path file, Path toDirectory) throws IOException {
        Files.createDirectories(toDirectory);
        try (FileInputStream fis = new FileInputStream(file.toFile())) {
            try (BufferedInputStream bis = new BufferedInputStream(fis)) {
                try (TarArchiveInputStream tarArchive = new TarArchiveInputStream(bis)) {
                    extractTar(toDirectory, tarArchive);
                }
            }
        }
    }

    private void extractTarBz2File(Path file, Path toDirectory) throws IOException {
        Files.createDirectories(toDirectory);
        try (FileInputStream fin = new FileInputStream(file.toFile())) {
            try (BufferedInputStream bin = new BufferedInputStream(fin)) {
                try (BZip2CompressorInputStream bzip2Archive = new BZip2CompressorInputStream(bin)) {
                    try (TarArchiveInputStream tarArchive = new TarArchiveInputStream(bzip2Archive)) {
                        extractTar(toDirectory, tarArchive);
                    }
                }
            }
        }
    }

    private void extractTar(Path toDirectory, TarArchiveInputStream tarArchive) throws IOException {
        for (TarArchiveEntry tarEntry = tarArchive.getNextTarEntry(); tarEntry != null; tarEntry = tarArchive.getNextTarEntry()) {
            if (tarEntry.isDirectory()) {
                Path fileToExtract = toDirectory.resolve(tarEntry.getName());
                if (extractPattern != null || !isPathSaveToUse(fileToExtract, toDirectory)) {
                    continue;
                }
                Files.createDirectories(fileToExtract);
            } else {
                if (tarEntry.isSymbolicLink()) {
                    continue;
                }
                extractPattern(toDirectory, tarArchive, tarEntry);
            }
        }
    }

    private void extractPattern(Path toDirectory, TarArchiveInputStream tarArchive, TarArchiveEntry tarEntry) throws IOException {
        Path fileToExtract = toDirectory.resolve(tarEntry.getName());
        if (extractPattern != null) {
            if (!extractPattern.matcher(tarEntry.getName()).matches()) {
                return;
            }
            Path filename = Paths.get(tarEntry.getName()).getFileName();
            fileToExtract = toDirectory.resolve(filename);
        }
        if (isPathSaveToUse(fileToExtract, toDirectory)) {
            // issue #50: directory entries may not have the D attribute set. we may have to create the directory first.
            if (!Files.isDirectory(fileToExtract.getParent())) {
                Files.createDirectories(fileToExtract.getParent());
            }
            Files.copy(tarArchive, fileToExtract);
        }
    }

    private void extractTarGzFile(Path file, Path toDirectory) throws IOException {
        Files.createDirectories(toDirectory);
        try (FileInputStream fin = new FileInputStream(file.toFile())) {
            try (BufferedInputStream bin = new BufferedInputStream(fin)) {
                try (GzipCompressorInputStream gzipArchive = new GzipCompressorInputStream(bin)) {
                    try (TarArchiveInputStream tarArchive = new TarArchiveInputStream(gzipArchive)) {
                        extractTar(toDirectory, tarArchive);
                    }
                }
            }
        }
    }

    private void extractZipFile(Path file, Path toDirectory) throws IOException {
        Files.createDirectories(toDirectory);
        try (FileInputStream fis = new FileInputStream(file.toFile())) {
            try (BufferedInputStream bis = new BufferedInputStream(fis)) {
                try (ZipArchiveInputStream zipArchive = new ZipArchiveInputStream(bis)) {
                    for (ZipArchiveEntry zipEntry = zipArchive.getNextZipEntry(); zipEntry != null; zipEntry = zipArchive.getNextZipEntry()) {

                        Path fileToExtract = toDirectory.resolve(zipEntry.getName());
                        if (zipEntry.isDirectory()) {
                            if (extractPattern != null || !isPathSaveToUse(fileToExtract, toDirectory)) {
                                continue;
                            }
                            Files.createDirectories(fileToExtract);
                        } else {
                            if (zipEntry.isUnixSymlink()) {
                                continue;
                            }
                            if (extractPattern != null) {
                                if (!extractPattern.matcher(zipEntry.getName()).matches()) {
                                    continue;
                                }
                                Path filename = Paths.get(zipEntry.getName()).getFileName();
                                fileToExtract = toDirectory.resolve(filename);
                            }
                            if (isPathSaveToUse(fileToExtract, toDirectory)) {
                                // issue #50: directory entries may not have the D attribute set. we may have to create the directory first.
                                if (!Files.isDirectory(fileToExtract.getParent())) {
                                    Files.createDirectories(fileToExtract.getParent());
                                }
                                Files.copy(zipArchive, fileToExtract);
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * tests that {@code pathToTest} is not outside of {@code expectedParent}
     *
     * @param pathToTest the path to test
     * @param expectedParent the directory {@code pathToTest} should be a child
     * node
     * @return if {@code pathToTest} is not outside of {@code expectedParent} and thus save to use
     */
    private static boolean isPathSaveToUse(Path pathToTest, Path expectedParent) {
        return pathToTest.normalize().startsWith(expectedParent);
    }
}
