package com.gizbel.excel.factory;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;

import com.gizbel.excel.annotations.ExcelBean;
import com.gizbel.excel.annotations.ExcelColumnHeader;
import com.gizbel.excel.annotations.ExcelColumnIndex;
import com.gizbel.excel.enums.ExcelFactoryType;

/**
 * Builds the excel factory on the specified class, The specified class must
 * have annotation type ExcelBean.<br>
 * Processes only first sheet at one time.<br>
 * Dependencies are latest version of apache POI<br>
 * @author Saket Kumar
 */

public class ExcelParser extends Parser{

    /**
     * Will hold the reference to the annotated class which is being populated.
     **/
    private Class clazz;

    /**
     * If set to true then first row in the excel sheet will be neglected.<br>
     **/
    private boolean skipHeader;

    /** Will store the reference to all the fields of the annotated class. **/
    private Map<String, Field> fieldsMap;

    /**
     * Will stop the row processing whenever an empty row is encountered, be
     * <b>default it is set to TRUE</b> in the constructor.
     **/
    private boolean breakAfterEmptyRow;
    

    /**
     * Initialize the excel parser.<br>
     * This constructor will also save the annotated class fields in to a map
     * for future use.
     * @param clazz The Excel Bean class.
     * @param excelFactoryType
     * @param file that needs to be parsed.
     * @throws Exception exception
     */
    public ExcelParser(Class clazz, ExcelFactoryType excelFactoryType,File file) throws Exception {

        super(excelFactoryType, file);
        this.clazz = clazz;
        this.breakAfterEmptyRow = true;

        /*
         * Check whether class has ExcelBean annotation present, If not present
         * then throw exception
         */
        if (clazz.isAnnotationPresent(ExcelBean.class)) {
            
            /*
             * Initialize the fields map as empty hash map, this will used to
             * store the reference to the class fields
             */
            this.fieldsMap = new HashMap<String, Field>();

            /* Get all declared fields for the annotated class */
            Field[] fields = clazz.getDeclaredFields();

            for (Field field : fields) {

                switch (getExcelFactoryType()) {
                    case COLUMN_INDEX_BASED_EXTRACTION: this.prepareColumnIndexBasedFieldMap(field);break;
                    case COLUMN_NAME_BASED_EXTRACTION:   this.prepareColumnHeaderBasedFieldMap(field);break;
                }
            }

        } else
            throw new Exception("Provided class is not annotated with ExcelBean");
        
    }
    
    
    /**
     * Prepares the field Map based on the column index.
     * @param field
     */
    private void prepareColumnIndexBasedFieldMap(Field field){
        if (field.isAnnotationPresent(ExcelColumnIndex.class)) {

            //Make the field accessible and save it into the fields map
            field.setAccessible(true);
            ExcelColumnIndex column = field.getAnnotation(ExcelColumnIndex.class);
            String key = String.valueOf(column.columnIndex());
            this.fieldsMap.put(key, field);
            
        }
    }
    
    /**
     * Prepares the field Map based on the column header.
     * @param field
     */
    private void prepareColumnHeaderBasedFieldMap(Field field){
        if(field.isAnnotationPresent(ExcelColumnHeader.class)){
            field.setAccessible(true);
            ExcelColumnHeader column = field.getAnnotation(ExcelColumnHeader.class);
            String key = column.columnHeader();
            this.fieldsMap.put(key, field);
        }
    }
    
    
    /**
     * Returns the dataType specified in the field.
     * @param field
     * @return String dataType
     */
    private String getDataTypeFor(Field field){
        String dataType = null;
        switch (getExcelFactoryType()) {
            case COLUMN_INDEX_BASED_EXTRACTION: ExcelColumnIndex indexColumn = field.getAnnotation(ExcelColumnIndex.class);
                                                dataType = indexColumn.dataType();
                                                break;
            case COLUMN_NAME_BASED_EXTRACTION:  ExcelColumnHeader headerColumn = field.getAnnotation(ExcelColumnHeader.class);
                                                dataType = headerColumn.dataType();
                                                break;
        }
        return dataType;
    }
    
    
    /**
     * Returns the default value specified in the field.
     * @param field
     * @return String dataType
     */
    private String getDefaultValueFor(Field field){
        String defaultValue = null;
        switch (getExcelFactoryType()) {
            case COLUMN_INDEX_BASED_EXTRACTION: ExcelColumnIndex indexColumn = field.getAnnotation(ExcelColumnIndex.class);
                                                defaultValue = indexColumn.defaultValue();
                                                break;
            case COLUMN_NAME_BASED_EXTRACTION:  ExcelColumnHeader headerColumn = field.getAnnotation(ExcelColumnHeader.class);
                                                defaultValue = headerColumn.defaultValue();
                                                break;
        }
        return defaultValue;
    }
    
    /**
     * Returns the expected column name or index for a specific field.
     * @param field
     * @return String dataType
     */
    private String getExpectedColumnOrIndexFor(Field field){
        String expectedValue = null;
        switch (getExcelFactoryType()) {
            case COLUMN_INDEX_BASED_EXTRACTION: ExcelColumnIndex indexColumn = field.getAnnotation(ExcelColumnIndex.class);
                                                expectedValue = indexColumn.expectedColumnName();
                                                break;
            case COLUMN_NAME_BASED_EXTRACTION:  ExcelColumnHeader headerColumn = field.getAnnotation(ExcelColumnHeader.class);
                                                expectedValue = headerColumn.expectedColumnIndex();
                                                break;
        }
        return expectedValue;
    }
    
    
    
    public List<String> validateFormat(){
        if(skipHeader){
            List<String> errors = new ArrayList<String>();
            for (Entry<String,Field> entry : fieldsMap.entrySet()) {
                String key = entry.getKey(); //For index based extraction this will index, for column name based extraction it will be column name
                Field field = entry.getValue();
                //Checks whether the header matched with the expectations or not
                //For index based the expected value will be column header name
                //For name based the expected value will be column header index
                String expectedValue = getExpectedColumnOrIndexFor(field);
                if(expectedValue!=null && !expectedValue.isEmpty()){
                    if(!expectedValue.equals( getHeadersMap().get(key) )){
                        switch (getExcelFactoryType()) {
                            case COLUMN_INDEX_BASED_EXTRACTION: errors.add("Expected Column Header mismatch at column "+key+ ", Expected column is \""+expectedValue+"\", found is \""+getHeadersMap().get(key)+"\"" );break;
                            case COLUMN_NAME_BASED_EXTRACTION: errors.add("Expected Column Header mismatch for header "+key+ ", Expected column index is \""+expectedValue+"\", found at index \""+getHeadersMap().get(key)+"\"" );break;
                        }
                        
                    }
                }
            }
            return errors;
        }
        return null;
    }
    
    
    /**
     * Reads and convert valid excel file into required format<br>
     * Will only process the first sheet, split multiple sheets into multiple
     * files.
     * @return List
     * @throws IOException IOException
     * @throws InvalidFormatException InvalidFormatException
     * @throws ParseException ParseException
     * @throws IllegalArgumentException IllegalArgumentException
     * @throws IllegalAccessException IllegalAccessException
     * @throws InstantiationException InstantiationException
     */
    public List<Object> parse() throws InvalidFormatException, IOException, InstantiationException,
            IllegalAccessException, IllegalArgumentException, ParseException {
        List<Object> result = new ArrayList<>();

        Sheet sheet = getWorkbook().getSheetAt(0);

        if (getExcelFactoryType() == ExcelFactoryType.COLUMN_NAME_BASED_EXTRACTION) {
            // Fetch the first row and save the field map with column index for
            // corresponding column headers
            Row firstRow = sheet.getRow(0);
            for (Cell column : firstRow) {

                Field field = this.fieldsMap.get(column.getStringCellValue());
                if (field != null) {
                    this.fieldsMap.remove(column.getStringCellValue());
                    this.fieldsMap.put(String.valueOf(column.getColumnIndex()), field);
                }
            }
        }

        for (Row row : sheet) {

            if (getExcelFactoryType() == ExcelFactoryType.COLUMN_INDEX_BASED_EXTRACTION) {
                if (row.getRowNum() == 0 && skipHeader)
                    continue;
            } else if (getExcelFactoryType() == ExcelFactoryType.COLUMN_NAME_BASED_EXTRACTION) {
                if (row.getRowNum() == 0)
                    continue;
            }

            // Process all non empty rows
            if (!isEmptyRow(row)) {
                Object beanObj = this.getBeanForARow(row);
                result.add(beanObj);
            } else {
                // If empty row found and user has opted to break whenever empty
                // row encountered then break the loop
                if (this.breakAfterEmptyRow)
                    break;
            }
        }
        return result;
    }


    /**
     * Fetches the cell details from the each row and sets its values based on
     * the instance variable defined by the annotation.
     * @param row excel row
     * @return Clazz object
     * @throws IllegalAccessException illegal access exception
     * @throws InstantiationException instantiation exception
     * @throws ParseException parse exception
     * @throws IllegalArgumentException illegal argument exception
     */
    public Object getBeanForARow(Row row)
            throws InstantiationException, IllegalAccessException, IllegalArgumentException, ParseException {

        final Object classObj = this.clazz.newInstance();
        for (int i = 0; i < row.getLastCellNum(); i++) {
            Cell cell = row.getCell(i);
            if (cell != null) {
                String value = getCellValue(cell);
                this.setCellValueBasedOnDesiredExcelFactoryType(classObj, value, i);
            } else
                this.setCellValueBasedOnDesiredExcelFactoryType(classObj, null, i);
        }

        return classObj;
    }

    
    /**
     * Parse the cell values to their specified dataType and sets into the java class field.
     * @param classObj
     * @param columnValue
     * @param columnIndex
     * @throws IllegalArgumentException IllegalArgumentException
     * @throws IllegalAccessException IllegalAccessException
     * @throws ParseException ParseException
     */
    private void setCellValueBasedOnDesiredExcelFactoryType(Object classObj, String columnValue, int columnIndex)
            throws IllegalArgumentException, IllegalAccessException, ParseException {

        Field field = this.fieldsMap.get(String.valueOf(columnIndex));
        if (field != null) {

            //If column value is null or empty then try to put the default value
            if( columnValue==null || columnValue.trim().isEmpty())
                columnValue = this.getDefaultValueFor(field);
            
            /**
             * Based on the dataType Specified convert it to primitive value<br>
             * But make sure that columnvalue is not null or empty
             **/
            if(columnValue!=null && !columnValue.trim().isEmpty()){
                String dataType = this.getDataTypeFor(field);
                switch (dataType) {
                case "int":
                    field.set(classObj, Integer.parseInt(columnValue));
                    break;
                case "long":
                    field.set(classObj, Long.parseLong(columnValue));
                    break;
                case "bool":
                    field.set(classObj, Boolean.parseBoolean(columnValue));
                    break;
                case "double":
                    Double data = Double.parseDouble(columnValue);
                    field.set(classObj, data);
                    break;
                case "date":
                    field.set(classObj, this.dateParser(columnValue));
                    break;
                default:
                    field.set(classObj, columnValue);
                    break;
                }
            }
        }

    }

    public boolean isBreakAfterEmptyRow() {
        return breakAfterEmptyRow;
    }

    public void setBreakAfterEmptyRow(boolean breakAfterEmptyRow) {
        this.breakAfterEmptyRow = breakAfterEmptyRow;
    }

    public boolean isSkipHeader() {
        return skipHeader;
    }

    public void setSkipHeader(boolean skipHeader) {
        this.skipHeader = skipHeader;
    }
}
