package com.gizbel.excel.factory;

import java.io.File;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.DateUtil;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;

import com.gizbel.excel.enums.ExcelFactoryType;

/**
 * Processes only first sheet at one time.<br>
 * Dependencies are latest version of apache POI.<br>
 * @author Saket Kumar
 */
public class Parser {

    /**
     * Excel factory type decides whether extraction is going to be column based
     * or index based. 
     */
    private ExcelFactoryType excelFactoryType;

    /** 
     * Will store the key value pairs of 
     * columnIndex,ColumnHeaderName in case if ExcelFactoryType is columnIndex based extraction.<br>
     * columnName,ColumnIndex in case if ExcelFactoryType is columnName based extraction.<br>
     * This will be helpful in validating the expected format of excel.
     */
    private Map<String,String> headersMap;

    
    /** Will hold the reference to the excel workbook file. **/
    private Workbook workbook;

    
    /**
     * Initialize the Parser, will create the headers map using the first row.
     * @param excelFactoryType the factory type for the parser.
     * @param file the file being parsed.
     * @throws Exception any handling exception
     */
    public Parser( ExcelFactoryType excelFactoryType,File file) throws Exception {

        this.excelFactoryType = excelFactoryType;
        this.headersMap = new HashMap<String,String>();
        this.workbook = WorkbookFactory.create(file);
        
        Sheet sheet = workbook.getSheetAt(0);
        Row firstRow = sheet.getRow(0);
        for (Cell column : firstRow) {
            String columnIndex = String.valueOf(column.getColumnIndex());
            String columnValue = getCellValue(column);
            
            //Prepare the header map based on the excel factory type, this will be helpful in validation
            switch (this.excelFactoryType) {
                case COLUMN_INDEX_BASED_EXTRACTION: headersMap.put(columnIndex, columnValue);break;
                case COLUMN_NAME_BASED_EXTRACTION:  headersMap.put(columnValue,columnIndex);break;
            }
        }
    }
    
    
    /**
     * Returns the text value of a column.<br>
     * If it's a date or number, it will be converted to string and then returned. 
     * @param cell the apache poi cell for excel cell
     * @return String cell value
     */
    protected String getCellValue(Cell cell){
        if (cell != null) {
            if (cell.getCellType() == Cell.CELL_TYPE_NUMERIC) {
                if (DateUtil.isCellDateFormatted(cell)) {
                    java.util.Date date = cell.getDateCellValue();
                    cell.setCellType(Cell.CELL_TYPE_STRING);
                    cell.setCellValue(new SimpleDateFormat("dd-MM-yyyy").format(date));
                } else
                    cell.setCellType(Cell.CELL_TYPE_STRING);
            } else
                cell.setCellType(Cell.CELL_TYPE_STRING);

            return (cell.getStringCellValue() == null ? null : cell.getStringCellValue().trim());
        } else
            return null;
    }
    

    /**
     * Parses the date columns in dd-MM-YYYY format. Customize the format in
     * case.
     * @param value the date value in normal string
     * @return Date the java util date object
     */
    protected Date dateParser(String value) {
        if (value != null && !value.isEmpty()) {
            String[] formats = new String[] { "dd-MM-yyyy" };
            java.util.Date date;
            try {
                date = DateUtils.parseDate(value, formats);
                return date;
            } catch (ParseException e) {
                e.printStackTrace();
            }
            return null;
        } else
            return null;
    }


    /**
     * Checks whether an encountered row is empty or not.
     * @param row excel row
     * @return boolean true means row is empty else not empty.
     */
    protected boolean isEmptyRow(Row row) {
        boolean isEmptyRow = true;
        for (int cellNum = row.getFirstCellNum(); cellNum < row.getLastCellNum(); cellNum++) {
            Cell cell = row.getCell(cellNum);
            if (cell != null && cell.getCellType() != Cell.CELL_TYPE_BLANK && StringUtils.isNotBlank(cell.toString())) {
                isEmptyRow = false;
            }
        }
        return isEmptyRow;
    }


    protected Map<String, String> getHeadersMap() {
        return headersMap;
    }

    protected Workbook getWorkbook() {
        return workbook;
    }

    protected ExcelFactoryType getExcelFactoryType() {
        return excelFactoryType;
    }
}
