/*
 * Decompiled with CFR 0.152.
 */
package com.globalmentor.calendar.calculator;

import com.globalmentor.application.Application;
import com.globalmentor.application.BaseCliApplication;
import com.globalmentor.calendar.calculator.CalendarCalculator;
import com.globalmentor.io.BOMInputStreamReader;
import com.globalmentor.java.Conditions;
import com.globalmentor.model.Count;
import com.globalmentor.model.Range;
import java.io.IOException;
import java.io.LineNumberReader;
import java.io.Reader;
import java.io.UncheckedIOException;
import java.time.LocalDate;
import java.time.MonthDay;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoUnit;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.SortedMap;
import javax.annotation.Nonnull;
import picocli.CommandLine;

@CommandLine.Command(name="print-day-totals", description={"A console application to print the totals of days overlapping some ranges."})
public class PrintDayTotals
extends BaseCliApplication {
    @CommandLine.Option(names={"--date"}, paramLabel="<date>", description={"The ending date that the program will use for the calculations. If no date is provided, the current local date will be used. If no year is provided (i.e., if a date on the format [MM-dd] is provided), it will default to the current year."})
    private String date;
    @CommandLine.Option(names={"--from", "-f"}, paramLabel="<fromDate>", description={"The initial date to be used for the calculations. This will set up the window size automatically. If no year is provided (i.e., if a date on the format [MM-dd] is provided), it will default to the last occurrence of the provided date."})
    private String fromDate;
    @CommandLine.Option(names={"--window", "-w"}, paramLabel="<windowSize>", description={"The number of days back to include in each total. If no window size is provided, the number of days between the given date and the same date a year before will be used."})
    private Integer windowSize;
    @CommandLine.Option(names={"--max", "-x"}, paramLabel="<maxDays>", description={"The maximum number of days to be included. If no maximum number is provided, all the days will be included."})
    private Integer maxDays;
    @CommandLine.Option(names={"--history", "-c"}, paramLabel="<historyCount>", description={"The number of day totals to include. If no history count is provided, the window size will be used."})
    private Integer historyCount;
    @CommandLine.Option(names={"--range-lower-bound"}, description={"Whether the first date in each range should be included in the totals.%nValid values: ${COMPLETION-CANDIDATES}.%nDefaults to @|bold ${DEFAULT-VALUE}|@."}, defaultValue="inclusive", arity="0..1")
    private RangeBoundType rangeLowerBound;
    private LocalDate now = LocalDate.now();

    public PrintDayTotals(@Nonnull String[] args) {
        super(args);
    }

    public static void main(@Nonnull String[] args) {
        Application.start((Application)new PrintDayTotals(args));
    }

    public void run() {
        LocalDate date = null;
        LocalDate resetDate = null;
        try {
            date = this.getDate();
            resetDate = this.findInitialDate().orElse(null);
        }
        catch (DateTimeParseException dateTimeParseException) {
            System.err.println("The provided date was in an invalid format. Please, make sure that the given date is correctly in the ISO-8601 format. i.e.: YYYY-MM-DD.");
            System.exit(1);
        }
        assert (date != null) : "<date> should not be null at this point of the program";
        int windowSize = this.getWindowSize();
        int historyCount = this.getHistoryCount();
        HashSet<Range<LocalDate>> ranges = new HashSet<Range<LocalDate>>();
        try {
            String line;
            LineNumberReader reader = new LineNumberReader((Reader)new BOMInputStreamReader(System.in));
            while ((line = reader.readLine()) != null) {
                String[] lineComponents = line.split(",");
                Conditions.checkArgument((lineComponents.length == 2 ? 1 : 0) != 0, (String)"Expected two components on line %d: %s", (Object[])new Object[]{reader.getLineNumber(), line});
                ranges.add((Range<LocalDate>)new Range((Comparable)LocalDate.parse(lineComponents[0]), (Comparable)LocalDate.parse(lineComponents[1])));
            }
        }
        catch (IOException ioException) {
            throw new UncheckedIOException(ioException);
        }
        boolean isRangeLowerInclusive = this.rangeLowerBound == RangeBoundType.inclusive;
        SortedMap<LocalDate, Count> dayCounts = CalendarCalculator.getDayCounts(ranges, isRangeLowerInclusive);
        SortedMap<LocalDate, Long> dayTotals = CalendarCalculator.getDayTotals(date, resetDate, windowSize, historyCount, dayCounts);
        long runTotal = 0L;
        for (Map.Entry dayTotal : dayTotals.entrySet()) {
            LocalDate day = (LocalDate)dayTotal.getKey();
            Count count = (Count)dayCounts.get(day);
            if (count != null && count.getCount() > 0L) {
                System.out.print('*');
                runTotal += count.getCount();
            } else {
                runTotal = 0L;
            }
            System.out.print(',');
            long windowTotal = (Long)dayTotal.getValue();
            System.out.print(day + ",");
            if (count != null) {
                System.out.print(count);
            }
            System.out.print(',');
            if (runTotal != 0L) {
                System.out.print(runTotal);
            }
            System.out.print(',');
            System.out.print(windowTotal);
            this.findMaxDays().ifPresent(maxDays -> System.out.print("," + ((long)maxDays.intValue() - windowTotal)));
            System.out.println();
        }
    }

    void setNow(@Nonnull LocalDate now) {
        this.now = Objects.requireNonNull(now);
    }

    LocalDate getDate() throws DateTimeParseException {
        if (this.date != null) {
            try {
                return LocalDate.parse(this.date);
            }
            catch (DateTimeParseException dateTimeParseExceptionLocalISO) {
                return LocalDate.parse(String.format("%d-%s", this.now.getYear(), this.date));
            }
        }
        return this.now;
    }

    Optional<LocalDate> findInitialDate() throws DateTimeParseException {
        LocalDate initialDate = null;
        if (this.fromDate != null) {
            try {
                initialDate = LocalDate.parse(this.fromDate);
            }
            catch (DateTimeParseException dateTimeParseExceptionLocalISO) {
                LocalDate date = this.getDate();
                int year = date.getYear();
                MonthDay fromDateWithoutYear = MonthDay.parse(String.format("--%s", this.fromDate));
                if (fromDateWithoutYear.isAfter(MonthDay.from(date))) {
                    --year;
                }
                initialDate = fromDateWithoutYear.atYear(year);
            }
            return Optional.of(initialDate);
        }
        return Optional.empty();
    }

    int getWindowSize() {
        int windowSize;
        if (this.windowSize != null) {
            Conditions.checkArgument((this.windowSize >= 0 ? 1 : 0) != 0, (String)"<windowSize> cannot be after less than 0", (Object[])new Object[0]);
            windowSize = this.windowSize;
        } else {
            LocalDate initialDate;
            LocalDate finalDate = this.getDate();
            Conditions.checkArgument((finalDate.compareTo(initialDate = this.findInitialDate().orElse(finalDate.minusYears(1L))) >= 0 ? 1 : 0) != 0, (String)"<fromDate> cannot be after <date>", (Object[])new Object[0]);
            windowSize = (int)ChronoUnit.DAYS.between(initialDate, finalDate);
        }
        return windowSize;
    }

    int getHistoryCount() {
        if (this.historyCount != null) {
            return this.historyCount;
        }
        return this.getWindowSize();
    }

    Optional<Integer> findMaxDays() {
        if (this.maxDays != null) {
            return Optional.of(Conditions.checkArgumentNotNegative((int)this.maxDays));
        }
        return Optional.empty();
    }

    public static enum RangeBoundType {
        inclusive,
        exclusive;

    }
}

