/*
 * Copyright 2017 Globus Ltd.
 * <p/>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p/>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p/>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.globusltd.recyclerview.choice;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentActivity;
import android.support.v7.app.AppCompatActivity;
import android.support.v7.view.ActionMode;

/**
 * Represents a contextual mode of the user interface. Action modes can be used to provide
 * alternative interaction modes and replace parts of the normal UI until finished.
 * Examples of good action modes include text selection and contextual actions.
 * <div class="special reference">
 * <p>
 * <h3>Developer Guides</h3>
 * <p>For information about how to provide contextual actions with {@code ActionMode},
 * read the <a href="{@docRoot}guide/topics/ui/menus.html#context-menu">Menus</a>
 * developer guide.</p>
 * <p>
 * </div>
 */
public abstract class ActionModeCompat {
    
    /**
     * Creates ActionModeCompat for {@link Fragment}.
     * Notice that fragment should be attached to the {@link AppCompatActivity}.
     */
    @NonNull
    public static ActionModeCompat from(@NonNull final Fragment fragment) {
        return new FragmentActionModeCompat(fragment);
    }
    
    /**
     * Creates ActionModeCompat for {@link AppCompatActivity}.
     */
    @NonNull
    public static ActionModeCompat from(@NonNull final AppCompatActivity activity) {
        return new ActivityActionModeCompat(activity);
    }
    
    private ActionModeCompat() {
    }
    
    /**
     * Start an action mode.
     *
     * @param callback Callback that will manage lifecycle events for this context mode.
     * @return The ContextMode that was started, or null if it was canceled.
     */
    @Nullable
    public abstract ActionMode startActionMode(@NonNull final ActionMode.Callback callback);
    
    private static class FragmentActionModeCompat extends ActionModeCompat {
        
        @NonNull
        private final Fragment mFragment;
        
        FragmentActionModeCompat(@NonNull final Fragment fragment) {
            super();
            mFragment = fragment;
        }
        
        @Nullable
        @Override
        public ActionMode startActionMode(@NonNull final ActionMode.Callback callback) {
            final FragmentActivity activity = mFragment.getActivity();
            if (activity instanceof AppCompatActivity) {
                final AppCompatActivity appCompatActivity = (AppCompatActivity) activity;
                return appCompatActivity.startSupportActionMode(callback);
            } else {
                throw new IllegalStateException("Your activity should extend AppCompatActivity.");
            }
        }
        
    }
    
    private static class ActivityActionModeCompat extends ActionModeCompat {
        
        @NonNull
        private final AppCompatActivity mActivity;
        
        ActivityActionModeCompat(@NonNull final AppCompatActivity activity) {
            super();
            mActivity = activity;
        }
        
        @Nullable
        @Override
        public ActionMode startActionMode(@NonNull final ActionMode.Callback callback) {
            return mActivity.startSupportActionMode(callback);
        }
        
    }
    
}
