/*
 * Decompiled with CFR 0.152.
 */
package com.amazonaws.mobileconnectors.s3.transferutility;

import com.amazonaws.mobileconnectors.s3.transferutility.Record;
import com.amazonaws.mobileconnectors.s3.transferutility.S3ClientReference;
import com.amazonaws.mobileconnectors.s3.transferutility.Service;
import com.amazonaws.mobileconnectors.s3.transferutility.TransferDBUtil;
import com.amazonaws.mobileconnectors.s3.transferutility.TransferRecord;
import com.amazonaws.mobileconnectors.s3.transferutility.TransferState;
import com.amazonaws.mobileconnectors.s3.transferutility.TransferStatusUpdater;
import com.amazonaws.mobileconnectors.s3.transferutility.TransferThreadPool;
import com.amazonaws.mobileconnectors.s3.transferutility.TransferUtilityOptions;
import com.amazonaws.services.s3.AmazonS3;
import java.io.FileDescriptor;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Date;
import java.util.Map;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class TransferService {
    private static final Log LOGGER = LogFactory.getLog(TransferService.class);
    static final int MSG_EXEC = 100;
    static final int MSG_CHECK = 200;
    static final int MSG_DISCONNECT = 300;
    static final int MSG_CONNECT = 400;
    static final String INTENT_ACTION_TRANSFER_ADD = "add_transfer";
    static final String INTENT_ACTION_TRANSFER_PAUSE = "pause_transfer";
    static final String INTENT_ACTION_TRANSFER_RESUME = "resume_transfer";
    static final String INTENT_ACTION_TRANSFER_CANCEL = "cancel_transfer";
    static final String INTENT_BUNDLE_TRANSFER_ID = "id";
    static final String INTENT_BUNDLE_TRANSFER_UTILITY_OPTIONS = "transfer_utility_options";
    static final TransferState[] UNFINISHED_TRANSFER_STATES = new TransferState[]{TransferState.WAITING, TransferState.WAITING_FOR_NETWORK, TransferState.IN_PROGRESS, TransferState.RESUMED_WAITING};
    private NetworkInfoReceiver networkInfoReceiver;
    private boolean shouldScan = true;
    private boolean isReceiverNotRegistered = true;
    private volatile long lastActiveTime;
    private volatile int startId;
    private final TransferDBUtil dbUtil;
    TransferStatusUpdater updater;
    private final UpdateHandler updateHandler;
    private long transferServiceCheckTimeInterval;

    public TransferService() {
        LOGGER.debug((Object)"Starting Transfer Service");
        this.dbUtil = new TransferDBUtil();
        this.updater = new TransferStatusUpdater(this.dbUtil);
        this.updateHandler = new UpdateHandler();
        this.networkInfoReceiver = new NetworkInfoReceiver();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int onStartCommand(String action, Integer id, TransferUtilityOptions options, int flags, int startId) {
        this.startId = startId;
        if (this.isReceiverNotRegistered) {
            try {
                LOGGER.info((Object)"registering receiver");
            }
            catch (IllegalArgumentException iae) {
                LOGGER.warn((Object)"Ignoring the exception trying to register the receiver for connectivity change.");
            }
            catch (IllegalStateException ise) {
                LOGGER.warn((Object)"Ignoring the leak in registering the receiver.");
            }
            finally {
                this.isReceiverNotRegistered = false;
            }
        }
        if (id < 0) {
            LOGGER.error((Object)"The intent sent by the TransferUtility doesn't have the id.");
            return -1;
        }
        AmazonS3 s3 = S3ClientReference.get(id);
        if (s3 == null) {
            LOGGER.error((Object)"TransferService can't get s3 client and not acting on the id.");
            return -1;
        }
        TransferUtilityOptions tuOptions = options;
        TransferThreadPool.init(tuOptions.getTransferThreadPoolSize());
        this.transferServiceCheckTimeInterval = tuOptions.getTransferServiceCheckTimeInterval();
        LOGGER.debug((Object)("ThreadPoolSize: " + tuOptions.getTransferThreadPoolSize() + " transferServiceCheckTimeInterval: " + tuOptions.getTransferServiceCheckTimeInterval()));
        this.updateHandler.handleMessage(100, action, id);
        return -1;
    }

    public void onDestroy() {
        try {
            if (this.networkInfoReceiver != null) {
                LOGGER.info((Object)"unregistering receiver");
                this.isReceiverNotRegistered = true;
            }
        }
        catch (IllegalArgumentException iae) {
            LOGGER.warn((Object)"exception trying to destroy the service");
        }
        this.pauseAll();
        TransferThreadPool.closeThreadPool();
        S3ClientReference.clear();
        LOGGER.info((Object)"Closing the database.");
    }

    void checkTransfers() {
        if (this.shouldScan && this.networkInfoReceiver.isNetworkConnected()) {
            this.loadAndResumeTransfersFromDB(UNFINISHED_TRANSFER_STATES);
            this.shouldScan = false;
        }
        if (this.isActive()) {
            this.lastActiveTime = System.currentTimeMillis();
        } else {
            LOGGER.debug((Object)"Stop self");
        }
    }

    void checkTransfersOnNetworkReconnect() {
        if (this.networkInfoReceiver.isNetworkConnected()) {
            this.loadAndResumeTransfersFromDB(new TransferState[]{TransferState.WAITING_FOR_NETWORK});
        } else {
            LOGGER.error((Object)"Network Connect message received but not connected to network.");
        }
    }

    void execCommand(String intentAction, Integer id) {
        this.lastActiveTime = System.currentTimeMillis();
        String action = intentAction;
        AmazonS3 s3 = S3ClientReference.get(id);
        if (INTENT_ACTION_TRANSFER_ADD.equals(action)) {
            if (this.updater.getTransfer(id) != null) {
                LOGGER.warn((Object)("Transfer has already been added: " + id));
            } else {
                TransferRecord transfer = this.dbUtil.getTransferById(id);
                if (transfer != null) {
                    this.updater.addTransfer(transfer);
                    transfer.start(s3, this.dbUtil, this.updater, this.networkInfoReceiver);
                } else {
                    LOGGER.error((Object)("Can't find transfer: " + id));
                }
            }
        } else if (INTENT_ACTION_TRANSFER_PAUSE.equals(action)) {
            TransferRecord transfer = this.updater.getTransfer(id);
            if (transfer == null) {
                transfer = this.dbUtil.getTransferById(id);
            }
            if (transfer != null) {
                transfer.pause(s3, this.updater);
            }
        } else if (INTENT_ACTION_TRANSFER_RESUME.equals(action)) {
            TransferRecord transfer = this.updater.getTransfer(id);
            if (transfer == null) {
                transfer = this.dbUtil.getTransferById(id);
                if (transfer != null) {
                    this.updater.addTransfer(transfer);
                } else {
                    LOGGER.error((Object)("Can't find transfer: " + id));
                }
            }
            if (transfer != null) {
                transfer.start(s3, this.dbUtil, this.updater, this.networkInfoReceiver);
            }
        } else if (INTENT_ACTION_TRANSFER_CANCEL.equals(action)) {
            TransferRecord transfer = this.updater.getTransfer(id);
            if (transfer == null) {
                transfer = this.dbUtil.getTransferById(id);
            }
            if (transfer != null) {
                transfer.cancel(s3, this.updater);
            }
        } else {
            LOGGER.error((Object)("Unknown action: " + action));
        }
    }

    private boolean isActive() {
        if (this.shouldScan) {
            return true;
        }
        for (TransferRecord transfer : this.updater.getTransfers().values()) {
            if (!transfer.isRunning()) continue;
            return true;
        }
        return System.currentTimeMillis() - this.lastActiveTime < this.transferServiceCheckTimeInterval;
    }

    void loadAndResumeTransfersFromDB(TransferState[] transferStates) {
        TransferRecord transfer;
        LOGGER.debug((Object)"Loading transfers from database...");
        int count = 0;
        ArrayList<Integer> transferIds = new ArrayList<Integer>();
        for (Record record : Service.getInstance().getRecords()) {
            int id = record.getId();
            if (this.updater.getTransfer(id) == null) {
                transfer = new TransferRecord();
                transfer.updateFromDB(record);
                this.updater.addTransfer(transfer);
                ++count;
            }
            transferIds.add(id);
        }
        try {
            for (Integer id : transferIds) {
                AmazonS3 s3 = S3ClientReference.get(id);
                if (s3 == null || (transfer = this.updater.getTransfer(id)) == null || transfer.isRunning()) continue;
                transfer.start(s3, this.dbUtil, this.updater, this.networkInfoReceiver);
            }
        }
        catch (Exception exception) {
            LOGGER.error((Object)("Error in resuming the transfers." + exception.getMessage()));
        }
        LOGGER.debug((Object)(count + " transfers are loaded from database."));
    }

    void pauseAll() {
        for (TransferRecord transferRecord : this.updater.getTransfers().values()) {
            AmazonS3 s3 = S3ClientReference.get(transferRecord.getRecord().getId());
            if (s3 == null) continue;
            transferRecord.pause(s3, this.updater);
        }
    }

    void pauseAllForNetwork() {
        for (TransferRecord transferRecord : this.updater.getTransfers().values()) {
            AmazonS3 s3 = S3ClientReference.get(transferRecord.getRecord().getId());
            if (s3 == null || !transferRecord.pause(s3, this.updater)) continue;
            this.updater.updateState(transferRecord.getRecord().getId(), TransferState.WAITING_FOR_NETWORK);
        }
        this.shouldScan = true;
    }

    protected void dump(FileDescriptor fd, PrintWriter writer, String[] args) {
        writer.printf("start id: %d\n", this.startId);
        writer.printf("network status: %s\n", this.networkInfoReceiver.isNetworkConnected());
        writer.printf("lastActiveTime: %s, shouldScan: %s\n", new Date(this.lastActiveTime), this.shouldScan);
        Map<Integer, TransferRecord> transfers = this.updater.getTransfers();
        writer.printf("# of active transfers: %d\n", transfers.size());
        for (TransferRecord transfer : transfers.values()) {
            Record r = transfer.getRecord();
            writer.printf("bucket: %s, key: %s, status: %s, total size: %d, current: %d\n", new Object[]{r.getBucketName(), r.getKey(), r.getState(), r.getBytesTotal(), r.getBytesCurrent()});
        }
        writer.flush();
    }

    class UpdateHandler {
        UpdateHandler() {
        }

        public void handleMessage(int msgWhat, String intentAction, Integer id) {
            LOGGER.info((Object)("Mesage " + msgWhat + ", action: " + intentAction + ", id: " + id));
            switch (msgWhat) {
                case 200: {
                    TransferService.this.checkTransfers();
                    break;
                }
                case 100: {
                    TransferService.this.execCommand(intentAction, id);
                    break;
                }
                case 300: {
                    TransferService.this.pauseAllForNetwork();
                    break;
                }
                case 400: {
                    TransferService.this.checkTransfersOnNetworkReconnect();
                    break;
                }
                default: {
                    LOGGER.error((Object)("Unknown command: " + msgWhat));
                }
            }
        }
    }

    static class NetworkInfoReceiver {
        NetworkInfoReceiver() {
        }

        boolean isNetworkConnected() {
            return true;
        }
    }
}

