/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.lazy.primitive;

import java.io.IOException;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.LongIterable;
import com.gs.collections.api.LazyLongIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.block.function.primitive.LongToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.LongPredicate;
import com.gs.collections.api.block.procedure.primitive.LongProcedure;
import com.gs.collections.api.iterator.LongIterator;
import com.gs.collections.api.bag.primitive.MutableLongBag;
import com.gs.collections.api.list.primitive.MutableLongList;
import com.gs.collections.api.set.primitive.MutableLongSet;
import com.gs.collections.impl.bag.mutable.primitive.LongHashBag;
import com.gs.collections.impl.set.mutable.primitive.LongHashSet;
import com.gs.collections.impl.list.mutable.primitive.LongArrayList;
import com.gs.collections.impl.block.factory.primitive.LongPredicates;

/**
 * This file was automatically generated from template file primitiveSelectIterable.stg.
 */
public class SelectLongIterable
    implements LazyLongIterable
{
    private final LongIterable delegate;
    private final LongPredicate predicate;

    public SelectLongIterable(LongIterable delegate, LongPredicate predicate)
    {
        this.delegate = delegate;
        this.predicate = predicate;
    }

    public LongIterator longIterator()
    {
        return new SelectLongIterator(this.delegate, this.predicate);
    }

    public void forEach(LongProcedure procedure)
    {
        this.delegate.forEach(new IfLongProcedure(procedure));
    }

    public int size()
    {
        return this.delegate.count(this.predicate);
    }

    public boolean isEmpty()
    {
        return !this.longIterator().hasNext();
    }

    public boolean notEmpty()
    {
        return this.longIterator().hasNext();
    }

    public int count(LongPredicate predicate)
    {
        CountLongProcedure countLongProcedure = new CountLongProcedure(predicate);
        this.forEach(countLongProcedure);
        return countLongProcedure.getCount();
    }

    public boolean anySatisfy(LongPredicate predicate)
    {
        return this.delegate.anySatisfy(LongPredicates.and(this.predicate, predicate));
    }

    public boolean allSatisfy(LongPredicate predicate)
    {
        return this.noneSatisfy(LongPredicates.not(predicate));
    }

    public boolean noneSatisfy(LongPredicate predicate)
    {
        return !this.anySatisfy(predicate);
    }

    public LazyLongIterable select(LongPredicate predicate)
    {
        return new SelectLongIterable(this, predicate);
    }

    public LazyLongIterable reject(LongPredicate predicate)
    {
        return new SelectLongIterable(this, LongPredicates.not(predicate));
    }

    public long detectIfNone(LongPredicate predicate, long ifNone)
    {
        for (LongIterator longIterator = this.longIterator(); longIterator.hasNext() ;)
        {
            long item = longIterator.next();
            if (predicate.accept(item))
            {
                return item;
            }
        }
        return ifNone;
    }

    public <V> LazyIterable<V> collect(LongToObjectFunction<? extends V> function)
    {
        return new CollectLongToObjectIterable<V>(this, function);
    }

        public long sum()
        {
            long sum = 0L;
            for (LongIterator longIterator = this.longIterator(); longIterator.hasNext() ;)
            {
                sum += longIterator.next();
            }
            return sum;
        }

        public long max()
        {
            LongIterator longIterator = this.longIterator();
            long max = longIterator.next();
            while (longIterator.hasNext())
            {
                max = (long) Math.max(max, longIterator.next());
            }
            return max;
        }

        public long min()
        {
            LongIterator longIterator = this.longIterator();
            long min = longIterator.next();
            while (longIterator.hasNext())
            {
                min = (long) Math.min(min, longIterator.next());
            }
            return min;
        }

        public long minIfEmpty(long defaultValue)
        {
            try
            {
                return this.min();
            }
            catch (NoSuchElementException ex)
            {
            }
            return defaultValue;
        }

        public long maxIfEmpty(long defaultValue)
        {
            try
            {
                return this.max();
            }
            catch (NoSuchElementException ex)
            {
            }
            return defaultValue;
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return (double) this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            long[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                long first = sortedArray[middleIndex];
                long second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public long[] toSortedArray()
        {
            long[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public MutableLongList toSortedList()
        {
            return LongArrayList.newList(this).sortThis();
        }

    public long[] toArray()
    {
        final long[] array = new long[this.size()];
        this.forEach(new LongProcedure()
        {
            @SuppressWarnings("FieldMayBeFinal")
            private int index = 0;
            public void value(long each)
            {
                array[this.index++] = each;
            }
        });
        return array;
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            LongIterator iterator = this.longIterator();
            if (iterator.hasNext())
            {
                appendable.append(String.valueOf(iterator.next()));
                while (iterator.hasNext())
                {
                    appendable.append(separator);
                    appendable.append(String.valueOf(iterator.next()));
                }
            }

            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public boolean contains(long value)
    {
        return this.anySatisfy(LongPredicates.equal(value));
    }

    public boolean containsAll(long... source)
    {
        for (long value : source)
        {
            if (!this.contains(value))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(LongIterable source)
    {
        for (LongIterator iterator = source.longIterator(); iterator.hasNext(); )
        {
            if (!this.contains(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public MutableLongList toList()
    {
        return LongArrayList.newList(this);
    }

    public MutableLongSet toSet()
    {
        return LongHashSet.newSet(this);
    }

    public MutableLongBag toBag()
    {
        return LongHashBag.newBag(this);
    }

    public LazyLongIterable asLazy()
    {
        return this;
    }

    private static final class CountLongProcedure implements LongProcedure
    {
        private static final long serialVersionUID = 1L;
        private final LongPredicate predicate;
        private int counter = 0;

        private CountLongProcedure(LongPredicate predicate)
        {
            this.predicate = predicate;
        }

        public void value(long each)
        {
            if (this.predicate.accept(each))
            {
                this.counter++;
            }
        }

        public int getCount()
        {
            return this.counter;
        }
    }

    private final class IfLongProcedure implements LongProcedure
    {
        private static final long serialVersionUID = 1L;
        private final LongProcedure procedure;

        private IfLongProcedure(LongProcedure procedure)
        {
            this.procedure = procedure;
        }

        public void value(long each)
        {
            if (SelectLongIterable.this.predicate.accept(each))
            {
                this.procedure.value(each);
            }
        }
    }

    private static final class SelectLongIterator
            implements LongIterator
    {
        private final LongIterator iterator;
        private final LongPredicate predicate;
        private long next;
        private boolean verifiedHasNext = false;

        private SelectLongIterator(LongIterable iterable, LongPredicate predicate)
        {
            this(iterable.longIterator(), predicate);
        }

        private SelectLongIterator(LongIterator iterator, LongPredicate predicate)
        {
            this.iterator = iterator;
            this.predicate = predicate;
        }

        public boolean hasNext()
        {
            if (this.verifiedHasNext)
            {
                return true;
            }
            while (this.iterator.hasNext())
            {
                long temp = this.iterator.next();
                if (this.predicate.accept(temp))
                {
                    this.next = temp;
                    this.verifiedHasNext = true;
                    return true;
                }
            }
            return false;
        }

        public long next()
        {
            if (this.verifiedHasNext || this.hasNext())
            {
                this.verifiedHasNext = false;
                return this.next;
            }
            throw new NoSuchElementException();
        }
    }
}
