/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.list.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.IntIterable;
import com.gs.collections.api.LazyIntIterable;
import com.gs.collections.api.RichIterable;
import com.gs.collections.api.bag.primitive.MutableIntBag;
import com.gs.collections.api.block.function.primitive.ObjectIntIntToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectIntToObjectFunction;
import com.gs.collections.api.block.function.primitive.IntToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.IntPredicate;
import com.gs.collections.api.block.procedure.primitive.IntIntProcedure;
import com.gs.collections.api.block.procedure.primitive.IntProcedure;
import com.gs.collections.api.iterator.IntIterator;
import com.gs.collections.api.list.MutableList;
import com.gs.collections.api.list.primitive.IntList;
import com.gs.collections.api.list.primitive.ImmutableIntList;
import com.gs.collections.api.list.primitive.MutableIntList;
import com.gs.collections.api.set.primitive.MutableIntSet;
import com.gs.collections.impl.bag.mutable.primitive.IntHashBag;
import com.gs.collections.impl.factory.primitive.IntLists;
import com.gs.collections.impl.lazy.primitive.LazyIntIterableAdapter;
import com.gs.collections.impl.list.mutable.FastList;
import com.gs.collections.impl.set.mutable.primitive.IntHashSet;
import net.jcip.annotations.NotThreadSafe;

/**
 * IntArrayList is similar to {@link FastList}, and is memory-optimized for int primitives.
 * This file was automatically generated from template file primitiveArrayList.stg.
 *
 * @since 3.0.
 */
@NotThreadSafe
public final class IntArrayList
        implements MutableIntList, Externalizable
{
    private static final long serialVersionUID = 1L;
    private static final int[] DEFAULT_SIZED_EMPTY_ARRAY = {};
    private static final int[] ZERO_SIZED_ARRAY = {};
    private static final int MAXIMUM_ARRAY_SIZE = Integer.MAX_VALUE - 8;
    private int size;
    private transient int[] items = DEFAULT_SIZED_EMPTY_ARRAY;

    public IntArrayList()
    {
    }

    public IntArrayList(int initialCapacity)
    {
        this.items = initialCapacity == 0 ? ZERO_SIZED_ARRAY : new int[initialCapacity];
    }

    public IntArrayList(int... array)
    {
        this.size = array.length;
        this.items = array;
    }

    /**
     * Creates a new list using the passed {@code elements} argument as the backing store.
     * <p/>
     * !!! WARNING: This method uses the passed in array, so can be very unsafe if the original
     * array is held onto anywhere else. !!!
     */
    public static IntArrayList newListWith(int... elements)
    {
        return new IntArrayList(elements);
    }

    public static IntArrayList newList(IntIterable source)
    {
        return IntArrayList.newListWith(source.toArray());
    }

    public static IntArrayList newWithNValues(int size, int value)
    {
        IntArrayList newList = new IntArrayList(size);
        for (int i = 0; i < size; i++)
        {
            newList.add(value);
        }
        return newList;
    }

    public int size()
    {
        return this.size;
    }

    public boolean isEmpty()
    {
        return this.size == 0;
    }

    public boolean notEmpty()
    {
        return this.size > 0;
    }

    public void clear()
    {
        Arrays.fill(this.items, 0);
        this.size = 0;
    }

    public boolean contains(int value)
    {
        for (int i = 0; i < this.size; i++)
        {
            if (this.items[i] == value)
            {
                return true;
            }
        }
        return false;
    }

    public boolean containsAll(int... source)
    {
        for (int value : source)
        {
            if (!this.contains(value))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(IntIterable source)
    {
        for (IntIterator iterator = source.intIterator(); iterator.hasNext(); )
        {
            if (!this.contains(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public int get(int index)
    {
        if (index < this.size)
        {
            return this.items[index];
        }
        throw this.newIndexOutOfBoundsException(index);
    }

    private IndexOutOfBoundsException newIndexOutOfBoundsException(int index)
    {
        return new IndexOutOfBoundsException("Index: " + index + " Size: " + this.size);
    }

    public int getFirst()
    {
        this.checkEmpty();
        return this.items[0];
    }

    public int getLast()
    {
        this.checkEmpty();
        return this.items[this.size() - 1];
    }

    private void checkEmpty()
    {
        if (this.isEmpty())
        {
            throw this.newIndexOutOfBoundsException(0);
        }
    }

    public int indexOf(int value)
    {
        for (int i = 0; i < this.size; i++)
        {
            if (this.items[i] == value)
            {
                return i;
            }
        }
        return -1;
    }

    public int lastIndexOf(int value)
    {
        for (int i = this.size - 1; i >= 0; i--)
        {
            if (this.items[i] == value)
            {
                return i;
            }
        }
        return -1;
    }

    public void trimToSize()
    {
        if (this.size < this.items.length)
        {
            this.transferItemsToNewArrayWithCapacity(this.size);
        }
    }

    private void transferItemsToNewArrayWithCapacity(int newCapacity)
    {
        this.items = this.copyItemsWithNewCapacity(newCapacity);
    }

    private int[] copyItemsWithNewCapacity(int newCapacity)
    {
        int[] newItems = new int[newCapacity];
        System.arraycopy(this.items, 0, newItems, 0, Math.min(this.size, newCapacity));
        return newItems;
    }

    private int sizePlusFiftyPercent(int oldSize)
    {
        int result = oldSize + (oldSize >> 1) + 1;
        return result < oldSize ? MAXIMUM_ARRAY_SIZE : result;
    }

    public void ensureCapacity(int minCapacity)
    {
        int oldCapacity = this.items.length;
        if (minCapacity > oldCapacity)
        {
            int newCapacity = Math.max(this.sizePlusFiftyPercent(oldCapacity), minCapacity);
            this.transferItemsToNewArrayWithCapacity(newCapacity);
        }
    }

    private void ensureCapacityForAdd()
    {
        if (this.items == DEFAULT_SIZED_EMPTY_ARRAY)
        {
            this.items = new int[10];
        }
        else
        {
            this.transferItemsToNewArrayWithCapacity(this.sizePlusFiftyPercent(this.size));
        }
    }

    public boolean add(int newItem)
    {
        if (this.items.length == this.size)
        {
            this.ensureCapacityForAdd();
        }
        this.items[this.size] = newItem;
        this.size++;
        return true;
    }

    public boolean addAll(int... source)
    {
        if (source.length < 1)
        {
            return false;
        }
        int sourceSize = source.length;
        int newSize = this.size + sourceSize;
        this.ensureCapacity(newSize);
        System.arraycopy(source, 0, this.items, this.size, sourceSize);
        this.size = newSize;
        return true;
    }

    public boolean addAll(IntIterable source)
    {
        return this.addAll(source.toArray());
    }

    private void throwOutOfBounds(int index)
    {
        throw this.newIndexOutOfBoundsException(index);
    }

    public void addAtIndex(int index, int element)
    {
        if (index > -1 && index < this.size)
        {
            this.addAtIndexLessThanSize(index, element);
        }
        else if (index == this.size)
        {
            this.add(element);
        }
        else
        {
            this.throwOutOfBounds(index);
        }
    }

    private void addAtIndexLessThanSize(int index, int element)
    {
        int oldSize = this.size;
        this.size++;
        if (this.items.length == oldSize)
        {
            int[] newItems = new int[this.sizePlusFiftyPercent(oldSize)];
            if (index > 0)
            {
                System.arraycopy(this.items, 0, newItems, 0, index);
            }
            System.arraycopy(this.items, index, newItems, index + 1, oldSize - index);
            this.items = newItems;
        }
        else
        {
            System.arraycopy(this.items, index, this.items, index + 1, oldSize - index);
        }
        this.items[index] = element;
    }

    public boolean addAllAtIndex(int index, int... source)
    {
        if (index > this.size || index < 0)
        {
            this.throwOutOfBounds(index);
        }
        if (source.length == 0)
        {
            return false;
        }
        int sourceSize = source.length;
        int newSize = this.size + sourceSize;
        this.ensureCapacity(newSize);
        this.shiftElementsAtIndex(index, sourceSize);
        System.arraycopy(source, 0, this.items, index, sourceSize);
        this.size = newSize;
        return true;
    }

    public boolean addAllAtIndex(int index, IntIterable source)
    {
        return this.addAllAtIndex(index, source.toArray());
    }

    private void shiftElementsAtIndex(int index, int sourceSize)
    {
        int numberToMove = this.size - index;
        if (numberToMove > 0)
        {
            System.arraycopy(this.items, index, this.items, index + sourceSize, numberToMove);
        }
    }

    public boolean remove(int value)
    {
        int index = this.indexOf(value);
        if (index >= 0)
        {
            this.removeAtIndex(index);
            return true;
        }
        return false;
    }

    public boolean removeAll(IntIterable source)
    {
        boolean modified = false;
        for (int index = 0; index < this.size; index++)
        {
            if (source.contains(this.get(index)))
            {
                this.removeAtIndex(index);
                index--;
                modified = true;
            }
        }
        return modified;
    }

    public boolean removeAll(int... source)
    {
        IntHashSet set = IntHashSet.newSetWith(source);
        int[] newItems = new int[this.size];
        int count = 0;
        int oldSize = this.size;
        for (int index = 0; index < this.size; index++)
        {
            if (!set.contains(this.items[index]))
            {
                newItems[count] = this.items[index];
                count++;
            }
        }
        this.items = newItems;
        this.size = count;
        return oldSize != this.size;
    }

    public int removeAtIndex(int index)
    {
        int previous = this.get(index);
        int totalOffset = this.size - index - 1;
        if (totalOffset > 0)
        {
            System.arraycopy(this.items, index + 1, this.items, index, totalOffset);
        }
        --this.size;
        this.items[this.size] = 0;
        return previous;
    }

    public int set(int index, int element)
    {
        int previous = this.get(index);
        this.items[index] = element;
        return previous;
    }

    public IntArrayList with(int element)
    {
        this.add(element);
        return this;
    }

    public IntArrayList without(int element)
    {
        this.remove(element);
        return this;
    }

    public IntArrayList withAll(IntIterable elements)
    {
        this.addAll(elements.toArray());
        return this;
    }

    public IntArrayList withoutAll(IntIterable elements)
    {
        this.removeAll(elements);
        return this;
    }

    public IntArrayList with(int element1, int element2)
    {
        this.add(element1);
        this.add(element2);
        return this;
    }

    public IntArrayList with(int element1, int element2, int element3)
    {
        this.add(element1);
        this.add(element2);
        this.add(element3);
        return this;
    }

    public IntArrayList with(int element1, int element2, int element3, int... elements)
    {
        this.add(element1);
        this.add(element2);
        this.add(element3);
        return this.withArrayCopy(elements, 0, elements.length);
    }

    private IntArrayList withArrayCopy(int[] elements, int begin, int length)
    {
        this.ensureCapacity(this.size + length);
        System.arraycopy(elements, begin, this.items, this.size, length);
        this.size += length;
        return this;
    }

    public IntIterator intIterator()
    {
        return new InternalIntIterator();
    }

    public void forEach(IntProcedure procedure)
    {
        for (int i = 0; i < this.size; i++)
        {
            procedure.value(this.items[i]);
        }
    }

    public void forEachWithIndex(IntIntProcedure procedure)
    {
        for (int i = 0; i < this.size; i++)
        {
            procedure.value(this.items[i], i);
        }
    }

    public <T> T injectInto(T injectedValue, ObjectIntToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        for (int i = 0; i < this.size; i++)
        {
            result = function.valueOf(result, this.items[i]);
        }
        return result;
    }

    public <T> T injectIntoWithIndex(T injectedValue, ObjectIntIntToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        for (int i = 0; i < this.size; i++)
        {
            result = function.valueOf(result, this.items[i], i);
        }
        return result;
    }

    public int count(IntPredicate predicate)
    {
        int count = 0;
        for (int i = 0; i < this.size; i++)
        {
            if (predicate.accept(this.items[i]))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(IntPredicate predicate)
    {
        for (int i = 0; i < this.size; i++)
        {
            if (predicate.accept(this.items[i]))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(IntPredicate predicate)
    {
        for (int i = 0; i < this.size; i++)
        {
            if (!predicate.accept(this.items[i]))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(IntPredicate predicate)
    {
        for (int i = 0; i < this.size; i++)
        {
            if (predicate.accept(this.items[i]))
            {
                return false;
            }
        }
        return true;
    }

    public IntArrayList select(IntPredicate predicate)
    {
        IntArrayList result = new IntArrayList();
        for (int i = 0; i < this.size; i++)
        {
            int item = this.items[i];
            if (predicate.accept(item))
            {
                result.add(item);
            }
        }
        return result;
    }

    public IntArrayList reject(IntPredicate predicate)
    {
        IntArrayList result = new IntArrayList();
        for (int i = 0; i < this.size; i++)
        {
            int item = this.items[i];
            if (!predicate.accept(item))
            {
                result.add(item);
            }
        }
        return result;
    }

    public int detectIfNone(IntPredicate predicate, int ifNone)
    {
        for (int i = 0; i < this.size; i++)
        {
            int item = this.items[i];
            if (predicate.accept(item))
            {
                return item;
            }
        }
        return ifNone;
    }

    public <V> MutableList<V> collect(IntToObjectFunction<? extends V> function)
    {
        FastList<V> target = FastList.newList(this.size);
        for (int i = 0; i < this.size; i++)
        {
            target.add(function.valueOf(this.items[i]));
        }
        return target;
    }

    public int max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        int max = this.items[0];
        for (int i = 1; i < this.size; i++)
        {
            int value = this.items[i];
            if (max < value)
            {
                max = value;
            }
        }
        return max;
    }

    public int min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        int min = this.items[0];
        for (int i = 1; i < this.size; i++)
        {
            int value = this.items[i];
            if (value < min)
            {
                min = value;
            }
        }
        return min;
    }

    public int minIfEmpty(int defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public int maxIfEmpty(int defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public long sum()
    {
        long result = 0L;
        for (int i = 0; i < this.size; i++)
        {
            result += this.items[i];
        }
        return result;
    }

    public long dotProduct(IntList list)
    {
        if (this.size != list.size())
        {
            throw new IllegalArgumentException("Lists used in dotProduct must be the same size");
        }
        long sum = 0L;
        for (int i = 0; i < this.size; i++)
        {
            sum += (long) this.items[i] * list.get(i);
        }
        return sum;
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return (double) this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        int[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            int first = sortedArray[middleIndex];
            int second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public int[] toArray()
    {
        int[] newItems = new int[this.size];
        System.arraycopy(this.items, 0, newItems, 0, this.size);
        return newItems;
    }

    public int[] toSortedArray()
    {
        int[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    @Override
    public boolean equals(Object otherList)
    {
        if (otherList == this)
        {
            return true;
        }
        if (!(otherList instanceof IntList))
        {
            return false;
        }
        IntList list = (IntList) otherList;
        if (this.size != list.size())
        {
            return false;
        }
        for (int i = 0; i < this.size; i++)
        {
            if (this.items[i] != list.get(i))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int hashCode = 1;
        for (int i = 0; i < this.size; i++)
        {
            int item = this.items[i];
            hashCode = 31 * hashCode + item;
        }
        return hashCode;
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(
            Appendable appendable,
            String start,
            String separator,
            String end)
    {
        try
        {
            appendable.append(start);
            for (int i = 0; i < this.size; i++)
            {
                if (i > 0)
                {
                    appendable.append(separator);
                }
                int value = this.items[i];
                appendable.append(String.valueOf(value));
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public MutableIntList toList()
    {
        return IntArrayList.newList(this);
    }

    public MutableIntList toSortedList()
    {
        return IntArrayList.newList(this).sortThis();
    }

    public MutableIntSet toSet()
    {
        return IntHashSet.newSet(this);
    }

    public MutableIntBag toBag()
    {
        return IntHashBag.newBag(this);
    }

    public LazyIntIterable asLazy()
    {
        return new LazyIntIterableAdapter(this);
    }

    public MutableIntList asUnmodifiable()
    {
        return new UnmodifiableIntList(this);
    }

    public MutableIntList asSynchronized()
    {
        return new SynchronizedIntList(this);
    }

    public ImmutableIntList toImmutable()
    {
        if (this.items.length == 0)
        {
            return IntLists.immutable.with();
        }
        if (this.items.length == 1)
        {
            return IntLists.immutable.with(this.items[0]);
        }
        return IntLists.immutable.with(this.toArray());
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size);
        for (int i = 0; i < this.size; i++)
        {
            out.writeInt(this.items[i]);
        }
    }

    public void readExternal(ObjectInput in) throws IOException
    {
        this.size = in.readInt();
        this.items = new int[this.size];
        for (int i = 0; i < this.size; i++)
        {
            this.items[i] = in.readInt();
        }
    }

    public IntIterable asReversed()
    {
        return new ReverseIntIterable();
    }

    public IntArrayList reverseThis()
    {
        int size = this.items.length;
        int endIndex = size - 1;
        for (int i = 0; i < size / 2; i++)
        {
            int tempSwapValue = this.items[i];
            this.items[i] = this.items[endIndex - i];
            this.items[endIndex - i] = tempSwapValue;
        }
        return this;
    }

    public IntArrayList sortThis()
    {
        Arrays.sort(this.items, 0, this.items.length);
        return this;
    }

    public IntArrayList toReversed()
    {
        return IntArrayList.newList(this.asReversed());
    }

    private class InternalIntIterator implements IntIterator
    {
        /**
         * Index of element to be returned by subsequent call to next.
         */
        private int currentIndex;

        public boolean hasNext()
        {
            return this.currentIndex != IntArrayList.this.size();
        }

        public int next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException();
            }
            int next = IntArrayList.this.items[this.currentIndex];
            this.currentIndex++;
            return next;
        }
    }

    private class ReverseIntIterable implements IntIterable
    {
        public IntIterator intIterator()
        {
            return new ReverseIntIterator();
        }

        public void forEach(IntProcedure procedure)
        {
            IntIterator iterator = this.intIterator();
            while (iterator.hasNext())
            {
                procedure.value(iterator.next());
            }
        }

        public int count(IntPredicate predicate)
        {
            int count = 0;
            IntIterator iterator = this.intIterator();
            while (iterator.hasNext())
            {
                if (predicate.accept(iterator.next()))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(IntPredicate predicate)
        {
            IntIterator iterator = this.intIterator();
            while (iterator.hasNext())
            {
                if (predicate.accept(iterator.next()))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(IntPredicate predicate)
        {
            IntIterator iterator = this.intIterator();
            while (iterator.hasNext())
            {
                if (!predicate.accept(iterator.next()))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(IntPredicate predicate)
        {
            IntIterator iterator = this.intIterator();
            while (iterator.hasNext())
            {
                if (predicate.accept(iterator.next()))
                {
                    return false;
                }
            }
            return true;
        }

        public IntIterable select(IntPredicate predicate)
        {
            IntArrayList result = new IntArrayList(IntArrayList.this.size);
            IntIterator iterator = this.intIterator();
            while (iterator.hasNext())
            {
                int value = iterator.next();
                if (predicate.accept(value))
                {
                    result.add(value);
                }
            }
            return result;
        }

        public IntIterable reject(IntPredicate predicate)
        {
            IntArrayList result = new IntArrayList(IntArrayList.this.size);
            IntIterator iterator = this.intIterator();
            while (iterator.hasNext())
            {
                int value = iterator.next();
                if (!predicate.accept(value))
                {
                    result.add(value);
                }
            }
            return result;
        }

        public int detectIfNone(IntPredicate predicate, int ifNone)
        {
            IntIterator iterator = this.intIterator();
            while (iterator.hasNext())
            {
                int value = iterator.next();
                if (predicate.accept(value))
                {
                    return value;
                }
            }
            return ifNone;
        }

        public <V> RichIterable<V> collect(IntToObjectFunction<? extends V> function)
        {
            FastList<V> results = FastList.newList(IntArrayList.this.size);
            IntIterator iterator = this.intIterator();
            while (iterator.hasNext())
            {
                results.add(function.valueOf(iterator.next()));
            }
            return results;
        }

        public long sum()
        {
            return IntArrayList.this.sum();
        }

        public int max()
        {
            return IntArrayList.this.max();
        }

        public int min()
        {
            return IntArrayList.this.min();
        }

        public int minIfEmpty(int defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.min();
        }

        public int maxIfEmpty(int defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.max();
        }

        public double average()
        {
            return IntArrayList.this.average();
        }

        public double median()
        {
            return IntArrayList.this.median();
        }

        public int[] toSortedArray()
        {
            return IntArrayList.this.toSortedArray();
        }

        public int[] toArray()
        {
            int[] results = new int[IntArrayList.this.size];
            int index = 0;
            IntIterator iterator = this.intIterator();
            while (iterator.hasNext())
            {
                results[index] = iterator.next();
                index++;
            }
            return results;
        }

        public boolean contains(int value)
        {
            return IntArrayList.this.contains(value);
        }

        public boolean containsAll(int... source)
        {
            return IntArrayList.this.containsAll(source);
        }

        public boolean containsAll(IntIterable source)
        {
            return IntArrayList.this.containsAll(source);
        }

        public int size()
        {
            return IntArrayList.this.size;
        }

        public boolean isEmpty()
        {
            return IntArrayList.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return IntArrayList.this.notEmpty();
        }

        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                IntIterator iterator = this.intIterator();
                boolean appendSeparator = false;
                while (iterator.hasNext())
                {
                    if (appendSeparator)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(iterator.next()));
                    appendSeparator = true;
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public MutableIntList toList()
        {
            return IntArrayList.newList(this);
        }

        public MutableIntList toSortedList()
        {
            return IntArrayList.newList(this).sortThis();
        }

        public MutableIntSet toSet()
        {
            return IntHashSet.newSet(this);
        }

        public MutableIntBag toBag()
        {
            return IntHashBag.newBag(this);
        }

        public LazyIntIterable asLazy()
        {
            return new LazyIntIterableAdapter(this);
        }

        private class ReverseIntIterator implements IntIterator
        {
            /**
             * Index of element to be returned by subsequent call to next.
             */
            private int currentIndex = IntArrayList.this.size - 1;

            public boolean hasNext()
            {
                return this.currentIndex != -1;
            }

            public int next()
            {
                if (!this.hasNext())
                {
                    throw new NoSuchElementException();
                }
                int next = IntArrayList.this.items[this.currentIndex];
                this.currentIndex--;
                return next;
            }
        }
    }
}
