/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.list.mutable.primitive;

import java.util.Collection;
import java.util.Collections;

import com.gs.collections.api.IntIterable;
import com.gs.collections.api.LazyIntIterable;
import com.gs.collections.api.block.function.primitive.IntToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.IntPredicate;
import com.gs.collections.api.iterator.IntIterator;
import com.gs.collections.api.list.MutableList;
import com.gs.collections.api.list.primitive.IntList;
import com.gs.collections.api.list.primitive.ImmutableIntList;
import com.gs.collections.api.list.primitive.MutableIntList;
import com.gs.collections.impl.collection.mutable.primitive.AbstractSynchronizedIntCollection;
import com.gs.collections.impl.factory.primitive.IntLists;
import com.gs.collections.impl.lazy.primitive.LazyIntIterableAdapter;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;

/**
 * A synchronized view of a {@link MutableIntList}. It is imperative that the user manually synchronize on the collection when iterating over it using the
 * {@link IntIterator}, as per {@link Collections#synchronizedCollection(Collection)}.
 * <p>
 * This file was automatically generated from template file synchronizedPrimitiveList.stg.
 * </p>
 *
 * @see MutableIntList#asSynchronized()
 * @see MutableList#asSynchronized()
 * @since 3.1.
 */
@ThreadSafe
public final class SynchronizedIntList
        extends AbstractSynchronizedIntCollection
        implements MutableIntList
{
    private static final long serialVersionUID = 1L;

    SynchronizedIntList(MutableIntList list)
    {
        super(list);
    }

    SynchronizedIntList(MutableIntList list, Object newLock)
    {
        super(list, newLock);
    }

    @GuardedBy("getLock()")
    private MutableIntList getMutableIntList()
    {
        return (MutableIntList) this.getIntCollection();
    }

    public int get(int index)
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntList().get(index);
        }
    }

    public int getFirst()
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntList().getFirst();
        }
    }

    public int getLast()
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntList().getLast();
        }
    }

    public int indexOf(int value)
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntList().indexOf(value);
        }
    }

    public int lastIndexOf(int value)
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntList().lastIndexOf(value);
        }
    }

    public void addAtIndex(int index, int element)
    {
        synchronized (this.getLock())
        {
            this.getMutableIntList().addAtIndex(index, element);
        }
    }

    public boolean addAllAtIndex(int index, int... source)
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntList().addAllAtIndex(index, source);
        }
    }

    public boolean addAllAtIndex(int index, IntIterable source)
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntList().addAllAtIndex(index, source);
        }
    }

    public int removeAtIndex(int index)
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntList().removeAtIndex(index);
        }
    }

    public int set(int index, int element)
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntList().set(index, element);
        }
    }

    @Override
    public SynchronizedIntList with(int element)
    {
        synchronized (this.getLock())
        {
            this.getMutableIntList().add(element);
        }
        return this;
    }

    @Override
    public SynchronizedIntList without(int element)
    {
        synchronized (this.getLock())
        {
            this.getMutableIntList().remove(element);
        }
        return this;
    }

    @Override
    public SynchronizedIntList withAll(IntIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableIntList().addAll(elements.toArray());
        }
        return this;
    }

    @Override
    public SynchronizedIntList withoutAll(IntIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableIntList().removeAll(elements);
        }
        return this;
    }

    @Override
    public MutableIntList select(IntPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntList().select(predicate);
        }
    }

    @Override
    public MutableIntList reject(IntPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntList().reject(predicate);
        }
    }

    @Override
    public <V> MutableList<V> collect(IntToObjectFunction<? extends V> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntList().collect(function);
        }
    }

    public MutableIntList sortThis()
    {
        synchronized (this.getLock())
        {
            this.getMutableIntList().sortThis();
        }
        return this;
    }

    public long dotProduct(IntList list)
    {
        return this.getMutableIntList().dotProduct(list);
    }

    @Override
    public boolean equals(Object otherList)
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntList().equals(otherList);
        }
    }

    @Override
    public int hashCode()
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntList().hashCode();
        }
    }

    @Override
    public LazyIntIterable asLazy()
    {
        synchronized (this.getLock())
        {
            return new LazyIntIterableAdapter(this);
        }
    }

    @Override
    public MutableIntList asUnmodifiable()
    {
        return new UnmodifiableIntList(this);
    }

    @Override
    public MutableIntList asSynchronized()
    {
        return this;
    }

    @Override
    public ImmutableIntList toImmutable()
    {
        int size = this.size();
        if (size == 0)
        {
            return IntLists.immutable.with();
        }
        if (size == 1)
        {
            return IntLists.immutable.with(this.getFirst());
        }
        return IntLists.immutable.with(this.toArray());
    }

    public MutableIntList reverseThis()
    {
        synchronized (this.getLock())
        {
            this.getMutableIntList().reverseThis();
        }
        return this;
    }

    public MutableIntList toReversed()
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntList().toReversed();
        }
    }
}
