/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.list.mutable.primitive;

import java.util.Collection;
import java.util.Collections;

import com.gs.collections.api.LongIterable;
import com.gs.collections.api.LazyLongIterable;
import com.gs.collections.api.block.function.primitive.LongToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.LongPredicate;
import com.gs.collections.api.iterator.LongIterator;
import com.gs.collections.api.list.MutableList;
import com.gs.collections.api.list.primitive.LongList;
import com.gs.collections.api.list.primitive.ImmutableLongList;
import com.gs.collections.api.list.primitive.MutableLongList;
import com.gs.collections.impl.collection.mutable.primitive.AbstractSynchronizedLongCollection;
import com.gs.collections.impl.factory.primitive.LongLists;
import com.gs.collections.impl.lazy.primitive.LazyLongIterableAdapter;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;

/**
 * A synchronized view of a {@link MutableLongList}. It is imperative that the user manually synchronize on the collection when iterating over it using the
 * {@link LongIterator}, as per {@link Collections#synchronizedCollection(Collection)}.
 * <p>
 * This file was automatically generated from template file synchronizedPrimitiveList.stg.
 * </p>
 *
 * @see MutableLongList#asSynchronized()
 * @see MutableList#asSynchronized()
 * @since 3.1.
 */
@ThreadSafe
public final class SynchronizedLongList
        extends AbstractSynchronizedLongCollection
        implements MutableLongList
{
    private static final long serialVersionUID = 1L;

    SynchronizedLongList(MutableLongList list)
    {
        super(list);
    }

    SynchronizedLongList(MutableLongList list, Object newLock)
    {
        super(list, newLock);
    }

    @GuardedBy("getLock()")
    private MutableLongList getMutableLongList()
    {
        return (MutableLongList) this.getLongCollection();
    }

    public long get(int index)
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongList().get(index);
        }
    }

    public long getFirst()
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongList().getFirst();
        }
    }

    public long getLast()
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongList().getLast();
        }
    }

    public int indexOf(long value)
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongList().indexOf(value);
        }
    }

    public int lastIndexOf(long value)
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongList().lastIndexOf(value);
        }
    }

    public void addAtIndex(int index, long element)
    {
        synchronized (this.getLock())
        {
            this.getMutableLongList().addAtIndex(index, element);
        }
    }

    public boolean addAllAtIndex(int index, long... source)
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongList().addAllAtIndex(index, source);
        }
    }

    public boolean addAllAtIndex(int index, LongIterable source)
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongList().addAllAtIndex(index, source);
        }
    }

    public long removeAtIndex(int index)
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongList().removeAtIndex(index);
        }
    }

    public long set(int index, long element)
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongList().set(index, element);
        }
    }

    @Override
    public SynchronizedLongList with(long element)
    {
        synchronized (this.getLock())
        {
            this.getMutableLongList().add(element);
        }
        return this;
    }

    @Override
    public SynchronizedLongList without(long element)
    {
        synchronized (this.getLock())
        {
            this.getMutableLongList().remove(element);
        }
        return this;
    }

    @Override
    public SynchronizedLongList withAll(LongIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableLongList().addAll(elements.toArray());
        }
        return this;
    }

    @Override
    public SynchronizedLongList withoutAll(LongIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableLongList().removeAll(elements);
        }
        return this;
    }

    @Override
    public MutableLongList select(LongPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongList().select(predicate);
        }
    }

    @Override
    public MutableLongList reject(LongPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongList().reject(predicate);
        }
    }

    @Override
    public <V> MutableList<V> collect(LongToObjectFunction<? extends V> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongList().collect(function);
        }
    }

    public MutableLongList sortThis()
    {
        synchronized (this.getLock())
        {
            this.getMutableLongList().sortThis();
        }
        return this;
    }

    public long dotProduct(LongList list)
    {
        return this.getMutableLongList().dotProduct(list);
    }

    @Override
    public boolean equals(Object otherList)
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongList().equals(otherList);
        }
    }

    @Override
    public int hashCode()
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongList().hashCode();
        }
    }

    @Override
    public LazyLongIterable asLazy()
    {
        synchronized (this.getLock())
        {
            return new LazyLongIterableAdapter(this);
        }
    }

    @Override
    public MutableLongList asUnmodifiable()
    {
        return new UnmodifiableLongList(this);
    }

    @Override
    public MutableLongList asSynchronized()
    {
        return this;
    }

    @Override
    public ImmutableLongList toImmutable()
    {
        int size = this.size();
        if (size == 0)
        {
            return LongLists.immutable.with();
        }
        if (size == 1)
        {
            return LongLists.immutable.with(this.getFirst());
        }
        return LongLists.immutable.with(this.toArray());
    }

    public MutableLongList reverseThis()
    {
        synchronized (this.getLock())
        {
            this.getMutableLongList().reverseThis();
        }
        return this;
    }

    public MutableLongList toReversed()
    {
        synchronized (this.getLock())
        {
            return this.getMutableLongList().toReversed();
        }
    }
}
