/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.map.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Arrays;
import java.util.BitSet;
import java.util.NoSuchElementException;

import com.gs.collections.api.BooleanIterable;
import com.gs.collections.api.ByteIterable;
import com.gs.collections.api.LazyBooleanIterable;
import com.gs.collections.api.LazyByteIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.bag.primitive.MutableBooleanBag;
import com.gs.collections.api.bag.primitive.MutableByteBag;
import com.gs.collections.api.block.function.primitive.BooleanFunction;
import com.gs.collections.api.block.function.primitive.BooleanFunction0;
import com.gs.collections.api.block.function.primitive.BooleanToBooleanFunction;
import com.gs.collections.api.block.function.primitive.BooleanToObjectFunction;
import com.gs.collections.api.block.function.primitive.ByteToBooleanFunction;
import com.gs.collections.api.block.function.primitive.ByteToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectBooleanToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.BooleanPredicate;
import com.gs.collections.api.block.predicate.primitive.ByteBooleanPredicate;
import com.gs.collections.api.block.predicate.primitive.BytePredicate;
import com.gs.collections.api.block.procedure.primitive.BooleanProcedure;
import com.gs.collections.api.block.procedure.primitive.ByteBooleanProcedure;
import com.gs.collections.api.block.procedure.primitive.ByteProcedure;
import com.gs.collections.api.collection.MutableCollection;
import com.gs.collections.api.collection.primitive.ImmutableBooleanCollection;
import com.gs.collections.api.collection.primitive.MutableBooleanCollection;
import com.gs.collections.api.iterator.BooleanIterator;
import com.gs.collections.api.iterator.ByteIterator;
import com.gs.collections.api.list.primitive.MutableBooleanList;
import com.gs.collections.api.list.primitive.MutableByteList;
import com.gs.collections.api.map.primitive.ByteBooleanMap;
import com.gs.collections.api.map.primitive.ImmutableByteBooleanMap;
import com.gs.collections.api.map.primitive.MutableByteBooleanMap;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.ByteSet;
import com.gs.collections.api.set.primitive.ImmutableByteSet;
import com.gs.collections.api.set.primitive.MutableBooleanSet;
import com.gs.collections.api.set.primitive.MutableByteSet;
import com.gs.collections.impl.bag.mutable.primitive.BooleanHashBag;
import com.gs.collections.impl.bag.mutable.primitive.ByteHashBag;
import com.gs.collections.impl.block.factory.primitive.BytePredicates;
import com.gs.collections.impl.collection.mutable.primitive.SynchronizedBooleanCollection;
import com.gs.collections.impl.collection.mutable.primitive.UnmodifiableBooleanCollection;
import com.gs.collections.impl.factory.Sets;
import com.gs.collections.impl.factory.primitive.BooleanLists;
import com.gs.collections.impl.factory.primitive.ByteBooleanMaps;
import com.gs.collections.impl.factory.primitive.ByteSets;
import com.gs.collections.impl.lazy.primitive.CollectByteToObjectIterable;
import com.gs.collections.impl.lazy.primitive.LazyBooleanIterableAdapter;
import com.gs.collections.impl.lazy.primitive.LazyByteIterableAdapter;
import com.gs.collections.impl.lazy.primitive.SelectByteIterable;
import com.gs.collections.impl.list.mutable.FastList;
import com.gs.collections.impl.list.mutable.primitive.BooleanArrayList;
import com.gs.collections.impl.list.mutable.primitive.ByteArrayList;
import com.gs.collections.impl.set.mutable.primitive.BooleanHashSet;
import com.gs.collections.impl.set.mutable.primitive.ByteHashSet;
import com.gs.collections.impl.set.mutable.primitive.SynchronizedByteSet;
import com.gs.collections.impl.set.mutable.primitive.UnmodifiableByteSet;

/**
 * This file was automatically generated from template file primitiveBooleanHashMap.stg.
 *
 * @since 3.0.
 */
public class ByteBooleanHashMap implements MutableByteBooleanMap, Externalizable
{
    static final boolean EMPTY_VALUE = false;
    private static final long serialVersionUID = 1L;
    private static final byte EMPTY_KEY = (byte) 0;
    private static final byte REMOVED_KEY = (byte) 1;

    private static final float DEFAULT_LOAD_FACTOR = 0.5f;
    private static final int DEFAULT_INITIAL_CAPACITY = 8;
    private int occupied;
    private int maxSize;

    private byte[] keys;
    private BitSet values;
    private float loadFactor = DEFAULT_LOAD_FACTOR;
    private SentinelValues sentinelValues;

    public ByteBooleanHashMap()
    {
        this.allocate(DEFAULT_INITIAL_CAPACITY << 1);
    }

    public ByteBooleanHashMap(int initialCapacity)
    {
        this(initialCapacity, DEFAULT_LOAD_FACTOR);
    }

    public ByteBooleanHashMap(ByteBooleanMap map)
    {
        this(Math.max(map.size(), DEFAULT_INITIAL_CAPACITY), DEFAULT_LOAD_FACTOR);

        this.putAll(map);
    }

    public ByteBooleanHashMap(int initialCapacity, float loadFactor)
    {
        if (initialCapacity < 0)
        {
            throw new IllegalArgumentException("initial capacity cannot be less than 0");
        }
        this.loadFactor = loadFactor;
        this.init(this.fastCeil(initialCapacity / loadFactor));
    }

    public MutableByteBooleanMap asUnmodifiable()
    {
        return new UnmodifiableByteBooleanMap(this);
    }

    public MutableByteBooleanMap asSynchronized()
    {
        return new SynchronizedByteBooleanMap(this);
    }

    public ImmutableByteBooleanMap toImmutable()
    {
        return ByteBooleanMaps.immutable.withAll(this);
    }

    public static ByteBooleanHashMap newWithKeysValues(byte key1, boolean value1)
    {
        return new ByteBooleanHashMap(1).withKeyValue(key1, value1);
    }

    public static ByteBooleanHashMap newWithKeysValues(byte key1, boolean value1, byte key2, boolean value2)
    {
        return new ByteBooleanHashMap(2).withKeysValues(key1, value1, key2, value2);
    }

    public static ByteBooleanHashMap newWithKeysValues(byte key1, boolean value1, byte key2, boolean value2, byte key3, boolean value3)
    {
        return new ByteBooleanHashMap(3).withKeysValues(key1, value1, key2, value2, key3, value3);
    }

    public static ByteBooleanHashMap newWithKeysValues(byte key1, boolean value1, byte key2, boolean value2, byte key3, boolean value3, byte key4, boolean value4)
    {
        return new ByteBooleanHashMap(4).withKeysValues(key1, value1, key2, value2, key3, value3, key4, value4);
    }

    public ByteBooleanHashMap withKeyValue(byte key1, boolean value1)
    {
        this.put(key1, value1);
        return this;
    }

    public ByteBooleanHashMap withKeysValues(byte key1, boolean value1, byte key2, boolean value2)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        return this;
    }

    public ByteBooleanHashMap withKeysValues(byte key1, boolean value1, byte key2, boolean value2, byte key3, boolean value3)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        return this;
    }

    public ByteBooleanHashMap withKeysValues(byte key1, boolean value1, byte key2, boolean value2, byte key3, boolean value3, byte key4, boolean value4)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        this.put(key4, value4);
        return this;
    }

    public ByteBooleanHashMap withoutKey(byte key)
    {
        this.removeKey(key);
        return this;
    }

    public ByteBooleanHashMap withoutAllKeys(ByteIterable keys)
    {
        keys.forEach(new ByteProcedure()
        {
            public void value(byte key)
            {
                ByteBooleanHashMap.this.removeKey(key);
            }
        });
        return this;
    }

    private int init(int initialCapacity)
    {
        int capacity = 1;
        while (capacity < initialCapacity)
        {
            capacity <<= 1;
        }
        return this.allocate(capacity);
    }

    private int fastCeil(float v)
    {
        int possibleResult = (int) v;
        if (v - possibleResult > 0.0F)
        {
            possibleResult++;
        }
        return possibleResult;
    }

    private static boolean isEmptyKey(byte key)
    {
        return key == EMPTY_KEY;
    }

    private static boolean isRemovedKey(byte key)
    {
        return key == REMOVED_KEY;
    }

    private static boolean isNonSentinel(byte key)
    {
        return !isEmptyKey(key) && !isRemovedKey(key);
    }

    private int allocate(int capacity)
    {
        this.allocateTable(capacity);
        this.computeMaxSize(capacity);
        return capacity;
    }

    private void allocateTable(int sizeToAllocate)
    {
        this.keys = new byte[sizeToAllocate];
        this.values = new BitSet(sizeToAllocate);
    }

    private void computeMaxSize(int capacity)
    {
        // need at least one free slot for open addressing
        this.maxSize = Math.min(capacity - 1, (int) (capacity * this.loadFactor));
    }

    public int size()
    {
        return this.occupied + (this.sentinelValues == null ? 0 : this.sentinelValues.size());
    }

    public boolean isEmpty()
    {
        return this.occupied == 0 && (this.sentinelValues == null || this.sentinelValues.size() == 0);
    }

    public boolean notEmpty()
    {
        return this.occupied != 0 || (this.sentinelValues != null && this.sentinelValues.size() != 0);
    }

    private void rehash()
    {
        this.rehash(this.keys.length << 1);
    }

    private void rehash(int newCapacity)
    {
        int oldLength = this.keys.length;
        byte[] old = this.keys;
        BitSet oldValues = this.values;
        this.allocate(newCapacity);
        this.occupied = 0;

        for (int i = 0; i < oldLength; i++)
        {
            if (isNonSentinel(old[i]))
            {
                this.put(old[i], oldValues.get(i));
            }
        }
    }

    // exposed for testing
    int probe(byte element)
    {
        int index = this.spread(element);
        byte keyAtIndex = this.keys[index];

        if (keyAtIndex == element || keyAtIndex == EMPTY_KEY)
        {
            return index;
        }

        int removedIndex = keyAtIndex == REMOVED_KEY ? index : -1;
        int nextIndex = index;
        int probe = 17;

        // loop until an empty slot is reached
        while (true)
        {
            // Probe algorithm: 17*n*(n+1)/2 where n = number of collisions
            nextIndex += probe;
            probe += 17;
            nextIndex &= this.keys.length - 1;

            if (this.keys[nextIndex] == element)
            {
                return nextIndex;
            }
            if (this.keys[nextIndex] == REMOVED_KEY)
            {
                if (removedIndex == -1)
                {
                    removedIndex = nextIndex;
                }
            }
            else if (this.keys[nextIndex] == EMPTY_KEY)
            {
                return removedIndex == -1 ? nextIndex : removedIndex;
            }
        }
    }

    // exposed for testing
    int spread(byte element)
    {
        // No spreading necessary for 8-bit types
        return element & (this.keys.length - 1);
    }

    public void clear()
    {
        this.sentinelValues = null;
        this.occupied = 0;
        Arrays.fill(this.keys, EMPTY_KEY);
        this.values.clear();
    }

    public void put(byte key, boolean value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return;
        }

        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return;
        }

        int index = this.probe(key);

        if (this.keys[index] == key)
        {
            // key already present in map
            this.values.set(index, value);
            return;
        }

        this.keys[index] = key;
        this.values.set(index, value);
        ++this.occupied;
        if (this.occupied > this.maxSize)
        {
            this.rehash();
        }
    }

    public void putAll(ByteBooleanMap map)
    {
        map.forEachKeyValue(new ByteBooleanProcedure()
        {
            public void value(byte key, boolean value)
            {
                ByteBooleanHashMap.this.put(key, value);
            }
        });
    }

    public boolean containsKey(byte key)
    {
        if (isEmptyKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsZeroKey;
        }
        if (isRemovedKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsOneKey;
        }
        return this.keys[this.probe(key)] == key;
    }

    public boolean containsValue(boolean value)
    {
        if (this.sentinelValues != null && this.sentinelValues.containsValue(value))
        {
            return true;
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && this.values.get(i) == value)
            {
                return true;
            }
        }
        return false;
    }

    public boolean contains(boolean value)
    {
        return this.containsValue(value);
    }

    public boolean containsAll(boolean... source)
    {
        for (boolean each : source)
        {
            if (!this.contains(each))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(BooleanIterable source)
    {
        return source.allSatisfy(new BooleanPredicate()
        {
            public boolean accept(boolean value)
            {
                return ByteBooleanHashMap.this.contains(value);
            }
        });
    }

    public boolean get(byte key)
    {
        return this.getIfAbsent(key, EMPTY_VALUE);
    }

    public boolean getIfAbsent(byte key, boolean ifAbsent)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]))
        {
            return this.values.get(index);
        }
        return ifAbsent;
    }

    public boolean getOrThrow(byte key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]))
        {
            return this.values.get(index);
        }
        throw new IllegalStateException("Key " + key + " not present.");
    }

    public boolean getIfAbsentPut(byte key, boolean value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values.get(index);
        }
        this.keys[index] = key;
        this.occupied++;
        this.values.set(index, value);
        return value;
    }

    public boolean getIfAbsentPut(byte key, BooleanFunction0 function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                boolean value = function.value();
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            boolean value = function.value();
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                boolean value = function.value();
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            boolean value = function.value();
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values.get(index);
        }
        this.keys[index] = key;
        this.occupied++;
        boolean value = function.value();
        this.values.set(index, value);
        return value;
    }

    public <P> boolean getIfAbsentPutWith(byte key, BooleanFunction<? super P> function, P parameter)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                boolean value = function.booleanValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            boolean value = function.booleanValueOf(parameter);
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                boolean value = function.booleanValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            boolean value = function.booleanValueOf(parameter);
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values.get(index);
        }
        this.keys[index] = key;
        this.occupied++;
        boolean value = function.booleanValueOf(parameter);
        this.values.set(index, value);
        return value;
    }

    public boolean getIfAbsentPutWithKey(byte key, ByteToBooleanFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                boolean value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            boolean value = function.valueOf(key);
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                boolean value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            boolean value = function.valueOf(key);
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values.get(index);
        }
        this.keys[index] = key;
        this.occupied++;
        boolean value = function.valueOf(key);
        this.values.set(index, value);
        return value;
    }

    public boolean updateValue(byte key, boolean initialValueIfAbsent, BooleanToBooleanFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = function.valueOf(initialValueIfAbsent);
            }
            else if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.zeroValue = function.valueOf(this.sentinelValues.zeroValue);
            }
            else
            {
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = function.valueOf(initialValueIfAbsent);
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = function.valueOf(initialValueIfAbsent);
            }
            else if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.oneValue = function.valueOf(this.sentinelValues.oneValue);
            }
            else
            {
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = function.valueOf(initialValueIfAbsent);
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.values.set(index, function.valueOf(this.values.get(index)));
        }
        else
        {
            this.keys[index] = key;
            this.occupied++;
            this.values.set(index, function.valueOf(initialValueIfAbsent));
        }
        return this.values.get(index);
    }

    public void removeKey(byte key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return;
            }
            if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.containsZeroKey = false;
                this.sentinelValues.zeroValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.containsOneKey = false;
                this.sentinelValues.oneValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.keys[index] = REMOVED_KEY;
            this.occupied--;
            this.values.set(index, EMPTY_VALUE);
        }
    }

    public void remove(byte key)
    {
        this.removeKey(key);
    }

    public boolean removeKeyIfAbsent(byte key, boolean value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return value;
            }
            boolean oldValue = this.sentinelValues.zeroValue;
            if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.containsZeroKey = false;
                this.sentinelValues.zeroValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return oldValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return value;
            }
            boolean oldValue = this.sentinelValues.oneValue;
            if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.containsOneKey = false;
                this.sentinelValues.oneValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return oldValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.keys[index] = REMOVED_KEY;
            this.occupied--;
            boolean oldValue = this.values.get(index);
            this.values.set(index, EMPTY_VALUE);
            return oldValue;
        }
        return value;
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }

        if (!(obj instanceof ByteBooleanMap))
        {
            return false;
        }

        ByteBooleanMap other = (ByteBooleanMap) obj;

        if (this.size() != other.size())
        {
            return false;
        }

        if (this.sentinelValues == null)
        {
            if (other.containsKey(EMPTY_KEY) || other.containsKey(REMOVED_KEY))
            {
                return false;
            }
        }
        else
        {
            if (this.sentinelValues.containsZeroKey && (!other.containsKey(EMPTY_KEY) || this.sentinelValues.zeroValue != other.getOrThrow(EMPTY_KEY)))
            {
                return false;
            }

            if (this.sentinelValues.containsOneKey && (!other.containsKey(REMOVED_KEY) || this.sentinelValues.oneValue != other.getOrThrow(REMOVED_KEY)))
            {
                return false;
            }
        }

        for (int i = 0; i < this.keys.length; i++)
        {
            byte key = this.keys[i];
            if (isNonSentinel(key) && (!other.containsKey(key) || this.values.get(i) != other.getOrThrow(key)))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int result = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += (int) EMPTY_KEY ^ (this.sentinelValues.zeroValue ? 1231 : 1237);
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += (int) REMOVED_KEY ^ (this.sentinelValues.oneValue ? 1231 : 1237);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += (int) this.keys[i] ^ (this.values.get(i) ? 1231 : 1237);
            }
        }

        return result;
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public BooleanIterator booleanIterator()
    {
        return new InternalBooleanIterator();
    }

    public void forEach(BooleanProcedure procedure)
    {
        this.forEachValue(procedure);
    }

    public void forEachValue(BooleanProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.values.get(i));
            }
        }
    }

    public void forEachKey(ByteProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY);
            }
        }
        for (byte key : this.keys)
        {
            if (isNonSentinel(key))
            {
                procedure.value(key);
            }
        }
    }

    public void forEachKeyValue(ByteBooleanProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.keys[i], this.values.get(i));
            }
        }
    }

    public ByteBooleanHashMap select(ByteBooleanPredicate predicate)
    {
        ByteBooleanHashMap result = new ByteBooleanHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.keys[i], this.values.get(i)))
            {
                result.put(this.keys[i], this.values.get(i));
            }
        }

        return result;
    }

    public ByteBooleanHashMap reject(ByteBooleanPredicate predicate)
    {
        ByteBooleanHashMap result = new ByteBooleanHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.keys[i], this.values.get(i)))
            {
                result.put(this.keys[i], this.values.get(i));
            }
        }
        return result;
    }

    public <V> V injectInto(V injectedValue, ObjectBooleanToObjectFunction<? super V, ? extends V> function)
    {
        V result = injectedValue;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result = function.valueOf(result, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                result = function.valueOf(result, this.sentinelValues.oneValue);
            }
        }

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result = function.valueOf(result, this.values.get(i));
            }
        }

        return result;
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            boolean first = true;

            if (this.sentinelValues != null)
            {
                if (this.sentinelValues.containsZeroKey)
                {
                    appendable.append(String.valueOf(EMPTY_KEY)).append("=").append(String.valueOf(this.sentinelValues.zeroValue));
                    first = false;
                }
                if (this.sentinelValues.containsOneKey)
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(REMOVED_KEY)).append("=").append(String.valueOf(this.sentinelValues.oneValue));
                    first = false;
                }
            }
            for (int i = 0; i < this.keys.length; i++)
            {
                byte key = this.keys[i];
                if (isNonSentinel(key))
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(key)).append("=").append(String.valueOf(this.values.get(i)));
                    first = false;
                }
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public MutableBooleanCollection select(BooleanPredicate predicate)
    {
        BooleanArrayList result = new BooleanArrayList();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values.get(i)))
            {
                result.add(this.values.get(i));
            }
        }

        return result;
    }

    public MutableBooleanCollection reject(BooleanPredicate predicate)
    {
        BooleanArrayList result = new BooleanArrayList();
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values.get(i)))
            {
                result.add(this.values.get(i));
            }
        }
        return result;
    }

    public boolean detectIfNone(BooleanPredicate predicate, boolean value)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values.get(i)))
            {
                return this.values.get(i);
            }
        }
        return value;
    }

    public <V> MutableCollection<V> collect(BooleanToObjectFunction<? extends V> function)
    {
        FastList<V> target = FastList.newList(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                target.add(function.valueOf(this.sentinelValues.zeroValue));
            }
            if (this.sentinelValues.containsOneKey)
            {
                target.add(function.valueOf(this.sentinelValues.oneValue));
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                target.add(function.valueOf(this.values.get(i)));
            }
        }
        return target;
    }

    public int count(BooleanPredicate predicate)
    {
        int count = 0;
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                count++;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                count++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values.get(i)))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(BooleanPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return true;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return true;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values.get(i)))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(BooleanPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values.get(i)))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(BooleanPredicate predicate)
    {
        return !this.anySatisfy(predicate);
    }

    public boolean[] toArray()
    {
        boolean[] array = new boolean[this.size()];
        int index = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                array[index] = this.sentinelValues.zeroValue;
                index++;
            }
            if (this.sentinelValues.containsOneKey)
            {
                array[index] = this.sentinelValues.oneValue;
                index++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                array[index] = this.values.get(i);
                index++;
            }
        }

        return array;
    }

    public MutableBooleanList toList()
    {
        return BooleanArrayList.newList(this);
    }

    public MutableBooleanSet toSet()
    {
        return BooleanHashSet.newSet(this);
    }

    public MutableBooleanBag toBag()
    {
        return BooleanHashBag.newBag(this);
    }

    public LazyBooleanIterable asLazy()
    {
        return new LazyBooleanIterableAdapter(this);
    }

    public LazyByteIterable keysView()
    {
        return new KeysView();
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());
        out.writeFloat(this.loadFactor);
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                out.writeByte(EMPTY_KEY);
                out.writeBoolean(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                out.writeByte(REMOVED_KEY);
                out.writeBoolean(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                out.writeByte(this.keys[i]);
                out.writeBoolean(this.values.get(i));
            }
        }
    }

    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
    {
        int size = in.readInt();
        this.loadFactor = in.readFloat();
        this.init(Math.max((int) (size / this.loadFactor) + 1,
                DEFAULT_INITIAL_CAPACITY));
        for (int i = 0; i < size; i++)
        {
            this.put(in.readByte(), in.readBoolean());
        }
    }

    private static final class SentinelValues
    {
        private boolean containsZeroKey;
        private boolean containsOneKey;
        private boolean zeroValue;
        private boolean oneValue;

        public int size()
        {
            return (this.containsZeroKey ? 1 : 0) + (this.containsOneKey ? 1 : 0);
        }

        public boolean containsValue(boolean value)
        {
            boolean valueEqualsZeroValue = this.containsZeroKey && this.zeroValue == value;
            boolean valueEqualsOneValue = this.containsOneKey && this.oneValue == value;
            return valueEqualsZeroValue || valueEqualsOneValue;
        }
    }

    private class InternalBooleanIterator implements BooleanIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < ByteBooleanHashMap.this.size();
        }

        public boolean next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (ByteBooleanHashMap.this.containsKey(EMPTY_KEY))
                {
                    return ByteBooleanHashMap.this.sentinelValues.zeroValue;
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (ByteBooleanHashMap.this.containsKey(REMOVED_KEY))
                {
                    return ByteBooleanHashMap.this.sentinelValues.oneValue;
                }
            }

            byte[] keys = ByteBooleanHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            boolean result = ByteBooleanHashMap.this.values.get(this.position);
            this.position++;
            return result;
        }
    }

    private class KeysView implements LazyByteIterable
    {
        public boolean isEmpty()
        {
            return ByteBooleanHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return ByteBooleanHashMap.this.notEmpty();
        }

        public int size()
        {
            return ByteBooleanHashMap.this.size();
        }

        public boolean contains(byte key)
        {
            return ByteBooleanHashMap.this.containsKey(key);
        }

        public boolean containsAll(byte... keys)
        {
            for (byte key : keys)
            {
                if (!ByteBooleanHashMap.this.containsKey(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(ByteIterable source)
        {
            return source.allSatisfy(new BytePredicate()
            {
                public boolean accept(byte key)
                {
                    return ByteBooleanHashMap.this.containsKey(key);
                }
            });
        }

        public ByteIterator byteIterator()
        {
            return new KeySetIterator();
        }

        public void forEach(ByteProcedure procedure)
        {
            ByteBooleanHashMap.this.forEachKey(procedure);
        }

        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (ByteBooleanHashMap.this.sentinelValues != null)
                {
                    if (ByteBooleanHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (ByteBooleanHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (byte key : ByteBooleanHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public int count(BytePredicate predicate)
        {
            int count = 0;
            if (ByteBooleanHashMap.this.sentinelValues != null)
            {
                if (ByteBooleanHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (ByteBooleanHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (byte key : ByteBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(BytePredicate predicate)
        {
            if (ByteBooleanHashMap.this.sentinelValues != null)
            {
                if (ByteBooleanHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (ByteBooleanHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (byte key : ByteBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(BytePredicate predicate)
        {
            if (ByteBooleanHashMap.this.sentinelValues != null)
            {
                if (ByteBooleanHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (ByteBooleanHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (byte key : ByteBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(BytePredicate predicate)
        {
            return !this.anySatisfy(predicate);
        }

        public LazyByteIterable select(BytePredicate predicate)
        {
            return new SelectByteIterable(this, predicate);
        }

        public LazyByteIterable reject(BytePredicate predicate)
        {
            return new SelectByteIterable(this, BytePredicates.not(predicate));
        }

        public byte detectIfNone(BytePredicate predicate, byte value)
        {
            if (ByteBooleanHashMap.this.sentinelValues != null)
            {
                if (ByteBooleanHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (ByteBooleanHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (byte key : ByteBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return value;
        }

        public <V> LazyIterable<V> collect(ByteToObjectFunction<? extends V> function)
        {
            return new CollectByteToObjectIterable<V>(this, function);
        }

        public long sum()
        {
            long result = 0L;

            if (ByteBooleanHashMap.this.sentinelValues != null)
            {
                if (ByteBooleanHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += EMPTY_KEY;
                }
                if (ByteBooleanHashMap.this.sentinelValues.containsOneKey)
                {
                    result += REMOVED_KEY;
                }
            }
            for (byte key : ByteBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result += key;
                }
            }

            return result;
        }

        public byte max()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            ByteIterator iterator = this.byteIterator();
            byte max = iterator.next();
            while (iterator.hasNext())
            {
                byte value = iterator.next();
                if (max < value)
                {
                    max = value;
                }
            }
            return max;
        }

        public byte min()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            ByteIterator iterator = this.byteIterator();
            byte min = iterator.next();
            while (iterator.hasNext())
            {
                byte value = iterator.next();
                if (value < min)
                {
                    min = value;
                }
            }
            return min;
        }

        public byte minIfEmpty(byte defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.min();
        }

        public byte maxIfEmpty(byte defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.max();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return (double) this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            byte[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                byte first = sortedArray[middleIndex];
                byte second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public byte[] toSortedArray()
        {
            byte[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public byte[] toArray()
        {
            int size = ByteBooleanHashMap.this.size();
            final byte[] result = new byte[size];
            ByteBooleanHashMap.this.forEachKey(new ByteProcedure()
            {
                private int index;

                public void value(byte each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public MutableByteList toList()
        {
            return ByteArrayList.newList(this);
        }

        public MutableByteList toSortedList()
        {
            return ByteArrayList.newList(this).sortThis();
        }

        public MutableByteSet toSet()
        {
            return ByteHashSet.newSet(this);
        }

        public MutableByteBag toBag()
        {
            return ByteHashBag.newBag(this);
        }

        public LazyByteIterable asLazy()
        {
            return this;
        }
    }

    public MutableByteSet keySet()
    {
        return new KeySet();
    }

    private class KeySet implements MutableByteSet
    {
        public ByteIterator byteIterator()
        {
            return new KeySetIterator();
        }

        public void forEach(ByteProcedure procedure)
        {
            ByteBooleanHashMap.this.forEachKey(procedure);
        }

        public int count(BytePredicate predicate)
        {
            int count = 0;
            if (ByteBooleanHashMap.this.sentinelValues != null)
            {
                if (ByteBooleanHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (ByteBooleanHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (byte key : ByteBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(BytePredicate predicate)
        {
            if (ByteBooleanHashMap.this.sentinelValues != null)
            {
                if (ByteBooleanHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (ByteBooleanHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (byte key : ByteBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(BytePredicate predicate)
        {
            if (ByteBooleanHashMap.this.sentinelValues != null)
            {
                if (ByteBooleanHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (ByteBooleanHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (byte key : ByteBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(BytePredicate predicate)
        {
            if (ByteBooleanHashMap.this.sentinelValues != null)
            {
                if (ByteBooleanHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (ByteBooleanHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (byte key : ByteBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean add(byte element)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(byte... source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(ByteIterable source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(byte key)
        {
            int oldSize = ByteBooleanHashMap.this.size();
            ByteBooleanHashMap.this.removeKey(key);
            return oldSize != ByteBooleanHashMap.this.size();
        }

        public boolean removeAll(ByteIterable source)
        {
            int oldSize = ByteBooleanHashMap.this.size();
            ByteIterator iterator = source.byteIterator();
            while (iterator.hasNext())
            {
                ByteBooleanHashMap.this.removeKey(iterator.next());
            }
            return oldSize != ByteBooleanHashMap.this.size();
        }

        public boolean removeAll(byte... source)
        {
            int oldSize = ByteBooleanHashMap.this.size();
            for (byte item : source)
            {
                ByteBooleanHashMap.this.removeKey(item);
            }
            return oldSize != ByteBooleanHashMap.this.size();
        }

        public void clear()
        {
            ByteBooleanHashMap.this.clear();
        }

        public MutableByteSet select(BytePredicate predicate)
        {
            MutableByteSet result = new ByteHashSet();
            if (ByteBooleanHashMap.this.sentinelValues != null)
            {
                if (ByteBooleanHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (ByteBooleanHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (byte key : ByteBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableByteSet reject(BytePredicate predicate)
        {
            MutableByteSet result = new ByteHashSet();
            if (ByteBooleanHashMap.this.sentinelValues != null)
            {
                if (ByteBooleanHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (ByteBooleanHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (byte key : ByteBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableByteSet with(byte element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableByteSet without(byte element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableByteSet withAll(ByteIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableByteSet withoutAll(ByteIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public byte detectIfNone(BytePredicate predicate, byte ifNone)
        {
            if (ByteBooleanHashMap.this.sentinelValues != null)
            {
                if (ByteBooleanHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (ByteBooleanHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (byte key : ByteBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return ifNone;
        }

        public <V> MutableSet<V> collect(ByteToObjectFunction<? extends V> function)
        {
            MutableSet<V> result = Sets.mutable.with();
            if (ByteBooleanHashMap.this.sentinelValues != null)
            {
                if (ByteBooleanHashMap.this.sentinelValues.containsZeroKey)
                {
                    result.add(function.valueOf(EMPTY_KEY));
                }
                if (ByteBooleanHashMap.this.sentinelValues.containsOneKey)
                {
                    result.add(function.valueOf(REMOVED_KEY));
                }
            }
            for (byte key : ByteBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result.add(function.valueOf(key));
                }
            }
            return result;
        }

        public MutableByteSet asUnmodifiable()
        {
            return UnmodifiableByteSet.of(this);
        }

        public MutableByteSet asSynchronized()
        {
            return SynchronizedByteSet.of(this);
        }

        public long sum()
        {
            long sum = 0L;
            if (ByteBooleanHashMap.this.sentinelValues != null)
            {
                if (ByteBooleanHashMap.this.sentinelValues.containsZeroKey)
                {
                    sum += EMPTY_KEY;
                }
                if (ByteBooleanHashMap.this.sentinelValues.containsOneKey)
                {
                    sum += REMOVED_KEY;
                }
            }
            for (byte key : ByteBooleanHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    sum += key;
                }
            }
            return sum;
        }

        public byte max()
        {
            if (ByteBooleanHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            byte max = 0;
            boolean isMaxSet = false;

            if (ByteBooleanHashMap.this.sentinelValues != null)
            {
                if (ByteBooleanHashMap.this.sentinelValues.containsZeroKey)
                {
                    max = EMPTY_KEY;
                    isMaxSet = true;
                }
                if (ByteBooleanHashMap.this.sentinelValues.containsOneKey && (!isMaxSet || max < REMOVED_KEY))
                {
                    max = REMOVED_KEY;
                    isMaxSet = true;
                }
            }
            for (int i = 0; i < ByteBooleanHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ByteBooleanHashMap.this.keys[i]) && (!isMaxSet || max < ByteBooleanHashMap.this.keys[i]))
                {
                    max = ByteBooleanHashMap.this.keys[i];
                    isMaxSet = true;
                }
            }
            return max;
        }

        public byte maxIfEmpty(byte defaultValue)
        {
            if (ByteBooleanHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.max();
        }

        public byte min()
        {
            if (ByteBooleanHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            byte min = 0;
            boolean isMinSet = false;

            if (ByteBooleanHashMap.this.sentinelValues != null)
            {
                if (ByteBooleanHashMap.this.sentinelValues.containsZeroKey)
                {
                    min = EMPTY_KEY;
                    isMinSet = true;
                }
                if (ByteBooleanHashMap.this.sentinelValues.containsOneKey && (!isMinSet || REMOVED_KEY < min))
                {
                    min = REMOVED_KEY;
                    isMinSet = true;
                }
            }
            for (int i = 0; i < ByteBooleanHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ByteBooleanHashMap.this.keys[i]) && (!isMinSet || ByteBooleanHashMap.this.keys[i] < min))
                {
                    min = ByteBooleanHashMap.this.keys[i];
                    isMinSet = true;
                }
            }
            return min;
        }

        public byte minIfEmpty(byte defaultValue)
        {
            if (ByteBooleanHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.min();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return (double) this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            byte[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                byte first = sortedArray[middleIndex];
                byte second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public byte[] toSortedArray()
        {
            byte[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public MutableByteList toSortedList()
        {
            return ByteArrayList.newList(this).sortThis();
        }

        public byte[] toArray()
        {
            int size = ByteBooleanHashMap.this.size();
            final byte[] result = new byte[size];
            ByteBooleanHashMap.this.forEachKey(new ByteProcedure()
            {
                private int index;

                public void value(byte each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public boolean contains(byte value)
        {
            return ByteBooleanHashMap.this.containsKey(value);
        }

        public boolean containsAll(byte... source)
        {
            for (byte item : source)
            {
                if (!ByteBooleanHashMap.this.containsKey(item))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(ByteIterable source)
        {
            ByteIterator iterator = source.byteIterator();
            while (iterator.hasNext())
            {
                if (!ByteBooleanHashMap.this.containsKey(iterator.next()))
                {
                    return false;
                }
            }
            return true;
        }

        public MutableByteList toList()
        {
            return ByteArrayList.newList(this);
        }

        public MutableByteSet toSet()
        {
            return ByteHashSet.newSet(this);
        }

        public MutableByteBag toBag()
        {
            return ByteHashBag.newBag(this);
        }

        public LazyByteIterable asLazy()
        {
            return new LazyByteIterableAdapter(this);
        }

        public ByteSet freeze()
        {
            throw new UnsupportedOperationException();
        }

        public ImmutableByteSet toImmutable()
        {
            return ByteSets.immutable.withAll(this);
        }

        public int size()
        {
            return ByteBooleanHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return ByteBooleanHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return ByteBooleanHashMap.this.notEmpty();
        }

        @Override
        public boolean equals(Object obj)
        {
            if (this == obj)
            {
                return true;
            }

            if (!(obj instanceof ByteSet))
            {
                return false;
            }

            ByteSet other = (ByteSet) obj;
            return this.size() == other.size() && this.containsAll(other.toArray());
        }

        @Override
        public int hashCode()
        {
            int result = 0;

            if (ByteBooleanHashMap.this.sentinelValues != null)
            {
                if (ByteBooleanHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += (int) EMPTY_KEY;
                }
                if (ByteBooleanHashMap.this.sentinelValues.containsOneKey)
                {
                    result += (int) REMOVED_KEY;
                }
            }
            for (int i = 0; i < ByteBooleanHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ByteBooleanHashMap.this.keys[i]))
                {
                    result += (int) ByteBooleanHashMap.this.keys[i];
                }
            }

            return result;
        }

        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (ByteBooleanHashMap.this.sentinelValues != null)
                {
                    if (ByteBooleanHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (ByteBooleanHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (byte key : ByteBooleanHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }
    }

    public MutableBooleanCollection values()
    {
        return new ValuesCollection();
    }

    private class ValuesCollection implements MutableBooleanCollection
    {
        public void clear()
        {
            ByteBooleanHashMap.this.clear();
        }

        public MutableBooleanCollection select(BooleanPredicate predicate)
        {
            return ByteBooleanHashMap.this.select(predicate);
        }

        public MutableBooleanCollection reject(BooleanPredicate predicate)
        {
            return ByteBooleanHashMap.this.reject(predicate);
        }

        public boolean detectIfNone(BooleanPredicate predicate, boolean ifNone)
        {
            return ByteBooleanHashMap.this.detectIfNone(predicate, ifNone);
        }

        public <V> MutableCollection<V> collect(BooleanToObjectFunction<? extends V> function)
        {
            return ByteBooleanHashMap.this.collect(function);
        }

        public MutableBooleanCollection with(boolean element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableBooleanCollection without(boolean element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableBooleanCollection withAll(BooleanIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableBooleanCollection withoutAll(BooleanIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableBooleanCollection asSynchronized()
        {
            return SynchronizedBooleanCollection.of(this);
        }

        public MutableBooleanCollection asUnmodifiable()
        {
            return UnmodifiableBooleanCollection.of(this);
        }

        public ImmutableBooleanCollection toImmutable()
        {
            return BooleanLists.immutable.withAll(this);
        }

        public boolean contains(boolean value)
        {
            return ByteBooleanHashMap.this.containsValue(value);
        }

        public boolean containsAll(boolean... source)
        {
            return ByteBooleanHashMap.this.containsAll(source);
        }

        public boolean containsAll(BooleanIterable source)
        {
            return ByteBooleanHashMap.this.containsAll(source);
        }

        public MutableBooleanList toList()
        {
            return ByteBooleanHashMap.this.toList();
        }

        public MutableBooleanSet toSet()
        {
            return ByteBooleanHashMap.this.toSet();
        }

        public MutableBooleanBag toBag()
        {
            return ByteBooleanHashMap.this.toBag();
        }

        public LazyBooleanIterable asLazy()
        {
            return new LazyBooleanIterableAdapter(this);
        }

        public boolean isEmpty()
        {
            return ByteBooleanHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return ByteBooleanHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);

                boolean first = true;

                if (ByteBooleanHashMap.this.sentinelValues != null)
                {
                    if (ByteBooleanHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(ByteBooleanHashMap.this.sentinelValues.zeroValue));
                        first = false;
                    }
                    if (ByteBooleanHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(ByteBooleanHashMap.this.sentinelValues.oneValue));
                        first = false;
                    }
                }
                for (int i = 0; i < ByteBooleanHashMap.this.keys.length; i++)
                {
                    byte key = ByteBooleanHashMap.this.keys[i];
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(ByteBooleanHashMap.this.values.get(i)));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public BooleanIterator booleanIterator()
        {
            return ByteBooleanHashMap.this.booleanIterator();
        }

        public void forEach(BooleanProcedure procedure)
        {
            ByteBooleanHashMap.this.forEach(procedure);
        }

        public int count(BooleanPredicate predicate)
        {
            return ByteBooleanHashMap.this.count(predicate);
        }

        public boolean anySatisfy(BooleanPredicate predicate)
        {
            return ByteBooleanHashMap.this.anySatisfy(predicate);
        }

        public boolean allSatisfy(BooleanPredicate predicate)
        {
            return ByteBooleanHashMap.this.allSatisfy(predicate);
        }

        public boolean noneSatisfy(BooleanPredicate predicate)
        {
            return ByteBooleanHashMap.this.noneSatisfy(predicate);
        }

        public boolean add(boolean element)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(boolean... source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(BooleanIterable source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(boolean item)
        {
            int oldSize = ByteBooleanHashMap.this.size();

            if (ByteBooleanHashMap.this.sentinelValues != null)
            {
                if (ByteBooleanHashMap.this.sentinelValues.containsZeroKey && item == ByteBooleanHashMap.this.sentinelValues.zeroValue)
                {
                    ByteBooleanHashMap.this.removeKey(EMPTY_KEY);
                }
                if (ByteBooleanHashMap.this.sentinelValues.containsOneKey && item == ByteBooleanHashMap.this.sentinelValues.oneValue)
                {
                    ByteBooleanHashMap.this.removeKey(REMOVED_KEY);
                }
            }
            for (int i = 0; i < ByteBooleanHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ByteBooleanHashMap.this.keys[i]) && item == ByteBooleanHashMap.this.values.get(i))
                {
                    ByteBooleanHashMap.this.removeKey(ByteBooleanHashMap.this.keys[i]);
                }
            }
            return oldSize != ByteBooleanHashMap.this.size();
        }

        public boolean removeAll(BooleanIterable source)
        {
            int oldSize = ByteBooleanHashMap.this.size();

            BooleanIterator iterator = source.booleanIterator();
            while (iterator.hasNext())
            {
                this.remove(iterator.next());
            }
            return oldSize != ByteBooleanHashMap.this.size();
        }

        public boolean removeAll(boolean... source)
        {
            int oldSize = ByteBooleanHashMap.this.size();

            for (boolean item : source)
            {
                this.remove(item);
            }
            return oldSize != ByteBooleanHashMap.this.size();
        }

        public int size()
        {
            return ByteBooleanHashMap.this.size();
        }

        public boolean[] toArray()
        {
            return ByteBooleanHashMap.this.toArray();
        }
    }

    private class KeySetIterator implements ByteIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < ByteBooleanHashMap.this.size();
        }

        public byte next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (ByteBooleanHashMap.this.containsKey(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (ByteBooleanHashMap.this.containsKey(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }

            byte[] keys = ByteBooleanHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            byte result = keys[this.position];
            this.position++;
            return result;
        }
    }
}
