/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.map.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.ByteIterable;
import com.gs.collections.api.LazyByteIterable;
import com.gs.collections.api.LazyFloatIterable;
import com.gs.collections.api.FloatIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.bag.primitive.MutableByteBag;
import com.gs.collections.api.bag.primitive.MutableFloatBag;
import com.gs.collections.api.block.function.primitive.ByteToObjectFunction;
import com.gs.collections.api.block.function.primitive.ByteToFloatFunction;
import com.gs.collections.api.block.function.primitive.FloatFunction;
import com.gs.collections.api.block.function.primitive.FloatFunction0;
import com.gs.collections.api.block.function.primitive.FloatToFloatFunction;
import com.gs.collections.api.block.function.primitive.FloatToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectFloatToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.ByteFloatPredicate;
import com.gs.collections.api.block.predicate.primitive.BytePredicate;
import com.gs.collections.api.block.predicate.primitive.FloatPredicate;
import com.gs.collections.api.block.procedure.primitive.ByteProcedure;
import com.gs.collections.api.block.procedure.primitive.ByteFloatProcedure;
import com.gs.collections.api.block.procedure.primitive.FloatProcedure;
import com.gs.collections.api.collection.MutableCollection;
import com.gs.collections.api.collection.primitive.ImmutableFloatCollection;
import com.gs.collections.api.collection.primitive.MutableFloatCollection;
import com.gs.collections.api.iterator.ByteIterator;
import com.gs.collections.api.iterator.FloatIterator;
import com.gs.collections.api.list.primitive.MutableByteList;
import com.gs.collections.api.list.primitive.MutableFloatList;
import com.gs.collections.api.map.primitive.ByteFloatMap;
import com.gs.collections.api.map.primitive.ImmutableByteFloatMap;
import com.gs.collections.api.map.primitive.MutableByteFloatMap;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.ByteSet;
import com.gs.collections.api.set.primitive.ImmutableByteSet;
import com.gs.collections.api.set.primitive.MutableByteSet;
import com.gs.collections.api.set.primitive.MutableFloatSet;
import com.gs.collections.impl.bag.mutable.primitive.ByteHashBag;
import com.gs.collections.impl.bag.mutable.primitive.FloatHashBag;
import com.gs.collections.impl.block.factory.primitive.BytePredicates;
import com.gs.collections.impl.collection.mutable.primitive.SynchronizedFloatCollection;
import com.gs.collections.impl.collection.mutable.primitive.UnmodifiableFloatCollection;
import com.gs.collections.impl.factory.Sets;
import com.gs.collections.impl.factory.primitive.FloatLists;
import com.gs.collections.impl.factory.primitive.ByteFloatMaps;
import com.gs.collections.impl.factory.primitive.ByteSets;
import com.gs.collections.impl.lazy.primitive.CollectByteToObjectIterable;
import com.gs.collections.impl.lazy.primitive.LazyFloatIterableAdapter;
import com.gs.collections.impl.lazy.primitive.LazyByteIterableAdapter;
import com.gs.collections.impl.lazy.primitive.SelectByteIterable;
import com.gs.collections.impl.list.mutable.FastList;
import com.gs.collections.impl.list.mutable.primitive.ByteArrayList;
import com.gs.collections.impl.list.mutable.primitive.FloatArrayList;
import com.gs.collections.impl.set.mutable.primitive.ByteHashSet;
import com.gs.collections.impl.set.mutable.primitive.SynchronizedByteSet;
import com.gs.collections.impl.set.mutable.primitive.UnmodifiableByteSet;
import com.gs.collections.impl.set.mutable.primitive.FloatHashSet;

/**
 * This file was automatically generated from template file primitivePrimitiveHashMap.stg.
 *
 * @since 3.0.
 */
public class ByteFloatHashMap implements MutableByteFloatMap, Externalizable
{
    static final float EMPTY_VALUE = 0.0f;
    private static final long serialVersionUID = 1L;
    private static final byte EMPTY_KEY = (byte) 0;
    private static final byte REMOVED_KEY = (byte) 1;

    private static final int OCCUPIED_DATA_RATIO = 2;
    private static final int OCCUPIED_SENTINEL_RATIO = 4;
    private static final int DEFAULT_INITIAL_CAPACITY = 8;

    private byte[] keys;
    private float[] values;

    private int occupiedWithData;
    private int occupiedWithSentinels;

    private SentinelValues sentinelValues;

    public ByteFloatHashMap()
    {
        this.allocateTable(DEFAULT_INITIAL_CAPACITY << 1);
    }

    public ByteFloatHashMap(int initialCapacity)
    {
        if (initialCapacity < 0)
        {
            throw new IllegalArgumentException("initial capacity cannot be less than 0");
        }
        int capacity = this.smallestPowerOfTwoGreaterThan(this.fastCeil(initialCapacity * OCCUPIED_DATA_RATIO));
        this.allocateTable(capacity);
    }

    private int smallestPowerOfTwoGreaterThan(int n)
    {
        return n > 1 ? Integer.highestOneBit(n - 1) << 1 : 1;
    }

    public ByteFloatHashMap(ByteFloatMap map)
    {
        this(Math.max(map.size(), DEFAULT_INITIAL_CAPACITY));
        this.putAll(map);
    }

    private int fastCeil(float v)
    {
        int possibleResult = (int) v;
        if (v - possibleResult > 0.0F)
        {
            possibleResult++;
        }
        return possibleResult;
    }

    public static ByteFloatHashMap newWithKeysValues(byte key1, float value1)
    {
        return new ByteFloatHashMap(1).withKeyValue(key1, value1);
    }

    public static ByteFloatHashMap newWithKeysValues(byte key1, float value1, byte key2, float value2)
    {
        return new ByteFloatHashMap(2).withKeysValues(key1, value1, key2, value2);
    }

    public static ByteFloatHashMap newWithKeysValues(byte key1, float value1, byte key2, float value2, byte key3, float value3)
    {
        return new ByteFloatHashMap(3).withKeysValues(key1, value1, key2, value2, key3, value3);
    }

    public static ByteFloatHashMap newWithKeysValues(byte key1, float value1, byte key2, float value2, byte key3, float value3, byte key4, float value4)
    {
        return new ByteFloatHashMap(4).withKeysValues(key1, value1, key2, value2, key3, value3, key4, value4);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }

        if (!(obj instanceof ByteFloatMap))
        {
            return false;
        }

        ByteFloatMap other = (ByteFloatMap) obj;

        if (this.size() != other.size())
        {
            return false;
        }

        if (this.sentinelValues == null)
        {
            if (other.containsKey(EMPTY_KEY) || other.containsKey(REMOVED_KEY))
            {
                return false;
            }
        }
        else
        {
            if (this.sentinelValues.containsZeroKey && (!other.containsKey(EMPTY_KEY) || Float.compare(this.sentinelValues.zeroValue, other.getOrThrow(EMPTY_KEY)) != 0))
            {
                return false;
            }

            if (this.sentinelValues.containsOneKey && (!other.containsKey(REMOVED_KEY) || Float.compare(this.sentinelValues.oneValue, other.getOrThrow(REMOVED_KEY)) != 0))
            {
                return false;
            }
        }

        for (int i = 0; i < this.keys.length; i++)
        {
            byte key = this.keys[i];
            if (isNonSentinel(key) && (!other.containsKey(key) || Float.compare(this.values[i], other.getOrThrow(key)) != 0))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int result = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += (int) EMPTY_KEY ^ Float.floatToIntBits(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += (int) REMOVED_KEY ^ Float.floatToIntBits(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += (int) this.keys[i] ^ Float.floatToIntBits(this.values[i]);
            }
        }

        return result;
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public int size()
    {
        return this.occupiedWithData + (this.sentinelValues == null ? 0 : this.sentinelValues.size());
    }

    public boolean isEmpty()
    {
        return this.occupiedWithData == 0 && (this.sentinelValues == null || this.sentinelValues.size() == 0);
    }

    public boolean notEmpty()
    {
        return this.occupiedWithData != 0 || (this.sentinelValues != null && this.sentinelValues.size() != 0);
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            boolean first = true;

            if (this.sentinelValues != null)
            {
                if (this.sentinelValues.containsZeroKey)
                {
                    appendable.append(String.valueOf(EMPTY_KEY)).append("=").append(String.valueOf(this.sentinelValues.zeroValue));
                    first = false;
                }
                if (this.sentinelValues.containsOneKey)
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(REMOVED_KEY)).append("=").append(String.valueOf(this.sentinelValues.oneValue));
                    first = false;
                }
            }
            for (int i = 0; i < this.keys.length; i++)
            {
                byte key = this.keys[i];
                if (isNonSentinel(key))
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(key)).append("=").append(String.valueOf(this.values[i]));
                    first = false;
                }
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public FloatIterator floatIterator()
    {
        return new InternalFloatIterator();
    }

    public float[] toArray()
    {
        float[] array = new float[this.size()];
        int index = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                array[index] = this.sentinelValues.zeroValue;
                index++;
            }
            if (this.sentinelValues.containsOneKey)
            {
                array[index] = this.sentinelValues.oneValue;
                index++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                array[index] = this.values[i];
                index++;
            }
        }

        return array;
    }

    public boolean contains(float value)
    {
        return this.containsValue(value);
    }

    public boolean containsAll(float... source)
    {
        for (float each : source)
        {
            if (!this.contains(each))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(FloatIterable source)
    {
        return source.allSatisfy(new FloatPredicate()
        {
            public boolean accept(float value)
            {
                return ByteFloatHashMap.this.contains(value);
            }
        });
    }

    public void forEach(FloatProcedure procedure)
    {
        this.forEachValue(procedure);
    }

    public MutableFloatCollection select(FloatPredicate predicate)
    {
        FloatArrayList result = new FloatArrayList();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }

        return result;
    }

    public MutableFloatCollection reject(FloatPredicate predicate)
    {
        FloatArrayList result = new FloatArrayList();
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public <V> MutableCollection<V> collect(FloatToObjectFunction<? extends V> function)
    {
        FastList<V> target = FastList.newList(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                target.add(function.valueOf(this.sentinelValues.zeroValue));
            }
            if (this.sentinelValues.containsOneKey)
            {
                target.add(function.valueOf(this.sentinelValues.oneValue));
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                target.add(function.valueOf(this.values[i]));
            }
        }
        return target;
    }

    public float detectIfNone(FloatPredicate predicate, float value)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return this.values[i];
            }
        }
        return value;
    }

    public int count(FloatPredicate predicate)
    {
        int count = 0;
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                count++;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                count++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(FloatPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return true;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return true;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(FloatPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(FloatPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public <V> V injectInto(V injectedValue, ObjectFloatToObjectFunction<? super V, ? extends V> function)
    {
        V result = injectedValue;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result = function.valueOf(result, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                result = function.valueOf(result, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result = function.valueOf(result, this.values[i]);
            }
        }

        return result;
    }

    public MutableFloatList toList()
    {
        return FloatArrayList.newList(this);
    }

    public MutableFloatSet toSet()
    {
        return FloatHashSet.newSet(this);
    }

    public MutableFloatBag toBag()
    {
        return FloatHashBag.newBag(this);
    }

    public LazyFloatIterable asLazy()
    {
        return new LazyFloatIterableAdapter(this);
    }

    public void clear()
    {
        this.sentinelValues = null;
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;
        Arrays.fill(this.keys, EMPTY_KEY);
        Arrays.fill(this.values, EMPTY_VALUE);
    }

    public void put(byte key, float value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return;
        }

        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return;
        }

        int index = this.probe(key);

        if (this.keys[index] == key)
        {
            // key already present in map
            this.values[index] = value;
            return;
        }

        if (this.keys[index] == REMOVED_KEY)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = key;
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
    }

    public void putAll(ByteFloatMap map)
    {
        map.forEachKeyValue(new ByteFloatProcedure()
        {
            public void value(byte key, float value)
            {
                ByteFloatHashMap.this.put(key, value);
            }
        });
    }

    public void removeKey(byte key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return;
            }
            if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.containsZeroKey = false;
                this.sentinelValues.zeroValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.containsOneKey = false;
                this.sentinelValues.oneValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.keys[index] = REMOVED_KEY;
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }
        }
    }

    public void remove(byte key)
    {
        this.removeKey(key);
    }

    public float removeKeyIfAbsent(byte key, float value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return value;
            }
            float oldValue = this.sentinelValues.zeroValue;
            if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.containsZeroKey = false;
                this.sentinelValues.zeroValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return oldValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return value;
            }
            float oldValue = this.sentinelValues.oneValue;
            if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.containsOneKey = false;
                this.sentinelValues.oneValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return oldValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.keys[index] = REMOVED_KEY;
            float oldValue = this.values[index];
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }

            return oldValue;
        }
        return value;
    }

    public float getIfAbsentPut(byte key, float value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public float getIfAbsentPut(byte key, FloatFunction0 function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                float value = function.value();
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            float value = function.value();
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                float value = function.value();
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            float value = function.value();
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        float value = function.value();
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public <P> float getIfAbsentPutWith(byte key, FloatFunction<? super P> function, P parameter)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                float value = function.floatValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            float value = function.floatValueOf(parameter);
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                float value = function.floatValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            float value = function.floatValueOf(parameter);
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        float value = function.floatValueOf(parameter);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public float getIfAbsentPutWithKey(byte key, ByteToFloatFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                float value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            float value = function.valueOf(key);
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                float value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            float value = function.valueOf(key);
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        float value = function.valueOf(key);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public float updateValue(byte key, float initialValueIfAbsent, FloatToFloatFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = function.valueOf(initialValueIfAbsent);
            }
            else if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.zeroValue = function.valueOf(this.sentinelValues.zeroValue);
            }
            else
            {
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = function.valueOf(initialValueIfAbsent);
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = function.valueOf(initialValueIfAbsent);
            }
            else if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.oneValue = function.valueOf(this.sentinelValues.oneValue);
            }
            else
            {
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = function.valueOf(initialValueIfAbsent);
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.values[index] = function.valueOf(this.values[index]);
            return this.values[index];
        }
        if (this.keys[index] == REMOVED_KEY)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = key;
        float value = function.valueOf(initialValueIfAbsent);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public ByteFloatHashMap withKeyValue(byte key1, float value1)
    {
        this.put(key1, value1);
        return this;
    }

    public ByteFloatHashMap withKeysValues(byte key1, float value1, byte key2, float value2)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        return this;
    }

    public ByteFloatHashMap withKeysValues(byte key1, float value1, byte key2, float value2, byte key3, float value3)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        return this;
    }

    public ByteFloatHashMap withKeysValues(byte key1, float value1, byte key2, float value2, byte key3, float value3, byte key4, float value4)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        this.put(key4, value4);
        return this;
    }

    public ByteFloatHashMap withoutKey(byte key)
    {
        this.removeKey(key);
        return this;
    }

    public ByteFloatHashMap withoutAllKeys(ByteIterable keys)
    {
        keys.forEach(new ByteProcedure()
        {
            public void value(byte key)
            {
                ByteFloatHashMap.this.removeKey(key);
            }
        });
        return this;
    }

    public MutableByteFloatMap asUnmodifiable()
    {
        return new UnmodifiableByteFloatMap(this);
    }

    public MutableByteFloatMap asSynchronized()
    {
        return new SynchronizedByteFloatMap(this);
    }

    public ImmutableByteFloatMap toImmutable()
    {
        return ByteFloatMaps.immutable.ofAll(this);
    }

    public float get(byte key)
    {
        return this.getIfAbsent(key, EMPTY_VALUE);
    }

    public float getIfAbsent(byte key, float ifAbsent)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        return ifAbsent;
    }

    public float getOrThrow(byte key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]))
        {
            return this.values[index];
        }
        throw new IllegalStateException("Key " + key + " not present.");
    }

    public boolean containsKey(byte key)
    {
        if (isEmptyKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsZeroKey;
        }
        if (isRemovedKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsOneKey;
        }
        return this.keys[this.probe(key)] == key;
    }

    public boolean containsValue(float value)
    {
        if (this.sentinelValues != null && this.sentinelValues.containsValue(value))
        {
            return true;
        }
        for (int i = 0; i < this.values.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && Float.compare(this.values[i], value) == 0)
            {
                return true;
            }
        }
        return false;
    }

    public void forEachValue(FloatProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.values[i]);
            }
        }
    }

    public void forEachKey(ByteProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY);
            }
        }
        for (byte key : this.keys)
        {
            if (isNonSentinel(key))
            {
                procedure.value(key);
            }
        }
    }

    public void forEachKeyValue(ByteFloatProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.keys[i], this.values[i]);
            }
        }
    }

    public LazyByteIterable keysView()
    {
        return new KeysView();
    }

    public ByteFloatHashMap select(ByteFloatPredicate predicate)
    {
        ByteFloatHashMap result = new ByteFloatHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.keys[i], this.values[i]))
            {
                result.put(this.keys[i], this.values[i]);
            }
        }

        return result;
    }

    public ByteFloatHashMap reject(ByteFloatPredicate predicate)
    {
        ByteFloatHashMap result = new ByteFloatHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.keys[i], this.values[i]))
            {
                result.put(this.keys[i], this.values[i]);
            }
        }
        return result;
    }

    public double sum()
    {
        double result = 0.0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += this.values[i];
            }
        }

        return result;
    }

    public float max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        FloatIterator iterator = this.floatIterator();
        float max = iterator.next();
        while (iterator.hasNext())
        {
            float value = iterator.next();
            if (Float.compare(max, value) < 0)
            {
                max = value;
            }
        }
        return max;
    }

    public float maxIfEmpty(float defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public float min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        FloatIterator iterator = this.floatIterator();
        float min = iterator.next();
        while (iterator.hasNext())
        {
            float value = iterator.next();
            if (Float.compare(value, min) < 0)
            {
                min = value;
            }
        }
        return min;
    }

    public float minIfEmpty(float defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        float[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            float first = sortedArray[middleIndex];
            float second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public float[] toSortedArray()
    {
        float[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public MutableFloatList toSortedList()
    {
        return FloatArrayList.newList(this).sortThis();
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                out.writeByte(EMPTY_KEY);
                out.writeFloat(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                out.writeByte(REMOVED_KEY);
                out.writeFloat(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                out.writeByte(this.keys[i]);
                out.writeFloat(this.values[i]);
            }
        }
    }

    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
    {
        int size = in.readInt();
        for (int i = 0; i < size; i++)
        {
            this.put(in.readByte(), in.readFloat());
        }
    }

    /**
     * Rehashes every element in the set into a new backing table of the smallest possible size and eliminating removed sentinels.
     */
    public void compact()
    {
        this.rehash(this.smallestPowerOfTwoGreaterThan(this.size()));
    }

    private void rehash()
    {
        this.rehash(this.keys.length);
    }

    private void rehashAndGrow()
    {
        this.rehash(this.keys.length << 1);
    }

    private void rehash(int newCapacity)
    {
        int oldLength = this.keys.length;
        byte[] old = this.keys;
        float[] oldValues = this.values;
        this.allocateTable(newCapacity);
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;

        for (int i = 0; i < oldLength; i++)
        {
            if (isNonSentinel(old[i]))
            {
                this.put(old[i], oldValues[i]);
            }
        }
    }

    // exposed for testing
    int probe(byte element)
    {
        int index = this.spread(element);
        byte keyAtIndex = this.keys[index];

        if (keyAtIndex == element || keyAtIndex == EMPTY_KEY)
        {
            return index;
        }

        int removedIndex = keyAtIndex == REMOVED_KEY ? index : -1;
        int nextIndex = index;
        int probe = 17;

        // loop until an empty slot is reached
        while (true)
        {
            // Probe algorithm: 17*n*(n+1)/2 where n = number of collisions
            nextIndex += probe;
            probe += 17;
            nextIndex &= this.keys.length - 1;

            if (this.keys[nextIndex] == element)
            {
                return nextIndex;
            }
            if (this.keys[nextIndex] == REMOVED_KEY)
            {
                if (removedIndex == -1)
                {
                    removedIndex = nextIndex;
                }
            }
            else if (this.keys[nextIndex] == EMPTY_KEY)
            {
                return removedIndex == -1 ? nextIndex : removedIndex;
            }
        }
    }

    // exposed for testing
    int spread(byte element)
    {
        // No spreading necessary for 8-bit types
        return element & (this.keys.length - 1);
    }

    private void allocateTable(int sizeToAllocate)
    {
        this.keys = new byte[sizeToAllocate];
        this.values = new float[sizeToAllocate];
    }

    private static boolean isEmptyKey(byte key)
    {
        return key == EMPTY_KEY;
    }

    private static boolean isRemovedKey(byte key)
    {
        return key == REMOVED_KEY;
    }

    private static boolean isNonSentinel(byte key)
    {
        return !isEmptyKey(key) && !isRemovedKey(key);
    }

    private int maxOccupiedWithData()
    {
        int capacity = this.keys.length;
        // need at least one free slot for open addressing
        return Math.min(capacity - 1, capacity / OCCUPIED_DATA_RATIO);
    }

    private int maxOccupiedWithSentinels()
    {
        return this.keys.length / OCCUPIED_SENTINEL_RATIO;
    }

    private static final class SentinelValues
    {
        private boolean containsZeroKey;
        private boolean containsOneKey;
        private float zeroValue;
        private float oneValue;

        public int size()
        {
            return (this.containsZeroKey ? 1 : 0) + (this.containsOneKey ? 1 : 0);
        }

        public boolean containsValue(float value)
        {
            boolean valueEqualsZeroValue = this.containsZeroKey && Float.compare(this.zeroValue, value) == 0;
            boolean valueEqualsOneValue = this.containsOneKey && Float.compare(this.oneValue, value) == 0;
            return valueEqualsZeroValue || valueEqualsOneValue;
        }
    }

    private class InternalFloatIterator implements FloatIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < ByteFloatHashMap.this.size();
        }

        public float next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (ByteFloatHashMap.this.containsKey(EMPTY_KEY))
                {
                    return ByteFloatHashMap.this.get(EMPTY_KEY);
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (ByteFloatHashMap.this.containsKey(REMOVED_KEY))
                {
                    return ByteFloatHashMap.this.get(REMOVED_KEY);
                }
            }

            byte[] keys = ByteFloatHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            float result = ByteFloatHashMap.this.values[this.position];
            this.position++;
            return result;
        }
    }

    private class KeysView implements LazyByteIterable
    {
        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public int size()
        {
            return ByteFloatHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return ByteFloatHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return ByteFloatHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (ByteFloatHashMap.this.sentinelValues != null)
                {
                    if (ByteFloatHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (ByteFloatHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (byte key : ByteFloatHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public ByteIterator byteIterator()
        {
            return new KeySetIterator();
        }

        public boolean contains(byte key)
        {
            return ByteFloatHashMap.this.containsKey(key);
        }

        public boolean containsAll(byte... keys)
        {
            for (byte key : keys)
            {
                if (!ByteFloatHashMap.this.containsKey(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(ByteIterable source)
        {
            return source.allSatisfy(new BytePredicate()
            {
                public boolean accept(byte key)
                {
                    return ByteFloatHashMap.this.containsKey(key);
                }
            });
        }

        public void forEach(ByteProcedure procedure)
        {
            ByteFloatHashMap.this.forEachKey(procedure);
        }

        public int count(BytePredicate predicate)
        {
            int count = 0;
            if (ByteFloatHashMap.this.sentinelValues != null)
            {
                if (ByteFloatHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (ByteFloatHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (byte key : ByteFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(BytePredicate predicate)
        {
            if (ByteFloatHashMap.this.sentinelValues != null)
            {
                if (ByteFloatHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (ByteFloatHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (byte key : ByteFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(BytePredicate predicate)
        {
            if (ByteFloatHashMap.this.sentinelValues != null)
            {
                if (ByteFloatHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (ByteFloatHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (byte key : ByteFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(BytePredicate predicate)
        {
            if (ByteFloatHashMap.this.sentinelValues != null)
            {
                if (ByteFloatHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (ByteFloatHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (byte key : ByteFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public LazyByteIterable select(BytePredicate predicate)
        {
            return new SelectByteIterable(this, predicate);
        }

        public LazyByteIterable reject(BytePredicate predicate)
        {
            return new SelectByteIterable(this, BytePredicates.not(predicate));
        }

        public byte detectIfNone(BytePredicate predicate, byte value)
        {
            if (ByteFloatHashMap.this.sentinelValues != null)
            {
                if (ByteFloatHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (ByteFloatHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (byte key : ByteFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return value;
        }

        public <V> LazyIterable<V> collect(ByteToObjectFunction<? extends V> function)
        {
            return new CollectByteToObjectIterable<V>(this, function);
        }

        public long sum()
        {
            long result = 0L;

            if (ByteFloatHashMap.this.sentinelValues != null)
            {
                if (ByteFloatHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += EMPTY_KEY;
                }
                if (ByteFloatHashMap.this.sentinelValues.containsOneKey)
                {
                    result += REMOVED_KEY;
                }
            }
            for (byte key : ByteFloatHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result += key;
                }
            }

            return result;
        }

        public byte max()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            ByteIterator iterator = this.byteIterator();
            byte max = iterator.next();
            while (iterator.hasNext())
            {
                byte value = iterator.next();
                if (max < value)
                {
                    max = value;
                }
            }
            return max;
        }

        public byte min()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            ByteIterator iterator = this.byteIterator();
            byte min = iterator.next();
            while (iterator.hasNext())
            {
                byte value = iterator.next();
                if (value < min)
                {
                    min = value;
                }
            }
            return min;
        }

        public byte minIfEmpty(byte defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.min();
        }

        public byte maxIfEmpty(byte defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.max();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return (double) this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            byte[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                byte first = sortedArray[middleIndex];
                byte second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public byte[] toSortedArray()
        {
            byte[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public byte[] toArray()
        {
            int size = ByteFloatHashMap.this.size();
            final byte[] result = new byte[size];
            ByteFloatHashMap.this.forEachKey(new ByteProcedure()
            {
                private int index;

                public void value(byte each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public MutableByteList toList()
        {
            return ByteArrayList.newList(this);
        }

        public MutableByteList toSortedList()
        {
            return ByteArrayList.newList(this).sortThis();
        }

        public MutableByteSet toSet()
        {
            return ByteHashSet.newSet(this);
        }

        public MutableByteBag toBag()
        {
            return ByteHashBag.newBag(this);
        }

        public LazyByteIterable asLazy()
        {
            return this;
        }
    }

    private class KeySetIterator implements ByteIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < ByteFloatHashMap.this.size();
        }

        public byte next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (ByteFloatHashMap.this.containsKey(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (ByteFloatHashMap.this.containsKey(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }

            byte[] keys = ByteFloatHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            byte result = keys[this.position];
            this.position++;
            return result;
        }
    }

    public MutableByteSet keySet()
    {
        return new KeySet();
    }

    private class KeySet implements MutableByteSet
    {
        public ByteIterator byteIterator()
        {
            return new KeySetIterator();
        }

        public void forEach(ByteProcedure procedure)
        {
            ByteFloatHashMap.this.forEachKey(procedure);
        }

        public int count(BytePredicate predicate)
        {
            int count = 0;
            if (ByteFloatHashMap.this.sentinelValues != null)
            {
                if (ByteFloatHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (ByteFloatHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (byte key : ByteFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(BytePredicate predicate)
        {
            if (ByteFloatHashMap.this.sentinelValues != null)
            {
                if (ByteFloatHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (ByteFloatHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (byte key : ByteFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(BytePredicate predicate)
        {
            if (ByteFloatHashMap.this.sentinelValues != null)
            {
                if (ByteFloatHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (ByteFloatHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (byte key : ByteFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(BytePredicate predicate)
        {
            if (ByteFloatHashMap.this.sentinelValues != null)
            {
                if (ByteFloatHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (ByteFloatHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (byte key : ByteFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean add(byte element)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(byte... source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(ByteIterable source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(byte key)
        {
            int oldSize = ByteFloatHashMap.this.size();
            ByteFloatHashMap.this.removeKey(key);
            return oldSize != ByteFloatHashMap.this.size();
        }

        public boolean removeAll(ByteIterable source)
        {
            int oldSize = ByteFloatHashMap.this.size();
            ByteIterator iterator = source.byteIterator();
            while (iterator.hasNext())
            {
                ByteFloatHashMap.this.removeKey(iterator.next());
            }
            return oldSize != ByteFloatHashMap.this.size();
        }

        public boolean removeAll(byte... source)
        {
            int oldSize = ByteFloatHashMap.this.size();
            for (byte item : source)
            {
                ByteFloatHashMap.this.removeKey(item);
            }
            return oldSize != ByteFloatHashMap.this.size();
        }

        public void clear()
        {
            ByteFloatHashMap.this.clear();
        }

        public MutableByteSet select(BytePredicate predicate)
        {
            MutableByteSet result = new ByteHashSet();
            if (ByteFloatHashMap.this.sentinelValues != null)
            {
                if (ByteFloatHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (ByteFloatHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (byte key : ByteFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableByteSet reject(BytePredicate predicate)
        {
            MutableByteSet result = new ByteHashSet();
            if (ByteFloatHashMap.this.sentinelValues != null)
            {
                if (ByteFloatHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (ByteFloatHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (byte key : ByteFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableByteSet with(byte element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableByteSet without(byte element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableByteSet withAll(ByteIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableByteSet withoutAll(ByteIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public byte detectIfNone(BytePredicate predicate, byte ifNone)
        {
            if (ByteFloatHashMap.this.sentinelValues != null)
            {
                if (ByteFloatHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (ByteFloatHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (byte key : ByteFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return ifNone;
        }

        public <V> MutableSet<V> collect(ByteToObjectFunction<? extends V> function)
        {
            MutableSet<V> result = Sets.mutable.with();
            if (ByteFloatHashMap.this.sentinelValues != null)
            {
                if (ByteFloatHashMap.this.sentinelValues.containsZeroKey)
                {
                    result.add(function.valueOf(EMPTY_KEY));
                }
                if (ByteFloatHashMap.this.sentinelValues.containsOneKey)
                {
                    result.add(function.valueOf(REMOVED_KEY));
                }
            }
            for (byte key : ByteFloatHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result.add(function.valueOf(key));
                }
            }
            return result;
        }

        public MutableByteSet asUnmodifiable()
        {
            return UnmodifiableByteSet.of(this);
        }

        public MutableByteSet asSynchronized()
        {
            return SynchronizedByteSet.of(this);
        }

        public long sum()
        {
            long sum = 0L;
            if (ByteFloatHashMap.this.sentinelValues != null)
            {
                if (ByteFloatHashMap.this.sentinelValues.containsZeroKey)
                {
                    sum += EMPTY_KEY;
                }
                if (ByteFloatHashMap.this.sentinelValues.containsOneKey)
                {
                    sum += REMOVED_KEY;
                }
            }
            for (byte key : ByteFloatHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    sum += key;
                }
            }
            return sum;
        }

        public byte max()
        {
            if (ByteFloatHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            byte max = 0;
            boolean isMaxSet = false;

            if (ByteFloatHashMap.this.sentinelValues != null)
            {
                if (ByteFloatHashMap.this.sentinelValues.containsZeroKey)
                {
                    max = EMPTY_KEY;
                    isMaxSet = true;
                }
                if (ByteFloatHashMap.this.sentinelValues.containsOneKey && (!isMaxSet || max < REMOVED_KEY))
                {
                    max = REMOVED_KEY;
                    isMaxSet = true;
                }
            }
            for (int i = 0; i < ByteFloatHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ByteFloatHashMap.this.keys[i]) && (!isMaxSet || max < ByteFloatHashMap.this.keys[i]))
                {
                    max = ByteFloatHashMap.this.keys[i];
                    isMaxSet = true;
                }
            }
            return max;
        }

        public byte maxIfEmpty(byte defaultValue)
        {
            if (ByteFloatHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.max();
        }

        public byte min()
        {
            if (ByteFloatHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            byte min = 0;
            boolean isMinSet = false;

            if (ByteFloatHashMap.this.sentinelValues != null)
            {
                if (ByteFloatHashMap.this.sentinelValues.containsZeroKey)
                {
                    min = EMPTY_KEY;
                    isMinSet = true;
                }
                if (ByteFloatHashMap.this.sentinelValues.containsOneKey && (!isMinSet || REMOVED_KEY < min))
                {
                    min = REMOVED_KEY;
                    isMinSet = true;
                }
            }
            for (int i = 0; i < ByteFloatHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ByteFloatHashMap.this.keys[i]) && (!isMinSet || ByteFloatHashMap.this.keys[i] < min))
                {
                    min = ByteFloatHashMap.this.keys[i];
                    isMinSet = true;
                }
            }
            return min;
        }

        public byte minIfEmpty(byte defaultValue)
        {
            if (ByteFloatHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.min();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return (double) this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            byte[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                byte first = sortedArray[middleIndex];
                byte second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public byte[] toSortedArray()
        {
            byte[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public MutableByteList toSortedList()
        {
            return ByteArrayList.newList(this).sortThis();
        }

        public byte[] toArray()
        {
            int size = ByteFloatHashMap.this.size();
            final byte[] result = new byte[size];
            ByteFloatHashMap.this.forEachKey(new ByteProcedure()
            {
                private int index;

                public void value(byte each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public boolean contains(byte value)
        {
            return ByteFloatHashMap.this.containsKey(value);
        }

        public boolean containsAll(byte... source)
        {
            for (byte item : source)
            {
                if (!ByteFloatHashMap.this.containsKey(item))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(ByteIterable source)
        {
            ByteIterator iterator = source.byteIterator();
            while (iterator.hasNext())
            {
                if (!ByteFloatHashMap.this.containsKey(iterator.next()))
                {
                    return false;
                }
            }
            return true;
        }

        public MutableByteList toList()
        {
            return ByteArrayList.newList(this);
        }

        public MutableByteSet toSet()
        {
            return ByteHashSet.newSet(this);
        }

        public MutableByteBag toBag()
        {
            return ByteHashBag.newBag(this);
        }

        public LazyByteIterable asLazy()
        {
            return new LazyByteIterableAdapter(this);
        }

        public ByteSet freeze()
        {
            throw new UnsupportedOperationException();
        }

        public ImmutableByteSet toImmutable()
        {
            return ByteSets.immutable.withAll(this);
        }

        public int size()
        {
            return ByteFloatHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return ByteFloatHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return ByteFloatHashMap.this.notEmpty();
        }

        @Override
        public boolean equals(Object obj)
        {
            if (this == obj)
            {
                return true;
            }

            if (!(obj instanceof ByteSet))
            {
                return false;
            }

            ByteSet other = (ByteSet) obj;
            return this.size() == other.size() && this.containsAll(other.toArray());
        }

        @Override
        public int hashCode()
        {
            int result = 0;

            if (ByteFloatHashMap.this.sentinelValues != null)
            {
                if (ByteFloatHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += (int) EMPTY_KEY;
                }
                if (ByteFloatHashMap.this.sentinelValues.containsOneKey)
                {
                    result += (int) REMOVED_KEY;
                }
            }
            for (int i = 0; i < ByteFloatHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ByteFloatHashMap.this.keys[i]))
                {
                    result += (int) ByteFloatHashMap.this.keys[i];
                }
            }

            return result;
        }

        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (ByteFloatHashMap.this.sentinelValues != null)
                {
                    if (ByteFloatHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (ByteFloatHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (byte key : ByteFloatHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }
    }

    public MutableFloatCollection values()
    {
        return new ValuesCollection();
    }

    private class ValuesCollection implements MutableFloatCollection
    {
        public void clear()
        {
            ByteFloatHashMap.this.clear();
        }

        public MutableFloatCollection select(FloatPredicate predicate)
        {
            return ByteFloatHashMap.this.select(predicate);
        }

        public MutableFloatCollection reject(FloatPredicate predicate)
        {
            return ByteFloatHashMap.this.reject(predicate);
        }

        public float detectIfNone(FloatPredicate predicate, float ifNone)
        {
            return ByteFloatHashMap.this.detectIfNone(predicate, ifNone);
        }

        public <V> MutableCollection<V> collect(FloatToObjectFunction<? extends V> function)
        {
            return ByteFloatHashMap.this.collect(function);
        }

        public double sum()
        {
            return ByteFloatHashMap.this.sum();
        }

        public float max()
        {
            return ByteFloatHashMap.this.max();
        }

        public float maxIfEmpty(float defaultValue)
        {
            return ByteFloatHashMap.this.maxIfEmpty(defaultValue);
        }

        public float min()
        {
            return ByteFloatHashMap.this.min();
        }

        public float minIfEmpty(float defaultValue)
        {
            return ByteFloatHashMap.this.minIfEmpty(defaultValue);
        }

        public double average()
        {
            return ByteFloatHashMap.this.average();
        }

        public double median()
        {
            return ByteFloatHashMap.this.median();
        }

        public float[] toSortedArray()
        {
            return ByteFloatHashMap.this.toSortedArray();
        }

        public MutableFloatList toSortedList()
        {
            return ByteFloatHashMap.this.toSortedList();
        }

        public MutableFloatCollection with(float element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableFloatCollection without(float element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableFloatCollection withAll(FloatIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableFloatCollection withoutAll(FloatIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableFloatCollection asUnmodifiable()
        {
            return UnmodifiableFloatCollection.of(this);
        }

        public MutableFloatCollection asSynchronized()
        {
            return SynchronizedFloatCollection.of(this);
        }

        public ImmutableFloatCollection toImmutable()
        {
            return FloatLists.immutable.withAll(this);
        }

        public boolean contains(float value)
        {
            return ByteFloatHashMap.this.containsValue(value);
        }

        public boolean containsAll(float... source)
        {
            return ByteFloatHashMap.this.containsAll(source);
        }

        public boolean containsAll(FloatIterable source)
        {
            return ByteFloatHashMap.this.containsAll(source);
        }

        public MutableFloatList toList()
        {
            return ByteFloatHashMap.this.toList();
        }

        public MutableFloatSet toSet()
        {
            return ByteFloatHashMap.this.toSet();
        }

        public MutableFloatBag toBag()
        {
            return ByteFloatHashMap.this.toBag();
        }

        public LazyFloatIterable asLazy()
        {
            return new LazyFloatIterableAdapter(this);
        }

        public boolean isEmpty()
        {
            return ByteFloatHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return ByteFloatHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);

                boolean first = true;

                if (ByteFloatHashMap.this.sentinelValues != null)
                {
                    if (ByteFloatHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(ByteFloatHashMap.this.sentinelValues.zeroValue));
                        first = false;
                    }
                    if (ByteFloatHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(ByteFloatHashMap.this.sentinelValues.oneValue));
                        first = false;
                    }
                }
                for (int i = 0; i < ByteFloatHashMap.this.keys.length; i++)
                {
                    byte key = ByteFloatHashMap.this.keys[i];
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(ByteFloatHashMap.this.values[i]));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public FloatIterator floatIterator()
        {
            return ByteFloatHashMap.this.floatIterator();
        }

        public void forEach(FloatProcedure procedure)
        {
            ByteFloatHashMap.this.forEach(procedure);
        }

        public int count(FloatPredicate predicate)
        {
            return ByteFloatHashMap.this.count(predicate);
        }

        public boolean anySatisfy(FloatPredicate predicate)
        {
            return ByteFloatHashMap.this.anySatisfy(predicate);
        }

        public boolean allSatisfy(FloatPredicate predicate)
        {
            return ByteFloatHashMap.this.allSatisfy(predicate);
        }

        public boolean noneSatisfy(FloatPredicate predicate)
        {
            return ByteFloatHashMap.this.noneSatisfy(predicate);
        }

        public boolean add(float element)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(float... source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(FloatIterable source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(float item)
        {
            int oldSize = ByteFloatHashMap.this.size();

            if (ByteFloatHashMap.this.sentinelValues != null)
            {
                if (ByteFloatHashMap.this.sentinelValues.containsZeroKey && Float.compare(item, ByteFloatHashMap.this.sentinelValues.zeroValue) == 0)
                {
                    ByteFloatHashMap.this.removeKey(EMPTY_KEY);
                }
                if (ByteFloatHashMap.this.sentinelValues.containsOneKey && Float.compare(item, ByteFloatHashMap.this.sentinelValues.oneValue) == 0)
                {
                    ByteFloatHashMap.this.removeKey(REMOVED_KEY);
                }
            }
            for (int i = 0; i < ByteFloatHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ByteFloatHashMap.this.keys[i]) && Float.compare(item, ByteFloatHashMap.this.values[i]) == 0)
                {
                    ByteFloatHashMap.this.removeKey(ByteFloatHashMap.this.keys[i]);
                }
            }
            return oldSize != ByteFloatHashMap.this.size();
        }

        public boolean removeAll(FloatIterable source)
        {
            int oldSize = ByteFloatHashMap.this.size();

            FloatIterator iterator = source.floatIterator();
            while (iterator.hasNext())
            {
                this.remove(iterator.next());
            }
            return oldSize != ByteFloatHashMap.this.size();
        }

        public boolean removeAll(float... source)
        {
            int oldSize = ByteFloatHashMap.this.size();

            for (float item : source)
            {
                this.remove(item);
            }
            return oldSize != ByteFloatHashMap.this.size();
        }

        public int size()
        {
            return ByteFloatHashMap.this.size();
        }

        public float[] toArray()
        {
            return ByteFloatHashMap.this.toArray();
        }
    }
}
