/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.map.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.CharIterable;
import com.gs.collections.api.LazyCharIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.bag.primitive.MutableCharBag;
import com.gs.collections.api.block.function.primitive.CharFunction;
import com.gs.collections.api.block.function.primitive.CharFunction0;
import com.gs.collections.api.block.function.primitive.CharToCharFunction;
import com.gs.collections.api.block.function.primitive.CharToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectCharToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.CharCharPredicate;
import com.gs.collections.api.block.predicate.primitive.CharPredicate;
import com.gs.collections.api.block.procedure.primitive.CharProcedure;
import com.gs.collections.api.block.procedure.primitive.CharCharProcedure;
import com.gs.collections.api.collection.MutableCollection;
import com.gs.collections.api.collection.primitive.ImmutableCharCollection;
import com.gs.collections.api.collection.primitive.MutableCharCollection;
import com.gs.collections.api.iterator.CharIterator;
import com.gs.collections.api.list.primitive.MutableCharList;
import com.gs.collections.api.map.primitive.CharCharMap;
import com.gs.collections.api.map.primitive.ImmutableCharCharMap;
import com.gs.collections.api.map.primitive.MutableCharCharMap;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.CharSet;
import com.gs.collections.api.set.primitive.ImmutableCharSet;
import com.gs.collections.api.set.primitive.MutableCharSet;
import com.gs.collections.impl.bag.mutable.primitive.CharHashBag;
import com.gs.collections.impl.block.factory.primitive.CharPredicates;
import com.gs.collections.impl.collection.mutable.primitive.SynchronizedCharCollection;
import com.gs.collections.impl.collection.mutable.primitive.UnmodifiableCharCollection;
import com.gs.collections.impl.factory.Sets;
import com.gs.collections.impl.factory.primitive.CharLists;
import com.gs.collections.impl.factory.primitive.CharCharMaps;
import com.gs.collections.impl.factory.primitive.CharSets;
import com.gs.collections.impl.lazy.primitive.CollectCharToObjectIterable;
import com.gs.collections.impl.lazy.primitive.LazyCharIterableAdapter;
import com.gs.collections.impl.lazy.primitive.SelectCharIterable;
import com.gs.collections.impl.list.mutable.FastList;
import com.gs.collections.impl.list.mutable.primitive.CharArrayList;
import com.gs.collections.impl.set.mutable.primitive.CharHashSet;
import com.gs.collections.impl.set.mutable.primitive.SynchronizedCharSet;
import com.gs.collections.impl.set.mutable.primitive.UnmodifiableCharSet;

/**
 * This file was automatically generated from template file primitivePrimitiveHashMap.stg.
 *
 * @since 3.0.
 */
public class CharCharHashMap implements MutableCharCharMap, Externalizable
{
    static final char EMPTY_VALUE = (char) 0;
    private static final long serialVersionUID = 1L;
    private static final char EMPTY_KEY = (char) 0;
    private static final char REMOVED_KEY = (char) 1;

    private static final int OCCUPIED_DATA_RATIO = 2;
    private static final int OCCUPIED_SENTINEL_RATIO = 4;
    private static final int DEFAULT_INITIAL_CAPACITY = 8;

    private char[] keys;
    private char[] values;

    private int occupiedWithData;
    private int occupiedWithSentinels;

    private SentinelValues sentinelValues;

    public CharCharHashMap()
    {
        this.allocateTable(DEFAULT_INITIAL_CAPACITY << 1);
    }

    public CharCharHashMap(int initialCapacity)
    {
        if (initialCapacity < 0)
        {
            throw new IllegalArgumentException("initial capacity cannot be less than 0");
        }
        int capacity = this.smallestPowerOfTwoGreaterThan(this.fastCeil(initialCapacity * OCCUPIED_DATA_RATIO));
        this.allocateTable(capacity);
    }

    private int smallestPowerOfTwoGreaterThan(int n)
    {
        return n > 1 ? Integer.highestOneBit(n - 1) << 1 : 1;
    }

    public CharCharHashMap(CharCharMap map)
    {
        this(Math.max(map.size(), DEFAULT_INITIAL_CAPACITY));
        this.putAll(map);
    }

    private int fastCeil(float v)
    {
        int possibleResult = (int) v;
        if (v - possibleResult > 0.0F)
        {
            possibleResult++;
        }
        return possibleResult;
    }

    public static CharCharHashMap newWithKeysValues(char key1, char value1)
    {
        return new CharCharHashMap(1).withKeyValue(key1, value1);
    }

    public static CharCharHashMap newWithKeysValues(char key1, char value1, char key2, char value2)
    {
        return new CharCharHashMap(2).withKeysValues(key1, value1, key2, value2);
    }

    public static CharCharHashMap newWithKeysValues(char key1, char value1, char key2, char value2, char key3, char value3)
    {
        return new CharCharHashMap(3).withKeysValues(key1, value1, key2, value2, key3, value3);
    }

    public static CharCharHashMap newWithKeysValues(char key1, char value1, char key2, char value2, char key3, char value3, char key4, char value4)
    {
        return new CharCharHashMap(4).withKeysValues(key1, value1, key2, value2, key3, value3, key4, value4);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }

        if (!(obj instanceof CharCharMap))
        {
            return false;
        }

        CharCharMap other = (CharCharMap) obj;

        if (this.size() != other.size())
        {
            return false;
        }

        if (this.sentinelValues == null)
        {
            if (other.containsKey(EMPTY_KEY) || other.containsKey(REMOVED_KEY))
            {
                return false;
            }
        }
        else
        {
            if (this.sentinelValues.containsZeroKey && (!other.containsKey(EMPTY_KEY) || this.sentinelValues.zeroValue != other.getOrThrow(EMPTY_KEY)))
            {
                return false;
            }

            if (this.sentinelValues.containsOneKey && (!other.containsKey(REMOVED_KEY) || this.sentinelValues.oneValue != other.getOrThrow(REMOVED_KEY)))
            {
                return false;
            }
        }

        for (int i = 0; i < this.keys.length; i++)
        {
            char key = this.keys[i];
            if (isNonSentinel(key) && (!other.containsKey(key) || this.values[i] != other.getOrThrow(key)))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int result = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += (int) EMPTY_KEY ^ (int) this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += (int) REMOVED_KEY ^ (int) this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += (int) this.keys[i] ^ (int) this.values[i];
            }
        }

        return result;
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public int size()
    {
        return this.occupiedWithData + (this.sentinelValues == null ? 0 : this.sentinelValues.size());
    }

    public boolean isEmpty()
    {
        return this.occupiedWithData == 0 && (this.sentinelValues == null || this.sentinelValues.size() == 0);
    }

    public boolean notEmpty()
    {
        return this.occupiedWithData != 0 || (this.sentinelValues != null && this.sentinelValues.size() != 0);
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            boolean first = true;

            if (this.sentinelValues != null)
            {
                if (this.sentinelValues.containsZeroKey)
                {
                    appendable.append(String.valueOf(EMPTY_KEY)).append("=").append(String.valueOf(this.sentinelValues.zeroValue));
                    first = false;
                }
                if (this.sentinelValues.containsOneKey)
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(REMOVED_KEY)).append("=").append(String.valueOf(this.sentinelValues.oneValue));
                    first = false;
                }
            }
            for (int i = 0; i < this.keys.length; i++)
            {
                char key = this.keys[i];
                if (isNonSentinel(key))
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(key)).append("=").append(String.valueOf(this.values[i]));
                    first = false;
                }
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public CharIterator charIterator()
    {
        return new InternalCharIterator();
    }

    public char[] toArray()
    {
        char[] array = new char[this.size()];
        int index = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                array[index] = this.sentinelValues.zeroValue;
                index++;
            }
            if (this.sentinelValues.containsOneKey)
            {
                array[index] = this.sentinelValues.oneValue;
                index++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                array[index] = this.values[i];
                index++;
            }
        }

        return array;
    }

    public boolean contains(char value)
    {
        return this.containsValue(value);
    }

    public boolean containsAll(char... source)
    {
        for (char each : source)
        {
            if (!this.contains(each))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(CharIterable source)
    {
        return source.allSatisfy(new CharPredicate()
        {
            public boolean accept(char value)
            {
                return CharCharHashMap.this.contains(value);
            }
        });
    }

    public void forEach(CharProcedure procedure)
    {
        this.forEachValue(procedure);
    }

    public MutableCharCollection select(CharPredicate predicate)
    {
        CharArrayList result = new CharArrayList();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }

        return result;
    }

    public MutableCharCollection reject(CharPredicate predicate)
    {
        CharArrayList result = new CharArrayList();
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public <V> MutableCollection<V> collect(CharToObjectFunction<? extends V> function)
    {
        FastList<V> target = FastList.newList(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                target.add(function.valueOf(this.sentinelValues.zeroValue));
            }
            if (this.sentinelValues.containsOneKey)
            {
                target.add(function.valueOf(this.sentinelValues.oneValue));
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                target.add(function.valueOf(this.values[i]));
            }
        }
        return target;
    }

    public char detectIfNone(CharPredicate predicate, char value)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return this.values[i];
            }
        }
        return value;
    }

    public int count(CharPredicate predicate)
    {
        int count = 0;
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                count++;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                count++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(CharPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return true;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return true;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(CharPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(CharPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public <V> V injectInto(V injectedValue, ObjectCharToObjectFunction<? super V, ? extends V> function)
    {
        V result = injectedValue;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result = function.valueOf(result, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                result = function.valueOf(result, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result = function.valueOf(result, this.values[i]);
            }
        }

        return result;
    }

    public MutableCharList toList()
    {
        return CharArrayList.newList(this);
    }

    public MutableCharSet toSet()
    {
        return CharHashSet.newSet(this);
    }

    public MutableCharBag toBag()
    {
        return CharHashBag.newBag(this);
    }

    public LazyCharIterable asLazy()
    {
        return new LazyCharIterableAdapter(this);
    }

    public void clear()
    {
        this.sentinelValues = null;
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;
        Arrays.fill(this.keys, EMPTY_KEY);
        Arrays.fill(this.values, EMPTY_VALUE);
    }

    public void put(char key, char value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return;
        }

        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return;
        }

        int index = this.probe(key);

        if (this.keys[index] == key)
        {
            // key already present in map
            this.values[index] = value;
            return;
        }

        if (this.keys[index] == REMOVED_KEY)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = key;
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
    }

    public void putAll(CharCharMap map)
    {
        map.forEachKeyValue(new CharCharProcedure()
        {
            public void value(char key, char value)
            {
                CharCharHashMap.this.put(key, value);
            }
        });
    }

    public void removeKey(char key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return;
            }
            if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.containsZeroKey = false;
                this.sentinelValues.zeroValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.containsOneKey = false;
                this.sentinelValues.oneValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.keys[index] = REMOVED_KEY;
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }
        }
    }

    public void remove(char key)
    {
        this.removeKey(key);
    }

    public char removeKeyIfAbsent(char key, char value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return value;
            }
            char oldValue = this.sentinelValues.zeroValue;
            if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.containsZeroKey = false;
                this.sentinelValues.zeroValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return oldValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return value;
            }
            char oldValue = this.sentinelValues.oneValue;
            if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.containsOneKey = false;
                this.sentinelValues.oneValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return oldValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.keys[index] = REMOVED_KEY;
            char oldValue = this.values[index];
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }

            return oldValue;
        }
        return value;
    }

    public char getIfAbsentPut(char key, char value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public char getIfAbsentPut(char key, CharFunction0 function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                char value = function.value();
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            char value = function.value();
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                char value = function.value();
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            char value = function.value();
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        char value = function.value();
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public <P> char getIfAbsentPutWith(char key, CharFunction<? super P> function, P parameter)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                char value = function.charValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            char value = function.charValueOf(parameter);
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                char value = function.charValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            char value = function.charValueOf(parameter);
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        char value = function.charValueOf(parameter);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public char getIfAbsentPutWithKey(char key, CharToCharFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                char value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            char value = function.valueOf(key);
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                char value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            char value = function.valueOf(key);
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        char value = function.valueOf(key);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public char updateValue(char key, char initialValueIfAbsent, CharToCharFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = function.valueOf(initialValueIfAbsent);
            }
            else if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.zeroValue = function.valueOf(this.sentinelValues.zeroValue);
            }
            else
            {
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = function.valueOf(initialValueIfAbsent);
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = function.valueOf(initialValueIfAbsent);
            }
            else if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.oneValue = function.valueOf(this.sentinelValues.oneValue);
            }
            else
            {
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = function.valueOf(initialValueIfAbsent);
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.values[index] = function.valueOf(this.values[index]);
            return this.values[index];
        }
        if (this.keys[index] == REMOVED_KEY)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = key;
        char value = function.valueOf(initialValueIfAbsent);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public CharCharHashMap withKeyValue(char key1, char value1)
    {
        this.put(key1, value1);
        return this;
    }

    public CharCharHashMap withKeysValues(char key1, char value1, char key2, char value2)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        return this;
    }

    public CharCharHashMap withKeysValues(char key1, char value1, char key2, char value2, char key3, char value3)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        return this;
    }

    public CharCharHashMap withKeysValues(char key1, char value1, char key2, char value2, char key3, char value3, char key4, char value4)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        this.put(key4, value4);
        return this;
    }

    public CharCharHashMap withoutKey(char key)
    {
        this.removeKey(key);
        return this;
    }

    public CharCharHashMap withoutAllKeys(CharIterable keys)
    {
        keys.forEach(new CharProcedure()
        {
            public void value(char key)
            {
                CharCharHashMap.this.removeKey(key);
            }
        });
        return this;
    }

    public MutableCharCharMap asUnmodifiable()
    {
        return new UnmodifiableCharCharMap(this);
    }

    public MutableCharCharMap asSynchronized()
    {
        return new SynchronizedCharCharMap(this);
    }

    public ImmutableCharCharMap toImmutable()
    {
        return CharCharMaps.immutable.ofAll(this);
    }

    public char get(char key)
    {
        return this.getIfAbsent(key, EMPTY_VALUE);
    }

    public char getIfAbsent(char key, char ifAbsent)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        return ifAbsent;
    }

    public char getOrThrow(char key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]))
        {
            return this.values[index];
        }
        throw new IllegalStateException("Key " + key + " not present.");
    }

    public boolean containsKey(char key)
    {
        if (isEmptyKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsZeroKey;
        }
        if (isRemovedKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsOneKey;
        }
        return this.keys[this.probe(key)] == key;
    }

    public boolean containsValue(char value)
    {
        if (this.sentinelValues != null && this.sentinelValues.containsValue(value))
        {
            return true;
        }
        for (int i = 0; i < this.values.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && this.values[i] == value)
            {
                return true;
            }
        }
        return false;
    }

    public void forEachValue(CharProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.values[i]);
            }
        }
    }

    public void forEachKey(CharProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY);
            }
        }
        for (char key : this.keys)
        {
            if (isNonSentinel(key))
            {
                procedure.value(key);
            }
        }
    }

    public void forEachKeyValue(CharCharProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.keys[i], this.values[i]);
            }
        }
    }

    public LazyCharIterable keysView()
    {
        return new KeysView();
    }

    public CharCharHashMap select(CharCharPredicate predicate)
    {
        CharCharHashMap result = new CharCharHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.keys[i], this.values[i]))
            {
                result.put(this.keys[i], this.values[i]);
            }
        }

        return result;
    }

    public CharCharHashMap reject(CharCharPredicate predicate)
    {
        CharCharHashMap result = new CharCharHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.keys[i], this.values[i]))
            {
                result.put(this.keys[i], this.values[i]);
            }
        }
        return result;
    }

    public long sum()
    {
        long result = 0L;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += this.values[i];
            }
        }

        return result;
    }

    public char max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        CharIterator iterator = this.charIterator();
        char max = iterator.next();
        while (iterator.hasNext())
        {
            char value = iterator.next();
            if (max < value)
            {
                max = value;
            }
        }
        return max;
    }

    public char maxIfEmpty(char defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public char min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        CharIterator iterator = this.charIterator();
        char min = iterator.next();
        while (iterator.hasNext())
        {
            char value = iterator.next();
            if (value < min)
            {
                min = value;
            }
        }
        return min;
    }

    public char minIfEmpty(char defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return (double)this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        char[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            char first = sortedArray[middleIndex];
            char second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public char[] toSortedArray()
    {
        char[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public MutableCharList toSortedList()
    {
        return CharArrayList.newList(this).sortThis();
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                out.writeChar(EMPTY_KEY);
                out.writeChar(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                out.writeChar(REMOVED_KEY);
                out.writeChar(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                out.writeChar(this.keys[i]);
                out.writeChar(this.values[i]);
            }
        }
    }

    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
    {
        int size = in.readInt();
        for (int i = 0; i < size; i++)
        {
            this.put(in.readChar(), in.readChar());
        }
    }

    /**
     * Rehashes every element in the set into a new backing table of the smallest possible size and eliminating removed sentinels.
     */
    public void compact()
    {
        this.rehash(this.smallestPowerOfTwoGreaterThan(this.size()));
    }

    private void rehash()
    {
        this.rehash(this.keys.length);
    }

    private void rehashAndGrow()
    {
        this.rehash(this.keys.length << 1);
    }

    private void rehash(int newCapacity)
    {
        int oldLength = this.keys.length;
        char[] old = this.keys;
        char[] oldValues = this.values;
        this.allocateTable(newCapacity);
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;

        for (int i = 0; i < oldLength; i++)
        {
            if (isNonSentinel(old[i]))
            {
                this.put(old[i], oldValues[i]);
            }
        }
    }

    // exposed for testing
    int probe(char element)
    {
        int index = this.spread(element);
        char keyAtIndex = this.keys[index];

        if (keyAtIndex == element || keyAtIndex == EMPTY_KEY)
        {
            return index;
        }

        int removedIndex = keyAtIndex == REMOVED_KEY ? index : -1;
        int nextIndex = index;
        int probe = 17;

        // loop until an empty slot is reached
        while (true)
        {
            // Probe algorithm: 17*n*(n+1)/2 where n = number of collisions
            nextIndex += probe;
            probe += 17;
            nextIndex &= this.keys.length - 1;

            if (this.keys[nextIndex] == element)
            {
                return nextIndex;
            }
            if (this.keys[nextIndex] == REMOVED_KEY)
            {
                if (removedIndex == -1)
                {
                    removedIndex = nextIndex;
                }
            }
            else if (this.keys[nextIndex] == EMPTY_KEY)
            {
                return removedIndex == -1 ? nextIndex : removedIndex;
            }
        }
    }

    // exposed for testing
    int spread(char element)
    {
        int code = (int) element;
        code ^= 61 ^ (code >> 16);
        code += code << 3;
        code ^= code >> 4;
        code *= 0x27d4eb2d;
        code ^= code >> 15;
        return code & (this.keys.length - 1);
    }

    private void allocateTable(int sizeToAllocate)
    {
        this.keys = new char[sizeToAllocate];
        this.values = new char[sizeToAllocate];
    }

    private static boolean isEmptyKey(char key)
    {
        return key == EMPTY_KEY;
    }

    private static boolean isRemovedKey(char key)
    {
        return key == REMOVED_KEY;
    }

    private static boolean isNonSentinel(char key)
    {
        return !isEmptyKey(key) && !isRemovedKey(key);
    }

    private int maxOccupiedWithData()
    {
        int capacity = this.keys.length;
        // need at least one free slot for open addressing
        return Math.min(capacity - 1, capacity / OCCUPIED_DATA_RATIO);
    }

    private int maxOccupiedWithSentinels()
    {
        return this.keys.length / OCCUPIED_SENTINEL_RATIO;
    }

    private static final class SentinelValues
    {
        private boolean containsZeroKey;
        private boolean containsOneKey;
        private char zeroValue;
        private char oneValue;

        public int size()
        {
            return (this.containsZeroKey ? 1 : 0) + (this.containsOneKey ? 1 : 0);
        }

        public boolean containsValue(char value)
        {
            boolean valueEqualsZeroValue = this.containsZeroKey && this.zeroValue == value;
            boolean valueEqualsOneValue = this.containsOneKey && this.oneValue == value;
            return valueEqualsZeroValue || valueEqualsOneValue;
        }
    }

    private class InternalCharIterator implements CharIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < CharCharHashMap.this.size();
        }

        public char next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (CharCharHashMap.this.containsKey(EMPTY_KEY))
                {
                    return CharCharHashMap.this.get(EMPTY_KEY);
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (CharCharHashMap.this.containsKey(REMOVED_KEY))
                {
                    return CharCharHashMap.this.get(REMOVED_KEY);
                }
            }

            char[] keys = CharCharHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            char result = CharCharHashMap.this.values[this.position];
            this.position++;
            return result;
        }
    }

    private class KeysView implements LazyCharIterable
    {
        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public int size()
        {
            return CharCharHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return CharCharHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return CharCharHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (CharCharHashMap.this.sentinelValues != null)
                {
                    if (CharCharHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (CharCharHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (char key : CharCharHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public CharIterator charIterator()
        {
            return new KeySetIterator();
        }

        public boolean contains(char key)
        {
            return CharCharHashMap.this.containsKey(key);
        }

        public boolean containsAll(char... keys)
        {
            for (char key : keys)
            {
                if (!CharCharHashMap.this.containsKey(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(CharIterable source)
        {
            return source.allSatisfy(new CharPredicate()
            {
                public boolean accept(char key)
                {
                    return CharCharHashMap.this.containsKey(key);
                }
            });
        }

        public void forEach(CharProcedure procedure)
        {
            CharCharHashMap.this.forEachKey(procedure);
        }

        public int count(CharPredicate predicate)
        {
            int count = 0;
            if (CharCharHashMap.this.sentinelValues != null)
            {
                if (CharCharHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (CharCharHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (char key : CharCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(CharPredicate predicate)
        {
            if (CharCharHashMap.this.sentinelValues != null)
            {
                if (CharCharHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (CharCharHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (char key : CharCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(CharPredicate predicate)
        {
            if (CharCharHashMap.this.sentinelValues != null)
            {
                if (CharCharHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (CharCharHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (char key : CharCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(CharPredicate predicate)
        {
            if (CharCharHashMap.this.sentinelValues != null)
            {
                if (CharCharHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (CharCharHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (char key : CharCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public LazyCharIterable select(CharPredicate predicate)
        {
            return new SelectCharIterable(this, predicate);
        }

        public LazyCharIterable reject(CharPredicate predicate)
        {
            return new SelectCharIterable(this, CharPredicates.not(predicate));
        }

        public char detectIfNone(CharPredicate predicate, char value)
        {
            if (CharCharHashMap.this.sentinelValues != null)
            {
                if (CharCharHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (CharCharHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (char key : CharCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return value;
        }

        public <V> LazyIterable<V> collect(CharToObjectFunction<? extends V> function)
        {
            return new CollectCharToObjectIterable<V>(this, function);
        }

        public long sum()
        {
            long result = 0L;

            if (CharCharHashMap.this.sentinelValues != null)
            {
                if (CharCharHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += EMPTY_KEY;
                }
                if (CharCharHashMap.this.sentinelValues.containsOneKey)
                {
                    result += REMOVED_KEY;
                }
            }
            for (char key : CharCharHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result += key;
                }
            }

            return result;
        }

        public char max()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            CharIterator iterator = this.charIterator();
            char max = iterator.next();
            while (iterator.hasNext())
            {
                char value = iterator.next();
                if (max < value)
                {
                    max = value;
                }
            }
            return max;
        }

        public char min()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            CharIterator iterator = this.charIterator();
            char min = iterator.next();
            while (iterator.hasNext())
            {
                char value = iterator.next();
                if (value < min)
                {
                    min = value;
                }
            }
            return min;
        }

        public char minIfEmpty(char defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.min();
        }

        public char maxIfEmpty(char defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.max();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return (double)this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            char[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                char first = sortedArray[middleIndex];
                char second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public char[] toSortedArray()
        {
            char[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public char[] toArray()
        {
            int size = CharCharHashMap.this.size();
            final char[] result = new char[size];
            CharCharHashMap.this.forEachKey(new CharProcedure()
            {
                private int index;

                public void value(char each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public MutableCharList toList()
        {
            return CharArrayList.newList(this);
        }

        public MutableCharList toSortedList()
        {
            return CharArrayList.newList(this).sortThis();
        }

        public MutableCharSet toSet()
        {
            return CharHashSet.newSet(this);
        }

        public MutableCharBag toBag()
        {
            return CharHashBag.newBag(this);
        }

        public LazyCharIterable asLazy()
        {
            return this;
        }
    }

    private class KeySetIterator implements CharIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < CharCharHashMap.this.size();
        }

        public char next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (CharCharHashMap.this.containsKey(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (CharCharHashMap.this.containsKey(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }

            char[] keys = CharCharHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            char result = keys[this.position];
            this.position++;
            return result;
        }
    }

    public MutableCharSet keySet()
    {
        return new KeySet();
    }

    private class KeySet implements MutableCharSet
    {
        public CharIterator charIterator()
        {
            return new KeySetIterator();
        }

        public void forEach(CharProcedure procedure)
        {
            CharCharHashMap.this.forEachKey(procedure);
        }

        public int count(CharPredicate predicate)
        {
            int count = 0;
            if (CharCharHashMap.this.sentinelValues != null)
            {
                if (CharCharHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (CharCharHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (char key : CharCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(CharPredicate predicate)
        {
            if (CharCharHashMap.this.sentinelValues != null)
            {
                if (CharCharHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (CharCharHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (char key : CharCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(CharPredicate predicate)
        {
            if (CharCharHashMap.this.sentinelValues != null)
            {
                if (CharCharHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (CharCharHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (char key : CharCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(CharPredicate predicate)
        {
            if (CharCharHashMap.this.sentinelValues != null)
            {
                if (CharCharHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (CharCharHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (char key : CharCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean add(char element)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(char... source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(CharIterable source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(char key)
        {
            int oldSize = CharCharHashMap.this.size();
            CharCharHashMap.this.removeKey(key);
            return oldSize != CharCharHashMap.this.size();
        }

        public boolean removeAll(CharIterable source)
        {
            int oldSize = CharCharHashMap.this.size();
            CharIterator iterator = source.charIterator();
            while (iterator.hasNext())
            {
                CharCharHashMap.this.removeKey(iterator.next());
            }
            return oldSize != CharCharHashMap.this.size();
        }

        public boolean removeAll(char... source)
        {
            int oldSize = CharCharHashMap.this.size();
            for (char item : source)
            {
                CharCharHashMap.this.removeKey(item);
            }
            return oldSize != CharCharHashMap.this.size();
        }

        public void clear()
        {
            CharCharHashMap.this.clear();
        }

        public MutableCharSet select(CharPredicate predicate)
        {
            MutableCharSet result = new CharHashSet();
            if (CharCharHashMap.this.sentinelValues != null)
            {
                if (CharCharHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (CharCharHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (char key : CharCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableCharSet reject(CharPredicate predicate)
        {
            MutableCharSet result = new CharHashSet();
            if (CharCharHashMap.this.sentinelValues != null)
            {
                if (CharCharHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (CharCharHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (char key : CharCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableCharSet with(char element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableCharSet without(char element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableCharSet withAll(CharIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableCharSet withoutAll(CharIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public char detectIfNone(CharPredicate predicate, char ifNone)
        {
            if (CharCharHashMap.this.sentinelValues != null)
            {
                if (CharCharHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (CharCharHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (char key : CharCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return ifNone;
        }

        public <V> MutableSet<V> collect(CharToObjectFunction<? extends V> function)
        {
            MutableSet<V> result = Sets.mutable.with();
            if (CharCharHashMap.this.sentinelValues != null)
            {
                if (CharCharHashMap.this.sentinelValues.containsZeroKey)
                {
                    result.add(function.valueOf(EMPTY_KEY));
                }
                if (CharCharHashMap.this.sentinelValues.containsOneKey)
                {
                    result.add(function.valueOf(REMOVED_KEY));
                }
            }
            for (char key : CharCharHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result.add(function.valueOf(key));
                }
            }
            return result;
        }

        public MutableCharSet asUnmodifiable()
        {
            return UnmodifiableCharSet.of(this);
        }

        public MutableCharSet asSynchronized()
        {
            return SynchronizedCharSet.of(this);
        }

        public long sum()
        {
            long sum = 0L;
            if (CharCharHashMap.this.sentinelValues != null)
            {
                if (CharCharHashMap.this.sentinelValues.containsZeroKey)
                {
                    sum += EMPTY_KEY;
                }
                if (CharCharHashMap.this.sentinelValues.containsOneKey)
                {
                    sum += REMOVED_KEY;
                }
            }
            for (char key : CharCharHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    sum += key;
                }
            }
            return sum;
        }

        public char max()
        {
            if (CharCharHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            char max = 0;
            boolean isMaxSet = false;

            if (CharCharHashMap.this.sentinelValues != null)
            {
                if (CharCharHashMap.this.sentinelValues.containsZeroKey)
                {
                    max = EMPTY_KEY;
                    isMaxSet = true;
                }
                if (CharCharHashMap.this.sentinelValues.containsOneKey && (!isMaxSet || max < REMOVED_KEY))
                {
                    max = REMOVED_KEY;
                    isMaxSet = true;
                }
            }
            for (int i = 0; i < CharCharHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(CharCharHashMap.this.keys[i]) && (!isMaxSet || max < CharCharHashMap.this.keys[i]))
                {
                    max = CharCharHashMap.this.keys[i];
                    isMaxSet = true;
                }
            }
            return max;
        }

        public char maxIfEmpty(char defaultValue)
        {
            if (CharCharHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.max();
        }

        public char min()
        {
            if (CharCharHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            char min = 0;
            boolean isMinSet = false;

            if (CharCharHashMap.this.sentinelValues != null)
            {
                if (CharCharHashMap.this.sentinelValues.containsZeroKey)
                {
                    min = EMPTY_KEY;
                    isMinSet = true;
                }
                if (CharCharHashMap.this.sentinelValues.containsOneKey && (!isMinSet || REMOVED_KEY < min))
                {
                    min = REMOVED_KEY;
                    isMinSet = true;
                }
            }
            for (int i = 0; i < CharCharHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(CharCharHashMap.this.keys[i]) && (!isMinSet || CharCharHashMap.this.keys[i] < min))
                {
                    min = CharCharHashMap.this.keys[i];
                    isMinSet = true;
                }
            }
            return min;
        }

        public char minIfEmpty(char defaultValue)
        {
            if (CharCharHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.min();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return (double)this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            char[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                char first = sortedArray[middleIndex];
                char second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public char[] toSortedArray()
        {
            char[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public MutableCharList toSortedList()
        {
            return CharArrayList.newList(this).sortThis();
        }

        public char[] toArray()
        {
            int size = CharCharHashMap.this.size();
            final char[] result = new char[size];
            CharCharHashMap.this.forEachKey(new CharProcedure()
            {
                private int index;

                public void value(char each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public boolean contains(char value)
        {
            return CharCharHashMap.this.containsKey(value);
        }

        public boolean containsAll(char... source)
        {
            for (char item : source)
            {
                if (!CharCharHashMap.this.containsKey(item))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(CharIterable source)
        {
            CharIterator iterator = source.charIterator();
            while (iterator.hasNext())
            {
                if (!CharCharHashMap.this.containsKey(iterator.next()))
                {
                    return false;
                }
            }
            return true;
        }

        public MutableCharList toList()
        {
            return CharArrayList.newList(this);
        }

        public MutableCharSet toSet()
        {
            return CharHashSet.newSet(this);
        }

        public MutableCharBag toBag()
        {
            return CharHashBag.newBag(this);
        }

        public LazyCharIterable asLazy()
        {
            return new LazyCharIterableAdapter(this);
        }

        public CharSet freeze()
        {
            throw new UnsupportedOperationException();
        }

        public ImmutableCharSet toImmutable()
        {
            return CharSets.immutable.withAll(this);
        }

        public int size()
        {
            return CharCharHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return CharCharHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return CharCharHashMap.this.notEmpty();
        }

        @Override
        public boolean equals(Object obj)
        {
            if (this == obj)
            {
                return true;
            }

            if (!(obj instanceof CharSet))
            {
                return false;
            }

            CharSet other = (CharSet) obj;
            return this.size() == other.size() && this.containsAll(other.toArray());
        }

        @Override
        public int hashCode()
        {
            int result = 0;

            if (CharCharHashMap.this.sentinelValues != null)
            {
                if (CharCharHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += (int) EMPTY_KEY;
                }
                if (CharCharHashMap.this.sentinelValues.containsOneKey)
                {
                    result += (int) REMOVED_KEY;
                }
            }
            for (int i = 0; i < CharCharHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(CharCharHashMap.this.keys[i]))
                {
                    result += (int) CharCharHashMap.this.keys[i];
                }
            }

            return result;
        }

        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (CharCharHashMap.this.sentinelValues != null)
                {
                    if (CharCharHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (CharCharHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (char key : CharCharHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }
    }

    public MutableCharCollection values()
    {
        return new ValuesCollection();
    }

    private class ValuesCollection implements MutableCharCollection
    {
        public void clear()
        {
            CharCharHashMap.this.clear();
        }

        public MutableCharCollection select(CharPredicate predicate)
        {
            return CharCharHashMap.this.select(predicate);
        }

        public MutableCharCollection reject(CharPredicate predicate)
        {
            return CharCharHashMap.this.reject(predicate);
        }

        public char detectIfNone(CharPredicate predicate, char ifNone)
        {
            return CharCharHashMap.this.detectIfNone(predicate, ifNone);
        }

        public <V> MutableCollection<V> collect(CharToObjectFunction<? extends V> function)
        {
            return CharCharHashMap.this.collect(function);
        }

        public long sum()
        {
            return CharCharHashMap.this.sum();
        }

        public char max()
        {
            return CharCharHashMap.this.max();
        }

        public char maxIfEmpty(char defaultValue)
        {
            return CharCharHashMap.this.maxIfEmpty(defaultValue);
        }

        public char min()
        {
            return CharCharHashMap.this.min();
        }

        public char minIfEmpty(char defaultValue)
        {
            return CharCharHashMap.this.minIfEmpty(defaultValue);
        }

        public double average()
        {
            return CharCharHashMap.this.average();
        }

        public double median()
        {
            return CharCharHashMap.this.median();
        }

        public char[] toSortedArray()
        {
            return CharCharHashMap.this.toSortedArray();
        }

        public MutableCharList toSortedList()
        {
            return CharCharHashMap.this.toSortedList();
        }

        public MutableCharCollection with(char element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableCharCollection without(char element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableCharCollection withAll(CharIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableCharCollection withoutAll(CharIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableCharCollection asUnmodifiable()
        {
            return UnmodifiableCharCollection.of(this);
        }

        public MutableCharCollection asSynchronized()
        {
            return SynchronizedCharCollection.of(this);
        }

        public ImmutableCharCollection toImmutable()
        {
            return CharLists.immutable.withAll(this);
        }

        public boolean contains(char value)
        {
            return CharCharHashMap.this.containsValue(value);
        }

        public boolean containsAll(char... source)
        {
            return CharCharHashMap.this.containsAll(source);
        }

        public boolean containsAll(CharIterable source)
        {
            return CharCharHashMap.this.containsAll(source);
        }

        public MutableCharList toList()
        {
            return CharCharHashMap.this.toList();
        }

        public MutableCharSet toSet()
        {
            return CharCharHashMap.this.toSet();
        }

        public MutableCharBag toBag()
        {
            return CharCharHashMap.this.toBag();
        }

        public LazyCharIterable asLazy()
        {
            return new LazyCharIterableAdapter(this);
        }

        public boolean isEmpty()
        {
            return CharCharHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return CharCharHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);

                boolean first = true;

                if (CharCharHashMap.this.sentinelValues != null)
                {
                    if (CharCharHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(CharCharHashMap.this.sentinelValues.zeroValue));
                        first = false;
                    }
                    if (CharCharHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(CharCharHashMap.this.sentinelValues.oneValue));
                        first = false;
                    }
                }
                for (int i = 0; i < CharCharHashMap.this.keys.length; i++)
                {
                    char key = CharCharHashMap.this.keys[i];
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(CharCharHashMap.this.values[i]));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public CharIterator charIterator()
        {
            return CharCharHashMap.this.charIterator();
        }

        public void forEach(CharProcedure procedure)
        {
            CharCharHashMap.this.forEach(procedure);
        }

        public int count(CharPredicate predicate)
        {
            return CharCharHashMap.this.count(predicate);
        }

        public boolean anySatisfy(CharPredicate predicate)
        {
            return CharCharHashMap.this.anySatisfy(predicate);
        }

        public boolean allSatisfy(CharPredicate predicate)
        {
            return CharCharHashMap.this.allSatisfy(predicate);
        }

        public boolean noneSatisfy(CharPredicate predicate)
        {
            return CharCharHashMap.this.noneSatisfy(predicate);
        }

        public boolean add(char element)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(char... source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(CharIterable source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(char item)
        {
            int oldSize = CharCharHashMap.this.size();

            if (CharCharHashMap.this.sentinelValues != null)
            {
                if (CharCharHashMap.this.sentinelValues.containsZeroKey && item == CharCharHashMap.this.sentinelValues.zeroValue)
                {
                    CharCharHashMap.this.removeKey(EMPTY_KEY);
                }
                if (CharCharHashMap.this.sentinelValues.containsOneKey && item == CharCharHashMap.this.sentinelValues.oneValue)
                {
                    CharCharHashMap.this.removeKey(REMOVED_KEY);
                }
            }
            for (int i = 0; i < CharCharHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(CharCharHashMap.this.keys[i]) && item == CharCharHashMap.this.values[i])
                {
                    CharCharHashMap.this.removeKey(CharCharHashMap.this.keys[i]);
                }
            }
            return oldSize != CharCharHashMap.this.size();
        }

        public boolean removeAll(CharIterable source)
        {
            int oldSize = CharCharHashMap.this.size();

            CharIterator iterator = source.charIterator();
            while (iterator.hasNext())
            {
                this.remove(iterator.next());
            }
            return oldSize != CharCharHashMap.this.size();
        }

        public boolean removeAll(char... source)
        {
            int oldSize = CharCharHashMap.this.size();

            for (char item : source)
            {
                this.remove(item);
            }
            return oldSize != CharCharHashMap.this.size();
        }

        public int size()
        {
            return CharCharHashMap.this.size();
        }

        public char[] toArray()
        {
            return CharCharHashMap.this.toArray();
        }
    }
}
