/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.map.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.DoubleIterable;
import com.gs.collections.api.LazyDoubleIterable;
import com.gs.collections.api.LazyShortIterable;
import com.gs.collections.api.ShortIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.bag.primitive.MutableDoubleBag;
import com.gs.collections.api.bag.primitive.MutableShortBag;
import com.gs.collections.api.block.function.primitive.DoubleToObjectFunction;
import com.gs.collections.api.block.function.primitive.DoubleToShortFunction;
import com.gs.collections.api.block.function.primitive.ShortFunction;
import com.gs.collections.api.block.function.primitive.ShortFunction0;
import com.gs.collections.api.block.function.primitive.ShortToShortFunction;
import com.gs.collections.api.block.function.primitive.ShortToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectShortToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.DoubleShortPredicate;
import com.gs.collections.api.block.predicate.primitive.DoublePredicate;
import com.gs.collections.api.block.predicate.primitive.ShortPredicate;
import com.gs.collections.api.block.procedure.primitive.DoubleProcedure;
import com.gs.collections.api.block.procedure.primitive.DoubleShortProcedure;
import com.gs.collections.api.block.procedure.primitive.ShortProcedure;
import com.gs.collections.api.collection.MutableCollection;
import com.gs.collections.api.collection.primitive.ImmutableShortCollection;
import com.gs.collections.api.collection.primitive.MutableShortCollection;
import com.gs.collections.api.iterator.DoubleIterator;
import com.gs.collections.api.iterator.ShortIterator;
import com.gs.collections.api.list.primitive.MutableDoubleList;
import com.gs.collections.api.list.primitive.MutableShortList;
import com.gs.collections.api.map.primitive.DoubleShortMap;
import com.gs.collections.api.map.primitive.ImmutableDoubleShortMap;
import com.gs.collections.api.map.primitive.MutableDoubleShortMap;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.DoubleSet;
import com.gs.collections.api.set.primitive.ImmutableDoubleSet;
import com.gs.collections.api.set.primitive.MutableDoubleSet;
import com.gs.collections.api.set.primitive.MutableShortSet;
import com.gs.collections.impl.bag.mutable.primitive.DoubleHashBag;
import com.gs.collections.impl.bag.mutable.primitive.ShortHashBag;
import com.gs.collections.impl.block.factory.primitive.DoublePredicates;
import com.gs.collections.impl.collection.mutable.primitive.SynchronizedShortCollection;
import com.gs.collections.impl.collection.mutable.primitive.UnmodifiableShortCollection;
import com.gs.collections.impl.factory.Sets;
import com.gs.collections.impl.factory.primitive.ShortLists;
import com.gs.collections.impl.factory.primitive.DoubleShortMaps;
import com.gs.collections.impl.factory.primitive.DoubleSets;
import com.gs.collections.impl.lazy.primitive.CollectDoubleToObjectIterable;
import com.gs.collections.impl.lazy.primitive.LazyShortIterableAdapter;
import com.gs.collections.impl.lazy.primitive.LazyDoubleIterableAdapter;
import com.gs.collections.impl.lazy.primitive.SelectDoubleIterable;
import com.gs.collections.impl.list.mutable.FastList;
import com.gs.collections.impl.list.mutable.primitive.DoubleArrayList;
import com.gs.collections.impl.list.mutable.primitive.ShortArrayList;
import com.gs.collections.impl.set.mutable.primitive.DoubleHashSet;
import com.gs.collections.impl.set.mutable.primitive.SynchronizedDoubleSet;
import com.gs.collections.impl.set.mutable.primitive.UnmodifiableDoubleSet;
import com.gs.collections.impl.set.mutable.primitive.ShortHashSet;

/**
 * This file was automatically generated from template file primitivePrimitiveHashMap.stg.
 *
 * @since 3.0.
 */
public class DoubleShortHashMap implements MutableDoubleShortMap, Externalizable
{
    static final short EMPTY_VALUE = (short) 0;
    private static final long serialVersionUID = 1L;
    private static final double EMPTY_KEY = 0.0;
    private static final double REMOVED_KEY = 1.0;

    private static final int OCCUPIED_DATA_RATIO = 2;
    private static final int OCCUPIED_SENTINEL_RATIO = 4;
    private static final int DEFAULT_INITIAL_CAPACITY = 8;

    private double[] keys;
    private short[] values;

    private int occupiedWithData;
    private int occupiedWithSentinels;

    private SentinelValues sentinelValues;

    public DoubleShortHashMap()
    {
        this.allocateTable(DEFAULT_INITIAL_CAPACITY << 1);
    }

    public DoubleShortHashMap(int initialCapacity)
    {
        if (initialCapacity < 0)
        {
            throw new IllegalArgumentException("initial capacity cannot be less than 0");
        }
        int capacity = this.smallestPowerOfTwoGreaterThan(this.fastCeil(initialCapacity * OCCUPIED_DATA_RATIO));
        this.allocateTable(capacity);
    }

    private int smallestPowerOfTwoGreaterThan(int n)
    {
        return n > 1 ? Integer.highestOneBit(n - 1) << 1 : 1;
    }

    public DoubleShortHashMap(DoubleShortMap map)
    {
        this(Math.max(map.size(), DEFAULT_INITIAL_CAPACITY));
        this.putAll(map);
    }

    private int fastCeil(float v)
    {
        int possibleResult = (int) v;
        if (v - possibleResult > 0.0F)
        {
            possibleResult++;
        }
        return possibleResult;
    }

    public static DoubleShortHashMap newWithKeysValues(double key1, short value1)
    {
        return new DoubleShortHashMap(1).withKeyValue(key1, value1);
    }

    public static DoubleShortHashMap newWithKeysValues(double key1, short value1, double key2, short value2)
    {
        return new DoubleShortHashMap(2).withKeysValues(key1, value1, key2, value2);
    }

    public static DoubleShortHashMap newWithKeysValues(double key1, short value1, double key2, short value2, double key3, short value3)
    {
        return new DoubleShortHashMap(3).withKeysValues(key1, value1, key2, value2, key3, value3);
    }

    public static DoubleShortHashMap newWithKeysValues(double key1, short value1, double key2, short value2, double key3, short value3, double key4, short value4)
    {
        return new DoubleShortHashMap(4).withKeysValues(key1, value1, key2, value2, key3, value3, key4, value4);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }

        if (!(obj instanceof DoubleShortMap))
        {
            return false;
        }

        DoubleShortMap other = (DoubleShortMap) obj;

        if (this.size() != other.size())
        {
            return false;
        }

        if (this.sentinelValues == null)
        {
            if (other.containsKey(EMPTY_KEY) || other.containsKey(REMOVED_KEY))
            {
                return false;
            }
        }
        else
        {
            if (this.sentinelValues.containsZeroKey && (!other.containsKey(EMPTY_KEY) || this.sentinelValues.zeroValue != other.getOrThrow(EMPTY_KEY)))
            {
                return false;
            }

            if (this.sentinelValues.containsOneKey && (!other.containsKey(REMOVED_KEY) || this.sentinelValues.oneValue != other.getOrThrow(REMOVED_KEY)))
            {
                return false;
            }
        }

        for (int i = 0; i < this.keys.length; i++)
        {
            double key = this.keys[i];
            if (isNonSentinel(key) && (!other.containsKey(key) || this.values[i] != other.getOrThrow(key)))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int result = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += (int) (Double.doubleToLongBits(EMPTY_KEY) ^ Double.doubleToLongBits(EMPTY_KEY) >>> 32) ^ (int) this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += (int) (Double.doubleToLongBits(REMOVED_KEY) ^ Double.doubleToLongBits(REMOVED_KEY) >>> 32) ^ (int) this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += (int) (Double.doubleToLongBits(this.keys[i]) ^ Double.doubleToLongBits(this.keys[i]) >>> 32) ^ (int) this.values[i];
            }
        }

        return result;
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public int size()
    {
        return this.occupiedWithData + (this.sentinelValues == null ? 0 : this.sentinelValues.size());
    }

    public boolean isEmpty()
    {
        return this.occupiedWithData == 0 && (this.sentinelValues == null || this.sentinelValues.size() == 0);
    }

    public boolean notEmpty()
    {
        return this.occupiedWithData != 0 || (this.sentinelValues != null && this.sentinelValues.size() != 0);
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            boolean first = true;

            if (this.sentinelValues != null)
            {
                if (this.sentinelValues.containsZeroKey)
                {
                    appendable.append(String.valueOf(EMPTY_KEY)).append("=").append(String.valueOf(this.sentinelValues.zeroValue));
                    first = false;
                }
                if (this.sentinelValues.containsOneKey)
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(REMOVED_KEY)).append("=").append(String.valueOf(this.sentinelValues.oneValue));
                    first = false;
                }
            }
            for (int i = 0; i < this.keys.length; i++)
            {
                double key = this.keys[i];
                if (isNonSentinel(key))
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(key)).append("=").append(String.valueOf(this.values[i]));
                    first = false;
                }
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public ShortIterator shortIterator()
    {
        return new InternalShortIterator();
    }

    public short[] toArray()
    {
        short[] array = new short[this.size()];
        int index = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                array[index] = this.sentinelValues.zeroValue;
                index++;
            }
            if (this.sentinelValues.containsOneKey)
            {
                array[index] = this.sentinelValues.oneValue;
                index++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                array[index] = this.values[i];
                index++;
            }
        }

        return array;
    }

    public boolean contains(short value)
    {
        return this.containsValue(value);
    }

    public boolean containsAll(short... source)
    {
        for (short each : source)
        {
            if (!this.contains(each))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(ShortIterable source)
    {
        return source.allSatisfy(new ShortPredicate()
        {
            public boolean accept(short value)
            {
                return DoubleShortHashMap.this.contains(value);
            }
        });
    }

    public void forEach(ShortProcedure procedure)
    {
        this.forEachValue(procedure);
    }

    public MutableShortCollection select(ShortPredicate predicate)
    {
        ShortArrayList result = new ShortArrayList();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }

        return result;
    }

    public MutableShortCollection reject(ShortPredicate predicate)
    {
        ShortArrayList result = new ShortArrayList();
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public <V> MutableCollection<V> collect(ShortToObjectFunction<? extends V> function)
    {
        FastList<V> target = FastList.newList(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                target.add(function.valueOf(this.sentinelValues.zeroValue));
            }
            if (this.sentinelValues.containsOneKey)
            {
                target.add(function.valueOf(this.sentinelValues.oneValue));
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                target.add(function.valueOf(this.values[i]));
            }
        }
        return target;
    }

    public short detectIfNone(ShortPredicate predicate, short value)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return this.values[i];
            }
        }
        return value;
    }

    public int count(ShortPredicate predicate)
    {
        int count = 0;
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                count++;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                count++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(ShortPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return true;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return true;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(ShortPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(ShortPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public <V> V injectInto(V injectedValue, ObjectShortToObjectFunction<? super V, ? extends V> function)
    {
        V result = injectedValue;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result = function.valueOf(result, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                result = function.valueOf(result, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result = function.valueOf(result, this.values[i]);
            }
        }

        return result;
    }

    public MutableShortList toList()
    {
        return ShortArrayList.newList(this);
    }

    public MutableShortSet toSet()
    {
        return ShortHashSet.newSet(this);
    }

    public MutableShortBag toBag()
    {
        return ShortHashBag.newBag(this);
    }

    public LazyShortIterable asLazy()
    {
        return new LazyShortIterableAdapter(this);
    }

    public void clear()
    {
        this.sentinelValues = null;
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;
        Arrays.fill(this.keys, EMPTY_KEY);
        Arrays.fill(this.values, EMPTY_VALUE);
    }

    public void put(double key, short value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return;
        }

        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return;
        }

        int index = this.probe(key);

        if (Double.compare(this.keys[index], key) == 0)
        {
            // key already present in map
            this.values[index] = value;
            return;
        }

        if (Double.compare(this.keys[index], REMOVED_KEY) == 0)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = key;
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
    }

    public void putAll(DoubleShortMap map)
    {
        map.forEachKeyValue(new DoubleShortProcedure()
        {
            public void value(double key, short value)
            {
                DoubleShortHashMap.this.put(key, value);
            }
        });
    }

    public void removeKey(double key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return;
            }
            if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.containsZeroKey = false;
                this.sentinelValues.zeroValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.containsOneKey = false;
                this.sentinelValues.oneValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return;
        }
        int index = this.probe(key);
        if (Double.compare(this.keys[index], key) == 0)
        {
            this.keys[index] = REMOVED_KEY;
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }
        }
    }

    public void remove(double key)
    {
        this.removeKey(key);
    }

    public short removeKeyIfAbsent(double key, short value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return value;
            }
            short oldValue = this.sentinelValues.zeroValue;
            if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.containsZeroKey = false;
                this.sentinelValues.zeroValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return oldValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return value;
            }
            short oldValue = this.sentinelValues.oneValue;
            if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.containsOneKey = false;
                this.sentinelValues.oneValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return oldValue;
        }
        int index = this.probe(key);
        if (Double.compare(this.keys[index], key) == 0)
        {
            this.keys[index] = REMOVED_KEY;
            short oldValue = this.values[index];
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }

            return oldValue;
        }
        return value;
    }

    public short getIfAbsentPut(double key, short value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (Double.compare(this.keys[index], key) == 0)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public short getIfAbsentPut(double key, ShortFunction0 function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                short value = function.value();
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            short value = function.value();
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                short value = function.value();
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            short value = function.value();
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (Double.compare(this.keys[index], key) == 0)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        short value = function.value();
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public <P> short getIfAbsentPutWith(double key, ShortFunction<? super P> function, P parameter)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                short value = function.shortValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            short value = function.shortValueOf(parameter);
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                short value = function.shortValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            short value = function.shortValueOf(parameter);
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (Double.compare(this.keys[index], key) == 0)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        short value = function.shortValueOf(parameter);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public short getIfAbsentPutWithKey(double key, DoubleToShortFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                short value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            short value = function.valueOf(key);
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                short value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            short value = function.valueOf(key);
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (Double.compare(this.keys[index], key) == 0)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        short value = function.valueOf(key);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public short updateValue(double key, short initialValueIfAbsent, ShortToShortFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = function.valueOf(initialValueIfAbsent);
            }
            else if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.zeroValue = function.valueOf(this.sentinelValues.zeroValue);
            }
            else
            {
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = function.valueOf(initialValueIfAbsent);
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = function.valueOf(initialValueIfAbsent);
            }
            else if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.oneValue = function.valueOf(this.sentinelValues.oneValue);
            }
            else
            {
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = function.valueOf(initialValueIfAbsent);
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (Double.compare(this.keys[index], key) == 0)
        {
            this.values[index] = function.valueOf(this.values[index]);
            return this.values[index];
        }
        if (Double.compare(this.keys[index], REMOVED_KEY) == 0)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = key;
        short value = function.valueOf(initialValueIfAbsent);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public DoubleShortHashMap withKeyValue(double key1, short value1)
    {
        this.put(key1, value1);
        return this;
    }

    public DoubleShortHashMap withKeysValues(double key1, short value1, double key2, short value2)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        return this;
    }

    public DoubleShortHashMap withKeysValues(double key1, short value1, double key2, short value2, double key3, short value3)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        return this;
    }

    public DoubleShortHashMap withKeysValues(double key1, short value1, double key2, short value2, double key3, short value3, double key4, short value4)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        this.put(key4, value4);
        return this;
    }

    public DoubleShortHashMap withoutKey(double key)
    {
        this.removeKey(key);
        return this;
    }

    public DoubleShortHashMap withoutAllKeys(DoubleIterable keys)
    {
        keys.forEach(new DoubleProcedure()
        {
            public void value(double key)
            {
                DoubleShortHashMap.this.removeKey(key);
            }
        });
        return this;
    }

    public MutableDoubleShortMap asUnmodifiable()
    {
        return new UnmodifiableDoubleShortMap(this);
    }

    public MutableDoubleShortMap asSynchronized()
    {
        return new SynchronizedDoubleShortMap(this);
    }

    public ImmutableDoubleShortMap toImmutable()
    {
        return DoubleShortMaps.immutable.ofAll(this);
    }

    public short get(double key)
    {
        return this.getIfAbsent(key, EMPTY_VALUE);
    }

    public short getIfAbsent(double key, short ifAbsent)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (Double.compare(this.keys[index], key) == 0)
        {
            return this.values[index];
        }
        return ifAbsent;
    }

    public short getOrThrow(double key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]))
        {
            return this.values[index];
        }
        throw new IllegalStateException("Key " + key + " not present.");
    }

    public boolean containsKey(double key)
    {
        if (isEmptyKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsZeroKey;
        }
        if (isRemovedKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsOneKey;
        }
        return Double.compare(this.keys[this.probe(key)], key) == 0;
    }

    public boolean containsValue(short value)
    {
        if (this.sentinelValues != null && this.sentinelValues.containsValue(value))
        {
            return true;
        }
        for (int i = 0; i < this.values.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && this.values[i] == value)
            {
                return true;
            }
        }
        return false;
    }

    public void forEachValue(ShortProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.values[i]);
            }
        }
    }

    public void forEachKey(DoubleProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY);
            }
        }
        for (double key : this.keys)
        {
            if (isNonSentinel(key))
            {
                procedure.value(key);
            }
        }
    }

    public void forEachKeyValue(DoubleShortProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.keys[i], this.values[i]);
            }
        }
    }

    public LazyDoubleIterable keysView()
    {
        return new KeysView();
    }

    public DoubleShortHashMap select(DoubleShortPredicate predicate)
    {
        DoubleShortHashMap result = new DoubleShortHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.keys[i], this.values[i]))
            {
                result.put(this.keys[i], this.values[i]);
            }
        }

        return result;
    }

    public DoubleShortHashMap reject(DoubleShortPredicate predicate)
    {
        DoubleShortHashMap result = new DoubleShortHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.keys[i], this.values[i]))
            {
                result.put(this.keys[i], this.values[i]);
            }
        }
        return result;
    }

    public long sum()
    {
        long result = 0L;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += this.values[i];
            }
        }

        return result;
    }

    public short max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        ShortIterator iterator = this.shortIterator();
        short max = iterator.next();
        while (iterator.hasNext())
        {
            short value = iterator.next();
            if (max < value)
            {
                max = value;
            }
        }
        return max;
    }

    public short maxIfEmpty(short defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public short min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        ShortIterator iterator = this.shortIterator();
        short min = iterator.next();
        while (iterator.hasNext())
        {
            short value = iterator.next();
            if (value < min)
            {
                min = value;
            }
        }
        return min;
    }

    public short minIfEmpty(short defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return (double) this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        short[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            short first = sortedArray[middleIndex];
            short second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public short[] toSortedArray()
    {
        short[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public MutableShortList toSortedList()
    {
        return ShortArrayList.newList(this).sortThis();
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                out.writeDouble(EMPTY_KEY);
                out.writeShort(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                out.writeDouble(REMOVED_KEY);
                out.writeShort(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                out.writeDouble(this.keys[i]);
                out.writeShort(this.values[i]);
            }
        }
    }

    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
    {
        int size = in.readInt();
        for (int i = 0; i < size; i++)
        {
            this.put(in.readDouble(), in.readShort());
        }
    }

    /**
     * Rehashes every element in the set into a new backing table of the smallest possible size and eliminating removed sentinels.
     */
    public void compact()
    {
        this.rehash(this.smallestPowerOfTwoGreaterThan(this.size()));
    }

    private void rehash()
    {
        this.rehash(this.keys.length);
    }

    private void rehashAndGrow()
    {
        this.rehash(this.keys.length << 1);
    }

    private void rehash(int newCapacity)
    {
        int oldLength = this.keys.length;
        double[] old = this.keys;
        short[] oldValues = this.values;
        this.allocateTable(newCapacity);
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;

        for (int i = 0; i < oldLength; i++)
        {
            if (isNonSentinel(old[i]))
            {
                this.put(old[i], oldValues[i]);
            }
        }
    }

    // exposed for testing
    int probe(double element)
    {
        int index = this.spread(element);
        double keyAtIndex = this.keys[index];

        if (Double.compare(keyAtIndex, element) == 0 || Double.compare(keyAtIndex, EMPTY_KEY) == 0)
        {
            return index;
        }

        int removedIndex = Double.compare(keyAtIndex, REMOVED_KEY) == 0 ? index : -1;
        int nextIndex = index;
        int probe = 17;

        // loop until an empty slot is reached
        while (true)
        {
            // Probe algorithm: 17*n*(n+1)/2 where n = number of collisions
            nextIndex += probe;
            probe += 17;
            nextIndex &= this.keys.length - 1;

            if (Double.compare(this.keys[nextIndex], element) == 0)
            {
                return nextIndex;
            }
            if (Double.compare(this.keys[nextIndex], REMOVED_KEY) == 0)
            {
                if (removedIndex == -1)
                {
                    removedIndex = nextIndex;
                }
            }
            else if (Double.compare(this.keys[nextIndex], EMPTY_KEY) == 0)
            {
                return removedIndex == -1 ? nextIndex : removedIndex;
            }
        }
    }

    // exposed for testing
    int spread(double element)
    {
        long code = Double.doubleToLongBits(element);
        code = ~code + (code << 18);
        code = (code << 18) - code - 1;
        code ^= code >>> 31;
        code *= 21;
        code += (code << 2) + (code << 4);
        code ^= code >>> 11;
        code += code << 6;
        code ^= code >>> 22;
        return (int) code & (this.keys.length - 1);
    }

    private void allocateTable(int sizeToAllocate)
    {
        this.keys = new double[sizeToAllocate];
        this.values = new short[sizeToAllocate];
    }

    private static boolean isEmptyKey(double key)
    {
        return Double.compare(key, EMPTY_KEY) == 0;
    }

    private static boolean isRemovedKey(double key)
    {
        return Double.compare(key, REMOVED_KEY) == 0;
    }

    private static boolean isNonSentinel(double key)
    {
        return !isEmptyKey(key) && !isRemovedKey(key);
    }

    private int maxOccupiedWithData()
    {
        int capacity = this.keys.length;
        // need at least one free slot for open addressing
        return Math.min(capacity - 1, capacity / OCCUPIED_DATA_RATIO);
    }

    private int maxOccupiedWithSentinels()
    {
        return this.keys.length / OCCUPIED_SENTINEL_RATIO;
    }

    private static final class SentinelValues
    {
        private boolean containsZeroKey;
        private boolean containsOneKey;
        private short zeroValue;
        private short oneValue;

        public int size()
        {
            return (this.containsZeroKey ? 1 : 0) + (this.containsOneKey ? 1 : 0);
        }

        public boolean containsValue(short value)
        {
            boolean valueEqualsZeroValue = this.containsZeroKey && this.zeroValue == value;
            boolean valueEqualsOneValue = this.containsOneKey && this.oneValue == value;
            return valueEqualsZeroValue || valueEqualsOneValue;
        }
    }

    private class InternalShortIterator implements ShortIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < DoubleShortHashMap.this.size();
        }

        public short next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (DoubleShortHashMap.this.containsKey(EMPTY_KEY))
                {
                    return DoubleShortHashMap.this.get(EMPTY_KEY);
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (DoubleShortHashMap.this.containsKey(REMOVED_KEY))
                {
                    return DoubleShortHashMap.this.get(REMOVED_KEY);
                }
            }

            double[] keys = DoubleShortHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            short result = DoubleShortHashMap.this.values[this.position];
            this.position++;
            return result;
        }
    }

    private class KeysView implements LazyDoubleIterable
    {
        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public int size()
        {
            return DoubleShortHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return DoubleShortHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return DoubleShortHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (DoubleShortHashMap.this.sentinelValues != null)
                {
                    if (DoubleShortHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (DoubleShortHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (double key : DoubleShortHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public DoubleIterator doubleIterator()
        {
            return new KeySetIterator();
        }

        public boolean contains(double key)
        {
            return DoubleShortHashMap.this.containsKey(key);
        }

        public boolean containsAll(double... keys)
        {
            for (double key : keys)
            {
                if (!DoubleShortHashMap.this.containsKey(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(DoubleIterable source)
        {
            return source.allSatisfy(new DoublePredicate()
            {
                public boolean accept(double key)
                {
                    return DoubleShortHashMap.this.containsKey(key);
                }
            });
        }

        public void forEach(DoubleProcedure procedure)
        {
            DoubleShortHashMap.this.forEachKey(procedure);
        }

        public int count(DoublePredicate predicate)
        {
            int count = 0;
            if (DoubleShortHashMap.this.sentinelValues != null)
            {
                if (DoubleShortHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (DoubleShortHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (double key : DoubleShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(DoublePredicate predicate)
        {
            if (DoubleShortHashMap.this.sentinelValues != null)
            {
                if (DoubleShortHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (DoubleShortHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (double key : DoubleShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(DoublePredicate predicate)
        {
            if (DoubleShortHashMap.this.sentinelValues != null)
            {
                if (DoubleShortHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (DoubleShortHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (double key : DoubleShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(DoublePredicate predicate)
        {
            if (DoubleShortHashMap.this.sentinelValues != null)
            {
                if (DoubleShortHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (DoubleShortHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (double key : DoubleShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public LazyDoubleIterable select(DoublePredicate predicate)
        {
            return new SelectDoubleIterable(this, predicate);
        }

        public LazyDoubleIterable reject(DoublePredicate predicate)
        {
            return new SelectDoubleIterable(this, DoublePredicates.not(predicate));
        }

        public double detectIfNone(DoublePredicate predicate, double value)
        {
            if (DoubleShortHashMap.this.sentinelValues != null)
            {
                if (DoubleShortHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (DoubleShortHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (double key : DoubleShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return value;
        }

        public <V> LazyIterable<V> collect(DoubleToObjectFunction<? extends V> function)
        {
            return new CollectDoubleToObjectIterable<V>(this, function);
        }

        public double sum()
        {
            double result = 0.0;

            if (DoubleShortHashMap.this.sentinelValues != null)
            {
                if (DoubleShortHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += EMPTY_KEY;
                }
                if (DoubleShortHashMap.this.sentinelValues.containsOneKey)
                {
                    result += REMOVED_KEY;
                }
            }
            for (double key : DoubleShortHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result += key;
                }
            }

            return result;
        }

        public double max()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            DoubleIterator iterator = this.doubleIterator();
            double max = iterator.next();
            while (iterator.hasNext())
            {
                double value = iterator.next();
                if (Double.compare(max, value) < 0)
                {
                    max = value;
                }
            }
            return max;
        }

        public double min()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            DoubleIterator iterator = this.doubleIterator();
            double min = iterator.next();
            while (iterator.hasNext())
            {
                double value = iterator.next();
                if (Double.compare(value, min) < 0)
                {
                    min = value;
                }
            }
            return min;
        }

        public double minIfEmpty(double defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.min();
        }

        public double maxIfEmpty(double defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.max();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            double[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                double first = sortedArray[middleIndex];
                double second = sortedArray[middleIndex - 1];
                return (first + second) / 2.0;
            }
            return sortedArray[middleIndex];
        }

        public double[] toSortedArray()
        {
            double[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public double[] toArray()
        {
            int size = DoubleShortHashMap.this.size();
            final double[] result = new double[size];
            DoubleShortHashMap.this.forEachKey(new DoubleProcedure()
            {
                private int index;

                public void value(double each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public MutableDoubleList toList()
        {
            return DoubleArrayList.newList(this);
        }

        public MutableDoubleList toSortedList()
        {
            return DoubleArrayList.newList(this).sortThis();
        }

        public MutableDoubleSet toSet()
        {
            return DoubleHashSet.newSet(this);
        }

        public MutableDoubleBag toBag()
        {
            return DoubleHashBag.newBag(this);
        }

        public LazyDoubleIterable asLazy()
        {
            return this;
        }
    }

    private class KeySetIterator implements DoubleIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < DoubleShortHashMap.this.size();
        }

        public double next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (DoubleShortHashMap.this.containsKey(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (DoubleShortHashMap.this.containsKey(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }

            double[] keys = DoubleShortHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            double result = keys[this.position];
            this.position++;
            return result;
        }
    }

    public MutableDoubleSet keySet()
    {
        return new KeySet();
    }

    private class KeySet implements MutableDoubleSet
    {
        public DoubleIterator doubleIterator()
        {
            return new KeySetIterator();
        }

        public void forEach(DoubleProcedure procedure)
        {
            DoubleShortHashMap.this.forEachKey(procedure);
        }

        public int count(DoublePredicate predicate)
        {
            int count = 0;
            if (DoubleShortHashMap.this.sentinelValues != null)
            {
                if (DoubleShortHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (DoubleShortHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (double key : DoubleShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(DoublePredicate predicate)
        {
            if (DoubleShortHashMap.this.sentinelValues != null)
            {
                if (DoubleShortHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (DoubleShortHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (double key : DoubleShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(DoublePredicate predicate)
        {
            if (DoubleShortHashMap.this.sentinelValues != null)
            {
                if (DoubleShortHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (DoubleShortHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (double key : DoubleShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(DoublePredicate predicate)
        {
            if (DoubleShortHashMap.this.sentinelValues != null)
            {
                if (DoubleShortHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (DoubleShortHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (double key : DoubleShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean add(double element)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(double... source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(DoubleIterable source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(double key)
        {
            int oldSize = DoubleShortHashMap.this.size();
            DoubleShortHashMap.this.removeKey(key);
            return oldSize != DoubleShortHashMap.this.size();
        }

        public boolean removeAll(DoubleIterable source)
        {
            int oldSize = DoubleShortHashMap.this.size();
            DoubleIterator iterator = source.doubleIterator();
            while (iterator.hasNext())
            {
                DoubleShortHashMap.this.removeKey(iterator.next());
            }
            return oldSize != DoubleShortHashMap.this.size();
        }

        public boolean removeAll(double... source)
        {
            int oldSize = DoubleShortHashMap.this.size();
            for (double item : source)
            {
                DoubleShortHashMap.this.removeKey(item);
            }
            return oldSize != DoubleShortHashMap.this.size();
        }

        public void clear()
        {
            DoubleShortHashMap.this.clear();
        }

        public MutableDoubleSet select(DoublePredicate predicate)
        {
            MutableDoubleSet result = new DoubleHashSet();
            if (DoubleShortHashMap.this.sentinelValues != null)
            {
                if (DoubleShortHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (DoubleShortHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (double key : DoubleShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableDoubleSet reject(DoublePredicate predicate)
        {
            MutableDoubleSet result = new DoubleHashSet();
            if (DoubleShortHashMap.this.sentinelValues != null)
            {
                if (DoubleShortHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (DoubleShortHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (double key : DoubleShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableDoubleSet with(double element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableDoubleSet without(double element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableDoubleSet withAll(DoubleIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableDoubleSet withoutAll(DoubleIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public double detectIfNone(DoublePredicate predicate, double ifNone)
        {
            if (DoubleShortHashMap.this.sentinelValues != null)
            {
                if (DoubleShortHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (DoubleShortHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (double key : DoubleShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return ifNone;
        }

        public <V> MutableSet<V> collect(DoubleToObjectFunction<? extends V> function)
        {
            MutableSet<V> result = Sets.mutable.with();
            if (DoubleShortHashMap.this.sentinelValues != null)
            {
                if (DoubleShortHashMap.this.sentinelValues.containsZeroKey)
                {
                    result.add(function.valueOf(EMPTY_KEY));
                }
                if (DoubleShortHashMap.this.sentinelValues.containsOneKey)
                {
                    result.add(function.valueOf(REMOVED_KEY));
                }
            }
            for (double key : DoubleShortHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result.add(function.valueOf(key));
                }
            }
            return result;
        }

        public MutableDoubleSet asUnmodifiable()
        {
            return UnmodifiableDoubleSet.of(this);
        }

        public MutableDoubleSet asSynchronized()
        {
            return SynchronizedDoubleSet.of(this);
        }

        public double sum()
        {
            double sum = 0.0;
            if (DoubleShortHashMap.this.sentinelValues != null)
            {
                if (DoubleShortHashMap.this.sentinelValues.containsZeroKey)
                {
                    sum += EMPTY_KEY;
                }
                if (DoubleShortHashMap.this.sentinelValues.containsOneKey)
                {
                    sum += REMOVED_KEY;
                }
            }
            for (double key : DoubleShortHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    sum += key;
                }
            }
            return sum;
        }

        public double max()
        {
            if (DoubleShortHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            double max = 0;
            boolean isMaxSet = false;

            if (DoubleShortHashMap.this.sentinelValues != null)
            {
                if (DoubleShortHashMap.this.sentinelValues.containsZeroKey)
                {
                    max = EMPTY_KEY;
                    isMaxSet = true;
                }
                if (DoubleShortHashMap.this.sentinelValues.containsOneKey && (!isMaxSet || Double.compare(max, REMOVED_KEY) < 0))
                {
                    max = REMOVED_KEY;
                    isMaxSet = true;
                }
            }
            for (int i = 0; i < DoubleShortHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(DoubleShortHashMap.this.keys[i]) && (!isMaxSet || Double.compare(max, DoubleShortHashMap.this.keys[i]) < 0))
                {
                    max = DoubleShortHashMap.this.keys[i];
                    isMaxSet = true;
                }
            }
            return max;
        }

        public double maxIfEmpty(double defaultValue)
        {
            if (DoubleShortHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.max();
        }

        public double min()
        {
            if (DoubleShortHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            double min = 0;
            boolean isMinSet = false;

            if (DoubleShortHashMap.this.sentinelValues != null)
            {
                if (DoubleShortHashMap.this.sentinelValues.containsZeroKey)
                {
                    min = EMPTY_KEY;
                    isMinSet = true;
                }
                if (DoubleShortHashMap.this.sentinelValues.containsOneKey && (!isMinSet || Double.compare(REMOVED_KEY, min) < 0))
                {
                    min = REMOVED_KEY;
                    isMinSet = true;
                }
            }
            for (int i = 0; i < DoubleShortHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(DoubleShortHashMap.this.keys[i]) && (!isMinSet || Double.compare(DoubleShortHashMap.this.keys[i], min) < 0))
                {
                    min = DoubleShortHashMap.this.keys[i];
                    isMinSet = true;
                }
            }
            return min;
        }

        public double minIfEmpty(double defaultValue)
        {
            if (DoubleShortHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.min();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            double[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                double first = sortedArray[middleIndex];
                double second = sortedArray[middleIndex - 1];
                return (first + second) / 2.0;
            }
            return sortedArray[middleIndex];
        }

        public double[] toSortedArray()
        {
            double[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public MutableDoubleList toSortedList()
        {
            return DoubleArrayList.newList(this).sortThis();
        }

        public double[] toArray()
        {
            int size = DoubleShortHashMap.this.size();
            final double[] result = new double[size];
            DoubleShortHashMap.this.forEachKey(new DoubleProcedure()
            {
                private int index;

                public void value(double each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public boolean contains(double value)
        {
            return DoubleShortHashMap.this.containsKey(value);
        }

        public boolean containsAll(double... source)
        {
            for (double item : source)
            {
                if (!DoubleShortHashMap.this.containsKey(item))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(DoubleIterable source)
        {
            DoubleIterator iterator = source.doubleIterator();
            while (iterator.hasNext())
            {
                if (!DoubleShortHashMap.this.containsKey(iterator.next()))
                {
                    return false;
                }
            }
            return true;
        }

        public MutableDoubleList toList()
        {
            return DoubleArrayList.newList(this);
        }

        public MutableDoubleSet toSet()
        {
            return DoubleHashSet.newSet(this);
        }

        public MutableDoubleBag toBag()
        {
            return DoubleHashBag.newBag(this);
        }

        public LazyDoubleIterable asLazy()
        {
            return new LazyDoubleIterableAdapter(this);
        }

        public DoubleSet freeze()
        {
            throw new UnsupportedOperationException();
        }

        public ImmutableDoubleSet toImmutable()
        {
            return DoubleSets.immutable.withAll(this);
        }

        public int size()
        {
            return DoubleShortHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return DoubleShortHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return DoubleShortHashMap.this.notEmpty();
        }

        @Override
        public boolean equals(Object obj)
        {
            if (this == obj)
            {
                return true;
            }

            if (!(obj instanceof DoubleSet))
            {
                return false;
            }

            DoubleSet other = (DoubleSet) obj;
            return this.size() == other.size() && this.containsAll(other.toArray());
        }

        @Override
        public int hashCode()
        {
            int result = 0;

            if (DoubleShortHashMap.this.sentinelValues != null)
            {
                if (DoubleShortHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += (int) (Double.doubleToLongBits(EMPTY_KEY) ^ Double.doubleToLongBits(EMPTY_KEY) >>> 32);
                }
                if (DoubleShortHashMap.this.sentinelValues.containsOneKey)
                {
                    result += (int) (Double.doubleToLongBits(REMOVED_KEY) ^ Double.doubleToLongBits(REMOVED_KEY) >>> 32);
                }
            }
            for (int i = 0; i < DoubleShortHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(DoubleShortHashMap.this.keys[i]))
                {
                    result += (int) (Double.doubleToLongBits(DoubleShortHashMap.this.keys[i]) ^ Double.doubleToLongBits(DoubleShortHashMap.this.keys[i]) >>> 32);
                }
            }

            return result;
        }

        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (DoubleShortHashMap.this.sentinelValues != null)
                {
                    if (DoubleShortHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (DoubleShortHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (double key : DoubleShortHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }
    }

    public MutableShortCollection values()
    {
        return new ValuesCollection();
    }

    private class ValuesCollection implements MutableShortCollection
    {
        public void clear()
        {
            DoubleShortHashMap.this.clear();
        }

        public MutableShortCollection select(ShortPredicate predicate)
        {
            return DoubleShortHashMap.this.select(predicate);
        }

        public MutableShortCollection reject(ShortPredicate predicate)
        {
            return DoubleShortHashMap.this.reject(predicate);
        }

        public short detectIfNone(ShortPredicate predicate, short ifNone)
        {
            return DoubleShortHashMap.this.detectIfNone(predicate, ifNone);
        }

        public <V> MutableCollection<V> collect(ShortToObjectFunction<? extends V> function)
        {
            return DoubleShortHashMap.this.collect(function);
        }

        public long sum()
        {
            return DoubleShortHashMap.this.sum();
        }

        public short max()
        {
            return DoubleShortHashMap.this.max();
        }

        public short maxIfEmpty(short defaultValue)
        {
            return DoubleShortHashMap.this.maxIfEmpty(defaultValue);
        }

        public short min()
        {
            return DoubleShortHashMap.this.min();
        }

        public short minIfEmpty(short defaultValue)
        {
            return DoubleShortHashMap.this.minIfEmpty(defaultValue);
        }

        public double average()
        {
            return DoubleShortHashMap.this.average();
        }

        public double median()
        {
            return DoubleShortHashMap.this.median();
        }

        public short[] toSortedArray()
        {
            return DoubleShortHashMap.this.toSortedArray();
        }

        public MutableShortList toSortedList()
        {
            return DoubleShortHashMap.this.toSortedList();
        }

        public MutableShortCollection with(short element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableShortCollection without(short element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableShortCollection withAll(ShortIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableShortCollection withoutAll(ShortIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableShortCollection asUnmodifiable()
        {
            return UnmodifiableShortCollection.of(this);
        }

        public MutableShortCollection asSynchronized()
        {
            return SynchronizedShortCollection.of(this);
        }

        public ImmutableShortCollection toImmutable()
        {
            return ShortLists.immutable.withAll(this);
        }

        public boolean contains(short value)
        {
            return DoubleShortHashMap.this.containsValue(value);
        }

        public boolean containsAll(short... source)
        {
            return DoubleShortHashMap.this.containsAll(source);
        }

        public boolean containsAll(ShortIterable source)
        {
            return DoubleShortHashMap.this.containsAll(source);
        }

        public MutableShortList toList()
        {
            return DoubleShortHashMap.this.toList();
        }

        public MutableShortSet toSet()
        {
            return DoubleShortHashMap.this.toSet();
        }

        public MutableShortBag toBag()
        {
            return DoubleShortHashMap.this.toBag();
        }

        public LazyShortIterable asLazy()
        {
            return new LazyShortIterableAdapter(this);
        }

        public boolean isEmpty()
        {
            return DoubleShortHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return DoubleShortHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);

                boolean first = true;

                if (DoubleShortHashMap.this.sentinelValues != null)
                {
                    if (DoubleShortHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(DoubleShortHashMap.this.sentinelValues.zeroValue));
                        first = false;
                    }
                    if (DoubleShortHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(DoubleShortHashMap.this.sentinelValues.oneValue));
                        first = false;
                    }
                }
                for (int i = 0; i < DoubleShortHashMap.this.keys.length; i++)
                {
                    double key = DoubleShortHashMap.this.keys[i];
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(DoubleShortHashMap.this.values[i]));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public ShortIterator shortIterator()
        {
            return DoubleShortHashMap.this.shortIterator();
        }

        public void forEach(ShortProcedure procedure)
        {
            DoubleShortHashMap.this.forEach(procedure);
        }

        public int count(ShortPredicate predicate)
        {
            return DoubleShortHashMap.this.count(predicate);
        }

        public boolean anySatisfy(ShortPredicate predicate)
        {
            return DoubleShortHashMap.this.anySatisfy(predicate);
        }

        public boolean allSatisfy(ShortPredicate predicate)
        {
            return DoubleShortHashMap.this.allSatisfy(predicate);
        }

        public boolean noneSatisfy(ShortPredicate predicate)
        {
            return DoubleShortHashMap.this.noneSatisfy(predicate);
        }

        public boolean add(short element)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(short... source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(ShortIterable source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(short item)
        {
            int oldSize = DoubleShortHashMap.this.size();

            if (DoubleShortHashMap.this.sentinelValues != null)
            {
                if (DoubleShortHashMap.this.sentinelValues.containsZeroKey && item == DoubleShortHashMap.this.sentinelValues.zeroValue)
                {
                    DoubleShortHashMap.this.removeKey(EMPTY_KEY);
                }
                if (DoubleShortHashMap.this.sentinelValues.containsOneKey && item == DoubleShortHashMap.this.sentinelValues.oneValue)
                {
                    DoubleShortHashMap.this.removeKey(REMOVED_KEY);
                }
            }
            for (int i = 0; i < DoubleShortHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(DoubleShortHashMap.this.keys[i]) && item == DoubleShortHashMap.this.values[i])
                {
                    DoubleShortHashMap.this.removeKey(DoubleShortHashMap.this.keys[i]);
                }
            }
            return oldSize != DoubleShortHashMap.this.size();
        }

        public boolean removeAll(ShortIterable source)
        {
            int oldSize = DoubleShortHashMap.this.size();

            ShortIterator iterator = source.shortIterator();
            while (iterator.hasNext())
            {
                this.remove(iterator.next());
            }
            return oldSize != DoubleShortHashMap.this.size();
        }

        public boolean removeAll(short... source)
        {
            int oldSize = DoubleShortHashMap.this.size();

            for (short item : source)
            {
                this.remove(item);
            }
            return oldSize != DoubleShortHashMap.this.size();
        }

        public int size()
        {
            return DoubleShortHashMap.this.size();
        }

        public short[] toArray()
        {
            return DoubleShortHashMap.this.toArray();
        }
    }
}
