/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.map.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.FloatIterable;
import com.gs.collections.api.LazyFloatIterable;
import com.gs.collections.api.LazyCharIterable;
import com.gs.collections.api.CharIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.bag.primitive.MutableFloatBag;
import com.gs.collections.api.bag.primitive.MutableCharBag;
import com.gs.collections.api.block.function.primitive.FloatToObjectFunction;
import com.gs.collections.api.block.function.primitive.FloatToCharFunction;
import com.gs.collections.api.block.function.primitive.CharFunction;
import com.gs.collections.api.block.function.primitive.CharFunction0;
import com.gs.collections.api.block.function.primitive.CharToCharFunction;
import com.gs.collections.api.block.function.primitive.CharToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectCharToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.FloatCharPredicate;
import com.gs.collections.api.block.predicate.primitive.FloatPredicate;
import com.gs.collections.api.block.predicate.primitive.CharPredicate;
import com.gs.collections.api.block.procedure.primitive.FloatProcedure;
import com.gs.collections.api.block.procedure.primitive.FloatCharProcedure;
import com.gs.collections.api.block.procedure.primitive.CharProcedure;
import com.gs.collections.api.collection.MutableCollection;
import com.gs.collections.api.collection.primitive.ImmutableCharCollection;
import com.gs.collections.api.collection.primitive.MutableCharCollection;
import com.gs.collections.api.iterator.FloatIterator;
import com.gs.collections.api.iterator.CharIterator;
import com.gs.collections.api.list.primitive.MutableFloatList;
import com.gs.collections.api.list.primitive.MutableCharList;
import com.gs.collections.api.map.primitive.FloatCharMap;
import com.gs.collections.api.map.primitive.ImmutableFloatCharMap;
import com.gs.collections.api.map.primitive.MutableFloatCharMap;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.FloatSet;
import com.gs.collections.api.set.primitive.ImmutableFloatSet;
import com.gs.collections.api.set.primitive.MutableFloatSet;
import com.gs.collections.api.set.primitive.MutableCharSet;
import com.gs.collections.impl.bag.mutable.primitive.FloatHashBag;
import com.gs.collections.impl.bag.mutable.primitive.CharHashBag;
import com.gs.collections.impl.block.factory.primitive.FloatPredicates;
import com.gs.collections.impl.collection.mutable.primitive.SynchronizedCharCollection;
import com.gs.collections.impl.collection.mutable.primitive.UnmodifiableCharCollection;
import com.gs.collections.impl.factory.Sets;
import com.gs.collections.impl.factory.primitive.CharLists;
import com.gs.collections.impl.factory.primitive.FloatCharMaps;
import com.gs.collections.impl.factory.primitive.FloatSets;
import com.gs.collections.impl.lazy.primitive.CollectFloatToObjectIterable;
import com.gs.collections.impl.lazy.primitive.LazyCharIterableAdapter;
import com.gs.collections.impl.lazy.primitive.LazyFloatIterableAdapter;
import com.gs.collections.impl.lazy.primitive.SelectFloatIterable;
import com.gs.collections.impl.list.mutable.FastList;
import com.gs.collections.impl.list.mutable.primitive.FloatArrayList;
import com.gs.collections.impl.list.mutable.primitive.CharArrayList;
import com.gs.collections.impl.set.mutable.primitive.FloatHashSet;
import com.gs.collections.impl.set.mutable.primitive.SynchronizedFloatSet;
import com.gs.collections.impl.set.mutable.primitive.UnmodifiableFloatSet;
import com.gs.collections.impl.set.mutable.primitive.CharHashSet;

/**
 * This file was automatically generated from template file primitivePrimitiveHashMap.stg.
 *
 * @since 3.0.
 */
public class FloatCharHashMap implements MutableFloatCharMap, Externalizable
{
    static final char EMPTY_VALUE = (char) 0;
    private static final long serialVersionUID = 1L;
    private static final float EMPTY_KEY = 0.0f;
    private static final float REMOVED_KEY = 1.0f;

    private static final int OCCUPIED_DATA_RATIO = 2;
    private static final int OCCUPIED_SENTINEL_RATIO = 4;
    private static final int DEFAULT_INITIAL_CAPACITY = 8;

    private float[] keys;
    private char[] values;

    private int occupiedWithData;
    private int occupiedWithSentinels;

    private SentinelValues sentinelValues;

    public FloatCharHashMap()
    {
        this.allocateTable(DEFAULT_INITIAL_CAPACITY << 1);
    }

    public FloatCharHashMap(int initialCapacity)
    {
        if (initialCapacity < 0)
        {
            throw new IllegalArgumentException("initial capacity cannot be less than 0");
        }
        int capacity = this.smallestPowerOfTwoGreaterThan(this.fastCeil(initialCapacity * OCCUPIED_DATA_RATIO));
        this.allocateTable(capacity);
    }

    private int smallestPowerOfTwoGreaterThan(int n)
    {
        return n > 1 ? Integer.highestOneBit(n - 1) << 1 : 1;
    }

    public FloatCharHashMap(FloatCharMap map)
    {
        this(Math.max(map.size(), DEFAULT_INITIAL_CAPACITY));
        this.putAll(map);
    }

    private int fastCeil(float v)
    {
        int possibleResult = (int) v;
        if (v - possibleResult > 0.0F)
        {
            possibleResult++;
        }
        return possibleResult;
    }

    public static FloatCharHashMap newWithKeysValues(float key1, char value1)
    {
        return new FloatCharHashMap(1).withKeyValue(key1, value1);
    }

    public static FloatCharHashMap newWithKeysValues(float key1, char value1, float key2, char value2)
    {
        return new FloatCharHashMap(2).withKeysValues(key1, value1, key2, value2);
    }

    public static FloatCharHashMap newWithKeysValues(float key1, char value1, float key2, char value2, float key3, char value3)
    {
        return new FloatCharHashMap(3).withKeysValues(key1, value1, key2, value2, key3, value3);
    }

    public static FloatCharHashMap newWithKeysValues(float key1, char value1, float key2, char value2, float key3, char value3, float key4, char value4)
    {
        return new FloatCharHashMap(4).withKeysValues(key1, value1, key2, value2, key3, value3, key4, value4);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }

        if (!(obj instanceof FloatCharMap))
        {
            return false;
        }

        FloatCharMap other = (FloatCharMap) obj;

        if (this.size() != other.size())
        {
            return false;
        }

        if (this.sentinelValues == null)
        {
            if (other.containsKey(EMPTY_KEY) || other.containsKey(REMOVED_KEY))
            {
                return false;
            }
        }
        else
        {
            if (this.sentinelValues.containsZeroKey && (!other.containsKey(EMPTY_KEY) || this.sentinelValues.zeroValue != other.getOrThrow(EMPTY_KEY)))
            {
                return false;
            }

            if (this.sentinelValues.containsOneKey && (!other.containsKey(REMOVED_KEY) || this.sentinelValues.oneValue != other.getOrThrow(REMOVED_KEY)))
            {
                return false;
            }
        }

        for (int i = 0; i < this.keys.length; i++)
        {
            float key = this.keys[i];
            if (isNonSentinel(key) && (!other.containsKey(key) || this.values[i] != other.getOrThrow(key)))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int result = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += Float.floatToIntBits(EMPTY_KEY) ^ (int) this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += Float.floatToIntBits(REMOVED_KEY) ^ (int) this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += Float.floatToIntBits(this.keys[i]) ^ (int) this.values[i];
            }
        }

        return result;
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public int size()
    {
        return this.occupiedWithData + (this.sentinelValues == null ? 0 : this.sentinelValues.size());
    }

    public boolean isEmpty()
    {
        return this.occupiedWithData == 0 && (this.sentinelValues == null || this.sentinelValues.size() == 0);
    }

    public boolean notEmpty()
    {
        return this.occupiedWithData != 0 || (this.sentinelValues != null && this.sentinelValues.size() != 0);
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            boolean first = true;

            if (this.sentinelValues != null)
            {
                if (this.sentinelValues.containsZeroKey)
                {
                    appendable.append(String.valueOf(EMPTY_KEY)).append("=").append(String.valueOf(this.sentinelValues.zeroValue));
                    first = false;
                }
                if (this.sentinelValues.containsOneKey)
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(REMOVED_KEY)).append("=").append(String.valueOf(this.sentinelValues.oneValue));
                    first = false;
                }
            }
            for (int i = 0; i < this.keys.length; i++)
            {
                float key = this.keys[i];
                if (isNonSentinel(key))
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(key)).append("=").append(String.valueOf(this.values[i]));
                    first = false;
                }
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public CharIterator charIterator()
    {
        return new InternalCharIterator();
    }

    public char[] toArray()
    {
        char[] array = new char[this.size()];
        int index = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                array[index] = this.sentinelValues.zeroValue;
                index++;
            }
            if (this.sentinelValues.containsOneKey)
            {
                array[index] = this.sentinelValues.oneValue;
                index++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                array[index] = this.values[i];
                index++;
            }
        }

        return array;
    }

    public boolean contains(char value)
    {
        return this.containsValue(value);
    }

    public boolean containsAll(char... source)
    {
        for (char each : source)
        {
            if (!this.contains(each))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(CharIterable source)
    {
        return source.allSatisfy(new CharPredicate()
        {
            public boolean accept(char value)
            {
                return FloatCharHashMap.this.contains(value);
            }
        });
    }

    public void forEach(CharProcedure procedure)
    {
        this.forEachValue(procedure);
    }

    public MutableCharCollection select(CharPredicate predicate)
    {
        CharArrayList result = new CharArrayList();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }

        return result;
    }

    public MutableCharCollection reject(CharPredicate predicate)
    {
        CharArrayList result = new CharArrayList();
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public <V> MutableCollection<V> collect(CharToObjectFunction<? extends V> function)
    {
        FastList<V> target = FastList.newList(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                target.add(function.valueOf(this.sentinelValues.zeroValue));
            }
            if (this.sentinelValues.containsOneKey)
            {
                target.add(function.valueOf(this.sentinelValues.oneValue));
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                target.add(function.valueOf(this.values[i]));
            }
        }
        return target;
    }

    public char detectIfNone(CharPredicate predicate, char value)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return this.values[i];
            }
        }
        return value;
    }

    public int count(CharPredicate predicate)
    {
        int count = 0;
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                count++;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                count++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(CharPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return true;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return true;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(CharPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(CharPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public <V> V injectInto(V injectedValue, ObjectCharToObjectFunction<? super V, ? extends V> function)
    {
        V result = injectedValue;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result = function.valueOf(result, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                result = function.valueOf(result, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result = function.valueOf(result, this.values[i]);
            }
        }

        return result;
    }

    public MutableCharList toList()
    {
        return CharArrayList.newList(this);
    }

    public MutableCharSet toSet()
    {
        return CharHashSet.newSet(this);
    }

    public MutableCharBag toBag()
    {
        return CharHashBag.newBag(this);
    }

    public LazyCharIterable asLazy()
    {
        return new LazyCharIterableAdapter(this);
    }

    public void clear()
    {
        this.sentinelValues = null;
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;
        Arrays.fill(this.keys, EMPTY_KEY);
        Arrays.fill(this.values, EMPTY_VALUE);
    }

    public void put(float key, char value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return;
        }

        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return;
        }

        int index = this.probe(key);

        if (Float.compare(this.keys[index], key) == 0)
        {
            // key already present in map
            this.values[index] = value;
            return;
        }

        if (Float.compare(this.keys[index], REMOVED_KEY) == 0)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = key;
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
    }

    public void putAll(FloatCharMap map)
    {
        map.forEachKeyValue(new FloatCharProcedure()
        {
            public void value(float key, char value)
            {
                FloatCharHashMap.this.put(key, value);
            }
        });
    }

    public void removeKey(float key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return;
            }
            if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.containsZeroKey = false;
                this.sentinelValues.zeroValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.containsOneKey = false;
                this.sentinelValues.oneValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return;
        }
        int index = this.probe(key);
        if (Float.compare(this.keys[index], key) == 0)
        {
            this.keys[index] = REMOVED_KEY;
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }
        }
    }

    public void remove(float key)
    {
        this.removeKey(key);
    }

    public char removeKeyIfAbsent(float key, char value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return value;
            }
            char oldValue = this.sentinelValues.zeroValue;
            if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.containsZeroKey = false;
                this.sentinelValues.zeroValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return oldValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return value;
            }
            char oldValue = this.sentinelValues.oneValue;
            if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.containsOneKey = false;
                this.sentinelValues.oneValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return oldValue;
        }
        int index = this.probe(key);
        if (Float.compare(this.keys[index], key) == 0)
        {
            this.keys[index] = REMOVED_KEY;
            char oldValue = this.values[index];
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }

            return oldValue;
        }
        return value;
    }

    public char getIfAbsentPut(float key, char value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (Float.compare(this.keys[index], key) == 0)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public char getIfAbsentPut(float key, CharFunction0 function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                char value = function.value();
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            char value = function.value();
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                char value = function.value();
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            char value = function.value();
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (Float.compare(this.keys[index], key) == 0)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        char value = function.value();
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public <P> char getIfAbsentPutWith(float key, CharFunction<? super P> function, P parameter)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                char value = function.charValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            char value = function.charValueOf(parameter);
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                char value = function.charValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            char value = function.charValueOf(parameter);
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (Float.compare(this.keys[index], key) == 0)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        char value = function.charValueOf(parameter);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public char getIfAbsentPutWithKey(float key, FloatToCharFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                char value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            char value = function.valueOf(key);
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                char value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            char value = function.valueOf(key);
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (Float.compare(this.keys[index], key) == 0)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        char value = function.valueOf(key);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public char updateValue(float key, char initialValueIfAbsent, CharToCharFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = function.valueOf(initialValueIfAbsent);
            }
            else if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.zeroValue = function.valueOf(this.sentinelValues.zeroValue);
            }
            else
            {
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = function.valueOf(initialValueIfAbsent);
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = function.valueOf(initialValueIfAbsent);
            }
            else if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.oneValue = function.valueOf(this.sentinelValues.oneValue);
            }
            else
            {
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = function.valueOf(initialValueIfAbsent);
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (Float.compare(this.keys[index], key) == 0)
        {
            this.values[index] = function.valueOf(this.values[index]);
            return this.values[index];
        }
        if (Float.compare(this.keys[index], REMOVED_KEY) == 0)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = key;
        char value = function.valueOf(initialValueIfAbsent);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public FloatCharHashMap withKeyValue(float key1, char value1)
    {
        this.put(key1, value1);
        return this;
    }

    public FloatCharHashMap withKeysValues(float key1, char value1, float key2, char value2)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        return this;
    }

    public FloatCharHashMap withKeysValues(float key1, char value1, float key2, char value2, float key3, char value3)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        return this;
    }

    public FloatCharHashMap withKeysValues(float key1, char value1, float key2, char value2, float key3, char value3, float key4, char value4)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        this.put(key4, value4);
        return this;
    }

    public FloatCharHashMap withoutKey(float key)
    {
        this.removeKey(key);
        return this;
    }

    public FloatCharHashMap withoutAllKeys(FloatIterable keys)
    {
        keys.forEach(new FloatProcedure()
        {
            public void value(float key)
            {
                FloatCharHashMap.this.removeKey(key);
            }
        });
        return this;
    }

    public MutableFloatCharMap asUnmodifiable()
    {
        return new UnmodifiableFloatCharMap(this);
    }

    public MutableFloatCharMap asSynchronized()
    {
        return new SynchronizedFloatCharMap(this);
    }

    public ImmutableFloatCharMap toImmutable()
    {
        return FloatCharMaps.immutable.ofAll(this);
    }

    public char get(float key)
    {
        return this.getIfAbsent(key, EMPTY_VALUE);
    }

    public char getIfAbsent(float key, char ifAbsent)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (Float.compare(this.keys[index], key) == 0)
        {
            return this.values[index];
        }
        return ifAbsent;
    }

    public char getOrThrow(float key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]))
        {
            return this.values[index];
        }
        throw new IllegalStateException("Key " + key + " not present.");
    }

    public boolean containsKey(float key)
    {
        if (isEmptyKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsZeroKey;
        }
        if (isRemovedKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsOneKey;
        }
        return Float.compare(this.keys[this.probe(key)], key) == 0;
    }

    public boolean containsValue(char value)
    {
        if (this.sentinelValues != null && this.sentinelValues.containsValue(value))
        {
            return true;
        }
        for (int i = 0; i < this.values.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && this.values[i] == value)
            {
                return true;
            }
        }
        return false;
    }

    public void forEachValue(CharProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.values[i]);
            }
        }
    }

    public void forEachKey(FloatProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY);
            }
        }
        for (float key : this.keys)
        {
            if (isNonSentinel(key))
            {
                procedure.value(key);
            }
        }
    }

    public void forEachKeyValue(FloatCharProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.keys[i], this.values[i]);
            }
        }
    }

    public LazyFloatIterable keysView()
    {
        return new KeysView();
    }

    public FloatCharHashMap select(FloatCharPredicate predicate)
    {
        FloatCharHashMap result = new FloatCharHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.keys[i], this.values[i]))
            {
                result.put(this.keys[i], this.values[i]);
            }
        }

        return result;
    }

    public FloatCharHashMap reject(FloatCharPredicate predicate)
    {
        FloatCharHashMap result = new FloatCharHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.keys[i], this.values[i]))
            {
                result.put(this.keys[i], this.values[i]);
            }
        }
        return result;
    }

    public long sum()
    {
        long result = 0L;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += this.values[i];
            }
        }

        return result;
    }

    public char max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        CharIterator iterator = this.charIterator();
        char max = iterator.next();
        while (iterator.hasNext())
        {
            char value = iterator.next();
            if (max < value)
            {
                max = value;
            }
        }
        return max;
    }

    public char maxIfEmpty(char defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public char min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        CharIterator iterator = this.charIterator();
        char min = iterator.next();
        while (iterator.hasNext())
        {
            char value = iterator.next();
            if (value < min)
            {
                min = value;
            }
        }
        return min;
    }

    public char minIfEmpty(char defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return (double)this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        char[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            char first = sortedArray[middleIndex];
            char second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public char[] toSortedArray()
    {
        char[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public MutableCharList toSortedList()
    {
        return CharArrayList.newList(this).sortThis();
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                out.writeFloat(EMPTY_KEY);
                out.writeChar(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                out.writeFloat(REMOVED_KEY);
                out.writeChar(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                out.writeFloat(this.keys[i]);
                out.writeChar(this.values[i]);
            }
        }
    }

    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
    {
        int size = in.readInt();
        for (int i = 0; i < size; i++)
        {
            this.put(in.readFloat(), in.readChar());
        }
    }

    /**
     * Rehashes every element in the set into a new backing table of the smallest possible size and eliminating removed sentinels.
     */
    public void compact()
    {
        this.rehash(this.smallestPowerOfTwoGreaterThan(this.size()));
    }

    private void rehash()
    {
        this.rehash(this.keys.length);
    }

    private void rehashAndGrow()
    {
        this.rehash(this.keys.length << 1);
    }

    private void rehash(int newCapacity)
    {
        int oldLength = this.keys.length;
        float[] old = this.keys;
        char[] oldValues = this.values;
        this.allocateTable(newCapacity);
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;

        for (int i = 0; i < oldLength; i++)
        {
            if (isNonSentinel(old[i]))
            {
                this.put(old[i], oldValues[i]);
            }
        }
    }

    // exposed for testing
    int probe(float element)
    {
        int index = this.spread(element);
        float keyAtIndex = this.keys[index];

        if (Float.compare(keyAtIndex, element) == 0 || Float.compare(keyAtIndex, EMPTY_KEY) == 0)
        {
            return index;
        }

        int removedIndex = Float.compare(keyAtIndex, REMOVED_KEY) == 0 ? index : -1;
        int nextIndex = index;
        int probe = 17;

        // loop until an empty slot is reached
        while (true)
        {
            // Probe algorithm: 17*n*(n+1)/2 where n = number of collisions
            nextIndex += probe;
            probe += 17;
            nextIndex &= this.keys.length - 1;

            if (Float.compare(this.keys[nextIndex], element) == 0)
            {
                return nextIndex;
            }
            if (Float.compare(this.keys[nextIndex], REMOVED_KEY) == 0)
            {
                if (removedIndex == -1)
                {
                    removedIndex = nextIndex;
                }
            }
            else if (Float.compare(this.keys[nextIndex], EMPTY_KEY) == 0)
            {
                return removedIndex == -1 ? nextIndex : removedIndex;
            }
        }
    }

    // exposed for testing
    int spread(float element)
    {
        int code = Float.floatToIntBits(element);
        code ^= 61 ^ (code >> 16);
        code += code << 3;
        code ^= code >> 4;
        code *= 0x27d4eb2d;
        code ^= code >> 15;
        return code & (this.keys.length - 1);
    }

    private void allocateTable(int sizeToAllocate)
    {
        this.keys = new float[sizeToAllocate];
        this.values = new char[sizeToAllocate];
    }

    private static boolean isEmptyKey(float key)
    {
        return Float.compare(key, EMPTY_KEY) == 0;
    }

    private static boolean isRemovedKey(float key)
    {
        return Float.compare(key, REMOVED_KEY) == 0;
    }

    private static boolean isNonSentinel(float key)
    {
        return !isEmptyKey(key) && !isRemovedKey(key);
    }

    private int maxOccupiedWithData()
    {
        int capacity = this.keys.length;
        // need at least one free slot for open addressing
        return Math.min(capacity - 1, capacity / OCCUPIED_DATA_RATIO);
    }

    private int maxOccupiedWithSentinels()
    {
        return this.keys.length / OCCUPIED_SENTINEL_RATIO;
    }

    private static final class SentinelValues
    {
        private boolean containsZeroKey;
        private boolean containsOneKey;
        private char zeroValue;
        private char oneValue;

        public int size()
        {
            return (this.containsZeroKey ? 1 : 0) + (this.containsOneKey ? 1 : 0);
        }

        public boolean containsValue(char value)
        {
            boolean valueEqualsZeroValue = this.containsZeroKey && this.zeroValue == value;
            boolean valueEqualsOneValue = this.containsOneKey && this.oneValue == value;
            return valueEqualsZeroValue || valueEqualsOneValue;
        }
    }

    private class InternalCharIterator implements CharIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < FloatCharHashMap.this.size();
        }

        public char next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (FloatCharHashMap.this.containsKey(EMPTY_KEY))
                {
                    return FloatCharHashMap.this.get(EMPTY_KEY);
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (FloatCharHashMap.this.containsKey(REMOVED_KEY))
                {
                    return FloatCharHashMap.this.get(REMOVED_KEY);
                }
            }

            float[] keys = FloatCharHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            char result = FloatCharHashMap.this.values[this.position];
            this.position++;
            return result;
        }
    }

    private class KeysView implements LazyFloatIterable
    {
        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public int size()
        {
            return FloatCharHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return FloatCharHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return FloatCharHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (FloatCharHashMap.this.sentinelValues != null)
                {
                    if (FloatCharHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (FloatCharHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (float key : FloatCharHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public FloatIterator floatIterator()
        {
            return new KeySetIterator();
        }

        public boolean contains(float key)
        {
            return FloatCharHashMap.this.containsKey(key);
        }

        public boolean containsAll(float... keys)
        {
            for (float key : keys)
            {
                if (!FloatCharHashMap.this.containsKey(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(FloatIterable source)
        {
            return source.allSatisfy(new FloatPredicate()
            {
                public boolean accept(float key)
                {
                    return FloatCharHashMap.this.containsKey(key);
                }
            });
        }

        public void forEach(FloatProcedure procedure)
        {
            FloatCharHashMap.this.forEachKey(procedure);
        }

        public int count(FloatPredicate predicate)
        {
            int count = 0;
            if (FloatCharHashMap.this.sentinelValues != null)
            {
                if (FloatCharHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (FloatCharHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (float key : FloatCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(FloatPredicate predicate)
        {
            if (FloatCharHashMap.this.sentinelValues != null)
            {
                if (FloatCharHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (FloatCharHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (float key : FloatCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(FloatPredicate predicate)
        {
            if (FloatCharHashMap.this.sentinelValues != null)
            {
                if (FloatCharHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (FloatCharHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (float key : FloatCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(FloatPredicate predicate)
        {
            if (FloatCharHashMap.this.sentinelValues != null)
            {
                if (FloatCharHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (FloatCharHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (float key : FloatCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public LazyFloatIterable select(FloatPredicate predicate)
        {
            return new SelectFloatIterable(this, predicate);
        }

        public LazyFloatIterable reject(FloatPredicate predicate)
        {
            return new SelectFloatIterable(this, FloatPredicates.not(predicate));
        }

        public float detectIfNone(FloatPredicate predicate, float value)
        {
            if (FloatCharHashMap.this.sentinelValues != null)
            {
                if (FloatCharHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (FloatCharHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (float key : FloatCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return value;
        }

        public <V> LazyIterable<V> collect(FloatToObjectFunction<? extends V> function)
        {
            return new CollectFloatToObjectIterable<V>(this, function);
        }

        public double sum()
        {
            double result = 0.0;

            if (FloatCharHashMap.this.sentinelValues != null)
            {
                if (FloatCharHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += EMPTY_KEY;
                }
                if (FloatCharHashMap.this.sentinelValues.containsOneKey)
                {
                    result += REMOVED_KEY;
                }
            }
            for (float key : FloatCharHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result += key;
                }
            }

            return result;
        }

        public float max()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            FloatIterator iterator = this.floatIterator();
            float max = iterator.next();
            while (iterator.hasNext())
            {
                float value = iterator.next();
                if (Float.compare(max, value) < 0)
                {
                    max = value;
                }
            }
            return max;
        }

        public float min()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            FloatIterator iterator = this.floatIterator();
            float min = iterator.next();
            while (iterator.hasNext())
            {
                float value = iterator.next();
                if (Float.compare(value, min) < 0)
                {
                    min = value;
                }
            }
            return min;
        }

        public float minIfEmpty(float defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.min();
        }

        public float maxIfEmpty(float defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.max();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            float[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                float first = sortedArray[middleIndex];
                float second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public float[] toSortedArray()
        {
            float[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public float[] toArray()
        {
            int size = FloatCharHashMap.this.size();
            final float[] result = new float[size];
            FloatCharHashMap.this.forEachKey(new FloatProcedure()
            {
                private int index;

                public void value(float each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public MutableFloatList toList()
        {
            return FloatArrayList.newList(this);
        }

        public MutableFloatList toSortedList()
        {
            return FloatArrayList.newList(this).sortThis();
        }

        public MutableFloatSet toSet()
        {
            return FloatHashSet.newSet(this);
        }

        public MutableFloatBag toBag()
        {
            return FloatHashBag.newBag(this);
        }

        public LazyFloatIterable asLazy()
        {
            return this;
        }
    }

    private class KeySetIterator implements FloatIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < FloatCharHashMap.this.size();
        }

        public float next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (FloatCharHashMap.this.containsKey(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (FloatCharHashMap.this.containsKey(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }

            float[] keys = FloatCharHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            float result = keys[this.position];
            this.position++;
            return result;
        }
    }

    public MutableFloatSet keySet()
    {
        return new KeySet();
    }

    private class KeySet implements MutableFloatSet
    {
        public FloatIterator floatIterator()
        {
            return new KeySetIterator();
        }

        public void forEach(FloatProcedure procedure)
        {
            FloatCharHashMap.this.forEachKey(procedure);
        }

        public int count(FloatPredicate predicate)
        {
            int count = 0;
            if (FloatCharHashMap.this.sentinelValues != null)
            {
                if (FloatCharHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (FloatCharHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (float key : FloatCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(FloatPredicate predicate)
        {
            if (FloatCharHashMap.this.sentinelValues != null)
            {
                if (FloatCharHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (FloatCharHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (float key : FloatCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(FloatPredicate predicate)
        {
            if (FloatCharHashMap.this.sentinelValues != null)
            {
                if (FloatCharHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (FloatCharHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (float key : FloatCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(FloatPredicate predicate)
        {
            if (FloatCharHashMap.this.sentinelValues != null)
            {
                if (FloatCharHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (FloatCharHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (float key : FloatCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean add(float element)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(float... source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(FloatIterable source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(float key)
        {
            int oldSize = FloatCharHashMap.this.size();
            FloatCharHashMap.this.removeKey(key);
            return oldSize != FloatCharHashMap.this.size();
        }

        public boolean removeAll(FloatIterable source)
        {
            int oldSize = FloatCharHashMap.this.size();
            FloatIterator iterator = source.floatIterator();
            while (iterator.hasNext())
            {
                FloatCharHashMap.this.removeKey(iterator.next());
            }
            return oldSize != FloatCharHashMap.this.size();
        }

        public boolean removeAll(float... source)
        {
            int oldSize = FloatCharHashMap.this.size();
            for (float item : source)
            {
                FloatCharHashMap.this.removeKey(item);
            }
            return oldSize != FloatCharHashMap.this.size();
        }

        public void clear()
        {
            FloatCharHashMap.this.clear();
        }

        public MutableFloatSet select(FloatPredicate predicate)
        {
            MutableFloatSet result = new FloatHashSet();
            if (FloatCharHashMap.this.sentinelValues != null)
            {
                if (FloatCharHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (FloatCharHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (float key : FloatCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableFloatSet reject(FloatPredicate predicate)
        {
            MutableFloatSet result = new FloatHashSet();
            if (FloatCharHashMap.this.sentinelValues != null)
            {
                if (FloatCharHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (FloatCharHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (float key : FloatCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableFloatSet with(float element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableFloatSet without(float element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableFloatSet withAll(FloatIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableFloatSet withoutAll(FloatIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public float detectIfNone(FloatPredicate predicate, float ifNone)
        {
            if (FloatCharHashMap.this.sentinelValues != null)
            {
                if (FloatCharHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (FloatCharHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (float key : FloatCharHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return ifNone;
        }

        public <V> MutableSet<V> collect(FloatToObjectFunction<? extends V> function)
        {
            MutableSet<V> result = Sets.mutable.with();
            if (FloatCharHashMap.this.sentinelValues != null)
            {
                if (FloatCharHashMap.this.sentinelValues.containsZeroKey)
                {
                    result.add(function.valueOf(EMPTY_KEY));
                }
                if (FloatCharHashMap.this.sentinelValues.containsOneKey)
                {
                    result.add(function.valueOf(REMOVED_KEY));
                }
            }
            for (float key : FloatCharHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result.add(function.valueOf(key));
                }
            }
            return result;
        }

        public MutableFloatSet asUnmodifiable()
        {
            return UnmodifiableFloatSet.of(this);
        }

        public MutableFloatSet asSynchronized()
        {
            return SynchronizedFloatSet.of(this);
        }

        public double sum()
        {
            double sum = 0.0;
            if (FloatCharHashMap.this.sentinelValues != null)
            {
                if (FloatCharHashMap.this.sentinelValues.containsZeroKey)
                {
                    sum += EMPTY_KEY;
                }
                if (FloatCharHashMap.this.sentinelValues.containsOneKey)
                {
                    sum += REMOVED_KEY;
                }
            }
            for (float key : FloatCharHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    sum += key;
                }
            }
            return sum;
        }

        public float max()
        {
            if (FloatCharHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            float max = 0;
            boolean isMaxSet = false;

            if (FloatCharHashMap.this.sentinelValues != null)
            {
                if (FloatCharHashMap.this.sentinelValues.containsZeroKey)
                {
                    max = EMPTY_KEY;
                    isMaxSet = true;
                }
                if (FloatCharHashMap.this.sentinelValues.containsOneKey && (!isMaxSet || Float.compare(max, REMOVED_KEY) < 0))
                {
                    max = REMOVED_KEY;
                    isMaxSet = true;
                }
            }
            for (int i = 0; i < FloatCharHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(FloatCharHashMap.this.keys[i]) && (!isMaxSet || Float.compare(max, FloatCharHashMap.this.keys[i]) < 0))
                {
                    max = FloatCharHashMap.this.keys[i];
                    isMaxSet = true;
                }
            }
            return max;
        }

        public float maxIfEmpty(float defaultValue)
        {
            if (FloatCharHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.max();
        }

        public float min()
        {
            if (FloatCharHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            float min = 0;
            boolean isMinSet = false;

            if (FloatCharHashMap.this.sentinelValues != null)
            {
                if (FloatCharHashMap.this.sentinelValues.containsZeroKey)
                {
                    min = EMPTY_KEY;
                    isMinSet = true;
                }
                if (FloatCharHashMap.this.sentinelValues.containsOneKey && (!isMinSet || Float.compare(REMOVED_KEY, min) < 0))
                {
                    min = REMOVED_KEY;
                    isMinSet = true;
                }
            }
            for (int i = 0; i < FloatCharHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(FloatCharHashMap.this.keys[i]) && (!isMinSet || Float.compare(FloatCharHashMap.this.keys[i], min) < 0))
                {
                    min = FloatCharHashMap.this.keys[i];
                    isMinSet = true;
                }
            }
            return min;
        }

        public float minIfEmpty(float defaultValue)
        {
            if (FloatCharHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.min();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            float[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                float first = sortedArray[middleIndex];
                float second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public float[] toSortedArray()
        {
            float[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public MutableFloatList toSortedList()
        {
            return FloatArrayList.newList(this).sortThis();
        }

        public float[] toArray()
        {
            int size = FloatCharHashMap.this.size();
            final float[] result = new float[size];
            FloatCharHashMap.this.forEachKey(new FloatProcedure()
            {
                private int index;

                public void value(float each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public boolean contains(float value)
        {
            return FloatCharHashMap.this.containsKey(value);
        }

        public boolean containsAll(float... source)
        {
            for (float item : source)
            {
                if (!FloatCharHashMap.this.containsKey(item))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(FloatIterable source)
        {
            FloatIterator iterator = source.floatIterator();
            while (iterator.hasNext())
            {
                if (!FloatCharHashMap.this.containsKey(iterator.next()))
                {
                    return false;
                }
            }
            return true;
        }

        public MutableFloatList toList()
        {
            return FloatArrayList.newList(this);
        }

        public MutableFloatSet toSet()
        {
            return FloatHashSet.newSet(this);
        }

        public MutableFloatBag toBag()
        {
            return FloatHashBag.newBag(this);
        }

        public LazyFloatIterable asLazy()
        {
            return new LazyFloatIterableAdapter(this);
        }

        public FloatSet freeze()
        {
            throw new UnsupportedOperationException();
        }

        public ImmutableFloatSet toImmutable()
        {
            return FloatSets.immutable.withAll(this);
        }

        public int size()
        {
            return FloatCharHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return FloatCharHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return FloatCharHashMap.this.notEmpty();
        }

        @Override
        public boolean equals(Object obj)
        {
            if (this == obj)
            {
                return true;
            }

            if (!(obj instanceof FloatSet))
            {
                return false;
            }

            FloatSet other = (FloatSet) obj;
            return this.size() == other.size() && this.containsAll(other.toArray());
        }

        @Override
        public int hashCode()
        {
            int result = 0;

            if (FloatCharHashMap.this.sentinelValues != null)
            {
                if (FloatCharHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += Float.floatToIntBits(EMPTY_KEY);
                }
                if (FloatCharHashMap.this.sentinelValues.containsOneKey)
                {
                    result += Float.floatToIntBits(REMOVED_KEY);
                }
            }
            for (int i = 0; i < FloatCharHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(FloatCharHashMap.this.keys[i]))
                {
                    result += Float.floatToIntBits(FloatCharHashMap.this.keys[i]);
                }
            }

            return result;
        }

        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (FloatCharHashMap.this.sentinelValues != null)
                {
                    if (FloatCharHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (FloatCharHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (float key : FloatCharHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }
    }

    public MutableCharCollection values()
    {
        return new ValuesCollection();
    }

    private class ValuesCollection implements MutableCharCollection
    {
        public void clear()
        {
            FloatCharHashMap.this.clear();
        }

        public MutableCharCollection select(CharPredicate predicate)
        {
            return FloatCharHashMap.this.select(predicate);
        }

        public MutableCharCollection reject(CharPredicate predicate)
        {
            return FloatCharHashMap.this.reject(predicate);
        }

        public char detectIfNone(CharPredicate predicate, char ifNone)
        {
            return FloatCharHashMap.this.detectIfNone(predicate, ifNone);
        }

        public <V> MutableCollection<V> collect(CharToObjectFunction<? extends V> function)
        {
            return FloatCharHashMap.this.collect(function);
        }

        public long sum()
        {
            return FloatCharHashMap.this.sum();
        }

        public char max()
        {
            return FloatCharHashMap.this.max();
        }

        public char maxIfEmpty(char defaultValue)
        {
            return FloatCharHashMap.this.maxIfEmpty(defaultValue);
        }

        public char min()
        {
            return FloatCharHashMap.this.min();
        }

        public char minIfEmpty(char defaultValue)
        {
            return FloatCharHashMap.this.minIfEmpty(defaultValue);
        }

        public double average()
        {
            return FloatCharHashMap.this.average();
        }

        public double median()
        {
            return FloatCharHashMap.this.median();
        }

        public char[] toSortedArray()
        {
            return FloatCharHashMap.this.toSortedArray();
        }

        public MutableCharList toSortedList()
        {
            return FloatCharHashMap.this.toSortedList();
        }

        public MutableCharCollection with(char element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableCharCollection without(char element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableCharCollection withAll(CharIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableCharCollection withoutAll(CharIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableCharCollection asUnmodifiable()
        {
            return UnmodifiableCharCollection.of(this);
        }

        public MutableCharCollection asSynchronized()
        {
            return SynchronizedCharCollection.of(this);
        }

        public ImmutableCharCollection toImmutable()
        {
            return CharLists.immutable.withAll(this);
        }

        public boolean contains(char value)
        {
            return FloatCharHashMap.this.containsValue(value);
        }

        public boolean containsAll(char... source)
        {
            return FloatCharHashMap.this.containsAll(source);
        }

        public boolean containsAll(CharIterable source)
        {
            return FloatCharHashMap.this.containsAll(source);
        }

        public MutableCharList toList()
        {
            return FloatCharHashMap.this.toList();
        }

        public MutableCharSet toSet()
        {
            return FloatCharHashMap.this.toSet();
        }

        public MutableCharBag toBag()
        {
            return FloatCharHashMap.this.toBag();
        }

        public LazyCharIterable asLazy()
        {
            return new LazyCharIterableAdapter(this);
        }

        public boolean isEmpty()
        {
            return FloatCharHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return FloatCharHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);

                boolean first = true;

                if (FloatCharHashMap.this.sentinelValues != null)
                {
                    if (FloatCharHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(FloatCharHashMap.this.sentinelValues.zeroValue));
                        first = false;
                    }
                    if (FloatCharHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(FloatCharHashMap.this.sentinelValues.oneValue));
                        first = false;
                    }
                }
                for (int i = 0; i < FloatCharHashMap.this.keys.length; i++)
                {
                    float key = FloatCharHashMap.this.keys[i];
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(FloatCharHashMap.this.values[i]));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public CharIterator charIterator()
        {
            return FloatCharHashMap.this.charIterator();
        }

        public void forEach(CharProcedure procedure)
        {
            FloatCharHashMap.this.forEach(procedure);
        }

        public int count(CharPredicate predicate)
        {
            return FloatCharHashMap.this.count(predicate);
        }

        public boolean anySatisfy(CharPredicate predicate)
        {
            return FloatCharHashMap.this.anySatisfy(predicate);
        }

        public boolean allSatisfy(CharPredicate predicate)
        {
            return FloatCharHashMap.this.allSatisfy(predicate);
        }

        public boolean noneSatisfy(CharPredicate predicate)
        {
            return FloatCharHashMap.this.noneSatisfy(predicate);
        }

        public boolean add(char element)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(char... source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(CharIterable source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(char item)
        {
            int oldSize = FloatCharHashMap.this.size();

            if (FloatCharHashMap.this.sentinelValues != null)
            {
                if (FloatCharHashMap.this.sentinelValues.containsZeroKey && item == FloatCharHashMap.this.sentinelValues.zeroValue)
                {
                    FloatCharHashMap.this.removeKey(EMPTY_KEY);
                }
                if (FloatCharHashMap.this.sentinelValues.containsOneKey && item == FloatCharHashMap.this.sentinelValues.oneValue)
                {
                    FloatCharHashMap.this.removeKey(REMOVED_KEY);
                }
            }
            for (int i = 0; i < FloatCharHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(FloatCharHashMap.this.keys[i]) && item == FloatCharHashMap.this.values[i])
                {
                    FloatCharHashMap.this.removeKey(FloatCharHashMap.this.keys[i]);
                }
            }
            return oldSize != FloatCharHashMap.this.size();
        }

        public boolean removeAll(CharIterable source)
        {
            int oldSize = FloatCharHashMap.this.size();

            CharIterator iterator = source.charIterator();
            while (iterator.hasNext())
            {
                this.remove(iterator.next());
            }
            return oldSize != FloatCharHashMap.this.size();
        }

        public boolean removeAll(char... source)
        {
            int oldSize = FloatCharHashMap.this.size();

            for (char item : source)
            {
                this.remove(item);
            }
            return oldSize != FloatCharHashMap.this.size();
        }

        public int size()
        {
            return FloatCharHashMap.this.size();
        }

        public char[] toArray()
        {
            return FloatCharHashMap.this.toArray();
        }
    }
}
