/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.map.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.IntIterable;
import com.gs.collections.api.LazyIntIterable;
import com.gs.collections.api.LazyLongIterable;
import com.gs.collections.api.LongIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.bag.primitive.MutableIntBag;
import com.gs.collections.api.bag.primitive.MutableLongBag;
import com.gs.collections.api.block.function.primitive.IntToObjectFunction;
import com.gs.collections.api.block.function.primitive.IntToLongFunction;
import com.gs.collections.api.block.function.primitive.LongFunction;
import com.gs.collections.api.block.function.primitive.LongFunction0;
import com.gs.collections.api.block.function.primitive.LongToLongFunction;
import com.gs.collections.api.block.function.primitive.LongToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectLongToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.IntLongPredicate;
import com.gs.collections.api.block.predicate.primitive.IntPredicate;
import com.gs.collections.api.block.predicate.primitive.LongPredicate;
import com.gs.collections.api.block.procedure.primitive.IntProcedure;
import com.gs.collections.api.block.procedure.primitive.IntLongProcedure;
import com.gs.collections.api.block.procedure.primitive.LongProcedure;
import com.gs.collections.api.collection.MutableCollection;
import com.gs.collections.api.collection.primitive.ImmutableLongCollection;
import com.gs.collections.api.collection.primitive.MutableLongCollection;
import com.gs.collections.api.iterator.IntIterator;
import com.gs.collections.api.iterator.LongIterator;
import com.gs.collections.api.list.primitive.MutableIntList;
import com.gs.collections.api.list.primitive.MutableLongList;
import com.gs.collections.api.map.primitive.IntLongMap;
import com.gs.collections.api.map.primitive.ImmutableIntLongMap;
import com.gs.collections.api.map.primitive.MutableIntLongMap;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.IntSet;
import com.gs.collections.api.set.primitive.ImmutableIntSet;
import com.gs.collections.api.set.primitive.MutableIntSet;
import com.gs.collections.api.set.primitive.MutableLongSet;
import com.gs.collections.impl.bag.mutable.primitive.IntHashBag;
import com.gs.collections.impl.bag.mutable.primitive.LongHashBag;
import com.gs.collections.impl.block.factory.primitive.IntPredicates;
import com.gs.collections.impl.collection.mutable.primitive.SynchronizedLongCollection;
import com.gs.collections.impl.collection.mutable.primitive.UnmodifiableLongCollection;
import com.gs.collections.impl.factory.Sets;
import com.gs.collections.impl.factory.primitive.LongLists;
import com.gs.collections.impl.factory.primitive.IntLongMaps;
import com.gs.collections.impl.factory.primitive.IntSets;
import com.gs.collections.impl.lazy.primitive.CollectIntToObjectIterable;
import com.gs.collections.impl.lazy.primitive.LazyLongIterableAdapter;
import com.gs.collections.impl.lazy.primitive.LazyIntIterableAdapter;
import com.gs.collections.impl.lazy.primitive.SelectIntIterable;
import com.gs.collections.impl.list.mutable.FastList;
import com.gs.collections.impl.list.mutable.primitive.IntArrayList;
import com.gs.collections.impl.list.mutable.primitive.LongArrayList;
import com.gs.collections.impl.set.mutable.primitive.IntHashSet;
import com.gs.collections.impl.set.mutable.primitive.SynchronizedIntSet;
import com.gs.collections.impl.set.mutable.primitive.UnmodifiableIntSet;
import com.gs.collections.impl.set.mutable.primitive.LongHashSet;

/**
 * This file was automatically generated from template file primitivePrimitiveHashMap.stg.
 *
 * @since 3.0.
 */
public class IntLongHashMap implements MutableIntLongMap, Externalizable
{
    static final long EMPTY_VALUE = 0L;
    private static final long serialVersionUID = 1L;
    private static final int EMPTY_KEY = 0;
    private static final int REMOVED_KEY = 1;

    private static final int OCCUPIED_DATA_RATIO = 2;
    private static final int OCCUPIED_SENTINEL_RATIO = 4;
    private static final int DEFAULT_INITIAL_CAPACITY = 8;

    private int[] keys;
    private long[] values;

    private int occupiedWithData;
    private int occupiedWithSentinels;

    private SentinelValues sentinelValues;

    public IntLongHashMap()
    {
        this.allocateTable(DEFAULT_INITIAL_CAPACITY << 1);
    }

    public IntLongHashMap(int initialCapacity)
    {
        if (initialCapacity < 0)
        {
            throw new IllegalArgumentException("initial capacity cannot be less than 0");
        }
        int capacity = this.smallestPowerOfTwoGreaterThan(this.fastCeil(initialCapacity * OCCUPIED_DATA_RATIO));
        this.allocateTable(capacity);
    }

    private int smallestPowerOfTwoGreaterThan(int n)
    {
        return n > 1 ? Integer.highestOneBit(n - 1) << 1 : 1;
    }

    public IntLongHashMap(IntLongMap map)
    {
        this(Math.max(map.size(), DEFAULT_INITIAL_CAPACITY));
        this.putAll(map);
    }

    private int fastCeil(float v)
    {
        int possibleResult = (int) v;
        if (v - possibleResult > 0.0F)
        {
            possibleResult++;
        }
        return possibleResult;
    }

    public static IntLongHashMap newWithKeysValues(int key1, long value1)
    {
        return new IntLongHashMap(1).withKeyValue(key1, value1);
    }

    public static IntLongHashMap newWithKeysValues(int key1, long value1, int key2, long value2)
    {
        return new IntLongHashMap(2).withKeysValues(key1, value1, key2, value2);
    }

    public static IntLongHashMap newWithKeysValues(int key1, long value1, int key2, long value2, int key3, long value3)
    {
        return new IntLongHashMap(3).withKeysValues(key1, value1, key2, value2, key3, value3);
    }

    public static IntLongHashMap newWithKeysValues(int key1, long value1, int key2, long value2, int key3, long value3, int key4, long value4)
    {
        return new IntLongHashMap(4).withKeysValues(key1, value1, key2, value2, key3, value3, key4, value4);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }

        if (!(obj instanceof IntLongMap))
        {
            return false;
        }

        IntLongMap other = (IntLongMap) obj;

        if (this.size() != other.size())
        {
            return false;
        }

        if (this.sentinelValues == null)
        {
            if (other.containsKey(EMPTY_KEY) || other.containsKey(REMOVED_KEY))
            {
                return false;
            }
        }
        else
        {
            if (this.sentinelValues.containsZeroKey && (!other.containsKey(EMPTY_KEY) || this.sentinelValues.zeroValue != other.getOrThrow(EMPTY_KEY)))
            {
                return false;
            }

            if (this.sentinelValues.containsOneKey && (!other.containsKey(REMOVED_KEY) || this.sentinelValues.oneValue != other.getOrThrow(REMOVED_KEY)))
            {
                return false;
            }
        }

        for (int i = 0; i < this.keys.length; i++)
        {
            int key = this.keys[i];
            if (isNonSentinel(key) && (!other.containsKey(key) || this.values[i] != other.getOrThrow(key)))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int result = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += EMPTY_KEY ^ (int) (this.sentinelValues.zeroValue ^ this.sentinelValues.zeroValue >>> 32);
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += REMOVED_KEY ^ (int) (this.sentinelValues.oneValue ^ this.sentinelValues.oneValue >>> 32);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += this.keys[i] ^ (int) (this.values[i] ^ this.values[i] >>> 32);
            }
        }

        return result;
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public int size()
    {
        return this.occupiedWithData + (this.sentinelValues == null ? 0 : this.sentinelValues.size());
    }

    public boolean isEmpty()
    {
        return this.occupiedWithData == 0 && (this.sentinelValues == null || this.sentinelValues.size() == 0);
    }

    public boolean notEmpty()
    {
        return this.occupiedWithData != 0 || (this.sentinelValues != null && this.sentinelValues.size() != 0);
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            boolean first = true;

            if (this.sentinelValues != null)
            {
                if (this.sentinelValues.containsZeroKey)
                {
                    appendable.append(String.valueOf(EMPTY_KEY)).append("=").append(String.valueOf(this.sentinelValues.zeroValue));
                    first = false;
                }
                if (this.sentinelValues.containsOneKey)
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(REMOVED_KEY)).append("=").append(String.valueOf(this.sentinelValues.oneValue));
                    first = false;
                }
            }
            for (int i = 0; i < this.keys.length; i++)
            {
                int key = this.keys[i];
                if (isNonSentinel(key))
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(key)).append("=").append(String.valueOf(this.values[i]));
                    first = false;
                }
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public LongIterator longIterator()
    {
        return new InternalLongIterator();
    }

    public long[] toArray()
    {
        long[] array = new long[this.size()];
        int index = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                array[index] = this.sentinelValues.zeroValue;
                index++;
            }
            if (this.sentinelValues.containsOneKey)
            {
                array[index] = this.sentinelValues.oneValue;
                index++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                array[index] = this.values[i];
                index++;
            }
        }

        return array;
    }

    public boolean contains(long value)
    {
        return this.containsValue(value);
    }

    public boolean containsAll(long... source)
    {
        for (long each : source)
        {
            if (!this.contains(each))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(LongIterable source)
    {
        return source.allSatisfy(new LongPredicate()
        {
            public boolean accept(long value)
            {
                return IntLongHashMap.this.contains(value);
            }
        });
    }

    public void forEach(LongProcedure procedure)
    {
        this.forEachValue(procedure);
    }

    public MutableLongCollection select(LongPredicate predicate)
    {
        LongArrayList result = new LongArrayList();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }

        return result;
    }

    public MutableLongCollection reject(LongPredicate predicate)
    {
        LongArrayList result = new LongArrayList();
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public <V> MutableCollection<V> collect(LongToObjectFunction<? extends V> function)
    {
        FastList<V> target = FastList.newList(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                target.add(function.valueOf(this.sentinelValues.zeroValue));
            }
            if (this.sentinelValues.containsOneKey)
            {
                target.add(function.valueOf(this.sentinelValues.oneValue));
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                target.add(function.valueOf(this.values[i]));
            }
        }
        return target;
    }

    public long detectIfNone(LongPredicate predicate, long value)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return this.values[i];
            }
        }
        return value;
    }

    public int count(LongPredicate predicate)
    {
        int count = 0;
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                count++;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                count++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(LongPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return true;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return true;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(LongPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(LongPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public <V> V injectInto(V injectedValue, ObjectLongToObjectFunction<? super V, ? extends V> function)
    {
        V result = injectedValue;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result = function.valueOf(result, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                result = function.valueOf(result, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result = function.valueOf(result, this.values[i]);
            }
        }

        return result;
    }

    public MutableLongList toList()
    {
        return LongArrayList.newList(this);
    }

    public MutableLongSet toSet()
    {
        return LongHashSet.newSet(this);
    }

    public MutableLongBag toBag()
    {
        return LongHashBag.newBag(this);
    }

    public LazyLongIterable asLazy()
    {
        return new LazyLongIterableAdapter(this);
    }

    public void clear()
    {
        this.sentinelValues = null;
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;
        Arrays.fill(this.keys, EMPTY_KEY);
        Arrays.fill(this.values, EMPTY_VALUE);
    }

    public void put(int key, long value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return;
        }

        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return;
        }

        int index = this.probe(key);

        if (this.keys[index] == key)
        {
            // key already present in map
            this.values[index] = value;
            return;
        }

        if (this.keys[index] == REMOVED_KEY)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = key;
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
    }

    public void putAll(IntLongMap map)
    {
        map.forEachKeyValue(new IntLongProcedure()
        {
            public void value(int key, long value)
            {
                IntLongHashMap.this.put(key, value);
            }
        });
    }

    public void removeKey(int key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return;
            }
            if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.containsZeroKey = false;
                this.sentinelValues.zeroValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.containsOneKey = false;
                this.sentinelValues.oneValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.keys[index] = REMOVED_KEY;
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }
        }
    }

    public void remove(int key)
    {
        this.removeKey(key);
    }

    public long removeKeyIfAbsent(int key, long value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return value;
            }
            long oldValue = this.sentinelValues.zeroValue;
            if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.containsZeroKey = false;
                this.sentinelValues.zeroValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return oldValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return value;
            }
            long oldValue = this.sentinelValues.oneValue;
            if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.containsOneKey = false;
                this.sentinelValues.oneValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return oldValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.keys[index] = REMOVED_KEY;
            long oldValue = this.values[index];
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }

            return oldValue;
        }
        return value;
    }

    public long getIfAbsentPut(int key, long value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public long getIfAbsentPut(int key, LongFunction0 function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                long value = function.value();
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            long value = function.value();
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                long value = function.value();
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            long value = function.value();
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        long value = function.value();
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public <P> long getIfAbsentPutWith(int key, LongFunction<? super P> function, P parameter)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                long value = function.longValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            long value = function.longValueOf(parameter);
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                long value = function.longValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            long value = function.longValueOf(parameter);
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        long value = function.longValueOf(parameter);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public long getIfAbsentPutWithKey(int key, IntToLongFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                long value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            long value = function.valueOf(key);
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                long value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            long value = function.valueOf(key);
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        long value = function.valueOf(key);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public long updateValue(int key, long initialValueIfAbsent, LongToLongFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = function.valueOf(initialValueIfAbsent);
            }
            else if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.zeroValue = function.valueOf(this.sentinelValues.zeroValue);
            }
            else
            {
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = function.valueOf(initialValueIfAbsent);
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = function.valueOf(initialValueIfAbsent);
            }
            else if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.oneValue = function.valueOf(this.sentinelValues.oneValue);
            }
            else
            {
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = function.valueOf(initialValueIfAbsent);
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.values[index] = function.valueOf(this.values[index]);
            return this.values[index];
        }
        if (this.keys[index] == REMOVED_KEY)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = key;
        long value = function.valueOf(initialValueIfAbsent);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public IntLongHashMap withKeyValue(int key1, long value1)
    {
        this.put(key1, value1);
        return this;
    }

    public IntLongHashMap withKeysValues(int key1, long value1, int key2, long value2)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        return this;
    }

    public IntLongHashMap withKeysValues(int key1, long value1, int key2, long value2, int key3, long value3)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        return this;
    }

    public IntLongHashMap withKeysValues(int key1, long value1, int key2, long value2, int key3, long value3, int key4, long value4)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        this.put(key4, value4);
        return this;
    }

    public IntLongHashMap withoutKey(int key)
    {
        this.removeKey(key);
        return this;
    }

    public IntLongHashMap withoutAllKeys(IntIterable keys)
    {
        keys.forEach(new IntProcedure()
        {
            public void value(int key)
            {
                IntLongHashMap.this.removeKey(key);
            }
        });
        return this;
    }

    public MutableIntLongMap asUnmodifiable()
    {
        return new UnmodifiableIntLongMap(this);
    }

    public MutableIntLongMap asSynchronized()
    {
        return new SynchronizedIntLongMap(this);
    }

    public ImmutableIntLongMap toImmutable()
    {
        return IntLongMaps.immutable.ofAll(this);
    }

    public long get(int key)
    {
        return this.getIfAbsent(key, EMPTY_VALUE);
    }

    public long getIfAbsent(int key, long ifAbsent)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        return ifAbsent;
    }

    public long getOrThrow(int key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]))
        {
            return this.values[index];
        }
        throw new IllegalStateException("Key " + key + " not present.");
    }

    public boolean containsKey(int key)
    {
        if (isEmptyKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsZeroKey;
        }
        if (isRemovedKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsOneKey;
        }
        return this.keys[this.probe(key)] == key;
    }

    public boolean containsValue(long value)
    {
        if (this.sentinelValues != null && this.sentinelValues.containsValue(value))
        {
            return true;
        }
        for (int i = 0; i < this.values.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && this.values[i] == value)
            {
                return true;
            }
        }
        return false;
    }

    public void forEachValue(LongProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.values[i]);
            }
        }
    }

    public void forEachKey(IntProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY);
            }
        }
        for (int key : this.keys)
        {
            if (isNonSentinel(key))
            {
                procedure.value(key);
            }
        }
    }

    public void forEachKeyValue(IntLongProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.keys[i], this.values[i]);
            }
        }
    }

    public LazyIntIterable keysView()
    {
        return new KeysView();
    }

    public IntLongHashMap select(IntLongPredicate predicate)
    {
        IntLongHashMap result = new IntLongHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.keys[i], this.values[i]))
            {
                result.put(this.keys[i], this.values[i]);
            }
        }

        return result;
    }

    public IntLongHashMap reject(IntLongPredicate predicate)
    {
        IntLongHashMap result = new IntLongHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.keys[i], this.values[i]))
            {
                result.put(this.keys[i], this.values[i]);
            }
        }
        return result;
    }

    public long sum()
    {
        long result = 0L;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += this.values[i];
            }
        }

        return result;
    }

    public long max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        LongIterator iterator = this.longIterator();
        long max = iterator.next();
        while (iterator.hasNext())
        {
            long value = iterator.next();
            if (max < value)
            {
                max = value;
            }
        }
        return max;
    }

    public long maxIfEmpty(long defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public long min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        LongIterator iterator = this.longIterator();
        long min = iterator.next();
        while (iterator.hasNext())
        {
            long value = iterator.next();
            if (value < min)
            {
                min = value;
            }
        }
        return min;
    }

    public long minIfEmpty(long defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return (double) this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        long[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            long first = sortedArray[middleIndex];
            long second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public long[] toSortedArray()
    {
        long[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public MutableLongList toSortedList()
    {
        return LongArrayList.newList(this).sortThis();
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                out.writeInt(EMPTY_KEY);
                out.writeLong(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                out.writeInt(REMOVED_KEY);
                out.writeLong(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                out.writeInt(this.keys[i]);
                out.writeLong(this.values[i]);
            }
        }
    }

    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
    {
        int size = in.readInt();
        for (int i = 0; i < size; i++)
        {
            this.put(in.readInt(), in.readLong());
        }
    }

    /**
     * Rehashes every element in the set into a new backing table of the smallest possible size and eliminating removed sentinels.
     */
    public void compact()
    {
        this.rehash(this.smallestPowerOfTwoGreaterThan(this.size()));
    }

    private void rehash()
    {
        this.rehash(this.keys.length);
    }

    private void rehashAndGrow()
    {
        this.rehash(this.keys.length << 1);
    }

    private void rehash(int newCapacity)
    {
        int oldLength = this.keys.length;
        int[] old = this.keys;
        long[] oldValues = this.values;
        this.allocateTable(newCapacity);
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;

        for (int i = 0; i < oldLength; i++)
        {
            if (isNonSentinel(old[i]))
            {
                this.put(old[i], oldValues[i]);
            }
        }
    }

    // exposed for testing
    int probe(int element)
    {
        int index = this.spread(element);
        int keyAtIndex = this.keys[index];

        if (keyAtIndex == element || keyAtIndex == EMPTY_KEY)
        {
            return index;
        }

        int removedIndex = keyAtIndex == REMOVED_KEY ? index : -1;
        int nextIndex = index;
        int probe = 17;

        // loop until an empty slot is reached
        while (true)
        {
            // Probe algorithm: 17*n*(n+1)/2 where n = number of collisions
            nextIndex += probe;
            probe += 17;
            nextIndex &= this.keys.length - 1;

            if (this.keys[nextIndex] == element)
            {
                return nextIndex;
            }
            if (this.keys[nextIndex] == REMOVED_KEY)
            {
                if (removedIndex == -1)
                {
                    removedIndex = nextIndex;
                }
            }
            else if (this.keys[nextIndex] == EMPTY_KEY)
            {
                return removedIndex == -1 ? nextIndex : removedIndex;
            }
        }
    }

    // exposed for testing
    int spread(int element)
    {
        int code = element;
        code ^= 61 ^ (code >> 16);
        code += code << 3;
        code ^= code >> 4;
        code *= 0x27d4eb2d;
        code ^= code >> 15;
        return code & (this.keys.length - 1);
    }

    private void allocateTable(int sizeToAllocate)
    {
        this.keys = new int[sizeToAllocate];
        this.values = new long[sizeToAllocate];
    }

    private static boolean isEmptyKey(int key)
    {
        return key == EMPTY_KEY;
    }

    private static boolean isRemovedKey(int key)
    {
        return key == REMOVED_KEY;
    }

    private static boolean isNonSentinel(int key)
    {
        return !isEmptyKey(key) && !isRemovedKey(key);
    }

    private int maxOccupiedWithData()
    {
        int capacity = this.keys.length;
        // need at least one free slot for open addressing
        return Math.min(capacity - 1, capacity / OCCUPIED_DATA_RATIO);
    }

    private int maxOccupiedWithSentinels()
    {
        return this.keys.length / OCCUPIED_SENTINEL_RATIO;
    }

    private static final class SentinelValues
    {
        private boolean containsZeroKey;
        private boolean containsOneKey;
        private long zeroValue;
        private long oneValue;

        public int size()
        {
            return (this.containsZeroKey ? 1 : 0) + (this.containsOneKey ? 1 : 0);
        }

        public boolean containsValue(long value)
        {
            boolean valueEqualsZeroValue = this.containsZeroKey && this.zeroValue == value;
            boolean valueEqualsOneValue = this.containsOneKey && this.oneValue == value;
            return valueEqualsZeroValue || valueEqualsOneValue;
        }
    }

    private class InternalLongIterator implements LongIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < IntLongHashMap.this.size();
        }

        public long next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (IntLongHashMap.this.containsKey(EMPTY_KEY))
                {
                    return IntLongHashMap.this.get(EMPTY_KEY);
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (IntLongHashMap.this.containsKey(REMOVED_KEY))
                {
                    return IntLongHashMap.this.get(REMOVED_KEY);
                }
            }

            int[] keys = IntLongHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            long result = IntLongHashMap.this.values[this.position];
            this.position++;
            return result;
        }
    }

    private class KeysView implements LazyIntIterable
    {
        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public int size()
        {
            return IntLongHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return IntLongHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return IntLongHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (IntLongHashMap.this.sentinelValues != null)
                {
                    if (IntLongHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (IntLongHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (int key : IntLongHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public IntIterator intIterator()
        {
            return new KeySetIterator();
        }

        public boolean contains(int key)
        {
            return IntLongHashMap.this.containsKey(key);
        }

        public boolean containsAll(int... keys)
        {
            for (int key : keys)
            {
                if (!IntLongHashMap.this.containsKey(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(IntIterable source)
        {
            return source.allSatisfy(new IntPredicate()
            {
                public boolean accept(int key)
                {
                    return IntLongHashMap.this.containsKey(key);
                }
            });
        }

        public void forEach(IntProcedure procedure)
        {
            IntLongHashMap.this.forEachKey(procedure);
        }

        public int count(IntPredicate predicate)
        {
            int count = 0;
            if (IntLongHashMap.this.sentinelValues != null)
            {
                if (IntLongHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (IntLongHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (int key : IntLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(IntPredicate predicate)
        {
            if (IntLongHashMap.this.sentinelValues != null)
            {
                if (IntLongHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (IntLongHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (int key : IntLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(IntPredicate predicate)
        {
            if (IntLongHashMap.this.sentinelValues != null)
            {
                if (IntLongHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (IntLongHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (int key : IntLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(IntPredicate predicate)
        {
            if (IntLongHashMap.this.sentinelValues != null)
            {
                if (IntLongHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (IntLongHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (int key : IntLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public LazyIntIterable select(IntPredicate predicate)
        {
            return new SelectIntIterable(this, predicate);
        }

        public LazyIntIterable reject(IntPredicate predicate)
        {
            return new SelectIntIterable(this, IntPredicates.not(predicate));
        }

        public int detectIfNone(IntPredicate predicate, int value)
        {
            if (IntLongHashMap.this.sentinelValues != null)
            {
                if (IntLongHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (IntLongHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (int key : IntLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return value;
        }

        public <V> LazyIterable<V> collect(IntToObjectFunction<? extends V> function)
        {
            return new CollectIntToObjectIterable<V>(this, function);
        }

        public long sum()
        {
            long result = 0L;

            if (IntLongHashMap.this.sentinelValues != null)
            {
                if (IntLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += EMPTY_KEY;
                }
                if (IntLongHashMap.this.sentinelValues.containsOneKey)
                {
                    result += REMOVED_KEY;
                }
            }
            for (int key : IntLongHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result += key;
                }
            }

            return result;
        }

        public int max()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            IntIterator iterator = this.intIterator();
            int max = iterator.next();
            while (iterator.hasNext())
            {
                int value = iterator.next();
                if (max < value)
                {
                    max = value;
                }
            }
            return max;
        }

        public int min()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            IntIterator iterator = this.intIterator();
            int min = iterator.next();
            while (iterator.hasNext())
            {
                int value = iterator.next();
                if (value < min)
                {
                    min = value;
                }
            }
            return min;
        }

        public int minIfEmpty(int defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.min();
        }

        public int maxIfEmpty(int defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.max();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return (double) this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            int[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                int first = sortedArray[middleIndex];
                int second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public int[] toSortedArray()
        {
            int[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public int[] toArray()
        {
            int size = IntLongHashMap.this.size();
            final int[] result = new int[size];
            IntLongHashMap.this.forEachKey(new IntProcedure()
            {
                private int index;

                public void value(int each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public MutableIntList toList()
        {
            return IntArrayList.newList(this);
        }

        public MutableIntList toSortedList()
        {
            return IntArrayList.newList(this).sortThis();
        }

        public MutableIntSet toSet()
        {
            return IntHashSet.newSet(this);
        }

        public MutableIntBag toBag()
        {
            return IntHashBag.newBag(this);
        }

        public LazyIntIterable asLazy()
        {
            return this;
        }
    }

    private class KeySetIterator implements IntIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < IntLongHashMap.this.size();
        }

        public int next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (IntLongHashMap.this.containsKey(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (IntLongHashMap.this.containsKey(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }

            int[] keys = IntLongHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            int result = keys[this.position];
            this.position++;
            return result;
        }
    }

    public MutableIntSet keySet()
    {
        return new KeySet();
    }

    private class KeySet implements MutableIntSet
    {
        public IntIterator intIterator()
        {
            return new KeySetIterator();
        }

        public void forEach(IntProcedure procedure)
        {
            IntLongHashMap.this.forEachKey(procedure);
        }

        public int count(IntPredicate predicate)
        {
            int count = 0;
            if (IntLongHashMap.this.sentinelValues != null)
            {
                if (IntLongHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (IntLongHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (int key : IntLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(IntPredicate predicate)
        {
            if (IntLongHashMap.this.sentinelValues != null)
            {
                if (IntLongHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (IntLongHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (int key : IntLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(IntPredicate predicate)
        {
            if (IntLongHashMap.this.sentinelValues != null)
            {
                if (IntLongHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (IntLongHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (int key : IntLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(IntPredicate predicate)
        {
            if (IntLongHashMap.this.sentinelValues != null)
            {
                if (IntLongHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (IntLongHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (int key : IntLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean add(int element)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(int... source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(IntIterable source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(int key)
        {
            int oldSize = IntLongHashMap.this.size();
            IntLongHashMap.this.removeKey(key);
            return oldSize != IntLongHashMap.this.size();
        }

        public boolean removeAll(IntIterable source)
        {
            int oldSize = IntLongHashMap.this.size();
            IntIterator iterator = source.intIterator();
            while (iterator.hasNext())
            {
                IntLongHashMap.this.removeKey(iterator.next());
            }
            return oldSize != IntLongHashMap.this.size();
        }

        public boolean removeAll(int... source)
        {
            int oldSize = IntLongHashMap.this.size();
            for (int item : source)
            {
                IntLongHashMap.this.removeKey(item);
            }
            return oldSize != IntLongHashMap.this.size();
        }

        public void clear()
        {
            IntLongHashMap.this.clear();
        }

        public MutableIntSet select(IntPredicate predicate)
        {
            MutableIntSet result = new IntHashSet();
            if (IntLongHashMap.this.sentinelValues != null)
            {
                if (IntLongHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (IntLongHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (int key : IntLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableIntSet reject(IntPredicate predicate)
        {
            MutableIntSet result = new IntHashSet();
            if (IntLongHashMap.this.sentinelValues != null)
            {
                if (IntLongHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (IntLongHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (int key : IntLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableIntSet with(int element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableIntSet without(int element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableIntSet withAll(IntIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableIntSet withoutAll(IntIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public int detectIfNone(IntPredicate predicate, int ifNone)
        {
            if (IntLongHashMap.this.sentinelValues != null)
            {
                if (IntLongHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (IntLongHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (int key : IntLongHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return ifNone;
        }

        public <V> MutableSet<V> collect(IntToObjectFunction<? extends V> function)
        {
            MutableSet<V> result = Sets.mutable.with();
            if (IntLongHashMap.this.sentinelValues != null)
            {
                if (IntLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    result.add(function.valueOf(EMPTY_KEY));
                }
                if (IntLongHashMap.this.sentinelValues.containsOneKey)
                {
                    result.add(function.valueOf(REMOVED_KEY));
                }
            }
            for (int key : IntLongHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result.add(function.valueOf(key));
                }
            }
            return result;
        }

        public MutableIntSet asUnmodifiable()
        {
            return UnmodifiableIntSet.of(this);
        }

        public MutableIntSet asSynchronized()
        {
            return SynchronizedIntSet.of(this);
        }

        public long sum()
        {
            long sum = 0L;
            if (IntLongHashMap.this.sentinelValues != null)
            {
                if (IntLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    sum += EMPTY_KEY;
                }
                if (IntLongHashMap.this.sentinelValues.containsOneKey)
                {
                    sum += REMOVED_KEY;
                }
            }
            for (int key : IntLongHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    sum += key;
                }
            }
            return sum;
        }

        public int max()
        {
            if (IntLongHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            int max = 0;
            boolean isMaxSet = false;

            if (IntLongHashMap.this.sentinelValues != null)
            {
                if (IntLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    max = EMPTY_KEY;
                    isMaxSet = true;
                }
                if (IntLongHashMap.this.sentinelValues.containsOneKey && (!isMaxSet || max < REMOVED_KEY))
                {
                    max = REMOVED_KEY;
                    isMaxSet = true;
                }
            }
            for (int i = 0; i < IntLongHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(IntLongHashMap.this.keys[i]) && (!isMaxSet || max < IntLongHashMap.this.keys[i]))
                {
                    max = IntLongHashMap.this.keys[i];
                    isMaxSet = true;
                }
            }
            return max;
        }

        public int maxIfEmpty(int defaultValue)
        {
            if (IntLongHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.max();
        }

        public int min()
        {
            if (IntLongHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            int min = 0;
            boolean isMinSet = false;

            if (IntLongHashMap.this.sentinelValues != null)
            {
                if (IntLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    min = EMPTY_KEY;
                    isMinSet = true;
                }
                if (IntLongHashMap.this.sentinelValues.containsOneKey && (!isMinSet || REMOVED_KEY < min))
                {
                    min = REMOVED_KEY;
                    isMinSet = true;
                }
            }
            for (int i = 0; i < IntLongHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(IntLongHashMap.this.keys[i]) && (!isMinSet || IntLongHashMap.this.keys[i] < min))
                {
                    min = IntLongHashMap.this.keys[i];
                    isMinSet = true;
                }
            }
            return min;
        }

        public int minIfEmpty(int defaultValue)
        {
            if (IntLongHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.min();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return (double) this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            int[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                int first = sortedArray[middleIndex];
                int second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public int[] toSortedArray()
        {
            int[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public MutableIntList toSortedList()
        {
            return IntArrayList.newList(this).sortThis();
        }

        public int[] toArray()
        {
            int size = IntLongHashMap.this.size();
            final int[] result = new int[size];
            IntLongHashMap.this.forEachKey(new IntProcedure()
            {
                private int index;

                public void value(int each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public boolean contains(int value)
        {
            return IntLongHashMap.this.containsKey(value);
        }

        public boolean containsAll(int... source)
        {
            for (int item : source)
            {
                if (!IntLongHashMap.this.containsKey(item))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(IntIterable source)
        {
            IntIterator iterator = source.intIterator();
            while (iterator.hasNext())
            {
                if (!IntLongHashMap.this.containsKey(iterator.next()))
                {
                    return false;
                }
            }
            return true;
        }

        public MutableIntList toList()
        {
            return IntArrayList.newList(this);
        }

        public MutableIntSet toSet()
        {
            return IntHashSet.newSet(this);
        }

        public MutableIntBag toBag()
        {
            return IntHashBag.newBag(this);
        }

        public LazyIntIterable asLazy()
        {
            return new LazyIntIterableAdapter(this);
        }

        public IntSet freeze()
        {
            throw new UnsupportedOperationException();
        }

        public ImmutableIntSet toImmutable()
        {
            return IntSets.immutable.withAll(this);
        }

        public int size()
        {
            return IntLongHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return IntLongHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return IntLongHashMap.this.notEmpty();
        }

        @Override
        public boolean equals(Object obj)
        {
            if (this == obj)
            {
                return true;
            }

            if (!(obj instanceof IntSet))
            {
                return false;
            }

            IntSet other = (IntSet) obj;
            return this.size() == other.size() && this.containsAll(other.toArray());
        }

        @Override
        public int hashCode()
        {
            int result = 0;

            if (IntLongHashMap.this.sentinelValues != null)
            {
                if (IntLongHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += EMPTY_KEY;
                }
                if (IntLongHashMap.this.sentinelValues.containsOneKey)
                {
                    result += REMOVED_KEY;
                }
            }
            for (int i = 0; i < IntLongHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(IntLongHashMap.this.keys[i]))
                {
                    result += IntLongHashMap.this.keys[i];
                }
            }

            return result;
        }

        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (IntLongHashMap.this.sentinelValues != null)
                {
                    if (IntLongHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (IntLongHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (int key : IntLongHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }
    }

    public MutableLongCollection values()
    {
        return new ValuesCollection();
    }

    private class ValuesCollection implements MutableLongCollection
    {
        public void clear()
        {
            IntLongHashMap.this.clear();
        }

        public MutableLongCollection select(LongPredicate predicate)
        {
            return IntLongHashMap.this.select(predicate);
        }

        public MutableLongCollection reject(LongPredicate predicate)
        {
            return IntLongHashMap.this.reject(predicate);
        }

        public long detectIfNone(LongPredicate predicate, long ifNone)
        {
            return IntLongHashMap.this.detectIfNone(predicate, ifNone);
        }

        public <V> MutableCollection<V> collect(LongToObjectFunction<? extends V> function)
        {
            return IntLongHashMap.this.collect(function);
        }

        public long sum()
        {
            return IntLongHashMap.this.sum();
        }

        public long max()
        {
            return IntLongHashMap.this.max();
        }

        public long maxIfEmpty(long defaultValue)
        {
            return IntLongHashMap.this.maxIfEmpty(defaultValue);
        }

        public long min()
        {
            return IntLongHashMap.this.min();
        }

        public long minIfEmpty(long defaultValue)
        {
            return IntLongHashMap.this.minIfEmpty(defaultValue);
        }

        public double average()
        {
            return IntLongHashMap.this.average();
        }

        public double median()
        {
            return IntLongHashMap.this.median();
        }

        public long[] toSortedArray()
        {
            return IntLongHashMap.this.toSortedArray();
        }

        public MutableLongList toSortedList()
        {
            return IntLongHashMap.this.toSortedList();
        }

        public MutableLongCollection with(long element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableLongCollection without(long element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableLongCollection withAll(LongIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableLongCollection withoutAll(LongIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableLongCollection asUnmodifiable()
        {
            return UnmodifiableLongCollection.of(this);
        }

        public MutableLongCollection asSynchronized()
        {
            return SynchronizedLongCollection.of(this);
        }

        public ImmutableLongCollection toImmutable()
        {
            return LongLists.immutable.withAll(this);
        }

        public boolean contains(long value)
        {
            return IntLongHashMap.this.containsValue(value);
        }

        public boolean containsAll(long... source)
        {
            return IntLongHashMap.this.containsAll(source);
        }

        public boolean containsAll(LongIterable source)
        {
            return IntLongHashMap.this.containsAll(source);
        }

        public MutableLongList toList()
        {
            return IntLongHashMap.this.toList();
        }

        public MutableLongSet toSet()
        {
            return IntLongHashMap.this.toSet();
        }

        public MutableLongBag toBag()
        {
            return IntLongHashMap.this.toBag();
        }

        public LazyLongIterable asLazy()
        {
            return new LazyLongIterableAdapter(this);
        }

        public boolean isEmpty()
        {
            return IntLongHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return IntLongHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);

                boolean first = true;

                if (IntLongHashMap.this.sentinelValues != null)
                {
                    if (IntLongHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(IntLongHashMap.this.sentinelValues.zeroValue));
                        first = false;
                    }
                    if (IntLongHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(IntLongHashMap.this.sentinelValues.oneValue));
                        first = false;
                    }
                }
                for (int i = 0; i < IntLongHashMap.this.keys.length; i++)
                {
                    int key = IntLongHashMap.this.keys[i];
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(IntLongHashMap.this.values[i]));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public LongIterator longIterator()
        {
            return IntLongHashMap.this.longIterator();
        }

        public void forEach(LongProcedure procedure)
        {
            IntLongHashMap.this.forEach(procedure);
        }

        public int count(LongPredicate predicate)
        {
            return IntLongHashMap.this.count(predicate);
        }

        public boolean anySatisfy(LongPredicate predicate)
        {
            return IntLongHashMap.this.anySatisfy(predicate);
        }

        public boolean allSatisfy(LongPredicate predicate)
        {
            return IntLongHashMap.this.allSatisfy(predicate);
        }

        public boolean noneSatisfy(LongPredicate predicate)
        {
            return IntLongHashMap.this.noneSatisfy(predicate);
        }

        public boolean add(long element)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(long... source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(LongIterable source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(long item)
        {
            int oldSize = IntLongHashMap.this.size();

            if (IntLongHashMap.this.sentinelValues != null)
            {
                if (IntLongHashMap.this.sentinelValues.containsZeroKey && item == IntLongHashMap.this.sentinelValues.zeroValue)
                {
                    IntLongHashMap.this.removeKey(EMPTY_KEY);
                }
                if (IntLongHashMap.this.sentinelValues.containsOneKey && item == IntLongHashMap.this.sentinelValues.oneValue)
                {
                    IntLongHashMap.this.removeKey(REMOVED_KEY);
                }
            }
            for (int i = 0; i < IntLongHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(IntLongHashMap.this.keys[i]) && item == IntLongHashMap.this.values[i])
                {
                    IntLongHashMap.this.removeKey(IntLongHashMap.this.keys[i]);
                }
            }
            return oldSize != IntLongHashMap.this.size();
        }

        public boolean removeAll(LongIterable source)
        {
            int oldSize = IntLongHashMap.this.size();

            LongIterator iterator = source.longIterator();
            while (iterator.hasNext())
            {
                this.remove(iterator.next());
            }
            return oldSize != IntLongHashMap.this.size();
        }

        public boolean removeAll(long... source)
        {
            int oldSize = IntLongHashMap.this.size();

            for (long item : source)
            {
                this.remove(item);
            }
            return oldSize != IntLongHashMap.this.size();
        }

        public int size()
        {
            return IntLongHashMap.this.size();
        }

        public long[] toArray()
        {
            return IntLongHashMap.this.toArray();
        }
    }
}
