/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.map.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.IntIterable;
import com.gs.collections.api.LazyIntIterable;
import com.gs.collections.api.LazyShortIterable;
import com.gs.collections.api.ShortIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.bag.primitive.MutableIntBag;
import com.gs.collections.api.bag.primitive.MutableShortBag;
import com.gs.collections.api.block.function.primitive.IntToObjectFunction;
import com.gs.collections.api.block.function.primitive.IntToShortFunction;
import com.gs.collections.api.block.function.primitive.ShortFunction;
import com.gs.collections.api.block.function.primitive.ShortFunction0;
import com.gs.collections.api.block.function.primitive.ShortToShortFunction;
import com.gs.collections.api.block.function.primitive.ShortToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectShortToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.IntShortPredicate;
import com.gs.collections.api.block.predicate.primitive.IntPredicate;
import com.gs.collections.api.block.predicate.primitive.ShortPredicate;
import com.gs.collections.api.block.procedure.primitive.IntProcedure;
import com.gs.collections.api.block.procedure.primitive.IntShortProcedure;
import com.gs.collections.api.block.procedure.primitive.ShortProcedure;
import com.gs.collections.api.collection.MutableCollection;
import com.gs.collections.api.collection.primitive.ImmutableShortCollection;
import com.gs.collections.api.collection.primitive.MutableShortCollection;
import com.gs.collections.api.iterator.IntIterator;
import com.gs.collections.api.iterator.ShortIterator;
import com.gs.collections.api.list.primitive.MutableIntList;
import com.gs.collections.api.list.primitive.MutableShortList;
import com.gs.collections.api.map.primitive.IntShortMap;
import com.gs.collections.api.map.primitive.ImmutableIntShortMap;
import com.gs.collections.api.map.primitive.MutableIntShortMap;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.IntSet;
import com.gs.collections.api.set.primitive.ImmutableIntSet;
import com.gs.collections.api.set.primitive.MutableIntSet;
import com.gs.collections.api.set.primitive.MutableShortSet;
import com.gs.collections.impl.bag.mutable.primitive.IntHashBag;
import com.gs.collections.impl.bag.mutable.primitive.ShortHashBag;
import com.gs.collections.impl.block.factory.primitive.IntPredicates;
import com.gs.collections.impl.collection.mutable.primitive.SynchronizedShortCollection;
import com.gs.collections.impl.collection.mutable.primitive.UnmodifiableShortCollection;
import com.gs.collections.impl.factory.Sets;
import com.gs.collections.impl.factory.primitive.ShortLists;
import com.gs.collections.impl.factory.primitive.IntShortMaps;
import com.gs.collections.impl.factory.primitive.IntSets;
import com.gs.collections.impl.lazy.primitive.CollectIntToObjectIterable;
import com.gs.collections.impl.lazy.primitive.LazyShortIterableAdapter;
import com.gs.collections.impl.lazy.primitive.LazyIntIterableAdapter;
import com.gs.collections.impl.lazy.primitive.SelectIntIterable;
import com.gs.collections.impl.list.mutable.FastList;
import com.gs.collections.impl.list.mutable.primitive.IntArrayList;
import com.gs.collections.impl.list.mutable.primitive.ShortArrayList;
import com.gs.collections.impl.set.mutable.primitive.IntHashSet;
import com.gs.collections.impl.set.mutable.primitive.SynchronizedIntSet;
import com.gs.collections.impl.set.mutable.primitive.UnmodifiableIntSet;
import com.gs.collections.impl.set.mutable.primitive.ShortHashSet;

/**
 * This file was automatically generated from template file primitivePrimitiveHashMap.stg.
 *
 * @since 3.0.
 */
public class IntShortHashMap implements MutableIntShortMap, Externalizable
{
    static final short EMPTY_VALUE = (short) 0;
    private static final long serialVersionUID = 1L;
    private static final int EMPTY_KEY = 0;
    private static final int REMOVED_KEY = 1;

    private static final int OCCUPIED_DATA_RATIO = 2;
    private static final int OCCUPIED_SENTINEL_RATIO = 4;
    private static final int DEFAULT_INITIAL_CAPACITY = 8;

    private int[] keys;
    private short[] values;

    private int occupiedWithData;
    private int occupiedWithSentinels;

    private SentinelValues sentinelValues;

    public IntShortHashMap()
    {
        this.allocateTable(DEFAULT_INITIAL_CAPACITY << 1);
    }

    public IntShortHashMap(int initialCapacity)
    {
        if (initialCapacity < 0)
        {
            throw new IllegalArgumentException("initial capacity cannot be less than 0");
        }
        int capacity = this.smallestPowerOfTwoGreaterThan(this.fastCeil(initialCapacity * OCCUPIED_DATA_RATIO));
        this.allocateTable(capacity);
    }

    private int smallestPowerOfTwoGreaterThan(int n)
    {
        return n > 1 ? Integer.highestOneBit(n - 1) << 1 : 1;
    }

    public IntShortHashMap(IntShortMap map)
    {
        this(Math.max(map.size(), DEFAULT_INITIAL_CAPACITY));
        this.putAll(map);
    }

    private int fastCeil(float v)
    {
        int possibleResult = (int) v;
        if (v - possibleResult > 0.0F)
        {
            possibleResult++;
        }
        return possibleResult;
    }

    public static IntShortHashMap newWithKeysValues(int key1, short value1)
    {
        return new IntShortHashMap(1).withKeyValue(key1, value1);
    }

    public static IntShortHashMap newWithKeysValues(int key1, short value1, int key2, short value2)
    {
        return new IntShortHashMap(2).withKeysValues(key1, value1, key2, value2);
    }

    public static IntShortHashMap newWithKeysValues(int key1, short value1, int key2, short value2, int key3, short value3)
    {
        return new IntShortHashMap(3).withKeysValues(key1, value1, key2, value2, key3, value3);
    }

    public static IntShortHashMap newWithKeysValues(int key1, short value1, int key2, short value2, int key3, short value3, int key4, short value4)
    {
        return new IntShortHashMap(4).withKeysValues(key1, value1, key2, value2, key3, value3, key4, value4);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }

        if (!(obj instanceof IntShortMap))
        {
            return false;
        }

        IntShortMap other = (IntShortMap) obj;

        if (this.size() != other.size())
        {
            return false;
        }

        if (this.sentinelValues == null)
        {
            if (other.containsKey(EMPTY_KEY) || other.containsKey(REMOVED_KEY))
            {
                return false;
            }
        }
        else
        {
            if (this.sentinelValues.containsZeroKey && (!other.containsKey(EMPTY_KEY) || this.sentinelValues.zeroValue != other.getOrThrow(EMPTY_KEY)))
            {
                return false;
            }

            if (this.sentinelValues.containsOneKey && (!other.containsKey(REMOVED_KEY) || this.sentinelValues.oneValue != other.getOrThrow(REMOVED_KEY)))
            {
                return false;
            }
        }

        for (int i = 0; i < this.keys.length; i++)
        {
            int key = this.keys[i];
            if (isNonSentinel(key) && (!other.containsKey(key) || this.values[i] != other.getOrThrow(key)))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int result = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += EMPTY_KEY ^ (int) this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += REMOVED_KEY ^ (int) this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += this.keys[i] ^ (int) this.values[i];
            }
        }

        return result;
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public int size()
    {
        return this.occupiedWithData + (this.sentinelValues == null ? 0 : this.sentinelValues.size());
    }

    public boolean isEmpty()
    {
        return this.occupiedWithData == 0 && (this.sentinelValues == null || this.sentinelValues.size() == 0);
    }

    public boolean notEmpty()
    {
        return this.occupiedWithData != 0 || (this.sentinelValues != null && this.sentinelValues.size() != 0);
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            boolean first = true;

            if (this.sentinelValues != null)
            {
                if (this.sentinelValues.containsZeroKey)
                {
                    appendable.append(String.valueOf(EMPTY_KEY)).append("=").append(String.valueOf(this.sentinelValues.zeroValue));
                    first = false;
                }
                if (this.sentinelValues.containsOneKey)
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(REMOVED_KEY)).append("=").append(String.valueOf(this.sentinelValues.oneValue));
                    first = false;
                }
            }
            for (int i = 0; i < this.keys.length; i++)
            {
                int key = this.keys[i];
                if (isNonSentinel(key))
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(key)).append("=").append(String.valueOf(this.values[i]));
                    first = false;
                }
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public ShortIterator shortIterator()
    {
        return new InternalShortIterator();
    }

    public short[] toArray()
    {
        short[] array = new short[this.size()];
        int index = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                array[index] = this.sentinelValues.zeroValue;
                index++;
            }
            if (this.sentinelValues.containsOneKey)
            {
                array[index] = this.sentinelValues.oneValue;
                index++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                array[index] = this.values[i];
                index++;
            }
        }

        return array;
    }

    public boolean contains(short value)
    {
        return this.containsValue(value);
    }

    public boolean containsAll(short... source)
    {
        for (short each : source)
        {
            if (!this.contains(each))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(ShortIterable source)
    {
        return source.allSatisfy(new ShortPredicate()
        {
            public boolean accept(short value)
            {
                return IntShortHashMap.this.contains(value);
            }
        });
    }

    public void forEach(ShortProcedure procedure)
    {
        this.forEachValue(procedure);
    }

    public MutableShortCollection select(ShortPredicate predicate)
    {
        ShortArrayList result = new ShortArrayList();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }

        return result;
    }

    public MutableShortCollection reject(ShortPredicate predicate)
    {
        ShortArrayList result = new ShortArrayList();
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public <V> MutableCollection<V> collect(ShortToObjectFunction<? extends V> function)
    {
        FastList<V> target = FastList.newList(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                target.add(function.valueOf(this.sentinelValues.zeroValue));
            }
            if (this.sentinelValues.containsOneKey)
            {
                target.add(function.valueOf(this.sentinelValues.oneValue));
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                target.add(function.valueOf(this.values[i]));
            }
        }
        return target;
    }

    public short detectIfNone(ShortPredicate predicate, short value)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return this.values[i];
            }
        }
        return value;
    }

    public int count(ShortPredicate predicate)
    {
        int count = 0;
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                count++;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                count++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(ShortPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return true;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return true;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(ShortPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(ShortPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public <V> V injectInto(V injectedValue, ObjectShortToObjectFunction<? super V, ? extends V> function)
    {
        V result = injectedValue;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result = function.valueOf(result, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                result = function.valueOf(result, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result = function.valueOf(result, this.values[i]);
            }
        }

        return result;
    }

    public MutableShortList toList()
    {
        return ShortArrayList.newList(this);
    }

    public MutableShortSet toSet()
    {
        return ShortHashSet.newSet(this);
    }

    public MutableShortBag toBag()
    {
        return ShortHashBag.newBag(this);
    }

    public LazyShortIterable asLazy()
    {
        return new LazyShortIterableAdapter(this);
    }

    public void clear()
    {
        this.sentinelValues = null;
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;
        Arrays.fill(this.keys, EMPTY_KEY);
        Arrays.fill(this.values, EMPTY_VALUE);
    }

    public void put(int key, short value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return;
        }

        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return;
        }

        int index = this.probe(key);

        if (this.keys[index] == key)
        {
            // key already present in map
            this.values[index] = value;
            return;
        }

        if (this.keys[index] == REMOVED_KEY)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = key;
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
    }

    public void putAll(IntShortMap map)
    {
        map.forEachKeyValue(new IntShortProcedure()
        {
            public void value(int key, short value)
            {
                IntShortHashMap.this.put(key, value);
            }
        });
    }

    public void removeKey(int key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return;
            }
            if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.containsZeroKey = false;
                this.sentinelValues.zeroValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.containsOneKey = false;
                this.sentinelValues.oneValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.keys[index] = REMOVED_KEY;
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }
        }
    }

    public void remove(int key)
    {
        this.removeKey(key);
    }

    public short removeKeyIfAbsent(int key, short value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return value;
            }
            short oldValue = this.sentinelValues.zeroValue;
            if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.containsZeroKey = false;
                this.sentinelValues.zeroValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return oldValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return value;
            }
            short oldValue = this.sentinelValues.oneValue;
            if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.containsOneKey = false;
                this.sentinelValues.oneValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return oldValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.keys[index] = REMOVED_KEY;
            short oldValue = this.values[index];
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }

            return oldValue;
        }
        return value;
    }

    public short getIfAbsentPut(int key, short value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public short getIfAbsentPut(int key, ShortFunction0 function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                short value = function.value();
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            short value = function.value();
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                short value = function.value();
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            short value = function.value();
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        short value = function.value();
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public <P> short getIfAbsentPutWith(int key, ShortFunction<? super P> function, P parameter)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                short value = function.shortValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            short value = function.shortValueOf(parameter);
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                short value = function.shortValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            short value = function.shortValueOf(parameter);
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        short value = function.shortValueOf(parameter);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public short getIfAbsentPutWithKey(int key, IntToShortFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                short value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            short value = function.valueOf(key);
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                short value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            short value = function.valueOf(key);
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        this.keys[index] = key;
        short value = function.valueOf(key);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public short updateValue(int key, short initialValueIfAbsent, ShortToShortFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = function.valueOf(initialValueIfAbsent);
            }
            else if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.zeroValue = function.valueOf(this.sentinelValues.zeroValue);
            }
            else
            {
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = function.valueOf(initialValueIfAbsent);
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = function.valueOf(initialValueIfAbsent);
            }
            else if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.oneValue = function.valueOf(this.sentinelValues.oneValue);
            }
            else
            {
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = function.valueOf(initialValueIfAbsent);
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.values[index] = function.valueOf(this.values[index]);
            return this.values[index];
        }
        if (this.keys[index] == REMOVED_KEY)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = key;
        short value = function.valueOf(initialValueIfAbsent);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public IntShortHashMap withKeyValue(int key1, short value1)
    {
        this.put(key1, value1);
        return this;
    }

    public IntShortHashMap withKeysValues(int key1, short value1, int key2, short value2)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        return this;
    }

    public IntShortHashMap withKeysValues(int key1, short value1, int key2, short value2, int key3, short value3)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        return this;
    }

    public IntShortHashMap withKeysValues(int key1, short value1, int key2, short value2, int key3, short value3, int key4, short value4)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        this.put(key4, value4);
        return this;
    }

    public IntShortHashMap withoutKey(int key)
    {
        this.removeKey(key);
        return this;
    }

    public IntShortHashMap withoutAllKeys(IntIterable keys)
    {
        keys.forEach(new IntProcedure()
        {
            public void value(int key)
            {
                IntShortHashMap.this.removeKey(key);
            }
        });
        return this;
    }

    public MutableIntShortMap asUnmodifiable()
    {
        return new UnmodifiableIntShortMap(this);
    }

    public MutableIntShortMap asSynchronized()
    {
        return new SynchronizedIntShortMap(this);
    }

    public ImmutableIntShortMap toImmutable()
    {
        return IntShortMaps.immutable.ofAll(this);
    }

    public short get(int key)
    {
        return this.getIfAbsent(key, EMPTY_VALUE);
    }

    public short getIfAbsent(int key, short ifAbsent)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        return ifAbsent;
    }

    public short getOrThrow(int key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]))
        {
            return this.values[index];
        }
        throw new IllegalStateException("Key " + key + " not present.");
    }

    public boolean containsKey(int key)
    {
        if (isEmptyKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsZeroKey;
        }
        if (isRemovedKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsOneKey;
        }
        return this.keys[this.probe(key)] == key;
    }

    public boolean containsValue(short value)
    {
        if (this.sentinelValues != null && this.sentinelValues.containsValue(value))
        {
            return true;
        }
        for (int i = 0; i < this.values.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && this.values[i] == value)
            {
                return true;
            }
        }
        return false;
    }

    public void forEachValue(ShortProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.values[i]);
            }
        }
    }

    public void forEachKey(IntProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY);
            }
        }
        for (int key : this.keys)
        {
            if (isNonSentinel(key))
            {
                procedure.value(key);
            }
        }
    }

    public void forEachKeyValue(IntShortProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.keys[i], this.values[i]);
            }
        }
    }

    public LazyIntIterable keysView()
    {
        return new KeysView();
    }

    public IntShortHashMap select(IntShortPredicate predicate)
    {
        IntShortHashMap result = new IntShortHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.keys[i], this.values[i]))
            {
                result.put(this.keys[i], this.values[i]);
            }
        }

        return result;
    }

    public IntShortHashMap reject(IntShortPredicate predicate)
    {
        IntShortHashMap result = new IntShortHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.keys[i], this.values[i]))
            {
                result.put(this.keys[i], this.values[i]);
            }
        }
        return result;
    }

    public long sum()
    {
        long result = 0L;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += this.values[i];
            }
        }

        return result;
    }

    public short max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        ShortIterator iterator = this.shortIterator();
        short max = iterator.next();
        while (iterator.hasNext())
        {
            short value = iterator.next();
            if (max < value)
            {
                max = value;
            }
        }
        return max;
    }

    public short maxIfEmpty(short defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public short min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        ShortIterator iterator = this.shortIterator();
        short min = iterator.next();
        while (iterator.hasNext())
        {
            short value = iterator.next();
            if (value < min)
            {
                min = value;
            }
        }
        return min;
    }

    public short minIfEmpty(short defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return (double) this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        short[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            short first = sortedArray[middleIndex];
            short second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public short[] toSortedArray()
    {
        short[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public MutableShortList toSortedList()
    {
        return ShortArrayList.newList(this).sortThis();
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                out.writeInt(EMPTY_KEY);
                out.writeShort(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                out.writeInt(REMOVED_KEY);
                out.writeShort(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                out.writeInt(this.keys[i]);
                out.writeShort(this.values[i]);
            }
        }
    }

    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
    {
        int size = in.readInt();
        for (int i = 0; i < size; i++)
        {
            this.put(in.readInt(), in.readShort());
        }
    }

    /**
     * Rehashes every element in the set into a new backing table of the smallest possible size and eliminating removed sentinels.
     */
    public void compact()
    {
        this.rehash(this.smallestPowerOfTwoGreaterThan(this.size()));
    }

    private void rehash()
    {
        this.rehash(this.keys.length);
    }

    private void rehashAndGrow()
    {
        this.rehash(this.keys.length << 1);
    }

    private void rehash(int newCapacity)
    {
        int oldLength = this.keys.length;
        int[] old = this.keys;
        short[] oldValues = this.values;
        this.allocateTable(newCapacity);
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;

        for (int i = 0; i < oldLength; i++)
        {
            if (isNonSentinel(old[i]))
            {
                this.put(old[i], oldValues[i]);
            }
        }
    }

    // exposed for testing
    int probe(int element)
    {
        int index = this.spread(element);
        int keyAtIndex = this.keys[index];

        if (keyAtIndex == element || keyAtIndex == EMPTY_KEY)
        {
            return index;
        }

        int removedIndex = keyAtIndex == REMOVED_KEY ? index : -1;
        int nextIndex = index;
        int probe = 17;

        // loop until an empty slot is reached
        while (true)
        {
            // Probe algorithm: 17*n*(n+1)/2 where n = number of collisions
            nextIndex += probe;
            probe += 17;
            nextIndex &= this.keys.length - 1;

            if (this.keys[nextIndex] == element)
            {
                return nextIndex;
            }
            if (this.keys[nextIndex] == REMOVED_KEY)
            {
                if (removedIndex == -1)
                {
                    removedIndex = nextIndex;
                }
            }
            else if (this.keys[nextIndex] == EMPTY_KEY)
            {
                return removedIndex == -1 ? nextIndex : removedIndex;
            }
        }
    }

    // exposed for testing
    int spread(int element)
    {
        int code = element;
        code ^= 61 ^ (code >> 16);
        code += code << 3;
        code ^= code >> 4;
        code *= 0x27d4eb2d;
        code ^= code >> 15;
        return code & (this.keys.length - 1);
    }

    private void allocateTable(int sizeToAllocate)
    {
        this.keys = new int[sizeToAllocate];
        this.values = new short[sizeToAllocate];
    }

    private static boolean isEmptyKey(int key)
    {
        return key == EMPTY_KEY;
    }

    private static boolean isRemovedKey(int key)
    {
        return key == REMOVED_KEY;
    }

    private static boolean isNonSentinel(int key)
    {
        return !isEmptyKey(key) && !isRemovedKey(key);
    }

    private int maxOccupiedWithData()
    {
        int capacity = this.keys.length;
        // need at least one free slot for open addressing
        return Math.min(capacity - 1, capacity / OCCUPIED_DATA_RATIO);
    }

    private int maxOccupiedWithSentinels()
    {
        return this.keys.length / OCCUPIED_SENTINEL_RATIO;
    }

    private static final class SentinelValues
    {
        private boolean containsZeroKey;
        private boolean containsOneKey;
        private short zeroValue;
        private short oneValue;

        public int size()
        {
            return (this.containsZeroKey ? 1 : 0) + (this.containsOneKey ? 1 : 0);
        }

        public boolean containsValue(short value)
        {
            boolean valueEqualsZeroValue = this.containsZeroKey && this.zeroValue == value;
            boolean valueEqualsOneValue = this.containsOneKey && this.oneValue == value;
            return valueEqualsZeroValue || valueEqualsOneValue;
        }
    }

    private class InternalShortIterator implements ShortIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < IntShortHashMap.this.size();
        }

        public short next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (IntShortHashMap.this.containsKey(EMPTY_KEY))
                {
                    return IntShortHashMap.this.get(EMPTY_KEY);
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (IntShortHashMap.this.containsKey(REMOVED_KEY))
                {
                    return IntShortHashMap.this.get(REMOVED_KEY);
                }
            }

            int[] keys = IntShortHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            short result = IntShortHashMap.this.values[this.position];
            this.position++;
            return result;
        }
    }

    private class KeysView implements LazyIntIterable
    {
        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public int size()
        {
            return IntShortHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return IntShortHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return IntShortHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (IntShortHashMap.this.sentinelValues != null)
                {
                    if (IntShortHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (IntShortHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (int key : IntShortHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public IntIterator intIterator()
        {
            return new KeySetIterator();
        }

        public boolean contains(int key)
        {
            return IntShortHashMap.this.containsKey(key);
        }

        public boolean containsAll(int... keys)
        {
            for (int key : keys)
            {
                if (!IntShortHashMap.this.containsKey(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(IntIterable source)
        {
            return source.allSatisfy(new IntPredicate()
            {
                public boolean accept(int key)
                {
                    return IntShortHashMap.this.containsKey(key);
                }
            });
        }

        public void forEach(IntProcedure procedure)
        {
            IntShortHashMap.this.forEachKey(procedure);
        }

        public int count(IntPredicate predicate)
        {
            int count = 0;
            if (IntShortHashMap.this.sentinelValues != null)
            {
                if (IntShortHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (IntShortHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (int key : IntShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(IntPredicate predicate)
        {
            if (IntShortHashMap.this.sentinelValues != null)
            {
                if (IntShortHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (IntShortHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (int key : IntShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(IntPredicate predicate)
        {
            if (IntShortHashMap.this.sentinelValues != null)
            {
                if (IntShortHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (IntShortHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (int key : IntShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(IntPredicate predicate)
        {
            if (IntShortHashMap.this.sentinelValues != null)
            {
                if (IntShortHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (IntShortHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (int key : IntShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public LazyIntIterable select(IntPredicate predicate)
        {
            return new SelectIntIterable(this, predicate);
        }

        public LazyIntIterable reject(IntPredicate predicate)
        {
            return new SelectIntIterable(this, IntPredicates.not(predicate));
        }

        public int detectIfNone(IntPredicate predicate, int value)
        {
            if (IntShortHashMap.this.sentinelValues != null)
            {
                if (IntShortHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (IntShortHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (int key : IntShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return value;
        }

        public <V> LazyIterable<V> collect(IntToObjectFunction<? extends V> function)
        {
            return new CollectIntToObjectIterable<V>(this, function);
        }

        public long sum()
        {
            long result = 0L;

            if (IntShortHashMap.this.sentinelValues != null)
            {
                if (IntShortHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += EMPTY_KEY;
                }
                if (IntShortHashMap.this.sentinelValues.containsOneKey)
                {
                    result += REMOVED_KEY;
                }
            }
            for (int key : IntShortHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result += key;
                }
            }

            return result;
        }

        public int max()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            IntIterator iterator = this.intIterator();
            int max = iterator.next();
            while (iterator.hasNext())
            {
                int value = iterator.next();
                if (max < value)
                {
                    max = value;
                }
            }
            return max;
        }

        public int min()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            IntIterator iterator = this.intIterator();
            int min = iterator.next();
            while (iterator.hasNext())
            {
                int value = iterator.next();
                if (value < min)
                {
                    min = value;
                }
            }
            return min;
        }

        public int minIfEmpty(int defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.min();
        }

        public int maxIfEmpty(int defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.max();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return (double) this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            int[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                int first = sortedArray[middleIndex];
                int second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public int[] toSortedArray()
        {
            int[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public int[] toArray()
        {
            int size = IntShortHashMap.this.size();
            final int[] result = new int[size];
            IntShortHashMap.this.forEachKey(new IntProcedure()
            {
                private int index;

                public void value(int each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public MutableIntList toList()
        {
            return IntArrayList.newList(this);
        }

        public MutableIntList toSortedList()
        {
            return IntArrayList.newList(this).sortThis();
        }

        public MutableIntSet toSet()
        {
            return IntHashSet.newSet(this);
        }

        public MutableIntBag toBag()
        {
            return IntHashBag.newBag(this);
        }

        public LazyIntIterable asLazy()
        {
            return this;
        }
    }

    private class KeySetIterator implements IntIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < IntShortHashMap.this.size();
        }

        public int next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (IntShortHashMap.this.containsKey(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (IntShortHashMap.this.containsKey(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }

            int[] keys = IntShortHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            int result = keys[this.position];
            this.position++;
            return result;
        }
    }

    public MutableIntSet keySet()
    {
        return new KeySet();
    }

    private class KeySet implements MutableIntSet
    {
        public IntIterator intIterator()
        {
            return new KeySetIterator();
        }

        public void forEach(IntProcedure procedure)
        {
            IntShortHashMap.this.forEachKey(procedure);
        }

        public int count(IntPredicate predicate)
        {
            int count = 0;
            if (IntShortHashMap.this.sentinelValues != null)
            {
                if (IntShortHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (IntShortHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (int key : IntShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(IntPredicate predicate)
        {
            if (IntShortHashMap.this.sentinelValues != null)
            {
                if (IntShortHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (IntShortHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (int key : IntShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(IntPredicate predicate)
        {
            if (IntShortHashMap.this.sentinelValues != null)
            {
                if (IntShortHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (IntShortHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (int key : IntShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(IntPredicate predicate)
        {
            if (IntShortHashMap.this.sentinelValues != null)
            {
                if (IntShortHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (IntShortHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (int key : IntShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean add(int element)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(int... source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(IntIterable source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(int key)
        {
            int oldSize = IntShortHashMap.this.size();
            IntShortHashMap.this.removeKey(key);
            return oldSize != IntShortHashMap.this.size();
        }

        public boolean removeAll(IntIterable source)
        {
            int oldSize = IntShortHashMap.this.size();
            IntIterator iterator = source.intIterator();
            while (iterator.hasNext())
            {
                IntShortHashMap.this.removeKey(iterator.next());
            }
            return oldSize != IntShortHashMap.this.size();
        }

        public boolean removeAll(int... source)
        {
            int oldSize = IntShortHashMap.this.size();
            for (int item : source)
            {
                IntShortHashMap.this.removeKey(item);
            }
            return oldSize != IntShortHashMap.this.size();
        }

        public void clear()
        {
            IntShortHashMap.this.clear();
        }

        public MutableIntSet select(IntPredicate predicate)
        {
            MutableIntSet result = new IntHashSet();
            if (IntShortHashMap.this.sentinelValues != null)
            {
                if (IntShortHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (IntShortHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (int key : IntShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableIntSet reject(IntPredicate predicate)
        {
            MutableIntSet result = new IntHashSet();
            if (IntShortHashMap.this.sentinelValues != null)
            {
                if (IntShortHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (IntShortHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (int key : IntShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableIntSet with(int element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableIntSet without(int element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableIntSet withAll(IntIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableIntSet withoutAll(IntIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public int detectIfNone(IntPredicate predicate, int ifNone)
        {
            if (IntShortHashMap.this.sentinelValues != null)
            {
                if (IntShortHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (IntShortHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (int key : IntShortHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return ifNone;
        }

        public <V> MutableSet<V> collect(IntToObjectFunction<? extends V> function)
        {
            MutableSet<V> result = Sets.mutable.with();
            if (IntShortHashMap.this.sentinelValues != null)
            {
                if (IntShortHashMap.this.sentinelValues.containsZeroKey)
                {
                    result.add(function.valueOf(EMPTY_KEY));
                }
                if (IntShortHashMap.this.sentinelValues.containsOneKey)
                {
                    result.add(function.valueOf(REMOVED_KEY));
                }
            }
            for (int key : IntShortHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result.add(function.valueOf(key));
                }
            }
            return result;
        }

        public MutableIntSet asUnmodifiable()
        {
            return UnmodifiableIntSet.of(this);
        }

        public MutableIntSet asSynchronized()
        {
            return SynchronizedIntSet.of(this);
        }

        public long sum()
        {
            long sum = 0L;
            if (IntShortHashMap.this.sentinelValues != null)
            {
                if (IntShortHashMap.this.sentinelValues.containsZeroKey)
                {
                    sum += EMPTY_KEY;
                }
                if (IntShortHashMap.this.sentinelValues.containsOneKey)
                {
                    sum += REMOVED_KEY;
                }
            }
            for (int key : IntShortHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    sum += key;
                }
            }
            return sum;
        }

        public int max()
        {
            if (IntShortHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            int max = 0;
            boolean isMaxSet = false;

            if (IntShortHashMap.this.sentinelValues != null)
            {
                if (IntShortHashMap.this.sentinelValues.containsZeroKey)
                {
                    max = EMPTY_KEY;
                    isMaxSet = true;
                }
                if (IntShortHashMap.this.sentinelValues.containsOneKey && (!isMaxSet || max < REMOVED_KEY))
                {
                    max = REMOVED_KEY;
                    isMaxSet = true;
                }
            }
            for (int i = 0; i < IntShortHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(IntShortHashMap.this.keys[i]) && (!isMaxSet || max < IntShortHashMap.this.keys[i]))
                {
                    max = IntShortHashMap.this.keys[i];
                    isMaxSet = true;
                }
            }
            return max;
        }

        public int maxIfEmpty(int defaultValue)
        {
            if (IntShortHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.max();
        }

        public int min()
        {
            if (IntShortHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            int min = 0;
            boolean isMinSet = false;

            if (IntShortHashMap.this.sentinelValues != null)
            {
                if (IntShortHashMap.this.sentinelValues.containsZeroKey)
                {
                    min = EMPTY_KEY;
                    isMinSet = true;
                }
                if (IntShortHashMap.this.sentinelValues.containsOneKey && (!isMinSet || REMOVED_KEY < min))
                {
                    min = REMOVED_KEY;
                    isMinSet = true;
                }
            }
            for (int i = 0; i < IntShortHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(IntShortHashMap.this.keys[i]) && (!isMinSet || IntShortHashMap.this.keys[i] < min))
                {
                    min = IntShortHashMap.this.keys[i];
                    isMinSet = true;
                }
            }
            return min;
        }

        public int minIfEmpty(int defaultValue)
        {
            if (IntShortHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.min();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return (double) this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            int[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                int first = sortedArray[middleIndex];
                int second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public int[] toSortedArray()
        {
            int[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public MutableIntList toSortedList()
        {
            return IntArrayList.newList(this).sortThis();
        }

        public int[] toArray()
        {
            int size = IntShortHashMap.this.size();
            final int[] result = new int[size];
            IntShortHashMap.this.forEachKey(new IntProcedure()
            {
                private int index;

                public void value(int each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public boolean contains(int value)
        {
            return IntShortHashMap.this.containsKey(value);
        }

        public boolean containsAll(int... source)
        {
            for (int item : source)
            {
                if (!IntShortHashMap.this.containsKey(item))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(IntIterable source)
        {
            IntIterator iterator = source.intIterator();
            while (iterator.hasNext())
            {
                if (!IntShortHashMap.this.containsKey(iterator.next()))
                {
                    return false;
                }
            }
            return true;
        }

        public MutableIntList toList()
        {
            return IntArrayList.newList(this);
        }

        public MutableIntSet toSet()
        {
            return IntHashSet.newSet(this);
        }

        public MutableIntBag toBag()
        {
            return IntHashBag.newBag(this);
        }

        public LazyIntIterable asLazy()
        {
            return new LazyIntIterableAdapter(this);
        }

        public IntSet freeze()
        {
            throw new UnsupportedOperationException();
        }

        public ImmutableIntSet toImmutable()
        {
            return IntSets.immutable.withAll(this);
        }

        public int size()
        {
            return IntShortHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return IntShortHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return IntShortHashMap.this.notEmpty();
        }

        @Override
        public boolean equals(Object obj)
        {
            if (this == obj)
            {
                return true;
            }

            if (!(obj instanceof IntSet))
            {
                return false;
            }

            IntSet other = (IntSet) obj;
            return this.size() == other.size() && this.containsAll(other.toArray());
        }

        @Override
        public int hashCode()
        {
            int result = 0;

            if (IntShortHashMap.this.sentinelValues != null)
            {
                if (IntShortHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += EMPTY_KEY;
                }
                if (IntShortHashMap.this.sentinelValues.containsOneKey)
                {
                    result += REMOVED_KEY;
                }
            }
            for (int i = 0; i < IntShortHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(IntShortHashMap.this.keys[i]))
                {
                    result += IntShortHashMap.this.keys[i];
                }
            }

            return result;
        }

        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (IntShortHashMap.this.sentinelValues != null)
                {
                    if (IntShortHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (IntShortHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (int key : IntShortHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }
    }

    public MutableShortCollection values()
    {
        return new ValuesCollection();
    }

    private class ValuesCollection implements MutableShortCollection
    {
        public void clear()
        {
            IntShortHashMap.this.clear();
        }

        public MutableShortCollection select(ShortPredicate predicate)
        {
            return IntShortHashMap.this.select(predicate);
        }

        public MutableShortCollection reject(ShortPredicate predicate)
        {
            return IntShortHashMap.this.reject(predicate);
        }

        public short detectIfNone(ShortPredicate predicate, short ifNone)
        {
            return IntShortHashMap.this.detectIfNone(predicate, ifNone);
        }

        public <V> MutableCollection<V> collect(ShortToObjectFunction<? extends V> function)
        {
            return IntShortHashMap.this.collect(function);
        }

        public long sum()
        {
            return IntShortHashMap.this.sum();
        }

        public short max()
        {
            return IntShortHashMap.this.max();
        }

        public short maxIfEmpty(short defaultValue)
        {
            return IntShortHashMap.this.maxIfEmpty(defaultValue);
        }

        public short min()
        {
            return IntShortHashMap.this.min();
        }

        public short minIfEmpty(short defaultValue)
        {
            return IntShortHashMap.this.minIfEmpty(defaultValue);
        }

        public double average()
        {
            return IntShortHashMap.this.average();
        }

        public double median()
        {
            return IntShortHashMap.this.median();
        }

        public short[] toSortedArray()
        {
            return IntShortHashMap.this.toSortedArray();
        }

        public MutableShortList toSortedList()
        {
            return IntShortHashMap.this.toSortedList();
        }

        public MutableShortCollection with(short element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableShortCollection without(short element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableShortCollection withAll(ShortIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableShortCollection withoutAll(ShortIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableShortCollection asUnmodifiable()
        {
            return UnmodifiableShortCollection.of(this);
        }

        public MutableShortCollection asSynchronized()
        {
            return SynchronizedShortCollection.of(this);
        }

        public ImmutableShortCollection toImmutable()
        {
            return ShortLists.immutable.withAll(this);
        }

        public boolean contains(short value)
        {
            return IntShortHashMap.this.containsValue(value);
        }

        public boolean containsAll(short... source)
        {
            return IntShortHashMap.this.containsAll(source);
        }

        public boolean containsAll(ShortIterable source)
        {
            return IntShortHashMap.this.containsAll(source);
        }

        public MutableShortList toList()
        {
            return IntShortHashMap.this.toList();
        }

        public MutableShortSet toSet()
        {
            return IntShortHashMap.this.toSet();
        }

        public MutableShortBag toBag()
        {
            return IntShortHashMap.this.toBag();
        }

        public LazyShortIterable asLazy()
        {
            return new LazyShortIterableAdapter(this);
        }

        public boolean isEmpty()
        {
            return IntShortHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return IntShortHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);

                boolean first = true;

                if (IntShortHashMap.this.sentinelValues != null)
                {
                    if (IntShortHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(IntShortHashMap.this.sentinelValues.zeroValue));
                        first = false;
                    }
                    if (IntShortHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(IntShortHashMap.this.sentinelValues.oneValue));
                        first = false;
                    }
                }
                for (int i = 0; i < IntShortHashMap.this.keys.length; i++)
                {
                    int key = IntShortHashMap.this.keys[i];
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(IntShortHashMap.this.values[i]));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public ShortIterator shortIterator()
        {
            return IntShortHashMap.this.shortIterator();
        }

        public void forEach(ShortProcedure procedure)
        {
            IntShortHashMap.this.forEach(procedure);
        }

        public int count(ShortPredicate predicate)
        {
            return IntShortHashMap.this.count(predicate);
        }

        public boolean anySatisfy(ShortPredicate predicate)
        {
            return IntShortHashMap.this.anySatisfy(predicate);
        }

        public boolean allSatisfy(ShortPredicate predicate)
        {
            return IntShortHashMap.this.allSatisfy(predicate);
        }

        public boolean noneSatisfy(ShortPredicate predicate)
        {
            return IntShortHashMap.this.noneSatisfy(predicate);
        }

        public boolean add(short element)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(short... source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(ShortIterable source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(short item)
        {
            int oldSize = IntShortHashMap.this.size();

            if (IntShortHashMap.this.sentinelValues != null)
            {
                if (IntShortHashMap.this.sentinelValues.containsZeroKey && item == IntShortHashMap.this.sentinelValues.zeroValue)
                {
                    IntShortHashMap.this.removeKey(EMPTY_KEY);
                }
                if (IntShortHashMap.this.sentinelValues.containsOneKey && item == IntShortHashMap.this.sentinelValues.oneValue)
                {
                    IntShortHashMap.this.removeKey(REMOVED_KEY);
                }
            }
            for (int i = 0; i < IntShortHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(IntShortHashMap.this.keys[i]) && item == IntShortHashMap.this.values[i])
                {
                    IntShortHashMap.this.removeKey(IntShortHashMap.this.keys[i]);
                }
            }
            return oldSize != IntShortHashMap.this.size();
        }

        public boolean removeAll(ShortIterable source)
        {
            int oldSize = IntShortHashMap.this.size();

            ShortIterator iterator = source.shortIterator();
            while (iterator.hasNext())
            {
                this.remove(iterator.next());
            }
            return oldSize != IntShortHashMap.this.size();
        }

        public boolean removeAll(short... source)
        {
            int oldSize = IntShortHashMap.this.size();

            for (short item : source)
            {
                this.remove(item);
            }
            return oldSize != IntShortHashMap.this.size();
        }

        public int size()
        {
            return IntShortHashMap.this.size();
        }

        public short[] toArray()
        {
            return IntShortHashMap.this.toArray();
        }
    }
}
