/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.set.mutable.primitive;

import java.util.Collection;
import java.util.Collections;

import com.gs.collections.api.ShortIterable;
import com.gs.collections.api.LazyShortIterable;
import com.gs.collections.api.block.function.primitive.ShortToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.ShortPredicate;
import com.gs.collections.api.iterator.ShortIterator;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.ImmutableShortSet;
import com.gs.collections.api.set.primitive.ShortSet;
import com.gs.collections.api.set.primitive.MutableShortSet;
import com.gs.collections.impl.collection.mutable.primitive.AbstractSynchronizedShortCollection;
import com.gs.collections.impl.lazy.primitive.LazyShortIterableAdapter;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;

/**
 * A synchronized view of a {@link MutableShortSet}. It is imperative that the user manually synchronize on the collection when iterating over it using the
 * {@link ShortIterator}, as per {@link Collections#synchronizedCollection(Collection)}.
 * <p>
 * This file was automatically generated from template file synchronizedPrimitiveSet.stg.
 * </p>
 *
 * @see MutableShortSet#asSynchronized()
 * @see MutableSet#asSynchronized()
 * @since 3.1.
 */
@ThreadSafe
public final class SynchronizedShortSet
        extends AbstractSynchronizedShortCollection
        implements MutableShortSet
{
    private static final long serialVersionUID = 1L;

    SynchronizedShortSet(MutableShortSet set)
    {
        super(set);
    }

    SynchronizedShortSet(MutableShortSet set, Object newLock)
    {
        super(set, newLock);
    }

    /**
     * This method will take a MutableShortSet and wrap it directly in a SynchronizedShortSet.
     */
    public static SynchronizedShortSet of(MutableShortSet set)
    {
        return new SynchronizedShortSet(set);
    }

    /**
     * This method will take a MutableShortSet and wrap it directly in a SynchronizedShortSet.
     * Additionally, a developer specifies which lock to use with the collection.
     */
    public static SynchronizedShortSet of(MutableShortSet set, Object lock)
    {
        return new SynchronizedShortSet(set, lock);
    }

    @GuardedBy("getLock()")
    private MutableShortSet getMutableShortSet()
    {
        return (MutableShortSet) this.getShortCollection();
    }

    @Override
    public SynchronizedShortSet without(short element)
    {
        synchronized (this.getLock())
        {
            this.getMutableShortSet().remove(element);
        }
        return this;
    }

    @Override
    public SynchronizedShortSet with(short element)
    {
        synchronized (this.getLock())
        {
            this.getMutableShortSet().add(element);
        }
        return this;
    }

    @Override
    public SynchronizedShortSet withAll(ShortIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableShortSet().addAll(elements.toArray());
        }
        return this;
    }

    @Override
    public SynchronizedShortSet withoutAll(ShortIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableShortSet().removeAll(elements);
        }
        return this;
    }

    @Override
    public MutableShortSet select(ShortPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableShortSet().select(predicate);
        }
    }

    @Override
    public MutableShortSet reject(ShortPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableShortSet().reject(predicate);
        }
    }

    @Override
    public <V> MutableSet<V> collect(ShortToObjectFunction<? extends V> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableShortSet().collect(function);
        }
    }

    @Override
    public boolean equals(Object otherSet)
    {
        synchronized (this.getLock())
        {
            return this.getMutableShortSet().equals(otherSet);
        }
    }

    @Override
    public int hashCode()
    {
        synchronized (this.getLock())
        {
            return this.getMutableShortSet().hashCode();
        }
    }

    @Override
    public LazyShortIterable asLazy()
    {
        synchronized (this.getLock())
        {
            return new LazyShortIterableAdapter(this);
        }
    }

    @Override
    public MutableShortSet asUnmodifiable()
    {
        return new UnmodifiableShortSet(this);
    }

    @Override
    public MutableShortSet asSynchronized()
    {
        return this;
    }

    public ShortSet freeze()
    {
        synchronized (this.getLock())
        {
            return this.getMutableShortSet().freeze();
        }
    }

    @Override
    public ImmutableShortSet toImmutable()
    {
        synchronized (this.getLock())
        {
            return this.getMutableShortSet().toImmutable();
        }
    }
}
