/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.stack.immutable.primitive;

import java.io.IOException;
import java.io.Serializable;

import com.gs.collections.api.IntIterable;
import com.gs.collections.api.LazyIntIterable;
import com.gs.collections.api.bag.primitive.MutableIntBag;
import com.gs.collections.api.block.function.primitive.IntToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.IntPredicate;
import com.gs.collections.api.block.procedure.primitive.IntProcedure;
import com.gs.collections.api.iterator.IntIterator;
import com.gs.collections.api.list.primitive.IntList;
import com.gs.collections.api.list.primitive.MutableIntList;
import com.gs.collections.api.set.primitive.MutableIntSet;
import com.gs.collections.api.stack.ImmutableStack;
import com.gs.collections.api.stack.primitive.IntStack;
import com.gs.collections.api.stack.primitive.ImmutableIntStack;
import com.gs.collections.impl.bag.mutable.primitive.IntHashBag;
import com.gs.collections.impl.factory.Stacks;
import com.gs.collections.impl.factory.primitive.IntStacks;
import com.gs.collections.impl.lazy.primitive.LazyIntIterableAdapter;
import com.gs.collections.impl.list.mutable.primitive.IntArrayList;
import com.gs.collections.impl.set.mutable.primitive.IntHashSet;
import com.gs.collections.impl.stack.mutable.primitive.IntArrayStack;
import net.jcip.annotations.Immutable;

/**
 * ImmutableIntSingletonStack is an optimization for {@link ImmutableIntStack} of size 1.
 * This file was automatically generated from template file immutablePrimitiveSingletonStack.stg.
 */
@Immutable
final class ImmutableIntSingletonStack implements ImmutableIntStack, Serializable
{
    private static final long serialVersionUID = 1L;
    private final int element1;

    ImmutableIntSingletonStack(int element)
    {
        this.element1 = element;
    }

    public IntIterator intIterator()
    {
        return IntArrayStack.newStackWith(this.element1).intIterator();
    }

    public void forEach(IntProcedure procedure)
    {
        procedure.value(this.element1);
    }

    public int count(IntPredicate predicate)
    {
        return predicate.accept(this.element1) ? 1 : 0;
    }

    public boolean anySatisfy(IntPredicate predicate)
    {
        return predicate.accept(this.element1);
    }

    public boolean allSatisfy(IntPredicate predicate)
    {
        return predicate.accept(this.element1);
    }

    public boolean noneSatisfy(IntPredicate predicate)
    {
        return !predicate.accept(this.element1);
    }

    public int peek()
    {
        return this.element1;
    }

    public IntList peek(int count)
    {
        this.checkNegativeCount(count);
        if (count == 0)
        {
            return new IntArrayList(0);
        }
        if (count == 1)
        {
            return IntArrayList.newListWith(this.element1);
        }
        throw new IllegalArgumentException("Count must be less than or equal to size: Count = " + count + " Size = 1");
    }

    public int peekAt(int index)
    {
        this.checkNegativeCount(index);
        if (index == 0)
        {
            return this.element1;
        }
        throw new IllegalArgumentException("Index must be less than size: Index = " + index + " Size = 1");
    }

    public ImmutableIntStack select(IntPredicate predicate)
    {
        return predicate.accept(this.element1) ? IntStacks.immutable.with(this.element1)
                : IntStacks.immutable.with();
    }

    public ImmutableIntStack reject(IntPredicate predicate)
    {
        return predicate.accept(this.element1) ? IntStacks.immutable.with()
                : IntStacks.immutable.with(this.element1);
    }

    public int detectIfNone(IntPredicate predicate, int ifNone)
    {
        return predicate.accept(this.element1) ? this.element1 : ifNone;
    }

    public <V> ImmutableStack<V> collect(IntToObjectFunction<? extends V> function)
    {
        return Stacks.immutable.with(function.valueOf(this.element1));
    }

    public long sum()
    {
        return this.element1;
    }

    public int max()
    {
        return this.element1;
    }

    public int maxIfEmpty(int defaultValue)
    {
        return this.element1;
    }

    public int min()
    {
        return this.element1;
    }

    public int minIfEmpty(int defaultValue)
    {
        return this.element1;
    }

    public double average()
    {
        return this.element1;
    }

    public double median()
    {
        return this.element1;
    }

    public int[] toSortedArray()
    {
        return new int[]{this.element1};
    }

    public MutableIntList toSortedList()
    {
        return IntArrayList.newListWith(this.element1);
    }

    public int[] toArray()
    {
        return new int[]{this.element1};
    }

    public boolean contains(int value)
    {
        return this.element1 == value;
    }

    public boolean containsAll(int... source)
    {
        for (int value : source)
        {
            if (this.element1 != value)
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(IntIterable source)
    {
        for (IntIterator iterator = source.intIterator(); iterator.hasNext(); )
        {
            if (this.element1 != iterator.next())
            {
                return false;
            }
        }
        return true;
    }

    public MutableIntList toList()
    {
        return IntArrayList.newListWith(this.element1);
    }

    public MutableIntSet toSet()
    {
        return IntHashSet.newSetWith(this.element1);
    }

    public MutableIntBag toBag()
    {
        return IntHashBag.newBagWith(this.element1);
    }

    public LazyIntIterable asLazy()
    {
        return new LazyIntIterableAdapter(this);
    }

    public ImmutableIntStack toImmutable()
    {
        return this;
    }

    public ImmutableIntStack push(int element)
    {
        return IntStacks.immutable.with(this.element1, element);
    }

    public ImmutableIntStack pop()
    {
        return IntStacks.immutable.with();
    }

    public ImmutableIntStack pop(int count)
    {
        this.checkNegativeCount(count);
        if (count == 0)
        {
            return this;
        }
        if (count == 1)
        {
            return IntStacks.immutable.with();
        }
        throw new IllegalArgumentException("Count must be less than size: Count = " + count + " Size = 1");
    }

    private void checkNegativeCount(int count)
    {
        if (count < 0)
        {
            throw new IllegalArgumentException("Count must be positive but was " + count);
        }
    }

    public int size()
    {
        return 1;
    }

    public boolean isEmpty()
    {
        return false;
    }

    public boolean notEmpty()
    {
        return true;
    }

    @Override
    public boolean equals(Object otherStack)
    {
        if (otherStack == this)
        {
            return true;
        }
        if (!(otherStack instanceof IntStack))
        {
            return false;
        }
        IntStack stack = (IntStack) otherStack;
        if (stack.size() != 1)
        {
            return false;
        }
        return this.element1 == stack.peek();
    }

    @Override
    public int hashCode()
    {
        return 31 + this.element1;
    }

    @Override
    public String toString()
    {
        return "[" + this.element1 + ']';
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);
            appendable.append(String.valueOf(this.element1));
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }
}
