/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.bag.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.FloatIterable;
import com.gs.collections.api.LazyFloatIterable;
import com.gs.collections.api.bag.MutableBag;
import com.gs.collections.api.bag.primitive.FloatBag;
import com.gs.collections.api.bag.primitive.ImmutableFloatBag;
import com.gs.collections.api.bag.primitive.MutableFloatBag;
import com.gs.collections.api.block.function.primitive.FloatToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectFloatToObjectFunction;
import com.gs.collections.api.block.function.primitive.IntToIntFunction;
import com.gs.collections.api.block.predicate.primitive.FloatPredicate;
import com.gs.collections.api.block.procedure.primitive.FloatProcedure;
import com.gs.collections.api.block.procedure.primitive.FloatIntProcedure;
import com.gs.collections.api.iterator.FloatIterator;
import com.gs.collections.api.list.primitive.MutableFloatList;
import com.gs.collections.api.set.primitive.MutableFloatSet;
import com.gs.collections.impl.Counter;
import com.gs.collections.impl.bag.mutable.HashBag;
import com.gs.collections.impl.block.factory.primitive.IntToIntFunctions;
import com.gs.collections.impl.factory.primitive.FloatBags;
import com.gs.collections.impl.lazy.primitive.LazyFloatIterableAdapter;
import com.gs.collections.impl.list.mutable.primitive.FloatArrayList;
import com.gs.collections.impl.map.mutable.primitive.FloatIntHashMap;
import com.gs.collections.impl.set.mutable.primitive.FloatHashSet;
import net.jcip.annotations.NotThreadSafe;

/**
 * FloatHashBag is similar to {@link HashBag}, and is memory-optimized for float primitives.
 * This file was automatically generated from template file primitiveHashBag.stg.
 *
 * @since 3.0.
 */
@NotThreadSafe
public final class FloatHashBag implements MutableFloatBag, Externalizable
{
    private static final long serialVersionUID = 1L;

    private FloatIntHashMap items;
    private int size;

    public FloatHashBag()
    {
        this.items = new FloatIntHashMap();
    }

    public FloatHashBag(int size)
    {
        this.items = new FloatIntHashMap(size);
    }

    public FloatHashBag(FloatIterable iterable)
    {
        this();
        this.addAll(iterable);
    }

    public FloatHashBag(FloatHashBag bag)
    {
        this.items = new FloatIntHashMap(bag.sizeDistinct());
        bag.forEachWithOccurrences(new FloatIntProcedure()
        {
            public void value(float item, int occurrences)
            {
                FloatHashBag.this.addOccurrences(item, occurrences);
            }
        });
    }

    public static FloatHashBag newBag(int size)
    {
        return new FloatHashBag(size);
    }

    public static FloatHashBag newBagWith(float... source)
    {
        FloatHashBag result = new FloatHashBag();
        result.addAll(source);
        return result;
    }

    public static FloatHashBag newBag(FloatIterable source)
    {
        if (source instanceof FloatHashBag)
        {
            return new FloatHashBag((FloatHashBag) source);
        }

        return new FloatHashBag(source);
    }

    public static FloatHashBag newBag(FloatBag source)
    {
        final FloatHashBag result = new FloatHashBag();
        source.forEachWithOccurrences(new FloatIntProcedure()
        {
            public void value(float each, int occurrences)
            {
                result.addOccurrences(each, occurrences);
            }
        });
        return result;
    }

    public boolean isEmpty()
    {
        return this.items.isEmpty();
    }

    public boolean notEmpty()
    {
        return this.items.notEmpty();
    }

    public int size()
    {
        return this.size;
    }

    public int sizeDistinct()
    {
        return this.items.size();
    }

    public void clear()
    {
        this.items.clear();
        this.size = 0;
    }

    public FloatHashBag with(float element)
    {
        this.add(element);
        return this;
    }

    public FloatHashBag with(float element1, float element2)
    {
        this.add(element1);
        this.add(element2);
        return this;
    }

    public FloatHashBag with(float element1, float element2, float element3)
    {
        this.add(element1);
        this.add(element2);
        this.add(element3);
        return this;
    }

    public FloatHashBag withAll(FloatIterable iterable)
    {
        this.addAll(iterable);
        return this;
    }

    public FloatHashBag without(float element)
    {
        this.remove(element);
        return this;
    }

    public FloatHashBag withoutAll(FloatIterable iterable)
    {
        this.removeAll(iterable);
        return this;
    }

    public boolean contains(float value)
    {
        return this.items.containsKey(value);
    }

    public boolean containsAll(float... source)
    {
        for (float each : source)
        {
            if (!this.items.containsKey(each))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(FloatIterable source)
    {
        return source.allSatisfy(new FloatPredicate()
        {
            public boolean accept(float each)
            {
                return FloatHashBag.this.contains(each);
            }
        });
    }

    public int occurrencesOf(float item)
    {
        return this.items.get(item);
    }

    public void forEachWithOccurrences(FloatIntProcedure procedure)
    {
        this.items.forEachKeyValue(procedure);
    }

    public boolean add(float item)
    {
        this.items.updateValue(item, 0, IntToIntFunctions.increment());
        this.size++;
        return true;
    }

    public boolean remove(float item)
    {
        int newValue = this.items.updateValue(item, 0, IntToIntFunctions.decrement());
        if (newValue <= 0)
        {
            this.items.removeKey(item);
            if (newValue == 0)
            {
                this.size--;
            }
            return newValue == 0;
        }
        this.size--;
        return true;
    }

    public boolean addAll(float... source)
    {
        if (source.length == 0)
        {
            return false;
        }

        for (float each : source)
        {
            this.add(each);
        }
        return true;
    }

    public boolean addAll(FloatIterable source)
    {
        if (source.isEmpty())
        {
            return false;
        }
        if (source instanceof FloatBag)
        {
            FloatBag otherBag = (FloatBag) source;
            otherBag.forEachWithOccurrences(new FloatIntProcedure()
            {
                public void value(float each, int occurrences)
                {
                    FloatHashBag.this.addOccurrences(each, occurrences);
                }
            });
        }
        else
        {
            FloatIterator iterator = source.floatIterator();
            while (iterator.hasNext())
            {
                float each = iterator.next();
                this.add(each);
            }
        }
        return true;
    }

    public boolean removeAll(float... source)
    {
        if (source.length == 0)
        {
            return false;
        }
        int oldSize = this.size();
        for (float each : source)
        {
            int occurrences = this.items.removeKeyIfAbsent(each, 0);
            this.size -= occurrences;
        }
        return this.size() != oldSize;
    }

    public boolean removeAll(FloatIterable source)
    {
        if (source.isEmpty())
        {
            return false;
        }
        int oldSize = this.size();
        if (source instanceof FloatBag)
        {
            FloatBag otherBag = (FloatBag) source;
            otherBag.forEachWithOccurrences(new FloatIntProcedure()
            {
                public void value(float each, int occurrences)
                {
                    int oldOccurrences = FloatHashBag.this.items.removeKeyIfAbsent(each, 0);
                    FloatHashBag.this.size -= oldOccurrences;
                }
            });
        }
        else
        {
            FloatIterator iterator = source.floatIterator();
            while (iterator.hasNext())
            {
                float each = iterator.next();
                int occurrences = this.items.removeKeyIfAbsent(each, 0);
                this.size -= occurrences;
            }
        }
        return this.size() != oldSize;
    }

    public void addOccurrences(float item, final int occurrences)
    {
        if (occurrences < 0)
        {
            throw new IllegalArgumentException("Cannot add a negative number of occurrences");
        }
        if (occurrences > 0)
        {
            this.items.updateValue(item, 0, new IntToIntFunction()
            {
                public int valueOf(int intParameter)
                {
                    return intParameter + occurrences;
                }
            });
            this.size += occurrences;
        }
    }

    public boolean removeOccurrences(float item, final int occurrences)
    {
        if (occurrences < 0)
        {
            throw new IllegalArgumentException("Cannot remove a negative number of occurrences");
        }

        if (occurrences == 0)
        {
            return false;
        }

        int newValue = this.items.updateValue(item, 0, new IntToIntFunction()
        {
            public int valueOf(int intParameter)
            {
                return intParameter - occurrences;
            }
        });

        if (newValue <= 0)
        {
            this.size -= occurrences - newValue;
            this.items.removeKey(item);
            return newValue + occurrences != 0;
        }

        this.size -= occurrences;
        return true;
    }

    public void forEach(final FloatProcedure procedure)
    {
        this.items.forEachKeyValue(new FloatIntProcedure()
        {
            public void value(float key, int occurrences)
            {
                for (int i = 0; i < occurrences; i++)
                {
                    procedure.value(key);
                }
            }
        });
    }

    public MutableFloatBag select(final FloatPredicate predicate)
    {
        final MutableFloatBag result = new FloatHashBag();
        this.forEachWithOccurrences(new FloatIntProcedure()
        {
            public void value(float each, int occurrences)
            {
                if (predicate.accept(each))
                {
                    result.addOccurrences(each, occurrences);
                }
            }
        });
        return result;
    }

    public MutableFloatBag reject(final FloatPredicate predicate)
    {
        final MutableFloatBag result = new FloatHashBag();
        this.forEachWithOccurrences(new FloatIntProcedure()
        {
            public void value(float each, int occurrences)
            {
                if (!predicate.accept(each))
                {
                    result.addOccurrences(each, occurrences);
                }
            }
        });
        return result;
    }

    public <T> T injectInto(T injectedValue, ObjectFloatToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        FloatIterator it = this.floatIterator();
        while (it.hasNext())
        {
            result = function.valueOf(result, it.next());
        }
        return result;
    }

    @Override
    public boolean equals(Object otherBag)
    {
        if (otherBag == this)
        {
            return true;
        }
        if (!(otherBag instanceof FloatBag))
        {
            return false;
        }
        final FloatBag bag = (FloatBag) otherBag;
        if (this.sizeDistinct() != bag.sizeDistinct())
        {
            return false;
        }

        return this.items.keysView().allSatisfy(new FloatPredicate()
        {
            public boolean accept(float key)
            {
                return FloatHashBag.this.occurrencesOf(key) == bag.occurrencesOf(key);
            }
        });
    }

    @Override
    public int hashCode()
    {
        final Counter result = new Counter();
        this.forEachWithOccurrences(new FloatIntProcedure()
        {
            public void value(float eachItem, int occurrences)
            {
                result.add(Float.floatToIntBits(eachItem) ^ occurrences);
            }
        });
        return result.getCount();
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(
            final Appendable appendable,
            String start,
            final String separator,
            String end)
    {
        final boolean[] firstItem = {true};
        try
        {
            appendable.append(start);
            this.items.forEachKeyValue(new FloatIntProcedure()
            {
                public void value(float each, int occurrences)
                {
                    try
                    {
                        for (int i = 0; i < occurrences; i++)
                        {
                            if (!firstItem[0])
                            {
                                appendable.append(separator);
                            }
                            appendable.append(String.valueOf(each));
                            firstItem[0] = false;
                        }
                    }
                    catch (IOException e)
                    {
                        throw new RuntimeException(e);
                    }
                }
            });
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public int count(final FloatPredicate predicate)
    {
        final Counter result = new Counter();
        this.forEachWithOccurrences(new FloatIntProcedure()
        {
            public void value(float each, int occurrences)
            {
                if (predicate.accept(each))
                {
                    result.add(occurrences);
                }
            }
        });
        return result.getCount();
    }

    public boolean anySatisfy(FloatPredicate predicate)
    {
        return this.items.keysView().anySatisfy(predicate);
    }

    public boolean allSatisfy(FloatPredicate predicate)
    {
        return this.items.keysView().allSatisfy(predicate);
    }

    public boolean noneSatisfy(FloatPredicate predicate)
    {
        return this.items.keysView().noneSatisfy(predicate);
    }

    public float detectIfNone(FloatPredicate predicate, float ifNone)
    {
        return this.items.keysView().detectIfNone(predicate, ifNone);
    }

    public <V> MutableBag<V> collect(final FloatToObjectFunction<? extends V> function)
    {
        final HashBag<V> result = HashBag.newBag(this.items.size());
        this.forEachWithOccurrences(new FloatIntProcedure()
        {
            public void value(float each, int occurrences)
            {
                result.addOccurrences(function.valueOf(each), occurrences);
            }
        });
        return result;
    }

    public float max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        return this.items.keysView().max();
    }

    public float min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        return this.items.keysView().min();
    }

    public double sum()
    {
        final double[] result = {0.0};
        this.forEachWithOccurrences(new FloatIntProcedure()
        {
            public void value(float each, int occurrences)
            {
                result[0] += (double) each * occurrences;
            }
        });
        return result[0];
    }

    public float minIfEmpty(float defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public float maxIfEmpty(float defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        float[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            float first = sortedArray[middleIndex];
            float second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public float[] toArray()
    {
        final float[] array = new float[this.size()];
        final int[] index = {0};

        this.forEachWithOccurrences(new FloatIntProcedure()
        {
            public void value(float each, int occurrences)
            {
                for (int i = 0; i < occurrences; i++)
                {
                    array[index[0]] = each;
                    index[0]++;
                }
            }
        });
        return array;
    }

    public float[] toSortedArray()
    {
        float[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public MutableFloatList toList()
    {
        return FloatArrayList.newList(this);
    }

    public MutableFloatList toSortedList()
    {
        return FloatArrayList.newList(this).sortThis();
    }

    public MutableFloatSet toSet()
    {
        return FloatHashSet.newSet(this.items.keysView());
    }

    public MutableFloatBag toBag()
    {
        return FloatHashBag.newBag(this);
    }

    public LazyFloatIterable asLazy()
    {
        return new LazyFloatIterableAdapter(this);
    }

    public MutableFloatBag asUnmodifiable()
    {
        return new UnmodifiableFloatBag(this);
    }

    public MutableFloatBag asSynchronized()
    {
        return new SynchronizedFloatBag(this);
    }

    public ImmutableFloatBag toImmutable()
    {
        return FloatBags.immutable.withAll(this);
    }

    public FloatIterator floatIterator()
    {
        return new InternalIterator();
    }

    public void writeExternal(final ObjectOutput out) throws IOException
    {
        out.writeInt(this.items.size());
        try
        {
            this.items.forEachKeyValue(new FloatIntProcedure()
            {
                public void value(float each, int occurrences)
                {
                    try
                    {
                        out.writeFloat(each);
                        out.writeInt(occurrences);
                    }
                    catch (IOException e)
                    {
                        throw new RuntimeException(e);
                    }
                }
            });
        }
        catch (RuntimeException e)
        {
            if (e.getCause() instanceof IOException)
            {
                throw (IOException) e.getCause();
            }
            throw e;
        }
    }

    public void readExternal(ObjectInput in) throws IOException
    {
        int size = in.readInt();
        this.items = new FloatIntHashMap(size);
        for (int i = 0; i < size; i++)
        {
            this.addOccurrences(in.readFloat(), in.readInt());
        }
    }

    private class InternalIterator implements FloatIterator
    {
        private final FloatIterator floatIterator = FloatHashBag.this.items.keysView().floatIterator();

        private float currentItem;
        private int occurrences;

        public boolean hasNext()
        {
            return this.occurrences > 0 || this.floatIterator.hasNext();
        }

        public float next()
        {
            if (this.occurrences == 0)
            {
                this.currentItem = this.floatIterator.next();
                this.occurrences = FloatHashBag.this.occurrencesOf(this.currentItem);
            }
            this.occurrences--;
            return this.currentItem;
        }
    }
}