/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.bag.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.ShortIterable;
import com.gs.collections.api.LazyShortIterable;
import com.gs.collections.api.bag.MutableBag;
import com.gs.collections.api.bag.primitive.ShortBag;
import com.gs.collections.api.bag.primitive.ImmutableShortBag;
import com.gs.collections.api.bag.primitive.MutableShortBag;
import com.gs.collections.api.block.function.primitive.ShortToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectShortToObjectFunction;
import com.gs.collections.api.block.function.primitive.IntToIntFunction;
import com.gs.collections.api.block.predicate.primitive.ShortPredicate;
import com.gs.collections.api.block.procedure.primitive.ShortProcedure;
import com.gs.collections.api.block.procedure.primitive.ShortIntProcedure;
import com.gs.collections.api.iterator.ShortIterator;
import com.gs.collections.api.list.primitive.MutableShortList;
import com.gs.collections.api.set.primitive.MutableShortSet;
import com.gs.collections.impl.Counter;
import com.gs.collections.impl.bag.mutable.HashBag;
import com.gs.collections.impl.block.factory.primitive.IntToIntFunctions;
import com.gs.collections.impl.factory.primitive.ShortBags;
import com.gs.collections.impl.lazy.primitive.LazyShortIterableAdapter;
import com.gs.collections.impl.list.mutable.primitive.ShortArrayList;
import com.gs.collections.impl.map.mutable.primitive.ShortIntHashMap;
import com.gs.collections.impl.set.mutable.primitive.ShortHashSet;
import net.jcip.annotations.NotThreadSafe;

/**
 * ShortHashBag is similar to {@link HashBag}, and is memory-optimized for short primitives.
 * This file was automatically generated from template file primitiveHashBag.stg.
 *
 * @since 3.0.
 */
@NotThreadSafe
public final class ShortHashBag implements MutableShortBag, Externalizable
{
    private static final long serialVersionUID = 1L;

    private ShortIntHashMap items;
    private int size;

    public ShortHashBag()
    {
        this.items = new ShortIntHashMap();
    }

    public ShortHashBag(int size)
    {
        this.items = new ShortIntHashMap(size);
    }

    public ShortHashBag(ShortIterable iterable)
    {
        this();
        this.addAll(iterable);
    }

    public ShortHashBag(ShortHashBag bag)
    {
        this.items = new ShortIntHashMap(bag.sizeDistinct());
        bag.forEachWithOccurrences(new ShortIntProcedure()
        {
            public void value(short item, int occurrences)
            {
                ShortHashBag.this.addOccurrences(item, occurrences);
            }
        });
    }

    public static ShortHashBag newBag(int size)
    {
        return new ShortHashBag(size);
    }

    public static ShortHashBag newBagWith(short... source)
    {
        ShortHashBag result = new ShortHashBag();
        result.addAll(source);
        return result;
    }

    public static ShortHashBag newBag(ShortIterable source)
    {
        if (source instanceof ShortHashBag)
        {
            return new ShortHashBag((ShortHashBag) source);
        }

        return new ShortHashBag(source);
    }

    public static ShortHashBag newBag(ShortBag source)
    {
        final ShortHashBag result = new ShortHashBag();
        source.forEachWithOccurrences(new ShortIntProcedure()
        {
            public void value(short each, int occurrences)
            {
                result.addOccurrences(each, occurrences);
            }
        });
        return result;
    }

    public boolean isEmpty()
    {
        return this.items.isEmpty();
    }

    public boolean notEmpty()
    {
        return this.items.notEmpty();
    }

    public int size()
    {
        return this.size;
    }

    public int sizeDistinct()
    {
        return this.items.size();
    }

    public void clear()
    {
        this.items.clear();
        this.size = 0;
    }

    public ShortHashBag with(short element)
    {
        this.add(element);
        return this;
    }

    public ShortHashBag with(short element1, short element2)
    {
        this.add(element1);
        this.add(element2);
        return this;
    }

    public ShortHashBag with(short element1, short element2, short element3)
    {
        this.add(element1);
        this.add(element2);
        this.add(element3);
        return this;
    }

    public ShortHashBag withAll(ShortIterable iterable)
    {
        this.addAll(iterable);
        return this;
    }

    public ShortHashBag without(short element)
    {
        this.remove(element);
        return this;
    }

    public ShortHashBag withoutAll(ShortIterable iterable)
    {
        this.removeAll(iterable);
        return this;
    }

    public boolean contains(short value)
    {
        return this.items.containsKey(value);
    }

    public boolean containsAll(short... source)
    {
        for (short each : source)
        {
            if (!this.items.containsKey(each))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(ShortIterable source)
    {
        return source.allSatisfy(new ShortPredicate()
        {
            public boolean accept(short each)
            {
                return ShortHashBag.this.contains(each);
            }
        });
    }

    public int occurrencesOf(short item)
    {
        return this.items.get(item);
    }

    public void forEachWithOccurrences(ShortIntProcedure procedure)
    {
        this.items.forEachKeyValue(procedure);
    }

    public boolean add(short item)
    {
        this.items.updateValue(item, 0, IntToIntFunctions.increment());
        this.size++;
        return true;
    }

    public boolean remove(short item)
    {
        int newValue = this.items.updateValue(item, 0, IntToIntFunctions.decrement());
        if (newValue <= 0)
        {
            this.items.removeKey(item);
            if (newValue == 0)
            {
                this.size--;
            }
            return newValue == 0;
        }
        this.size--;
        return true;
    }

    public boolean addAll(short... source)
    {
        if (source.length == 0)
        {
            return false;
        }

        for (short each : source)
        {
            this.add(each);
        }
        return true;
    }

    public boolean addAll(ShortIterable source)
    {
        if (source.isEmpty())
        {
            return false;
        }
        if (source instanceof ShortBag)
        {
            ShortBag otherBag = (ShortBag) source;
            otherBag.forEachWithOccurrences(new ShortIntProcedure()
            {
                public void value(short each, int occurrences)
                {
                    ShortHashBag.this.addOccurrences(each, occurrences);
                }
            });
        }
        else
        {
            ShortIterator iterator = source.shortIterator();
            while (iterator.hasNext())
            {
                short each = iterator.next();
                this.add(each);
            }
        }
        return true;
    }

    public boolean removeAll(short... source)
    {
        if (source.length == 0)
        {
            return false;
        }
        int oldSize = this.size();
        for (short each : source)
        {
            int occurrences = this.items.removeKeyIfAbsent(each, 0);
            this.size -= occurrences;
        }
        return this.size() != oldSize;
    }

    public boolean removeAll(ShortIterable source)
    {
        if (source.isEmpty())
        {
            return false;
        }
        int oldSize = this.size();
        if (source instanceof ShortBag)
        {
            ShortBag otherBag = (ShortBag) source;
            otherBag.forEachWithOccurrences(new ShortIntProcedure()
            {
                public void value(short each, int occurrences)
                {
                    int oldOccurrences = ShortHashBag.this.items.removeKeyIfAbsent(each, 0);
                    ShortHashBag.this.size -= oldOccurrences;
                }
            });
        }
        else
        {
            ShortIterator iterator = source.shortIterator();
            while (iterator.hasNext())
            {
                short each = iterator.next();
                int occurrences = this.items.removeKeyIfAbsent(each, 0);
                this.size -= occurrences;
            }
        }
        return this.size() != oldSize;
    }

    public void addOccurrences(short item, final int occurrences)
    {
        if (occurrences < 0)
        {
            throw new IllegalArgumentException("Cannot add a negative number of occurrences");
        }
        if (occurrences > 0)
        {
            this.items.updateValue(item, 0, new IntToIntFunction()
            {
                public int valueOf(int intParameter)
                {
                    return intParameter + occurrences;
                }
            });
            this.size += occurrences;
        }
    }

    public boolean removeOccurrences(short item, final int occurrences)
    {
        if (occurrences < 0)
        {
            throw new IllegalArgumentException("Cannot remove a negative number of occurrences");
        }

        if (occurrences == 0)
        {
            return false;
        }

        int newValue = this.items.updateValue(item, 0, new IntToIntFunction()
        {
            public int valueOf(int intParameter)
            {
                return intParameter - occurrences;
            }
        });

        if (newValue <= 0)
        {
            this.size -= occurrences - newValue;
            this.items.removeKey(item);
            return newValue + occurrences != 0;
        }

        this.size -= occurrences;
        return true;
    }

    public void forEach(final ShortProcedure procedure)
    {
        this.items.forEachKeyValue(new ShortIntProcedure()
        {
            public void value(short key, int occurrences)
            {
                for (int i = 0; i < occurrences; i++)
                {
                    procedure.value(key);
                }
            }
        });
    }

    public MutableShortBag select(final ShortPredicate predicate)
    {
        final MutableShortBag result = new ShortHashBag();
        this.forEachWithOccurrences(new ShortIntProcedure()
        {
            public void value(short each, int occurrences)
            {
                if (predicate.accept(each))
                {
                    result.addOccurrences(each, occurrences);
                }
            }
        });
        return result;
    }

    public MutableShortBag reject(final ShortPredicate predicate)
    {
        final MutableShortBag result = new ShortHashBag();
        this.forEachWithOccurrences(new ShortIntProcedure()
        {
            public void value(short each, int occurrences)
            {
                if (!predicate.accept(each))
                {
                    result.addOccurrences(each, occurrences);
                }
            }
        });
        return result;
    }

    public <T> T injectInto(T injectedValue, ObjectShortToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        ShortIterator it = this.shortIterator();
        while (it.hasNext())
        {
            result = function.valueOf(result, it.next());
        }
        return result;
    }

    @Override
    public boolean equals(Object otherBag)
    {
        if (otherBag == this)
        {
            return true;
        }
        if (!(otherBag instanceof ShortBag))
        {
            return false;
        }
        final ShortBag bag = (ShortBag) otherBag;
        if (this.sizeDistinct() != bag.sizeDistinct())
        {
            return false;
        }

        return this.items.keysView().allSatisfy(new ShortPredicate()
        {
            public boolean accept(short key)
            {
                return ShortHashBag.this.occurrencesOf(key) == bag.occurrencesOf(key);
            }
        });
    }

    @Override
    public int hashCode()
    {
        final Counter result = new Counter();
        this.forEachWithOccurrences(new ShortIntProcedure()
        {
            public void value(short eachItem, int occurrences)
            {
                result.add((int) eachItem ^ occurrences);
            }
        });
        return result.getCount();
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(
            final Appendable appendable,
            String start,
            final String separator,
            String end)
    {
        final boolean[] firstItem = {true};
        try
        {
            appendable.append(start);
            this.items.forEachKeyValue(new ShortIntProcedure()
            {
                public void value(short each, int occurrences)
                {
                    try
                    {
                        for (int i = 0; i < occurrences; i++)
                        {
                            if (!firstItem[0])
                            {
                                appendable.append(separator);
                            }
                            appendable.append(String.valueOf(each));
                            firstItem[0] = false;
                        }
                    }
                    catch (IOException e)
                    {
                        throw new RuntimeException(e);
                    }
                }
            });
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public int count(final ShortPredicate predicate)
    {
        final Counter result = new Counter();
        this.forEachWithOccurrences(new ShortIntProcedure()
        {
            public void value(short each, int occurrences)
            {
                if (predicate.accept(each))
                {
                    result.add(occurrences);
                }
            }
        });
        return result.getCount();
    }

    public boolean anySatisfy(ShortPredicate predicate)
    {
        return this.items.keysView().anySatisfy(predicate);
    }

    public boolean allSatisfy(ShortPredicate predicate)
    {
        return this.items.keysView().allSatisfy(predicate);
    }

    public boolean noneSatisfy(ShortPredicate predicate)
    {
        return this.items.keysView().noneSatisfy(predicate);
    }

    public short detectIfNone(ShortPredicate predicate, short ifNone)
    {
        return this.items.keysView().detectIfNone(predicate, ifNone);
    }

    public <V> MutableBag<V> collect(final ShortToObjectFunction<? extends V> function)
    {
        final HashBag<V> result = HashBag.newBag(this.items.size());
        this.forEachWithOccurrences(new ShortIntProcedure()
        {
            public void value(short each, int occurrences)
            {
                result.addOccurrences(function.valueOf(each), occurrences);
            }
        });
        return result;
    }

    public short max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        return this.items.keysView().max();
    }

    public short min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        return this.items.keysView().min();
    }

    public long sum()
    {
        final long[] result = {0L};
        this.forEachWithOccurrences(new ShortIntProcedure()
        {
            public void value(short each, int occurrences)
            {
                result[0] += (long) each * occurrences;
            }
        });
        return result[0];
    }

    public short minIfEmpty(short defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public short maxIfEmpty(short defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return (double) this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        short[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            short first = sortedArray[middleIndex];
            short second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public short[] toArray()
    {
        final short[] array = new short[this.size()];
        final int[] index = {0};

        this.forEachWithOccurrences(new ShortIntProcedure()
        {
            public void value(short each, int occurrences)
            {
                for (int i = 0; i < occurrences; i++)
                {
                    array[index[0]] = each;
                    index[0]++;
                }
            }
        });
        return array;
    }

    public short[] toSortedArray()
    {
        short[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public MutableShortList toList()
    {
        return ShortArrayList.newList(this);
    }

    public MutableShortList toSortedList()
    {
        return ShortArrayList.newList(this).sortThis();
    }

    public MutableShortSet toSet()
    {
        return ShortHashSet.newSet(this.items.keysView());
    }

    public MutableShortBag toBag()
    {
        return ShortHashBag.newBag(this);
    }

    public LazyShortIterable asLazy()
    {
        return new LazyShortIterableAdapter(this);
    }

    public MutableShortBag asUnmodifiable()
    {
        return new UnmodifiableShortBag(this);
    }

    public MutableShortBag asSynchronized()
    {
        return new SynchronizedShortBag(this);
    }

    public ImmutableShortBag toImmutable()
    {
        return ShortBags.immutable.withAll(this);
    }

    public ShortIterator shortIterator()
    {
        return new InternalIterator();
    }

    public void writeExternal(final ObjectOutput out) throws IOException
    {
        out.writeInt(this.items.size());
        try
        {
            this.items.forEachKeyValue(new ShortIntProcedure()
            {
                public void value(short each, int occurrences)
                {
                    try
                    {
                        out.writeShort(each);
                        out.writeInt(occurrences);
                    }
                    catch (IOException e)
                    {
                        throw new RuntimeException(e);
                    }
                }
            });
        }
        catch (RuntimeException e)
        {
            if (e.getCause() instanceof IOException)
            {
                throw (IOException) e.getCause();
            }
            throw e;
        }
    }

    public void readExternal(ObjectInput in) throws IOException
    {
        int size = in.readInt();
        this.items = new ShortIntHashMap(size);
        for (int i = 0; i < size; i++)
        {
            this.addOccurrences(in.readShort(), in.readInt());
        }
    }

    private class InternalIterator implements ShortIterator
    {
        private final ShortIterator shortIterator = ShortHashBag.this.items.keysView().shortIterator();

        private short currentItem;
        private int occurrences;

        public boolean hasNext()
        {
            return this.occurrences > 0 || this.shortIterator.hasNext();
        }

        public short next()
        {
            if (this.occurrences == 0)
            {
                this.currentItem = this.shortIterator.next();
                this.occurrences = ShortHashBag.this.occurrencesOf(this.currentItem);
            }
            this.occurrences--;
            return this.currentItem;
        }
    }
}