/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.bag.mutable.primitive;

import java.util.Collection;
import java.util.Collections;

import com.gs.collections.api.ByteIterable;
import com.gs.collections.api.LazyByteIterable;
import com.gs.collections.api.bag.MutableBag;
import com.gs.collections.api.bag.primitive.ImmutableByteBag;
import com.gs.collections.api.bag.primitive.MutableByteBag;
import com.gs.collections.api.block.function.primitive.ByteToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.BytePredicate;
import com.gs.collections.api.block.procedure.primitive.ByteIntProcedure;
import com.gs.collections.api.iterator.ByteIterator;
import com.gs.collections.impl.collection.mutable.primitive.AbstractSynchronizedByteCollection;
import com.gs.collections.impl.factory.primitive.ByteBags;
import com.gs.collections.impl.lazy.primitive.LazyByteIterableAdapter;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;

/**
 * A synchronized view of a {@link MutableByteBag}. It is imperative that the user manually synchronize on the collection when iterating over it using the
 * {@link ByteIterator}, as per {@link Collections#synchronizedCollection(Collection)}.
 * <p>
 * This file was automatically generated from template file synchronizedPrimitiveBag.stg.
 * </p>
 *
 * @see MutableByteBag#asSynchronized()
 * @see MutableBag#asSynchronized()
 * @since 3.1.
 */
@ThreadSafe
public final class SynchronizedByteBag
        extends AbstractSynchronizedByteCollection
        implements MutableByteBag
{
    private static final long serialVersionUID = 1L;

    SynchronizedByteBag(MutableByteBag bag)
    {
        super(bag);
    }

    SynchronizedByteBag(MutableByteBag bag, Object newLock)
    {
        super(bag, newLock);
    }

    @GuardedBy("getLock()")
    private MutableByteBag getMutableByteBag()
    {
        return (MutableByteBag) this.getByteCollection();
    }

    @Override
    public SynchronizedByteBag with(byte element)
    {
        synchronized (this.getLock())
        {
            this.getMutableByteBag().add(element);
        }
        return this;
    }

    @Override
    public SynchronizedByteBag without(byte element)
    {
        synchronized (this.getLock())
        {
            this.getMutableByteBag().remove(element);
        }
        return this;
    }

    @Override
    public SynchronizedByteBag withAll(ByteIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableByteBag().addAll(elements.toArray());
        }
        return this;
    }

    @Override
    public SynchronizedByteBag withoutAll(ByteIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableByteBag().removeAll(elements);
        }
        return this;
    }

    public void addOccurrences(byte item, int occurrences)
    {
        synchronized (this.getLock())
        {
            this.getMutableByteBag().addOccurrences(item, occurrences);
        }
    }

    public boolean removeOccurrences(byte item, int occurrences)
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteBag().removeOccurrences(item, occurrences);
        }
    }

    public int sizeDistinct()
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteBag().sizeDistinct();
        }
    }

    public int occurrencesOf(byte item)
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteBag().occurrencesOf(item);
        }
    }

    public void forEachWithOccurrences(ByteIntProcedure procedure)
    {
        synchronized (this.getLock())
        {
            this.getMutableByteBag().forEachWithOccurrences(procedure);
        }
    }

    @Override
    public MutableByteBag select(BytePredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteBag().select(predicate);
        }
    }

    @Override
    public MutableByteBag reject(BytePredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteBag().reject(predicate);
        }
    }

    @Override
    public <V> MutableBag<V> collect(ByteToObjectFunction<? extends V> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteBag().collect(function);
        }
    }

    @Override
    public boolean equals(Object otherBag)
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteBag().equals(otherBag);
        }
    }

    @Override
    public int hashCode()
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteBag().hashCode();
        }
    }

    @Override
    public LazyByteIterable asLazy()
    {
        synchronized (this.getLock())
        {
            return new LazyByteIterableAdapter(this);
        }
    }

    @Override
    public MutableByteBag asUnmodifiable()
    {
        return new UnmodifiableByteBag(this);
    }

    @Override
    public MutableByteBag asSynchronized()
    {
        return this;
    }

    @Override
    public ImmutableByteBag toImmutable()
    {
        return ByteBags.immutable.withAll(this);
    }
}
