/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.lazy.primitive;

import java.io.IOException;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.FloatIterable;
import com.gs.collections.api.LazyFloatIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.block.function.primitive.FloatToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectFloatToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.FloatPredicate;
import com.gs.collections.api.block.procedure.primitive.FloatProcedure;
import com.gs.collections.api.iterator.FloatIterator;
import com.gs.collections.api.bag.primitive.MutableFloatBag;
import com.gs.collections.api.list.primitive.MutableFloatList;
import com.gs.collections.api.set.primitive.MutableFloatSet;
import com.gs.collections.impl.bag.mutable.primitive.FloatHashBag;
import com.gs.collections.impl.set.mutable.primitive.FloatHashSet;
import com.gs.collections.impl.list.mutable.primitive.FloatArrayList;
import com.gs.collections.impl.block.factory.primitive.FloatPredicates;

/**
 * This file was automatically generated from template file selectPrimitiveIterable.stg.
 */
public class SelectFloatIterable
    implements LazyFloatIterable
{
    private final FloatIterable delegate;
    private final FloatPredicate predicate;

    public SelectFloatIterable(FloatIterable delegate, FloatPredicate predicate)
    {
        this.delegate = delegate;
        this.predicate = predicate;
    }

    public FloatIterator floatIterator()
    {
        return new SelectFloatIterator(this.delegate, this.predicate);
    }

    public void forEach(FloatProcedure procedure)
    {
        this.delegate.forEach(new IfFloatProcedure(procedure));
    }

    public int size()
    {
        return this.delegate.count(this.predicate);
    }

    public boolean isEmpty()
    {
        return !this.floatIterator().hasNext();
    }

    public boolean notEmpty()
    {
        return this.floatIterator().hasNext();
    }

    public int count(FloatPredicate predicate)
    {
        CountFloatProcedure countFloatProcedure = new CountFloatProcedure(predicate);
        this.forEach(countFloatProcedure);
        return countFloatProcedure.getCount();
    }

    public boolean anySatisfy(FloatPredicate predicate)
    {
        return this.delegate.anySatisfy(FloatPredicates.and(this.predicate, predicate));
    }

    public boolean allSatisfy(FloatPredicate predicate)
    {
        return this.noneSatisfy(FloatPredicates.not(predicate));
    }

    public boolean noneSatisfy(FloatPredicate predicate)
    {
        return !this.anySatisfy(predicate);
    }

    public LazyFloatIterable select(FloatPredicate predicate)
    {
        return new SelectFloatIterable(this, predicate);
    }

    public LazyFloatIterable reject(FloatPredicate predicate)
    {
        return new SelectFloatIterable(this, FloatPredicates.not(predicate));
    }

    public float detectIfNone(FloatPredicate predicate, float ifNone)
    {
        for (FloatIterator floatIterator = this.floatIterator(); floatIterator.hasNext() ;)
        {
            float item = floatIterator.next();
            if (predicate.accept(item))
            {
                return item;
            }
        }
        return ifNone;
    }

    public <V> LazyIterable<V> collect(FloatToObjectFunction<? extends V> function)
    {
        return new CollectFloatToObjectIterable<V>(this, function);
    }

        public double sum()
        {
            double sum = 0.0;
            for (FloatIterator floatIterator = this.floatIterator(); floatIterator.hasNext() ;)
            {
                sum += floatIterator.next();
            }
            return sum;
        }

        public float max()
        {
            FloatIterator floatIterator = this.floatIterator();
            float max = floatIterator.next();
            while (floatIterator.hasNext())
            {
                max = (float) Math.max(max, floatIterator.next());
            }
            return max;
        }

        public float min()
        {
            FloatIterator floatIterator = this.floatIterator();
            float min = floatIterator.next();
            while (floatIterator.hasNext())
            {
                min = (float) Math.min(min, floatIterator.next());
            }
            return min;
        }

        public float minIfEmpty(float defaultValue)
        {
            try
            {
                return this.min();
            }
            catch (NoSuchElementException ex)
            {
            }
            return defaultValue;
        }

        public float maxIfEmpty(float defaultValue)
        {
            try
            {
                return this.max();
            }
            catch (NoSuchElementException ex)
            {
            }
            return defaultValue;
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            float[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                float first = sortedArray[middleIndex];
                float second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public float[] toSortedArray()
        {
            float[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public MutableFloatList toSortedList()
        {
            return FloatArrayList.newList(this).sortThis();
        }

    public float[] toArray()
    {
        final float[] array = new float[this.size()];
        this.forEach(new FloatProcedure()
        {
            @SuppressWarnings("FieldMayBeFinal")
            private int index = 0;
            public void value(float each)
            {
                array[this.index++] = each;
            }
        });
        return array;
    }

    public <T> T injectInto(T injectedValue, ObjectFloatToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        for (FloatIterator iterator = this.floatIterator(); iterator.hasNext(); )
        {
            result = function.valueOf(result, iterator.next());
        }
        return result;
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            FloatIterator iterator = this.floatIterator();
            if (iterator.hasNext())
            {
                appendable.append(String.valueOf(iterator.next()));
                while (iterator.hasNext())
                {
                    appendable.append(separator);
                    appendable.append(String.valueOf(iterator.next()));
                }
            }

            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public boolean contains(float value)
    {
        return this.anySatisfy(FloatPredicates.equal(value));
    }

    public boolean containsAll(float... source)
    {
        for (float value : source)
        {
            if (!this.contains(value))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(FloatIterable source)
    {
        for (FloatIterator iterator = source.floatIterator(); iterator.hasNext(); )
        {
            if (!this.contains(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public MutableFloatList toList()
    {
        return FloatArrayList.newList(this);
    }

    public MutableFloatSet toSet()
    {
        return FloatHashSet.newSet(this);
    }

    public MutableFloatBag toBag()
    {
        return FloatHashBag.newBag(this);
    }

    public LazyFloatIterable asLazy()
    {
        return this;
    }

    private static final class CountFloatProcedure implements FloatProcedure
    {
        private static final long serialVersionUID = 1L;
        private final FloatPredicate predicate;
        private int counter = 0;

        private CountFloatProcedure(FloatPredicate predicate)
        {
            this.predicate = predicate;
        }

        public void value(float each)
        {
            if (this.predicate.accept(each))
            {
                this.counter++;
            }
        }

        public int getCount()
        {
            return this.counter;
        }
    }

    private final class IfFloatProcedure implements FloatProcedure
    {
        private static final long serialVersionUID = 1L;
        private final FloatProcedure procedure;

        private IfFloatProcedure(FloatProcedure procedure)
        {
            this.procedure = procedure;
        }

        public void value(float each)
        {
            if (SelectFloatIterable.this.predicate.accept(each))
            {
                this.procedure.value(each);
            }
        }
    }

    private static final class SelectFloatIterator
            implements FloatIterator
    {
        private final FloatIterator iterator;
        private final FloatPredicate predicate;
        private float next;
        private boolean verifiedHasNext = false;

        private SelectFloatIterator(FloatIterable iterable, FloatPredicate predicate)
        {
            this(iterable.floatIterator(), predicate);
        }

        private SelectFloatIterator(FloatIterator iterator, FloatPredicate predicate)
        {
            this.iterator = iterator;
            this.predicate = predicate;
        }

        public boolean hasNext()
        {
            if (this.verifiedHasNext)
            {
                return true;
            }
            while (this.iterator.hasNext())
            {
                float temp = this.iterator.next();
                if (this.predicate.accept(temp))
                {
                    this.next = temp;
                    this.verifiedHasNext = true;
                    return true;
                }
            }
            return false;
        }

        public float next()
        {
            if (this.verifiedHasNext || this.hasNext())
            {
                this.verifiedHasNext = false;
                return this.next;
            }
            throw new NoSuchElementException();
        }
    }
}
