/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.lazy.primitive;

import java.io.IOException;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.IntIterable;
import com.gs.collections.api.LazyIntIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.block.function.primitive.IntToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectIntToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.IntPredicate;
import com.gs.collections.api.block.procedure.primitive.IntProcedure;
import com.gs.collections.api.iterator.IntIterator;
import com.gs.collections.api.bag.primitive.MutableIntBag;
import com.gs.collections.api.list.primitive.MutableIntList;
import com.gs.collections.api.set.primitive.MutableIntSet;
import com.gs.collections.impl.bag.mutable.primitive.IntHashBag;
import com.gs.collections.impl.set.mutable.primitive.IntHashSet;
import com.gs.collections.impl.list.mutable.primitive.IntArrayList;
import com.gs.collections.impl.block.factory.primitive.IntPredicates;

/**
 * This file was automatically generated from template file selectPrimitiveIterable.stg.
 */
public class SelectIntIterable
    implements LazyIntIterable
{
    private final IntIterable delegate;
    private final IntPredicate predicate;

    public SelectIntIterable(IntIterable delegate, IntPredicate predicate)
    {
        this.delegate = delegate;
        this.predicate = predicate;
    }

    public IntIterator intIterator()
    {
        return new SelectIntIterator(this.delegate, this.predicate);
    }

    public void forEach(IntProcedure procedure)
    {
        this.delegate.forEach(new IfIntProcedure(procedure));
    }

    public int size()
    {
        return this.delegate.count(this.predicate);
    }

    public boolean isEmpty()
    {
        return !this.intIterator().hasNext();
    }

    public boolean notEmpty()
    {
        return this.intIterator().hasNext();
    }

    public int count(IntPredicate predicate)
    {
        CountIntProcedure countIntProcedure = new CountIntProcedure(predicate);
        this.forEach(countIntProcedure);
        return countIntProcedure.getCount();
    }

    public boolean anySatisfy(IntPredicate predicate)
    {
        return this.delegate.anySatisfy(IntPredicates.and(this.predicate, predicate));
    }

    public boolean allSatisfy(IntPredicate predicate)
    {
        return this.noneSatisfy(IntPredicates.not(predicate));
    }

    public boolean noneSatisfy(IntPredicate predicate)
    {
        return !this.anySatisfy(predicate);
    }

    public LazyIntIterable select(IntPredicate predicate)
    {
        return new SelectIntIterable(this, predicate);
    }

    public LazyIntIterable reject(IntPredicate predicate)
    {
        return new SelectIntIterable(this, IntPredicates.not(predicate));
    }

    public int detectIfNone(IntPredicate predicate, int ifNone)
    {
        for (IntIterator intIterator = this.intIterator(); intIterator.hasNext() ;)
        {
            int item = intIterator.next();
            if (predicate.accept(item))
            {
                return item;
            }
        }
        return ifNone;
    }

    public <V> LazyIterable<V> collect(IntToObjectFunction<? extends V> function)
    {
        return new CollectIntToObjectIterable<V>(this, function);
    }

        public long sum()
        {
            long sum = 0L;
            for (IntIterator intIterator = this.intIterator(); intIterator.hasNext() ;)
            {
                sum += intIterator.next();
            }
            return sum;
        }

        public int max()
        {
            IntIterator intIterator = this.intIterator();
            int max = intIterator.next();
            while (intIterator.hasNext())
            {
                max = (int) Math.max(max, intIterator.next());
            }
            return max;
        }

        public int min()
        {
            IntIterator intIterator = this.intIterator();
            int min = intIterator.next();
            while (intIterator.hasNext())
            {
                min = (int) Math.min(min, intIterator.next());
            }
            return min;
        }

        public int minIfEmpty(int defaultValue)
        {
            try
            {
                return this.min();
            }
            catch (NoSuchElementException ex)
            {
            }
            return defaultValue;
        }

        public int maxIfEmpty(int defaultValue)
        {
            try
            {
                return this.max();
            }
            catch (NoSuchElementException ex)
            {
            }
            return defaultValue;
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return (double) this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            int[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                int first = sortedArray[middleIndex];
                int second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public int[] toSortedArray()
        {
            int[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public MutableIntList toSortedList()
        {
            return IntArrayList.newList(this).sortThis();
        }

    public int[] toArray()
    {
        final int[] array = new int[this.size()];
        this.forEach(new IntProcedure()
        {
            @SuppressWarnings("FieldMayBeFinal")
            private int index = 0;
            public void value(int each)
            {
                array[this.index++] = each;
            }
        });
        return array;
    }

    public <T> T injectInto(T injectedValue, ObjectIntToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        for (IntIterator iterator = this.intIterator(); iterator.hasNext(); )
        {
            result = function.valueOf(result, iterator.next());
        }
        return result;
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            IntIterator iterator = this.intIterator();
            if (iterator.hasNext())
            {
                appendable.append(String.valueOf(iterator.next()));
                while (iterator.hasNext())
                {
                    appendable.append(separator);
                    appendable.append(String.valueOf(iterator.next()));
                }
            }

            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public boolean contains(int value)
    {
        return this.anySatisfy(IntPredicates.equal(value));
    }

    public boolean containsAll(int... source)
    {
        for (int value : source)
        {
            if (!this.contains(value))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(IntIterable source)
    {
        for (IntIterator iterator = source.intIterator(); iterator.hasNext(); )
        {
            if (!this.contains(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public MutableIntList toList()
    {
        return IntArrayList.newList(this);
    }

    public MutableIntSet toSet()
    {
        return IntHashSet.newSet(this);
    }

    public MutableIntBag toBag()
    {
        return IntHashBag.newBag(this);
    }

    public LazyIntIterable asLazy()
    {
        return this;
    }

    private static final class CountIntProcedure implements IntProcedure
    {
        private static final long serialVersionUID = 1L;
        private final IntPredicate predicate;
        private int counter = 0;

        private CountIntProcedure(IntPredicate predicate)
        {
            this.predicate = predicate;
        }

        public void value(int each)
        {
            if (this.predicate.accept(each))
            {
                this.counter++;
            }
        }

        public int getCount()
        {
            return this.counter;
        }
    }

    private final class IfIntProcedure implements IntProcedure
    {
        private static final long serialVersionUID = 1L;
        private final IntProcedure procedure;

        private IfIntProcedure(IntProcedure procedure)
        {
            this.procedure = procedure;
        }

        public void value(int each)
        {
            if (SelectIntIterable.this.predicate.accept(each))
            {
                this.procedure.value(each);
            }
        }
    }

    private static final class SelectIntIterator
            implements IntIterator
    {
        private final IntIterator iterator;
        private final IntPredicate predicate;
        private int next;
        private boolean verifiedHasNext = false;

        private SelectIntIterator(IntIterable iterable, IntPredicate predicate)
        {
            this(iterable.intIterator(), predicate);
        }

        private SelectIntIterator(IntIterator iterator, IntPredicate predicate)
        {
            this.iterator = iterator;
            this.predicate = predicate;
        }

        public boolean hasNext()
        {
            if (this.verifiedHasNext)
            {
                return true;
            }
            while (this.iterator.hasNext())
            {
                int temp = this.iterator.next();
                if (this.predicate.accept(temp))
                {
                    this.next = temp;
                    this.verifiedHasNext = true;
                    return true;
                }
            }
            return false;
        }

        public int next()
        {
            if (this.verifiedHasNext || this.hasNext())
            {
                this.verifiedHasNext = false;
                return this.next;
            }
            throw new NoSuchElementException();
        }
    }
}
