/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.lazy.primitive;

import java.io.IOException;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.ShortIterable;
import com.gs.collections.api.LazyShortIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.block.function.primitive.ShortToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectShortToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.ShortPredicate;
import com.gs.collections.api.block.procedure.primitive.ShortProcedure;
import com.gs.collections.api.iterator.ShortIterator;
import com.gs.collections.api.bag.primitive.MutableShortBag;
import com.gs.collections.api.list.primitive.MutableShortList;
import com.gs.collections.api.set.primitive.MutableShortSet;
import com.gs.collections.impl.bag.mutable.primitive.ShortHashBag;
import com.gs.collections.impl.set.mutable.primitive.ShortHashSet;
import com.gs.collections.impl.list.mutable.primitive.ShortArrayList;
import com.gs.collections.impl.block.factory.primitive.ShortPredicates;

/**
 * This file was automatically generated from template file selectPrimitiveIterable.stg.
 */
public class SelectShortIterable
    implements LazyShortIterable
{
    private final ShortIterable delegate;
    private final ShortPredicate predicate;

    public SelectShortIterable(ShortIterable delegate, ShortPredicate predicate)
    {
        this.delegate = delegate;
        this.predicate = predicate;
    }

    public ShortIterator shortIterator()
    {
        return new SelectShortIterator(this.delegate, this.predicate);
    }

    public void forEach(ShortProcedure procedure)
    {
        this.delegate.forEach(new IfShortProcedure(procedure));
    }

    public int size()
    {
        return this.delegate.count(this.predicate);
    }

    public boolean isEmpty()
    {
        return !this.shortIterator().hasNext();
    }

    public boolean notEmpty()
    {
        return this.shortIterator().hasNext();
    }

    public int count(ShortPredicate predicate)
    {
        CountShortProcedure countShortProcedure = new CountShortProcedure(predicate);
        this.forEach(countShortProcedure);
        return countShortProcedure.getCount();
    }

    public boolean anySatisfy(ShortPredicate predicate)
    {
        return this.delegate.anySatisfy(ShortPredicates.and(this.predicate, predicate));
    }

    public boolean allSatisfy(ShortPredicate predicate)
    {
        return this.noneSatisfy(ShortPredicates.not(predicate));
    }

    public boolean noneSatisfy(ShortPredicate predicate)
    {
        return !this.anySatisfy(predicate);
    }

    public LazyShortIterable select(ShortPredicate predicate)
    {
        return new SelectShortIterable(this, predicate);
    }

    public LazyShortIterable reject(ShortPredicate predicate)
    {
        return new SelectShortIterable(this, ShortPredicates.not(predicate));
    }

    public short detectIfNone(ShortPredicate predicate, short ifNone)
    {
        for (ShortIterator shortIterator = this.shortIterator(); shortIterator.hasNext() ;)
        {
            short item = shortIterator.next();
            if (predicate.accept(item))
            {
                return item;
            }
        }
        return ifNone;
    }

    public <V> LazyIterable<V> collect(ShortToObjectFunction<? extends V> function)
    {
        return new CollectShortToObjectIterable<V>(this, function);
    }

        public long sum()
        {
            long sum = 0L;
            for (ShortIterator shortIterator = this.shortIterator(); shortIterator.hasNext() ;)
            {
                sum += shortIterator.next();
            }
            return sum;
        }

        public short max()
        {
            ShortIterator shortIterator = this.shortIterator();
            short max = shortIterator.next();
            while (shortIterator.hasNext())
            {
                max = (short) Math.max(max, shortIterator.next());
            }
            return max;
        }

        public short min()
        {
            ShortIterator shortIterator = this.shortIterator();
            short min = shortIterator.next();
            while (shortIterator.hasNext())
            {
                min = (short) Math.min(min, shortIterator.next());
            }
            return min;
        }

        public short minIfEmpty(short defaultValue)
        {
            try
            {
                return this.min();
            }
            catch (NoSuchElementException ex)
            {
            }
            return defaultValue;
        }

        public short maxIfEmpty(short defaultValue)
        {
            try
            {
                return this.max();
            }
            catch (NoSuchElementException ex)
            {
            }
            return defaultValue;
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return (double) this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            short[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                short first = sortedArray[middleIndex];
                short second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public short[] toSortedArray()
        {
            short[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public MutableShortList toSortedList()
        {
            return ShortArrayList.newList(this).sortThis();
        }

    public short[] toArray()
    {
        final short[] array = new short[this.size()];
        this.forEach(new ShortProcedure()
        {
            @SuppressWarnings("FieldMayBeFinal")
            private int index = 0;
            public void value(short each)
            {
                array[this.index++] = each;
            }
        });
        return array;
    }

    public <T> T injectInto(T injectedValue, ObjectShortToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        for (ShortIterator iterator = this.shortIterator(); iterator.hasNext(); )
        {
            result = function.valueOf(result, iterator.next());
        }
        return result;
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            ShortIterator iterator = this.shortIterator();
            if (iterator.hasNext())
            {
                appendable.append(String.valueOf(iterator.next()));
                while (iterator.hasNext())
                {
                    appendable.append(separator);
                    appendable.append(String.valueOf(iterator.next()));
                }
            }

            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public boolean contains(short value)
    {
        return this.anySatisfy(ShortPredicates.equal(value));
    }

    public boolean containsAll(short... source)
    {
        for (short value : source)
        {
            if (!this.contains(value))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(ShortIterable source)
    {
        for (ShortIterator iterator = source.shortIterator(); iterator.hasNext(); )
        {
            if (!this.contains(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public MutableShortList toList()
    {
        return ShortArrayList.newList(this);
    }

    public MutableShortSet toSet()
    {
        return ShortHashSet.newSet(this);
    }

    public MutableShortBag toBag()
    {
        return ShortHashBag.newBag(this);
    }

    public LazyShortIterable asLazy()
    {
        return this;
    }

    private static final class CountShortProcedure implements ShortProcedure
    {
        private static final long serialVersionUID = 1L;
        private final ShortPredicate predicate;
        private int counter = 0;

        private CountShortProcedure(ShortPredicate predicate)
        {
            this.predicate = predicate;
        }

        public void value(short each)
        {
            if (this.predicate.accept(each))
            {
                this.counter++;
            }
        }

        public int getCount()
        {
            return this.counter;
        }
    }

    private final class IfShortProcedure implements ShortProcedure
    {
        private static final long serialVersionUID = 1L;
        private final ShortProcedure procedure;

        private IfShortProcedure(ShortProcedure procedure)
        {
            this.procedure = procedure;
        }

        public void value(short each)
        {
            if (SelectShortIterable.this.predicate.accept(each))
            {
                this.procedure.value(each);
            }
        }
    }

    private static final class SelectShortIterator
            implements ShortIterator
    {
        private final ShortIterator iterator;
        private final ShortPredicate predicate;
        private short next;
        private boolean verifiedHasNext = false;

        private SelectShortIterator(ShortIterable iterable, ShortPredicate predicate)
        {
            this(iterable.shortIterator(), predicate);
        }

        private SelectShortIterator(ShortIterator iterator, ShortPredicate predicate)
        {
            this.iterator = iterator;
            this.predicate = predicate;
        }

        public boolean hasNext()
        {
            if (this.verifiedHasNext)
            {
                return true;
            }
            while (this.iterator.hasNext())
            {
                short temp = this.iterator.next();
                if (this.predicate.accept(temp))
                {
                    this.next = temp;
                    this.verifiedHasNext = true;
                    return true;
                }
            }
            return false;
        }

        public short next()
        {
            if (this.verifiedHasNext || this.hasNext())
            {
                this.verifiedHasNext = false;
                return this.next;
            }
            throw new NoSuchElementException();
        }
    }
}
