/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.list.immutable.primitive;

import java.io.IOException;
import java.io.Serializable;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.FloatIterable;
import com.gs.collections.api.LazyFloatIterable;
import com.gs.collections.api.bag.primitive.MutableFloatBag;
import com.gs.collections.api.block.function.primitive.FloatToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectFloatToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.FloatPredicate;
import com.gs.collections.api.block.procedure.primitive.FloatProcedure;
import com.gs.collections.api.iterator.FloatIterator;
import com.gs.collections.api.list.ImmutableList;
import com.gs.collections.api.list.primitive.ImmutableFloatList;
import com.gs.collections.api.list.primitive.FloatList;
import com.gs.collections.api.list.primitive.MutableFloatList;
import com.gs.collections.api.set.primitive.MutableFloatSet;
import com.gs.collections.impl.bag.mutable.primitive.FloatHashBag;
import com.gs.collections.impl.factory.primitive.FloatLists;
import com.gs.collections.impl.lazy.primitive.LazyFloatIterableAdapter;
import com.gs.collections.impl.list.mutable.FastList;
import com.gs.collections.impl.list.mutable.primitive.FloatArrayList;
import com.gs.collections.impl.set.mutable.primitive.FloatHashSet;
import net.jcip.annotations.Immutable;

/**
 * ImmutableFloatArrayList is the non-modifiable equivalent of {@link FloatArrayList}.
 * It wraps a Java float array.
 * This file was automatically generated from template file immutablePrimitiveArrayList.stg.
 *
 * @since 3.2.
 */
@Immutable
final class ImmutableFloatArrayList
        implements ImmutableFloatList, Serializable
{
    private static final long serialVersionUID = 1L;
    private final float[] items;

    private ImmutableFloatArrayList(float[] newElements)
    {
        if (newElements.length <= 1)
        {
            throw new IllegalArgumentException("Use FloatLists.immutable.with() to instantiate an optimized collection");
        }
        this.items = newElements;
    }

    public static ImmutableFloatArrayList newList(FloatIterable iterable)
    {
        return new ImmutableFloatArrayList(iterable.toArray());
    }

    public static ImmutableFloatArrayList newListWith(float... elements)
    {
        float[] newArray = new float[elements.length];
        System.arraycopy(elements, 0, newArray, 0, elements.length);
        return new ImmutableFloatArrayList(newArray);
    }

    public float get(int index)
    {
        return this.items[index];
    }

    public float getFirst()
    {
        return this.items[0];
    }

    public float getLast()
    {
        return this.items[this.items.length - 1];
    }

    public int indexOf(float value)
    {
        for (int i = 0; i < this.items.length; i++)
        {
            if (Float.compare(this.items[i], value) == 0)
            {
                return i;
            }
        }
        return -1;
    }

    public int lastIndexOf(float value)
    {
        for (int i = this.items.length - 1; i >= 0; i--)
        {
            if (Float.compare(this.items[i], value) == 0)
            {
                return i;
            }
        }
        return -1;
    }

    public FloatIterator floatIterator()
    {
        return new InternalFloatIterator();
    }

    public void forEach(FloatProcedure procedure)
    {
        for (float item : this.items)
        {
            procedure.value(item);
        }
    }

    public int count(FloatPredicate predicate)
    {
        int count = 0;
        for (float item : this.items)
        {
            if (predicate.accept(item))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(FloatPredicate predicate)
    {
        for (float item : this.items)
        {
            if (predicate.accept(item))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(FloatPredicate predicate)
    {
        for (float item : this.items)
        {
            if (!predicate.accept(item))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(FloatPredicate predicate)
    {
        for (float item : this.items)
        {
            if (predicate.accept(item))
            {
                return false;
            }
        }
        return true;
    }

    public ImmutableFloatList select(FloatPredicate predicate)
    {
        FloatArrayList result = new FloatArrayList();
        for (float item : this.items)
        {
            if (predicate.accept(item))
            {
                result.add(item);
            }
        }
        return result.toImmutable();
    }

    public ImmutableFloatList reject(FloatPredicate predicate)
    {
        FloatArrayList result = new FloatArrayList();
        for (float item : this.items)
        {
            if (!predicate.accept(item))
            {
                result.add(item);
            }
        }
        return result.toImmutable();
    }

    public float detectIfNone(FloatPredicate predicate, float ifNone)
    {
        for (float item : this.items)
        {
            if (predicate.accept(item))
            {
                return item;
            }
        }
        return ifNone;
    }

    public <V> ImmutableList<V> collect(FloatToObjectFunction<? extends V> function)
    {
        FastList<V> target = FastList.newList(this.items.length);
        for (float item : this.items)
        {
            target.add(function.valueOf(item));
        }
        return target.toImmutable();
    }

    public double sum()
    {
        double result = 0.0;
        for (float item : this.items)
        {
            result += item;
        }
        return result;
    }

    public float max()
    {
        float max = this.items[0];
        for (int i = 1; i < this.items.length; i++)
        {
            float value = this.items[i];
            if (Float.compare(max, value) < 0)
            {
                max = value;
            }
        }
        return max;
    }

    public float maxIfEmpty(float defaultValue)
    {
        return this.max();
    }

    public float min()
    {
        float min = this.items[0];
        for (int i = 1; i < this.items.length; i++)
        {
            float value = this.items[i];
            if (Float.compare(value, min) < 0)
            {
                min = value;
            }
        }
        return min;
    }

    public float minIfEmpty(float defaultValue)
    {
        return this.min();
    }

    public double average()
    {
        return this.sum() / (double) this.size();
    }

    public double median()
    {
        float[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            float first = sortedArray[middleIndex];
            float second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public float[] toSortedArray()
    {
        float[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public double dotProduct(FloatList list)
    {
        if (this.size() != list.size())
        {
            throw new IllegalArgumentException("Lists used in dotProduct must be the same size");
        }
        double sum = 0.0;
        for (int i = 0; i < this.size(); i++)
        {
            sum += (double) this.items[i] * list.get(i);
        }
        return sum;
    }

    public MutableFloatList toSortedList()
    {
        return FloatArrayList.newList(this).sortThis();
    }

    public float[] toArray()
    {
        float[] newItems = new float[this.items.length];
        System.arraycopy(this.items, 0, newItems, 0, this.items.length);
        return newItems;
    }

    public boolean contains(float value)
    {
        for (float item : this.items)
        {
            if (Float.compare(item, value) == 0)
            {
                return true;
            }
        }
        return false;
    }

    public boolean containsAll(float... source)
    {
        for (float value : source)
        {
            if (!this.contains(value))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(FloatIterable source)
    {
        for (FloatIterator iterator = source.floatIterator(); iterator.hasNext(); )
        {
            if (!this.contains(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public MutableFloatList toList()
    {
        return FloatArrayList.newList(this);
    }

    public MutableFloatSet toSet()
    {
        return FloatHashSet.newSet(this);
    }

    public MutableFloatBag toBag()
    {
        return FloatHashBag.newBag(this);
    }

    public LazyFloatIterable asLazy()
    {
        return new LazyFloatIterableAdapter(this);
    }

    public ImmutableFloatList toImmutable()
    {
        return this;
    }

    public ImmutableFloatList newWith(float element)
    {
        float[] newItems = new float[this.items.length + 1];
        System.arraycopy(this.items, 0, newItems, 0, this.items.length);
        newItems[this.items.length] = element;
        return new ImmutableFloatArrayList(newItems);
    }

    public ImmutableFloatList newWithout(float element)
    {
        int index = this.indexOf(element);
        if (index != -1)
        {
            float[] newItems = new float[this.items.length - 1];
            System.arraycopy(this.items, 0, newItems, 0, index);
            System.arraycopy(this.items, index + 1, newItems, index, this.items.length - index - 1);
            return FloatLists.immutable.with(newItems);
        }
        return this;
    }

    public ImmutableFloatList newWithAll(FloatIterable elements)
    {
        float[] newItems = new float[this.items.length + elements.size()];
        System.arraycopy(this.items, 0, newItems, 0, this.items.length);
        int index = 0;
        for (FloatIterator iterator = elements.floatIterator(); iterator.hasNext(); index++)
        {
            newItems[this.items.length + index] = iterator.next();
        }
        return new ImmutableFloatArrayList(newItems);
    }

    public ImmutableFloatList newWithoutAll(FloatIterable elements)
    {
        MutableFloatList mutableFloatList = this.toList();
        mutableFloatList.removeAll(elements);
        return mutableFloatList.toImmutable();
    }

    public int size()
    {
        return this.items.length;
    }

    public boolean isEmpty()
    {
        return false;
    }

    public boolean notEmpty()
    {
        return true;
    }

    public <T> T injectInto(T injectedValue, ObjectFloatToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        for (int i = 0; i < this.items.length; i++)
        {
            result = function.valueOf(result, this.items[i]);
        }
        return result;
    }

    @Override
    public boolean equals(Object otherList)
    {
        if (otherList == this)
        {
            return true;
        }
        if (!(otherList instanceof FloatList))
        {
            return false;
        }
        FloatList list = (FloatList) otherList;
        if (this.items.length != list.size())
        {
            return false;
        }
        for (int i = 0; i < this.items.length; i++)
        {
            if (Float.compare(this.items[i], list.get(i)) != 0)
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int hashCode = 1;
        for (float item : this.items)
        {
            hashCode = 31 * hashCode + Float.floatToIntBits(item);
        }
        return hashCode;
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);
            for (int i = 0; i < this.items.length; i++)
            {
                if (i > 0)
                {
                    appendable.append(separator);
                }
                float value = this.items[i];
                appendable.append(String.valueOf(value));
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    private class InternalFloatIterator implements FloatIterator
    {
        /**
         * Index of element to be returned by subsequent call to next.
         */
        private int currentIndex;

        public boolean hasNext()
        {
            return this.currentIndex != ImmutableFloatArrayList.this.items.length;
        }

        public float next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException();
            }
            float next = ImmutableFloatArrayList.this.items[this.currentIndex];
            this.currentIndex++;
            return next;
        }
    }
}
