/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.list.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.CharIterable;
import com.gs.collections.api.LazyCharIterable;
import com.gs.collections.api.RichIterable;
import com.gs.collections.api.bag.primitive.MutableCharBag;
import com.gs.collections.api.block.function.primitive.ObjectCharIntToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectCharToObjectFunction;
import com.gs.collections.api.block.function.primitive.CharToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.CharPredicate;
import com.gs.collections.api.block.procedure.primitive.CharIntProcedure;
import com.gs.collections.api.block.procedure.primitive.CharProcedure;
import com.gs.collections.api.iterator.CharIterator;
import com.gs.collections.api.list.MutableList;
import com.gs.collections.api.list.primitive.CharList;
import com.gs.collections.api.list.primitive.ImmutableCharList;
import com.gs.collections.api.list.primitive.MutableCharList;
import com.gs.collections.api.set.primitive.MutableCharSet;
import com.gs.collections.impl.bag.mutable.primitive.CharHashBag;
import com.gs.collections.impl.factory.primitive.CharLists;
import com.gs.collections.impl.lazy.primitive.LazyCharIterableAdapter;
import com.gs.collections.impl.list.mutable.FastList;
import com.gs.collections.impl.set.mutable.primitive.CharHashSet;
import net.jcip.annotations.NotThreadSafe;

/**
 * CharArrayList is similar to {@link FastList}, and is memory-optimized for char primitives.
 * This file was automatically generated from template file primitiveArrayList.stg.
 *
 * @since 3.0.
 */
@NotThreadSafe
public final class CharArrayList
        implements MutableCharList, Externalizable
{
    private static final long serialVersionUID = 1L;
    private static final char[] DEFAULT_SIZED_EMPTY_ARRAY = {};
    private static final char[] ZERO_SIZED_ARRAY = {};
    private static final int MAXIMUM_ARRAY_SIZE = Integer.MAX_VALUE - 8;
    private int size;
    private transient char[] items = DEFAULT_SIZED_EMPTY_ARRAY;

    public CharArrayList()
    {
    }

    public CharArrayList(int initialCapacity)
    {
        this.items = initialCapacity == 0 ? ZERO_SIZED_ARRAY : new char[initialCapacity];
    }

    public CharArrayList(char... array)
    {
        this.size = array.length;
        this.items = array;
    }

    /**
     * Creates a new list using the passed {@code elements} argument as the backing store.
     * <p/>
     * !!! WARNING: This method uses the passed in array, so can be very unsafe if the original
     * array is held onto anywhere else. !!!
     */
    public static CharArrayList newListWith(char... elements)
    {
        return new CharArrayList(elements);
    }

    public static CharArrayList newList(CharIterable source)
    {
        return CharArrayList.newListWith(source.toArray());
    }

    public static CharArrayList newWithNValues(int size, char value)
    {
        CharArrayList newList = new CharArrayList(size);
        for (int i = 0; i < size; i++)
        {
            newList.add(value);
        }
        return newList;
    }

    public int size()
    {
        return this.size;
    }

    public boolean isEmpty()
    {
        return this.size == 0;
    }

    public boolean notEmpty()
    {
        return this.size > 0;
    }

    public void clear()
    {
        Arrays.fill(this.items, '\0');
        this.size = 0;
    }

    public boolean contains(char value)
    {
        for (int i = 0; i < this.size; i++)
        {
            if (this.items[i] == value)
            {
                return true;
            }
        }
        return false;
    }

    public boolean containsAll(char... source)
    {
        for (char value : source)
        {
            if (!this.contains(value))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(CharIterable source)
    {
        for (CharIterator iterator = source.charIterator(); iterator.hasNext(); )
        {
            if (!this.contains(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public char get(int index)
    {
        if (index < this.size)
        {
            return this.items[index];
        }
        throw this.newIndexOutOfBoundsException(index);
    }

    private IndexOutOfBoundsException newIndexOutOfBoundsException(int index)
    {
        return new IndexOutOfBoundsException("Index: " + index + " Size: " + this.size);
    }

    public char getFirst()
    {
        this.checkEmpty();
        return this.items[0];
    }

    public char getLast()
    {
        this.checkEmpty();
        return this.items[this.size() - 1];
    }

    private void checkEmpty()
    {
        if (this.isEmpty())
        {
            throw this.newIndexOutOfBoundsException(0);
        }
    }

    public int indexOf(char value)
    {
        for (int i = 0; i < this.size; i++)
        {
            if (this.items[i] == value)
            {
                return i;
            }
        }
        return -1;
    }

    public int lastIndexOf(char value)
    {
        for (int i = this.size - 1; i >= 0; i--)
        {
            if (this.items[i] == value)
            {
                return i;
            }
        }
        return -1;
    }

    public void trimToSize()
    {
        if (this.size < this.items.length)
        {
            this.transferItemsToNewArrayWithCapacity(this.size);
        }
    }

    private void transferItemsToNewArrayWithCapacity(int newCapacity)
    {
        this.items = this.copyItemsWithNewCapacity(newCapacity);
    }

    private char[] copyItemsWithNewCapacity(int newCapacity)
    {
        char[] newItems = new char[newCapacity];
        System.arraycopy(this.items, 0, newItems, 0, Math.min(this.size, newCapacity));
        return newItems;
    }

    private int sizePlusFiftyPercent(int oldSize)
    {
        int result = oldSize + (oldSize >> 1) + 1;
        return result < oldSize ? MAXIMUM_ARRAY_SIZE : result;
    }

    public void ensureCapacity(int minCapacity)
    {
        int oldCapacity = this.items.length;
        if (minCapacity > oldCapacity)
        {
            int newCapacity = Math.max(this.sizePlusFiftyPercent(oldCapacity), minCapacity);
            this.transferItemsToNewArrayWithCapacity(newCapacity);
        }
    }

    private void ensureCapacityForAdd()
    {
        if (this.items == DEFAULT_SIZED_EMPTY_ARRAY)
        {
            this.items = new char[10];
        }
        else
        {
            this.transferItemsToNewArrayWithCapacity(this.sizePlusFiftyPercent(this.size));
        }
    }

    public boolean add(char newItem)
    {
        if (this.items.length == this.size)
        {
            this.ensureCapacityForAdd();
        }
        this.items[this.size] = newItem;
        this.size++;
        return true;
    }

    public boolean addAll(char... source)
    {
        if (source.length < 1)
        {
            return false;
        }
        int sourceSize = source.length;
        int newSize = this.size + sourceSize;
        this.ensureCapacity(newSize);
        System.arraycopy(source, 0, this.items, this.size, sourceSize);
        this.size = newSize;
        return true;
    }

    public boolean addAll(CharIterable source)
    {
        return this.addAll(source.toArray());
    }

    private void throwOutOfBounds(int index)
    {
        throw this.newIndexOutOfBoundsException(index);
    }

    public void addAtIndex(int index, char element)
    {
        if (index > -1 && index < this.size)
        {
            this.addAtIndexLessThanSize(index, element);
        }
        else if (index == this.size)
        {
            this.add(element);
        }
        else
        {
            this.throwOutOfBounds(index);
        }
    }

    private void addAtIndexLessThanSize(int index, char element)
    {
        int oldSize = this.size;
        this.size++;
        if (this.items.length == oldSize)
        {
            char[] newItems = new char[this.sizePlusFiftyPercent(oldSize)];
            if (index > 0)
            {
                System.arraycopy(this.items, 0, newItems, 0, index);
            }
            System.arraycopy(this.items, index, newItems, index + 1, oldSize - index);
            this.items = newItems;
        }
        else
        {
            System.arraycopy(this.items, index, this.items, index + 1, oldSize - index);
        }
        this.items[index] = element;
    }

    public boolean addAllAtIndex(int index, char... source)
    {
        if (index > this.size || index < 0)
        {
            this.throwOutOfBounds(index);
        }
        if (source.length == 0)
        {
            return false;
        }
        int sourceSize = source.length;
        int newSize = this.size + sourceSize;
        this.ensureCapacity(newSize);
        this.shiftElementsAtIndex(index, sourceSize);
        System.arraycopy(source, 0, this.items, index, sourceSize);
        this.size = newSize;
        return true;
    }

    public boolean addAllAtIndex(int index, CharIterable source)
    {
        return this.addAllAtIndex(index, source.toArray());
    }

    private void shiftElementsAtIndex(int index, int sourceSize)
    {
        int numberToMove = this.size - index;
        if (numberToMove > 0)
        {
            System.arraycopy(this.items, index, this.items, index + sourceSize, numberToMove);
        }
    }

    public boolean remove(char value)
    {
        int index = this.indexOf(value);
        if (index >= 0)
        {
            this.removeAtIndex(index);
            return true;
        }
        return false;
    }

    public boolean removeAll(CharIterable source)
    {
        boolean modified = false;
        for (int index = 0; index < this.size; index++)
        {
            if (source.contains(this.get(index)))
            {
                this.removeAtIndex(index);
                index--;
                modified = true;
            }
        }
        return modified;
    }

    public boolean removeAll(char... source)
    {
        CharHashSet set = CharHashSet.newSetWith(source);
        char[] newItems = new char[this.size];
        int count = 0;
        int oldSize = this.size;
        for (int index = 0; index < this.size; index++)
        {
            if (!set.contains(this.items[index]))
            {
                newItems[count] = this.items[index];
                count++;
            }
        }
        this.items = newItems;
        this.size = count;
        return oldSize != this.size;
    }

    public char removeAtIndex(int index)
    {
        char previous = this.get(index);
        int totalOffset = this.size - index - 1;
        if (totalOffset > 0)
        {
            System.arraycopy(this.items, index + 1, this.items, index, totalOffset);
        }
        --this.size;
        this.items[this.size] = '\0';
        return previous;
    }

    public char set(int index, char element)
    {
        char previous = this.get(index);
        this.items[index] = element;
        return previous;
    }

    public CharArrayList with(char element)
    {
        this.add(element);
        return this;
    }

    public CharArrayList without(char element)
    {
        this.remove(element);
        return this;
    }

    public CharArrayList withAll(CharIterable elements)
    {
        this.addAll(elements.toArray());
        return this;
    }

    public CharArrayList withoutAll(CharIterable elements)
    {
        this.removeAll(elements);
        return this;
    }

    public CharArrayList with(char element1, char element2)
    {
        this.add(element1);
        this.add(element2);
        return this;
    }

    public CharArrayList with(char element1, char element2, char element3)
    {
        this.add(element1);
        this.add(element2);
        this.add(element3);
        return this;
    }

    public CharArrayList with(char element1, char element2, char element3, char... elements)
    {
        this.add(element1);
        this.add(element2);
        this.add(element3);
        return this.withArrayCopy(elements, 0, elements.length);
    }

    private CharArrayList withArrayCopy(char[] elements, int begin, int length)
    {
        this.ensureCapacity(this.size + length);
        System.arraycopy(elements, begin, this.items, this.size, length);
        this.size += length;
        return this;
    }

    public CharIterator charIterator()
    {
        return new InternalCharIterator();
    }

    public void forEach(CharProcedure procedure)
    {
        for (int i = 0; i < this.size; i++)
        {
            procedure.value(this.items[i]);
        }
    }

    public void forEachWithIndex(CharIntProcedure procedure)
    {
        for (int i = 0; i < this.size; i++)
        {
            procedure.value(this.items[i], i);
        }
    }

    public <T> T injectInto(T injectedValue, ObjectCharToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        for (int i = 0; i < this.size; i++)
        {
            result = function.valueOf(result, this.items[i]);
        }
        return result;
    }

    public <T> T injectIntoWithIndex(T injectedValue, ObjectCharIntToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        for (int i = 0; i < this.size; i++)
        {
            result = function.valueOf(result, this.items[i], i);
        }
        return result;
    }

    public int count(CharPredicate predicate)
    {
        int count = 0;
        for (int i = 0; i < this.size; i++)
        {
            if (predicate.accept(this.items[i]))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(CharPredicate predicate)
    {
        for (int i = 0; i < this.size; i++)
        {
            if (predicate.accept(this.items[i]))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(CharPredicate predicate)
    {
        for (int i = 0; i < this.size; i++)
        {
            if (!predicate.accept(this.items[i]))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(CharPredicate predicate)
    {
        for (int i = 0; i < this.size; i++)
        {
            if (predicate.accept(this.items[i]))
            {
                return false;
            }
        }
        return true;
    }

    public CharArrayList select(CharPredicate predicate)
    {
        CharArrayList result = new CharArrayList();
        for (int i = 0; i < this.size; i++)
        {
            char item = this.items[i];
            if (predicate.accept(item))
            {
                result.add(item);
            }
        }
        return result;
    }

    public CharArrayList reject(CharPredicate predicate)
    {
        CharArrayList result = new CharArrayList();
        for (int i = 0; i < this.size; i++)
        {
            char item = this.items[i];
            if (!predicate.accept(item))
            {
                result.add(item);
            }
        }
        return result;
    }

    public char detectIfNone(CharPredicate predicate, char ifNone)
    {
        for (int i = 0; i < this.size; i++)
        {
            char item = this.items[i];
            if (predicate.accept(item))
            {
                return item;
            }
        }
        return ifNone;
    }

    public <V> MutableList<V> collect(CharToObjectFunction<? extends V> function)
    {
        FastList<V> target = FastList.newList(this.size);
        for (int i = 0; i < this.size; i++)
        {
            target.add(function.valueOf(this.items[i]));
        }
        return target;
    }

    public char max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        char max = this.items[0];
        for (int i = 1; i < this.size; i++)
        {
            char value = this.items[i];
            if (max < value)
            {
                max = value;
            }
        }
        return max;
    }

    public char min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        char min = this.items[0];
        for (int i = 1; i < this.size; i++)
        {
            char value = this.items[i];
            if (value < min)
            {
                min = value;
            }
        }
        return min;
    }

    public char minIfEmpty(char defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public char maxIfEmpty(char defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public long sum()
    {
        long result = 0L;
        for (int i = 0; i < this.size; i++)
        {
            result += this.items[i];
        }
        return result;
    }

    public long dotProduct(CharList list)
    {
        if (this.size != list.size())
        {
            throw new IllegalArgumentException("Lists used in dotProduct must be the same size");
        }
        long sum = 0L;
        for (int i = 0; i < this.size; i++)
        {
            sum += (long) this.items[i] * list.get(i);
        }
        return sum;
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return (double)this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        char[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            char first = sortedArray[middleIndex];
            char second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public char[] toArray()
    {
        char[] newItems = new char[this.size];
        System.arraycopy(this.items, 0, newItems, 0, this.size);
        return newItems;
    }

    public char[] toSortedArray()
    {
        char[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    @Override
    public boolean equals(Object otherList)
    {
        if (otherList == this)
        {
            return true;
        }
        if (!(otherList instanceof CharList))
        {
            return false;
        }
        CharList list = (CharList) otherList;
        if (this.size != list.size())
        {
            return false;
        }
        for (int i = 0; i < this.size; i++)
        {
            if (this.items[i] != list.get(i))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int hashCode = 1;
        for (int i = 0; i < this.size; i++)
        {
            char item = this.items[i];
            hashCode = 31 * hashCode + (int) item;
        }
        return hashCode;
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(
            Appendable appendable,
            String start,
            String separator,
            String end)
    {
        try
        {
            appendable.append(start);
            for (int i = 0; i < this.size; i++)
            {
                if (i > 0)
                {
                    appendable.append(separator);
                }
                char value = this.items[i];
                appendable.append(String.valueOf(value));
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public MutableCharList toList()
    {
        return CharArrayList.newList(this);
    }

    public MutableCharList toSortedList()
    {
        return CharArrayList.newList(this).sortThis();
    }

    public MutableCharSet toSet()
    {
        return CharHashSet.newSet(this);
    }

    public MutableCharBag toBag()
    {
        return CharHashBag.newBag(this);
    }

    public LazyCharIterable asLazy()
    {
        return new LazyCharIterableAdapter(this);
    }

    public MutableCharList asUnmodifiable()
    {
        return new UnmodifiableCharList(this);
    }

    public MutableCharList asSynchronized()
    {
        return new SynchronizedCharList(this);
    }

    public ImmutableCharList toImmutable()
    {
        if (this.items.length == 0)
        {
            return CharLists.immutable.with();
        }
        if (this.items.length == 1)
        {
            return CharLists.immutable.with(this.items[0]);
        }
        return CharLists.immutable.with(this.toArray());
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size);
        for (int i = 0; i < this.size; i++)
        {
            out.writeChar(this.items[i]);
        }
    }

    public void readExternal(ObjectInput in) throws IOException
    {
        this.size = in.readInt();
        this.items = new char[this.size];
        for (int i = 0; i < this.size; i++)
        {
            this.items[i] = in.readChar();
        }
    }

    public CharIterable asReversed()
    {
        return new ReverseCharIterable();
    }

    public CharArrayList reverseThis()
    {
        int size = this.items.length;
        int endIndex = size - 1;
        for (int i = 0; i < size / 2; i++)
        {
            char tempSwapValue = this.items[i];
            this.items[i] = this.items[endIndex - i];
            this.items[endIndex - i] = tempSwapValue;
        }
        return this;
    }

    public CharArrayList sortThis()
    {
        Arrays.sort(this.items, 0, this.items.length);
        return this;
    }

    public CharArrayList toReversed()
    {
        return CharArrayList.newList(this.asReversed());
    }

    private class InternalCharIterator implements CharIterator
    {
        /**
         * Index of element to be returned by subsequent call to next.
         */
        private int currentIndex;

        public boolean hasNext()
        {
            return this.currentIndex != CharArrayList.this.size();
        }

        public char next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException();
            }
            char next = CharArrayList.this.items[this.currentIndex];
            this.currentIndex++;
            return next;
        }
    }

    private class ReverseCharIterable implements CharIterable
    {
        public CharIterator charIterator()
        {
            return new ReverseCharIterator();
        }

        public void forEach(CharProcedure procedure)
        {
            CharIterator iterator = this.charIterator();
            while (iterator.hasNext())
            {
                procedure.value(iterator.next());
            }
        }

        public int count(CharPredicate predicate)
        {
            int count = 0;
            CharIterator iterator = this.charIterator();
            while (iterator.hasNext())
            {
                if (predicate.accept(iterator.next()))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(CharPredicate predicate)
        {
            CharIterator iterator = this.charIterator();
            while (iterator.hasNext())
            {
                if (predicate.accept(iterator.next()))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(CharPredicate predicate)
        {
            CharIterator iterator = this.charIterator();
            while (iterator.hasNext())
            {
                if (!predicate.accept(iterator.next()))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(CharPredicate predicate)
        {
            CharIterator iterator = this.charIterator();
            while (iterator.hasNext())
            {
                if (predicate.accept(iterator.next()))
                {
                    return false;
                }
            }
            return true;
        }

        public CharIterable select(CharPredicate predicate)
        {
            CharArrayList result = new CharArrayList(CharArrayList.this.size);
            CharIterator iterator = this.charIterator();
            while (iterator.hasNext())
            {
                char value = iterator.next();
                if (predicate.accept(value))
                {
                    result.add(value);
                }
            }
            return result;
        }

        public CharIterable reject(CharPredicate predicate)
        {
            CharArrayList result = new CharArrayList(CharArrayList.this.size);
            CharIterator iterator = this.charIterator();
            while (iterator.hasNext())
            {
                char value = iterator.next();
                if (!predicate.accept(value))
                {
                    result.add(value);
                }
            }
            return result;
        }

        public char detectIfNone(CharPredicate predicate, char ifNone)
        {
            CharIterator iterator = this.charIterator();
            while (iterator.hasNext())
            {
                char value = iterator.next();
                if (predicate.accept(value))
                {
                    return value;
                }
            }
            return ifNone;
        }

        public <V> RichIterable<V> collect(CharToObjectFunction<? extends V> function)
        {
            FastList<V> results = FastList.newList(CharArrayList.this.size);
            CharIterator iterator = this.charIterator();
            while (iterator.hasNext())
            {
                results.add(function.valueOf(iterator.next()));
            }
            return results;
        }

        public long sum()
        {
            return CharArrayList.this.sum();
        }

        public char max()
        {
            return CharArrayList.this.max();
        }

        public char min()
        {
            return CharArrayList.this.min();
        }

        public char minIfEmpty(char defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.min();
        }

        public char maxIfEmpty(char defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.max();
        }

        public double average()
        {
            return CharArrayList.this.average();
        }

        public double median()
        {
            return CharArrayList.this.median();
        }

        public char[] toSortedArray()
        {
            return CharArrayList.this.toSortedArray();
        }

        public char[] toArray()
        {
            char[] results = new char[CharArrayList.this.size];
            int index = 0;
            CharIterator iterator = this.charIterator();
            while (iterator.hasNext())
            {
                results[index] = iterator.next();
                index++;
            }
            return results;
        }

        public boolean contains(char value)
        {
            return CharArrayList.this.contains(value);
        }

        public boolean containsAll(char... source)
        {
            return CharArrayList.this.containsAll(source);
        }

        public boolean containsAll(CharIterable source)
        {
            return CharArrayList.this.containsAll(source);
        }

        public int size()
        {
            return CharArrayList.this.size;
        }

        public boolean isEmpty()
        {
            return CharArrayList.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return CharArrayList.this.notEmpty();
        }

        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                CharIterator iterator = this.charIterator();
                boolean appendSeparator = false;
                while (iterator.hasNext())
                {
                    if (appendSeparator)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(iterator.next()));
                    appendSeparator = true;
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public MutableCharList toList()
        {
            return CharArrayList.newList(this);
        }

        public MutableCharList toSortedList()
        {
            return CharArrayList.newList(this).sortThis();
        }

        public MutableCharSet toSet()
        {
            return CharHashSet.newSet(this);
        }

        public MutableCharBag toBag()
        {
            return CharHashBag.newBag(this);
        }

        public LazyCharIterable asLazy()
        {
            return new LazyCharIterableAdapter(this);
        }

        private class ReverseCharIterator implements CharIterator
        {
            /**
             * Index of element to be returned by subsequent call to next.
             */
            private int currentIndex = CharArrayList.this.size - 1;

            public boolean hasNext()
            {
                return this.currentIndex != -1;
            }

            public char next()
            {
                if (!this.hasNext())
                {
                    throw new NoSuchElementException();
                }
                char next = CharArrayList.this.items[this.currentIndex];
                this.currentIndex--;
                return next;
            }
        }
    }
}
