/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.map.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.FloatIterable;
import com.gs.collections.api.LazyFloatIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.bag.primitive.MutableFloatBag;
import com.gs.collections.api.block.function.primitive.FloatFunction;
import com.gs.collections.api.block.function.primitive.FloatFunction0;
import com.gs.collections.api.block.function.primitive.FloatToFloatFunction;
import com.gs.collections.api.block.function.primitive.FloatToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectFloatToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.FloatFloatPredicate;
import com.gs.collections.api.block.predicate.primitive.FloatPredicate;
import com.gs.collections.api.block.procedure.primitive.FloatProcedure;
import com.gs.collections.api.block.procedure.primitive.FloatFloatProcedure;
import com.gs.collections.api.collection.MutableCollection;
import com.gs.collections.api.collection.primitive.ImmutableFloatCollection;
import com.gs.collections.api.collection.primitive.MutableFloatCollection;
import com.gs.collections.api.iterator.FloatIterator;
import com.gs.collections.api.list.primitive.MutableFloatList;
import com.gs.collections.api.map.primitive.FloatFloatMap;
import com.gs.collections.api.map.primitive.ImmutableFloatFloatMap;
import com.gs.collections.api.map.primitive.MutableFloatFloatMap;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.FloatSet;
import com.gs.collections.api.set.primitive.ImmutableFloatSet;
import com.gs.collections.api.set.primitive.MutableFloatSet;
import com.gs.collections.impl.bag.mutable.primitive.FloatHashBag;
import com.gs.collections.impl.block.factory.primitive.FloatPredicates;
import com.gs.collections.impl.collection.mutable.primitive.SynchronizedFloatCollection;
import com.gs.collections.impl.collection.mutable.primitive.UnmodifiableFloatCollection;
import com.gs.collections.impl.factory.Sets;
import com.gs.collections.impl.factory.primitive.FloatLists;
import com.gs.collections.impl.factory.primitive.FloatFloatMaps;
import com.gs.collections.impl.factory.primitive.FloatSets;
import com.gs.collections.impl.lazy.primitive.CollectFloatToObjectIterable;
import com.gs.collections.impl.lazy.primitive.LazyFloatIterableAdapter;
import com.gs.collections.impl.lazy.primitive.SelectFloatIterable;
import com.gs.collections.impl.list.mutable.FastList;
import com.gs.collections.impl.list.mutable.primitive.FloatArrayList;
import com.gs.collections.impl.set.mutable.primitive.FloatHashSet;
import com.gs.collections.impl.set.mutable.primitive.SynchronizedFloatSet;
import com.gs.collections.impl.set.mutable.primitive.UnmodifiableFloatSet;

/**
 * This file was automatically generated from template file primitivePrimitiveHashMap.stg.
 *
 * @since 3.0.
 */
public class FloatFloatHashMap implements MutableFloatFloatMap, Externalizable
{
    static final float EMPTY_VALUE = 0.0f;
    private static final long serialVersionUID = 1L;
    private static final float EMPTY_KEY = 0.0f;
    private static final float REMOVED_KEY = 1.0f;

    private static final int OCCUPIED_DATA_RATIO = 2;
    private static final int OCCUPIED_SENTINEL_RATIO = 4;
    private static final int DEFAULT_INITIAL_CAPACITY = 8;

    private float[] keys;
    private float[] values;

    private int occupiedWithData;
    private int occupiedWithSentinels;

    private SentinelValues sentinelValues;

    public FloatFloatHashMap()
    {
        this.allocateTable(DEFAULT_INITIAL_CAPACITY << 1);
    }

    public FloatFloatHashMap(int initialCapacity)
    {
        if (initialCapacity < 0)
        {
            throw new IllegalArgumentException("initial capacity cannot be less than 0");
        }
        int capacity = this.smallestPowerOfTwoGreaterThan(this.fastCeil(initialCapacity * OCCUPIED_DATA_RATIO));
        this.allocateTable(capacity);
    }

    private int smallestPowerOfTwoGreaterThan(int n)
    {
        return n > 1 ? Integer.highestOneBit(n - 1) << 1 : 1;
    }

    public FloatFloatHashMap(FloatFloatMap map)
    {
        this(Math.max(map.size(), DEFAULT_INITIAL_CAPACITY));
        this.putAll(map);
    }

    private int fastCeil(float v)
    {
        int possibleResult = (int) v;
        if (v - possibleResult > 0.0F)
        {
            possibleResult++;
        }
        return possibleResult;
    }

    public static FloatFloatHashMap newWithKeysValues(float key1, float value1)
    {
        return new FloatFloatHashMap(1).withKeyValue(key1, value1);
    }

    public static FloatFloatHashMap newWithKeysValues(float key1, float value1, float key2, float value2)
    {
        return new FloatFloatHashMap(2).withKeysValues(key1, value1, key2, value2);
    }

    public static FloatFloatHashMap newWithKeysValues(float key1, float value1, float key2, float value2, float key3, float value3)
    {
        return new FloatFloatHashMap(3).withKeysValues(key1, value1, key2, value2, key3, value3);
    }

    public static FloatFloatHashMap newWithKeysValues(float key1, float value1, float key2, float value2, float key3, float value3, float key4, float value4)
    {
        return new FloatFloatHashMap(4).withKeysValues(key1, value1, key2, value2, key3, value3, key4, value4);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }

        if (!(obj instanceof FloatFloatMap))
        {
            return false;
        }

        FloatFloatMap other = (FloatFloatMap) obj;

        if (this.size() != other.size())
        {
            return false;
        }

        if (this.sentinelValues == null)
        {
            if (other.containsKey(EMPTY_KEY) || other.containsKey(REMOVED_KEY))
            {
                return false;
            }
        }
        else
        {
            if (this.sentinelValues.containsZeroKey && (!other.containsKey(EMPTY_KEY) || Float.compare(this.sentinelValues.zeroValue, other.getOrThrow(EMPTY_KEY)) != 0))
            {
                return false;
            }

            if (this.sentinelValues.containsOneKey && (!other.containsKey(REMOVED_KEY) || Float.compare(this.sentinelValues.oneValue, other.getOrThrow(REMOVED_KEY)) != 0))
            {
                return false;
            }
        }

        for (int i = 0; i < this.keys.length; i++)
        {
            float key = this.keys[i];
            if (isNonSentinel(key) && (!other.containsKey(key) || Float.compare(this.values[i], other.getOrThrow(key)) != 0))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int result = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += Float.floatToIntBits(EMPTY_KEY) ^ Float.floatToIntBits(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += Float.floatToIntBits(REMOVED_KEY) ^ Float.floatToIntBits(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += Float.floatToIntBits(this.keys[i]) ^ Float.floatToIntBits(this.values[i]);
            }
        }

        return result;
    }

    @Override
    public String toString()
    {
        return this.makeString("{", ", ", "}");
    }

    public int size()
    {
        return this.occupiedWithData + (this.sentinelValues == null ? 0 : this.sentinelValues.size());
    }

    public boolean isEmpty()
    {
        return this.occupiedWithData == 0 && (this.sentinelValues == null || this.sentinelValues.size() == 0);
    }

    public boolean notEmpty()
    {
        return this.occupiedWithData != 0 || (this.sentinelValues != null && this.sentinelValues.size() != 0);
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            boolean first = true;

            if (this.sentinelValues != null)
            {
                if (this.sentinelValues.containsZeroKey)
                {
                    appendable.append(String.valueOf(EMPTY_KEY)).append("=").append(String.valueOf(this.sentinelValues.zeroValue));
                    first = false;
                }
                if (this.sentinelValues.containsOneKey)
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(REMOVED_KEY)).append("=").append(String.valueOf(this.sentinelValues.oneValue));
                    first = false;
                }
            }
            for (int i = 0; i < this.keys.length; i++)
            {
                float key = this.keys[i];
                if (isNonSentinel(key))
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(key)).append("=").append(String.valueOf(this.values[i]));
                    first = false;
                }
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public FloatIterator floatIterator()
    {
        return new InternalFloatIterator();
    }

    public float[] toArray()
    {
        float[] array = new float[this.size()];
        int index = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                array[index] = this.sentinelValues.zeroValue;
                index++;
            }
            if (this.sentinelValues.containsOneKey)
            {
                array[index] = this.sentinelValues.oneValue;
                index++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                array[index] = this.values[i];
                index++;
            }
        }

        return array;
    }

    public boolean contains(float value)
    {
        return this.containsValue(value);
    }

    public boolean containsAll(float... source)
    {
        for (float each : source)
        {
            if (!this.contains(each))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(FloatIterable source)
    {
        return source.allSatisfy(new FloatPredicate()
        {
            public boolean accept(float value)
            {
                return FloatFloatHashMap.this.contains(value);
            }
        });
    }

    public void forEach(FloatProcedure procedure)
    {
        this.forEachValue(procedure);
    }

    public MutableFloatCollection select(FloatPredicate predicate)
    {
        FloatArrayList result = new FloatArrayList();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }

        return result;
    }

    public MutableFloatCollection reject(FloatPredicate predicate)
    {
        FloatArrayList result = new FloatArrayList();
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public <V> MutableCollection<V> collect(FloatToObjectFunction<? extends V> function)
    {
        FastList<V> target = FastList.newList(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                target.add(function.valueOf(this.sentinelValues.zeroValue));
            }
            if (this.sentinelValues.containsOneKey)
            {
                target.add(function.valueOf(this.sentinelValues.oneValue));
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                target.add(function.valueOf(this.values[i]));
            }
        }
        return target;
    }

    public float detectIfNone(FloatPredicate predicate, float value)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return this.values[i];
            }
        }
        return value;
    }

    public int count(FloatPredicate predicate)
    {
        int count = 0;
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                count++;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                count++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(FloatPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return true;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return true;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(FloatPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(FloatPredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public <V> V injectInto(V injectedValue, ObjectFloatToObjectFunction<? super V, ? extends V> function)
    {
        V result = injectedValue;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result = function.valueOf(result, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                result = function.valueOf(result, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result = function.valueOf(result, this.values[i]);
            }
        }

        return result;
    }

    public MutableFloatList toList()
    {
        return FloatArrayList.newList(this);
    }

    public MutableFloatSet toSet()
    {
        return FloatHashSet.newSet(this);
    }

    public MutableFloatBag toBag()
    {
        return FloatHashBag.newBag(this);
    }

    public LazyFloatIterable asLazy()
    {
        return new LazyFloatIterableAdapter(this);
    }

    public void clear()
    {
        this.sentinelValues = null;
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;
        Arrays.fill(this.keys, EMPTY_KEY);
        Arrays.fill(this.values, EMPTY_VALUE);
    }

    public void put(float key, float value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return;
        }

        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return;
        }

        int index = this.probe(key);

        if (Float.compare(this.keys[index], key) == 0)
        {
            // key already present in map
            this.values[index] = value;
            return;
        }

        if (Float.compare(this.keys[index], REMOVED_KEY) == 0)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = key;
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
    }

    public void putAll(FloatFloatMap map)
    {
        map.forEachKeyValue(new FloatFloatProcedure()
        {
            public void value(float key, float value)
            {
                FloatFloatHashMap.this.put(key, value);
            }
        });
    }

    public void removeKey(float key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return;
            }
            if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.containsZeroKey = false;
                this.sentinelValues.zeroValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.containsOneKey = false;
                this.sentinelValues.oneValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return;
        }
        int index = this.probe(key);
        if (Float.compare(this.keys[index], key) == 0)
        {
            this.keys[index] = REMOVED_KEY;
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }
        }
    }

    public void remove(float key)
    {
        this.removeKey(key);
    }

    public float removeKeyIfAbsent(float key, float value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return value;
            }
            float oldValue = this.sentinelValues.zeroValue;
            if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.containsZeroKey = false;
                this.sentinelValues.zeroValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return oldValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return value;
            }
            float oldValue = this.sentinelValues.oneValue;
            if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.containsOneKey = false;
                this.sentinelValues.oneValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return oldValue;
        }
        int index = this.probe(key);
        if (Float.compare(this.keys[index], key) == 0)
        {
            this.keys[index] = REMOVED_KEY;
            float oldValue = this.values[index];
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }

            return oldValue;
        }
        return value;
    }

    public float getIfAbsentPut(float key, float value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (Float.compare(this.keys[index], key) == 0)
        {
            return this.values[index];
        }
        if (Float.compare(this.keys[index], REMOVED_KEY) == 0)
        {
            this.occupiedWithSentinels--;
        }
        this.keys[index] = key;
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public float getIfAbsentPut(float key, FloatFunction0 function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                float value = function.value();
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            float value = function.value();
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                float value = function.value();
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            float value = function.value();
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (Float.compare(this.keys[index], key) == 0)
        {
            return this.values[index];
        }
        if (Float.compare(this.keys[index], REMOVED_KEY) == 0)
        {
            this.occupiedWithSentinels--;
        }
        this.keys[index] = key;
        float value = function.value();
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public <P> float getIfAbsentPutWith(float key, FloatFunction<? super P> function, P parameter)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                float value = function.floatValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            float value = function.floatValueOf(parameter);
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                float value = function.floatValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            float value = function.floatValueOf(parameter);
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (Float.compare(this.keys[index], key) == 0)
        {
            return this.values[index];
        }
        if (Float.compare(this.keys[index], REMOVED_KEY) == 0)
        {
            this.occupiedWithSentinels--;
        }
        this.keys[index] = key;
        float value = function.floatValueOf(parameter);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public float getIfAbsentPutWithKey(float key, FloatToFloatFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                float value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            float value = function.valueOf(key);
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                float value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            float value = function.valueOf(key);
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (Float.compare(this.keys[index], key) == 0)
        {
            return this.values[index];
        }
        if (Float.compare(this.keys[index], REMOVED_KEY) == 0)
        {
            this.occupiedWithSentinels--;
        }
        this.keys[index] = key;
        float value = function.valueOf(key);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public float updateValue(float key, float initialValueIfAbsent, FloatToFloatFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = function.valueOf(initialValueIfAbsent);
            }
            else if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.zeroValue = function.valueOf(this.sentinelValues.zeroValue);
            }
            else
            {
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = function.valueOf(initialValueIfAbsent);
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = function.valueOf(initialValueIfAbsent);
            }
            else if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.oneValue = function.valueOf(this.sentinelValues.oneValue);
            }
            else
            {
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = function.valueOf(initialValueIfAbsent);
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (Float.compare(this.keys[index], key) == 0)
        {
            this.values[index] = function.valueOf(this.values[index]);
            return this.values[index];
        }
        if (Float.compare(this.keys[index], REMOVED_KEY) == 0)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = key;
        float value = function.valueOf(initialValueIfAbsent);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public FloatFloatHashMap withKeyValue(float key1, float value1)
    {
        this.put(key1, value1);
        return this;
    }

    public FloatFloatHashMap withKeysValues(float key1, float value1, float key2, float value2)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        return this;
    }

    public FloatFloatHashMap withKeysValues(float key1, float value1, float key2, float value2, float key3, float value3)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        return this;
    }

    public FloatFloatHashMap withKeysValues(float key1, float value1, float key2, float value2, float key3, float value3, float key4, float value4)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        this.put(key4, value4);
        return this;
    }

    public FloatFloatHashMap withoutKey(float key)
    {
        this.removeKey(key);
        return this;
    }

    public FloatFloatHashMap withoutAllKeys(FloatIterable keys)
    {
        keys.forEach(new FloatProcedure()
        {
            public void value(float key)
            {
                FloatFloatHashMap.this.removeKey(key);
            }
        });
        return this;
    }

    public MutableFloatFloatMap asUnmodifiable()
    {
        return new UnmodifiableFloatFloatMap(this);
    }

    public MutableFloatFloatMap asSynchronized()
    {
        return new SynchronizedFloatFloatMap(this);
    }

    public ImmutableFloatFloatMap toImmutable()
    {
        return FloatFloatMaps.immutable.ofAll(this);
    }

    public float get(float key)
    {
        return this.getIfAbsent(key, EMPTY_VALUE);
    }

    public float getIfAbsent(float key, float ifAbsent)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (Float.compare(this.keys[index], key) == 0)
        {
            return this.values[index];
        }
        return ifAbsent;
    }

    public float getOrThrow(float key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]))
        {
            return this.values[index];
        }
        throw new IllegalStateException("Key " + key + " not present.");
    }

    public boolean containsKey(float key)
    {
        if (isEmptyKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsZeroKey;
        }
        if (isRemovedKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsOneKey;
        }
        return Float.compare(this.keys[this.probe(key)], key) == 0;
    }

    public boolean containsValue(float value)
    {
        if (this.sentinelValues != null && this.sentinelValues.containsValue(value))
        {
            return true;
        }
        for (int i = 0; i < this.values.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && Float.compare(this.values[i], value) == 0)
            {
                return true;
            }
        }
        return false;
    }

    public void forEachValue(FloatProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.values[i]);
            }
        }
    }

    public void forEachKey(FloatProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY);
            }
        }
        for (float key : this.keys)
        {
            if (isNonSentinel(key))
            {
                procedure.value(key);
            }
        }
    }

    public void forEachKeyValue(FloatFloatProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.keys[i], this.values[i]);
            }
        }
    }

    public LazyFloatIterable keysView()
    {
        return new KeysView();
    }

    public FloatFloatHashMap select(FloatFloatPredicate predicate)
    {
        FloatFloatHashMap result = new FloatFloatHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.keys[i], this.values[i]))
            {
                result.put(this.keys[i], this.values[i]);
            }
        }

        return result;
    }

    public FloatFloatHashMap reject(FloatFloatPredicate predicate)
    {
        FloatFloatHashMap result = new FloatFloatHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.keys[i], this.values[i]))
            {
                result.put(this.keys[i], this.values[i]);
            }
        }
        return result;
    }

    public double sum()
    {
        double result = 0.0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += this.values[i];
            }
        }

        return result;
    }

    public float max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        FloatIterator iterator = this.floatIterator();
        float max = iterator.next();
        while (iterator.hasNext())
        {
            float value = iterator.next();
            if (Float.compare(max, value) < 0)
            {
                max = value;
            }
        }
        return max;
    }

    public float maxIfEmpty(float defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public float min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        FloatIterator iterator = this.floatIterator();
        float min = iterator.next();
        while (iterator.hasNext())
        {
            float value = iterator.next();
            if (Float.compare(value, min) < 0)
            {
                min = value;
            }
        }
        return min;
    }

    public float minIfEmpty(float defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        float[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            float first = sortedArray[middleIndex];
            float second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public float[] toSortedArray()
    {
        float[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public MutableFloatList toSortedList()
    {
        return FloatArrayList.newList(this).sortThis();
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                out.writeFloat(EMPTY_KEY);
                out.writeFloat(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                out.writeFloat(REMOVED_KEY);
                out.writeFloat(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                out.writeFloat(this.keys[i]);
                out.writeFloat(this.values[i]);
            }
        }
    }

    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
    {
        int size = in.readInt();
        for (int i = 0; i < size; i++)
        {
            this.put(in.readFloat(), in.readFloat());
        }
    }

    /**
     * Rehashes every element in the set into a new backing table of the smallest possible size and eliminating removed sentinels.
     */
    public void compact()
    {
        this.rehash(this.smallestPowerOfTwoGreaterThan(this.size()));
    }

    private void rehash()
    {
        this.rehash(this.keys.length);
    }

    private void rehashAndGrow()
    {
        this.rehash(this.keys.length << 1);
    }

    private void rehash(int newCapacity)
    {
        int oldLength = this.keys.length;
        float[] old = this.keys;
        float[] oldValues = this.values;
        this.allocateTable(newCapacity);
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;

        for (int i = 0; i < oldLength; i++)
        {
            if (isNonSentinel(old[i]))
            {
                this.put(old[i], oldValues[i]);
            }
        }
    }

    // exposed for testing
    int probe(float element)
    {
        int index = this.spread(element);
        float keyAtIndex = this.keys[index];

        if (Float.compare(keyAtIndex, element) == 0 || Float.compare(keyAtIndex, EMPTY_KEY) == 0)
        {
            return index;
        }

        int removedIndex = Float.compare(keyAtIndex, REMOVED_KEY) == 0 ? index : -1;
        int nextIndex = index;
        int probe = 17;

        // loop until an empty slot is reached
        while (true)
        {
            // Probe algorithm: 17*n*(n+1)/2 where n = number of collisions
            nextIndex += probe;
            probe += 17;
            nextIndex &= this.keys.length - 1;

            if (Float.compare(this.keys[nextIndex], element) == 0)
            {
                return nextIndex;
            }
            if (Float.compare(this.keys[nextIndex], REMOVED_KEY) == 0)
            {
                if (removedIndex == -1)
                {
                    removedIndex = nextIndex;
                }
            }
            else if (Float.compare(this.keys[nextIndex], EMPTY_KEY) == 0)
            {
                return removedIndex == -1 ? nextIndex : removedIndex;
            }
        }
    }

    // exposed for testing
    int spread(float element)
    {
        int code = Float.floatToIntBits(element);
        code ^= 61 ^ (code >> 16);
        code += code << 3;
        code ^= code >> 4;
        code *= 0x27d4eb2d;
        code ^= code >> 15;
        return code & (this.keys.length - 1);
    }

    private void allocateTable(int sizeToAllocate)
    {
        this.keys = new float[sizeToAllocate];
        this.values = new float[sizeToAllocate];
    }

    private static boolean isEmptyKey(float key)
    {
        return Float.compare(key, EMPTY_KEY) == 0;
    }

    private static boolean isRemovedKey(float key)
    {
        return Float.compare(key, REMOVED_KEY) == 0;
    }

    private static boolean isNonSentinel(float key)
    {
        return !isEmptyKey(key) && !isRemovedKey(key);
    }

    private int maxOccupiedWithData()
    {
        int capacity = this.keys.length;
        // need at least one free slot for open addressing
        return Math.min(capacity - 1, capacity / OCCUPIED_DATA_RATIO);
    }

    private int maxOccupiedWithSentinels()
    {
        return this.keys.length / OCCUPIED_SENTINEL_RATIO;
    }

    private static final class SentinelValues
    {
        private boolean containsZeroKey;
        private boolean containsOneKey;
        private float zeroValue;
        private float oneValue;

        public int size()
        {
            return (this.containsZeroKey ? 1 : 0) + (this.containsOneKey ? 1 : 0);
        }

        public boolean containsValue(float value)
        {
            boolean valueEqualsZeroValue = this.containsZeroKey && Float.compare(this.zeroValue, value) == 0;
            boolean valueEqualsOneValue = this.containsOneKey && Float.compare(this.oneValue, value) == 0;
            return valueEqualsZeroValue || valueEqualsOneValue;
        }
    }

    private class InternalFloatIterator implements FloatIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < FloatFloatHashMap.this.size();
        }

        public float next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (FloatFloatHashMap.this.containsKey(EMPTY_KEY))
                {
                    return FloatFloatHashMap.this.get(EMPTY_KEY);
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (FloatFloatHashMap.this.containsKey(REMOVED_KEY))
                {
                    return FloatFloatHashMap.this.get(REMOVED_KEY);
                }
            }

            float[] keys = FloatFloatHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            float result = FloatFloatHashMap.this.values[this.position];
            this.position++;
            return result;
        }
    }

    private class KeysView implements LazyFloatIterable
    {
        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public int size()
        {
            return FloatFloatHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return FloatFloatHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return FloatFloatHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (FloatFloatHashMap.this.sentinelValues != null)
                {
                    if (FloatFloatHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (FloatFloatHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (float key : FloatFloatHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public FloatIterator floatIterator()
        {
            return new KeySetIterator();
        }

        public boolean contains(float key)
        {
            return FloatFloatHashMap.this.containsKey(key);
        }

        public boolean containsAll(float... keys)
        {
            for (float key : keys)
            {
                if (!FloatFloatHashMap.this.containsKey(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(FloatIterable source)
        {
            return source.allSatisfy(new FloatPredicate()
            {
                public boolean accept(float key)
                {
                    return FloatFloatHashMap.this.containsKey(key);
                }
            });
        }

        public void forEach(FloatProcedure procedure)
        {
            FloatFloatHashMap.this.forEachKey(procedure);
        }

        public int count(FloatPredicate predicate)
        {
            int count = 0;
            if (FloatFloatHashMap.this.sentinelValues != null)
            {
                if (FloatFloatHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (FloatFloatHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (float key : FloatFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(FloatPredicate predicate)
        {
            if (FloatFloatHashMap.this.sentinelValues != null)
            {
                if (FloatFloatHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (FloatFloatHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (float key : FloatFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(FloatPredicate predicate)
        {
            if (FloatFloatHashMap.this.sentinelValues != null)
            {
                if (FloatFloatHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (FloatFloatHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (float key : FloatFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(FloatPredicate predicate)
        {
            if (FloatFloatHashMap.this.sentinelValues != null)
            {
                if (FloatFloatHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (FloatFloatHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (float key : FloatFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public LazyFloatIterable select(FloatPredicate predicate)
        {
            return new SelectFloatIterable(this, predicate);
        }

        public LazyFloatIterable reject(FloatPredicate predicate)
        {
            return new SelectFloatIterable(this, FloatPredicates.not(predicate));
        }

        public float detectIfNone(FloatPredicate predicate, float value)
        {
            if (FloatFloatHashMap.this.sentinelValues != null)
            {
                if (FloatFloatHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (FloatFloatHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (float key : FloatFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return value;
        }

        public <V> LazyIterable<V> collect(FloatToObjectFunction<? extends V> function)
        {
            return new CollectFloatToObjectIterable<V>(this, function);
        }

        public double sum()
        {
            double result = 0.0;

            if (FloatFloatHashMap.this.sentinelValues != null)
            {
                if (FloatFloatHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += EMPTY_KEY;
                }
                if (FloatFloatHashMap.this.sentinelValues.containsOneKey)
                {
                    result += REMOVED_KEY;
                }
            }
            for (float key : FloatFloatHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result += key;
                }
            }

            return result;
        }

        public float max()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            FloatIterator iterator = this.floatIterator();
            float max = iterator.next();
            while (iterator.hasNext())
            {
                float value = iterator.next();
                if (Float.compare(max, value) < 0)
                {
                    max = value;
                }
            }
            return max;
        }

        public float min()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            FloatIterator iterator = this.floatIterator();
            float min = iterator.next();
            while (iterator.hasNext())
            {
                float value = iterator.next();
                if (Float.compare(value, min) < 0)
                {
                    min = value;
                }
            }
            return min;
        }

        public float minIfEmpty(float defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.min();
        }

        public float maxIfEmpty(float defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.max();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            float[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                float first = sortedArray[middleIndex];
                float second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public float[] toSortedArray()
        {
            float[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public float[] toArray()
        {
            int size = FloatFloatHashMap.this.size();
            final float[] result = new float[size];
            FloatFloatHashMap.this.forEachKey(new FloatProcedure()
            {
                private int index;

                public void value(float each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public <T> T injectInto(T injectedValue, ObjectFloatToObjectFunction<? super T, ? extends T> function)
        {
            T result = injectedValue;
            if (FloatFloatHashMap.this.sentinelValues != null)
            {
                if (FloatFloatHashMap.this.sentinelValues.containsZeroKey)
                {
                    result = function.valueOf(result, EMPTY_KEY);
                }
                if (FloatFloatHashMap.this.sentinelValues.containsOneKey)
                {
                    result = function.valueOf(result, REMOVED_KEY);
                }
            }
            for (int i = 0; i < FloatFloatHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(FloatFloatHashMap.this.keys[i]))
                {
                    result = function.valueOf(result, FloatFloatHashMap.this.keys[i]);
                }
            }
            return result;
        }

        public MutableFloatList toList()
        {
            return FloatArrayList.newList(this);
        }

        public MutableFloatList toSortedList()
        {
            return FloatArrayList.newList(this).sortThis();
        }

        public MutableFloatSet toSet()
        {
            return FloatHashSet.newSet(this);
        }

        public MutableFloatBag toBag()
        {
            return FloatHashBag.newBag(this);
        }

        public LazyFloatIterable asLazy()
        {
            return this;
        }
    }

    private class KeySetIterator implements FloatIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < FloatFloatHashMap.this.size();
        }

        public float next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (FloatFloatHashMap.this.containsKey(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (FloatFloatHashMap.this.containsKey(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }

            float[] keys = FloatFloatHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            float result = keys[this.position];
            this.position++;
            return result;
        }
    }

    public MutableFloatSet keySet()
    {
        return new KeySet();
    }

    private class KeySet implements MutableFloatSet
    {
        public FloatIterator floatIterator()
        {
            return new KeySetIterator();
        }

        public void forEach(FloatProcedure procedure)
        {
            FloatFloatHashMap.this.forEachKey(procedure);
        }

        public int count(FloatPredicate predicate)
        {
            int count = 0;
            if (FloatFloatHashMap.this.sentinelValues != null)
            {
                if (FloatFloatHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (FloatFloatHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (float key : FloatFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(FloatPredicate predicate)
        {
            if (FloatFloatHashMap.this.sentinelValues != null)
            {
                if (FloatFloatHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (FloatFloatHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (float key : FloatFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(FloatPredicate predicate)
        {
            if (FloatFloatHashMap.this.sentinelValues != null)
            {
                if (FloatFloatHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (FloatFloatHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (float key : FloatFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(FloatPredicate predicate)
        {
            if (FloatFloatHashMap.this.sentinelValues != null)
            {
                if (FloatFloatHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (FloatFloatHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (float key : FloatFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean add(float element)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(float... source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(FloatIterable source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(float key)
        {
            int oldSize = FloatFloatHashMap.this.size();
            FloatFloatHashMap.this.removeKey(key);
            return oldSize != FloatFloatHashMap.this.size();
        }

        public boolean removeAll(FloatIterable source)
        {
            int oldSize = FloatFloatHashMap.this.size();
            FloatIterator iterator = source.floatIterator();
            while (iterator.hasNext())
            {
                FloatFloatHashMap.this.removeKey(iterator.next());
            }
            return oldSize != FloatFloatHashMap.this.size();
        }

        public boolean removeAll(float... source)
        {
            int oldSize = FloatFloatHashMap.this.size();
            for (float item : source)
            {
                FloatFloatHashMap.this.removeKey(item);
            }
            return oldSize != FloatFloatHashMap.this.size();
        }

        public void clear()
        {
            FloatFloatHashMap.this.clear();
        }

        public MutableFloatSet select(FloatPredicate predicate)
        {
            MutableFloatSet result = new FloatHashSet();
            if (FloatFloatHashMap.this.sentinelValues != null)
            {
                if (FloatFloatHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (FloatFloatHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (float key : FloatFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableFloatSet reject(FloatPredicate predicate)
        {
            MutableFloatSet result = new FloatHashSet();
            if (FloatFloatHashMap.this.sentinelValues != null)
            {
                if (FloatFloatHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (FloatFloatHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (float key : FloatFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableFloatSet with(float element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableFloatSet without(float element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableFloatSet withAll(FloatIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableFloatSet withoutAll(FloatIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public float detectIfNone(FloatPredicate predicate, float ifNone)
        {
            if (FloatFloatHashMap.this.sentinelValues != null)
            {
                if (FloatFloatHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (FloatFloatHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (float key : FloatFloatHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return ifNone;
        }

        public <V> MutableSet<V> collect(FloatToObjectFunction<? extends V> function)
        {
            MutableSet<V> result = Sets.mutable.with();
            if (FloatFloatHashMap.this.sentinelValues != null)
            {
                if (FloatFloatHashMap.this.sentinelValues.containsZeroKey)
                {
                    result.add(function.valueOf(EMPTY_KEY));
                }
                if (FloatFloatHashMap.this.sentinelValues.containsOneKey)
                {
                    result.add(function.valueOf(REMOVED_KEY));
                }
            }
            for (float key : FloatFloatHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result.add(function.valueOf(key));
                }
            }
            return result;
        }

        public MutableFloatSet asUnmodifiable()
        {
            return UnmodifiableFloatSet.of(this);
        }

        public MutableFloatSet asSynchronized()
        {
            return SynchronizedFloatSet.of(this);
        }

        public double sum()
        {
            double sum = 0.0;
            if (FloatFloatHashMap.this.sentinelValues != null)
            {
                if (FloatFloatHashMap.this.sentinelValues.containsZeroKey)
                {
                    sum += EMPTY_KEY;
                }
                if (FloatFloatHashMap.this.sentinelValues.containsOneKey)
                {
                    sum += REMOVED_KEY;
                }
            }
            for (float key : FloatFloatHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    sum += key;
                }
            }
            return sum;
        }

        public float max()
        {
            if (FloatFloatHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            float max = 0;
            boolean isMaxSet = false;

            if (FloatFloatHashMap.this.sentinelValues != null)
            {
                if (FloatFloatHashMap.this.sentinelValues.containsZeroKey)
                {
                    max = EMPTY_KEY;
                    isMaxSet = true;
                }
                if (FloatFloatHashMap.this.sentinelValues.containsOneKey && (!isMaxSet || Float.compare(max, REMOVED_KEY) < 0))
                {
                    max = REMOVED_KEY;
                    isMaxSet = true;
                }
            }
            for (int i = 0; i < FloatFloatHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(FloatFloatHashMap.this.keys[i]) && (!isMaxSet || Float.compare(max, FloatFloatHashMap.this.keys[i]) < 0))
                {
                    max = FloatFloatHashMap.this.keys[i];
                    isMaxSet = true;
                }
            }
            return max;
        }

        public float maxIfEmpty(float defaultValue)
        {
            if (FloatFloatHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.max();
        }

        public float min()
        {
            if (FloatFloatHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            float min = 0;
            boolean isMinSet = false;

            if (FloatFloatHashMap.this.sentinelValues != null)
            {
                if (FloatFloatHashMap.this.sentinelValues.containsZeroKey)
                {
                    min = EMPTY_KEY;
                    isMinSet = true;
                }
                if (FloatFloatHashMap.this.sentinelValues.containsOneKey && (!isMinSet || Float.compare(REMOVED_KEY, min) < 0))
                {
                    min = REMOVED_KEY;
                    isMinSet = true;
                }
            }
            for (int i = 0; i < FloatFloatHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(FloatFloatHashMap.this.keys[i]) && (!isMinSet || Float.compare(FloatFloatHashMap.this.keys[i], min) < 0))
                {
                    min = FloatFloatHashMap.this.keys[i];
                    isMinSet = true;
                }
            }
            return min;
        }

        public float minIfEmpty(float defaultValue)
        {
            if (FloatFloatHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.min();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            float[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                float first = sortedArray[middleIndex];
                float second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public float[] toSortedArray()
        {
            float[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public MutableFloatList toSortedList()
        {
            return FloatArrayList.newList(this).sortThis();
        }

        public float[] toArray()
        {
            int size = FloatFloatHashMap.this.size();
            final float[] result = new float[size];
            FloatFloatHashMap.this.forEachKey(new FloatProcedure()
            {
                private int index;

                public void value(float each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public boolean contains(float value)
        {
            return FloatFloatHashMap.this.containsKey(value);
        }

        public boolean containsAll(float... source)
        {
            for (float item : source)
            {
                if (!FloatFloatHashMap.this.containsKey(item))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(FloatIterable source)
        {
            FloatIterator iterator = source.floatIterator();
            while (iterator.hasNext())
            {
                if (!FloatFloatHashMap.this.containsKey(iterator.next()))
                {
                    return false;
                }
            }
            return true;
        }

        public MutableFloatList toList()
        {
            return FloatArrayList.newList(this);
        }

        public MutableFloatSet toSet()
        {
            return FloatHashSet.newSet(this);
        }

        public MutableFloatBag toBag()
        {
            return FloatHashBag.newBag(this);
        }

        public LazyFloatIterable asLazy()
        {
            return new LazyFloatIterableAdapter(this);
        }

        public <T> T injectInto(T injectedValue, ObjectFloatToObjectFunction<? super T, ? extends T> function)
        {
            T result = injectedValue;
            if (FloatFloatHashMap.this.sentinelValues != null)
            {
                if (FloatFloatHashMap.this.sentinelValues.containsZeroKey)
                {
                    result = function.valueOf(result, EMPTY_KEY);
                }
                if (FloatFloatHashMap.this.sentinelValues.containsOneKey)
                {
                    result = function.valueOf(result, REMOVED_KEY);
                }
            }
            for (int i = 0; i < FloatFloatHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(FloatFloatHashMap.this.keys[i]))
                {
                    result = function.valueOf(result, FloatFloatHashMap.this.keys[i]);
                }
            }
            return result;
        }

        public FloatSet freeze()
        {
            throw new UnsupportedOperationException();
        }

        public ImmutableFloatSet toImmutable()
        {
            return FloatSets.immutable.withAll(this);
        }

        public int size()
        {
            return FloatFloatHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return FloatFloatHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return FloatFloatHashMap.this.notEmpty();
        }

        @Override
        public boolean equals(Object obj)
        {
            if (this == obj)
            {
                return true;
            }

            if (!(obj instanceof FloatSet))
            {
                return false;
            }

            FloatSet other = (FloatSet) obj;
            return this.size() == other.size() && this.containsAll(other.toArray());
        }

        @Override
        public int hashCode()
        {
            int result = 0;

            if (FloatFloatHashMap.this.sentinelValues != null)
            {
                if (FloatFloatHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += Float.floatToIntBits(EMPTY_KEY);
                }
                if (FloatFloatHashMap.this.sentinelValues.containsOneKey)
                {
                    result += Float.floatToIntBits(REMOVED_KEY);
                }
            }
            for (int i = 0; i < FloatFloatHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(FloatFloatHashMap.this.keys[i]))
                {
                    result += Float.floatToIntBits(FloatFloatHashMap.this.keys[i]);
                }
            }

            return result;
        }

        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (FloatFloatHashMap.this.sentinelValues != null)
                {
                    if (FloatFloatHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (FloatFloatHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (float key : FloatFloatHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }
    }

    public MutableFloatCollection values()
    {
        return new ValuesCollection();
    }

    private class ValuesCollection implements MutableFloatCollection
    {
        public void clear()
        {
            FloatFloatHashMap.this.clear();
        }

        public MutableFloatCollection select(FloatPredicate predicate)
        {
            return FloatFloatHashMap.this.select(predicate);
        }

        public MutableFloatCollection reject(FloatPredicate predicate)
        {
            return FloatFloatHashMap.this.reject(predicate);
        }

        public float detectIfNone(FloatPredicate predicate, float ifNone)
        {
            return FloatFloatHashMap.this.detectIfNone(predicate, ifNone);
        }

        public <V> MutableCollection<V> collect(FloatToObjectFunction<? extends V> function)
        {
            return FloatFloatHashMap.this.collect(function);
        }

        public <T> T injectInto(T injectedValue, ObjectFloatToObjectFunction<? super T, ? extends T> function)
        {
            return FloatFloatHashMap.this.injectInto(injectedValue, function);
        }

        public double sum()
        {
            return FloatFloatHashMap.this.sum();
        }

        public float max()
        {
            return FloatFloatHashMap.this.max();
        }

        public float maxIfEmpty(float defaultValue)
        {
            return FloatFloatHashMap.this.maxIfEmpty(defaultValue);
        }

        public float min()
        {
            return FloatFloatHashMap.this.min();
        }

        public float minIfEmpty(float defaultValue)
        {
            return FloatFloatHashMap.this.minIfEmpty(defaultValue);
        }

        public double average()
        {
            return FloatFloatHashMap.this.average();
        }

        public double median()
        {
            return FloatFloatHashMap.this.median();
        }

        public float[] toSortedArray()
        {
            return FloatFloatHashMap.this.toSortedArray();
        }

        public MutableFloatList toSortedList()
        {
            return FloatFloatHashMap.this.toSortedList();
        }

        public MutableFloatCollection with(float element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableFloatCollection without(float element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableFloatCollection withAll(FloatIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableFloatCollection withoutAll(FloatIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableFloatCollection asUnmodifiable()
        {
            return UnmodifiableFloatCollection.of(this);
        }

        public MutableFloatCollection asSynchronized()
        {
            return SynchronizedFloatCollection.of(this);
        }

        public ImmutableFloatCollection toImmutable()
        {
            return FloatLists.immutable.withAll(this);
        }

        public boolean contains(float value)
        {
            return FloatFloatHashMap.this.containsValue(value);
        }

        public boolean containsAll(float... source)
        {
            return FloatFloatHashMap.this.containsAll(source);
        }

        public boolean containsAll(FloatIterable source)
        {
            return FloatFloatHashMap.this.containsAll(source);
        }

        public MutableFloatList toList()
        {
            return FloatFloatHashMap.this.toList();
        }

        public MutableFloatSet toSet()
        {
            return FloatFloatHashMap.this.toSet();
        }

        public MutableFloatBag toBag()
        {
            return FloatFloatHashMap.this.toBag();
        }

        public LazyFloatIterable asLazy()
        {
            return new LazyFloatIterableAdapter(this);
        }

        public boolean isEmpty()
        {
            return FloatFloatHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return FloatFloatHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);

                boolean first = true;

                if (FloatFloatHashMap.this.sentinelValues != null)
                {
                    if (FloatFloatHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(FloatFloatHashMap.this.sentinelValues.zeroValue));
                        first = false;
                    }
                    if (FloatFloatHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(FloatFloatHashMap.this.sentinelValues.oneValue));
                        first = false;
                    }
                }
                for (int i = 0; i < FloatFloatHashMap.this.keys.length; i++)
                {
                    float key = FloatFloatHashMap.this.keys[i];
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(FloatFloatHashMap.this.values[i]));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public FloatIterator floatIterator()
        {
            return FloatFloatHashMap.this.floatIterator();
        }

        public void forEach(FloatProcedure procedure)
        {
            FloatFloatHashMap.this.forEach(procedure);
        }

        public int count(FloatPredicate predicate)
        {
            return FloatFloatHashMap.this.count(predicate);
        }

        public boolean anySatisfy(FloatPredicate predicate)
        {
            return FloatFloatHashMap.this.anySatisfy(predicate);
        }

        public boolean allSatisfy(FloatPredicate predicate)
        {
            return FloatFloatHashMap.this.allSatisfy(predicate);
        }

        public boolean noneSatisfy(FloatPredicate predicate)
        {
            return FloatFloatHashMap.this.noneSatisfy(predicate);
        }

        public boolean add(float element)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(float... source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(FloatIterable source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(float item)
        {
            int oldSize = FloatFloatHashMap.this.size();

            if (FloatFloatHashMap.this.sentinelValues != null)
            {
                if (FloatFloatHashMap.this.sentinelValues.containsZeroKey && Float.compare(item, FloatFloatHashMap.this.sentinelValues.zeroValue) == 0)
                {
                    FloatFloatHashMap.this.removeKey(EMPTY_KEY);
                }
                if (FloatFloatHashMap.this.sentinelValues.containsOneKey && Float.compare(item, FloatFloatHashMap.this.sentinelValues.oneValue) == 0)
                {
                    FloatFloatHashMap.this.removeKey(REMOVED_KEY);
                }
            }
            for (int i = 0; i < FloatFloatHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(FloatFloatHashMap.this.keys[i]) && Float.compare(item, FloatFloatHashMap.this.values[i]) == 0)
                {
                    FloatFloatHashMap.this.removeKey(FloatFloatHashMap.this.keys[i]);
                }
            }
            return oldSize != FloatFloatHashMap.this.size();
        }

        public boolean removeAll(FloatIterable source)
        {
            int oldSize = FloatFloatHashMap.this.size();

            FloatIterator iterator = source.floatIterator();
            while (iterator.hasNext())
            {
                this.remove(iterator.next());
            }
            return oldSize != FloatFloatHashMap.this.size();
        }

        public boolean removeAll(float... source)
        {
            int oldSize = FloatFloatHashMap.this.size();

            for (float item : source)
            {
                this.remove(item);
            }
            return oldSize != FloatFloatHashMap.this.size();
        }

        public int size()
        {
            return FloatFloatHashMap.this.size();
        }

        public float[] toArray()
        {
            return FloatFloatHashMap.this.toArray();
        }
    }
}
