/*
 * Copyright 2013 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.map.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.LongIterable;
import com.gs.collections.api.LazyLongIterable;
import com.gs.collections.api.LazyByteIterable;
import com.gs.collections.api.ByteIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.bag.primitive.MutableLongBag;
import com.gs.collections.api.bag.primitive.MutableByteBag;
import com.gs.collections.api.block.function.primitive.LongToObjectFunction;
import com.gs.collections.api.block.function.primitive.LongToByteFunction;
import com.gs.collections.api.block.function.primitive.ByteFunction;
import com.gs.collections.api.block.function.primitive.ByteFunction0;
import com.gs.collections.api.block.function.primitive.ByteToByteFunction;
import com.gs.collections.api.block.function.primitive.ByteToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectByteToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectLongToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.LongBytePredicate;
import com.gs.collections.api.block.predicate.primitive.LongPredicate;
import com.gs.collections.api.block.predicate.primitive.BytePredicate;
import com.gs.collections.api.block.procedure.primitive.LongProcedure;
import com.gs.collections.api.block.procedure.primitive.LongByteProcedure;
import com.gs.collections.api.block.procedure.primitive.ByteProcedure;
import com.gs.collections.api.collection.MutableCollection;
import com.gs.collections.api.collection.primitive.ImmutableByteCollection;
import com.gs.collections.api.collection.primitive.MutableByteCollection;
import com.gs.collections.api.iterator.LongIterator;
import com.gs.collections.api.iterator.ByteIterator;
import com.gs.collections.api.list.primitive.MutableLongList;
import com.gs.collections.api.list.primitive.MutableByteList;
import com.gs.collections.api.map.primitive.LongByteMap;
import com.gs.collections.api.map.primitive.ImmutableLongByteMap;
import com.gs.collections.api.map.primitive.MutableLongByteMap;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.LongSet;
import com.gs.collections.api.set.primitive.ImmutableLongSet;
import com.gs.collections.api.set.primitive.MutableLongSet;
import com.gs.collections.api.set.primitive.MutableByteSet;
import com.gs.collections.impl.bag.mutable.primitive.LongHashBag;
import com.gs.collections.impl.bag.mutable.primitive.ByteHashBag;
import com.gs.collections.impl.block.factory.primitive.LongPredicates;
import com.gs.collections.impl.collection.mutable.primitive.SynchronizedByteCollection;
import com.gs.collections.impl.collection.mutable.primitive.UnmodifiableByteCollection;
import com.gs.collections.impl.factory.Sets;
import com.gs.collections.impl.factory.primitive.ByteLists;
import com.gs.collections.impl.factory.primitive.LongByteMaps;
import com.gs.collections.impl.factory.primitive.LongSets;
import com.gs.collections.impl.lazy.primitive.CollectLongToObjectIterable;
import com.gs.collections.impl.lazy.primitive.LazyByteIterableAdapter;
import com.gs.collections.impl.lazy.primitive.LazyLongIterableAdapter;
import com.gs.collections.impl.lazy.primitive.SelectLongIterable;
import com.gs.collections.impl.list.mutable.FastList;
import com.gs.collections.impl.list.mutable.primitive.LongArrayList;
import com.gs.collections.impl.list.mutable.primitive.ByteArrayList;
import com.gs.collections.impl.set.mutable.primitive.LongHashSet;
import com.gs.collections.impl.set.mutable.primitive.SynchronizedLongSet;
import com.gs.collections.impl.set.mutable.primitive.UnmodifiableLongSet;
import com.gs.collections.impl.set.mutable.primitive.ByteHashSet;

/**
 * This file was automatically generated from template file primitivePrimitiveHashMap.stg.
 *
 * @since 3.0.
 */
public class LongByteHashMap implements MutableLongByteMap, Externalizable
{
    static final byte EMPTY_VALUE = (byte) 0;
    private static final long serialVersionUID = 1L;
    private static final long EMPTY_KEY = 0L;
    private static final long REMOVED_KEY = 1L;

    private static final int OCCUPIED_DATA_RATIO = 2;
    private static final int OCCUPIED_SENTINEL_RATIO = 4;
    private static final int DEFAULT_INITIAL_CAPACITY = 8;

    private long[] keys;
    private byte[] values;

    private int occupiedWithData;
    private int occupiedWithSentinels;

    private SentinelValues sentinelValues;

    public LongByteHashMap()
    {
        this.allocateTable(DEFAULT_INITIAL_CAPACITY << 1);
    }

    public LongByteHashMap(int initialCapacity)
    {
        if (initialCapacity < 0)
        {
            throw new IllegalArgumentException("initial capacity cannot be less than 0");
        }
        int capacity = this.smallestPowerOfTwoGreaterThan(this.fastCeil(initialCapacity * OCCUPIED_DATA_RATIO));
        this.allocateTable(capacity);
    }

    private int smallestPowerOfTwoGreaterThan(int n)
    {
        return n > 1 ? Integer.highestOneBit(n - 1) << 1 : 1;
    }

    public LongByteHashMap(LongByteMap map)
    {
        this(Math.max(map.size(), DEFAULT_INITIAL_CAPACITY));
        this.putAll(map);
    }

    private int fastCeil(float v)
    {
        int possibleResult = (int) v;
        if (v - possibleResult > 0.0F)
        {
            possibleResult++;
        }
        return possibleResult;
    }

    public static LongByteHashMap newWithKeysValues(long key1, byte value1)
    {
        return new LongByteHashMap(1).withKeyValue(key1, value1);
    }

    public static LongByteHashMap newWithKeysValues(long key1, byte value1, long key2, byte value2)
    {
        return new LongByteHashMap(2).withKeysValues(key1, value1, key2, value2);
    }

    public static LongByteHashMap newWithKeysValues(long key1, byte value1, long key2, byte value2, long key3, byte value3)
    {
        return new LongByteHashMap(3).withKeysValues(key1, value1, key2, value2, key3, value3);
    }

    public static LongByteHashMap newWithKeysValues(long key1, byte value1, long key2, byte value2, long key3, byte value3, long key4, byte value4)
    {
        return new LongByteHashMap(4).withKeysValues(key1, value1, key2, value2, key3, value3, key4, value4);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }

        if (!(obj instanceof LongByteMap))
        {
            return false;
        }

        LongByteMap other = (LongByteMap) obj;

        if (this.size() != other.size())
        {
            return false;
        }

        if (this.sentinelValues == null)
        {
            if (other.containsKey(EMPTY_KEY) || other.containsKey(REMOVED_KEY))
            {
                return false;
            }
        }
        else
        {
            if (this.sentinelValues.containsZeroKey && (!other.containsKey(EMPTY_KEY) || this.sentinelValues.zeroValue != other.getOrThrow(EMPTY_KEY)))
            {
                return false;
            }

            if (this.sentinelValues.containsOneKey && (!other.containsKey(REMOVED_KEY) || this.sentinelValues.oneValue != other.getOrThrow(REMOVED_KEY)))
            {
                return false;
            }
        }

        for (int i = 0; i < this.keys.length; i++)
        {
            long key = this.keys[i];
            if (isNonSentinel(key) && (!other.containsKey(key) || this.values[i] != other.getOrThrow(key)))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int result = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += (int) (EMPTY_KEY ^ EMPTY_KEY >>> 32) ^ (int) this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += (int) (REMOVED_KEY ^ REMOVED_KEY >>> 32) ^ (int) this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += (int) (this.keys[i] ^ this.keys[i] >>> 32) ^ (int) this.values[i];
            }
        }

        return result;
    }

    @Override
    public String toString()
    {
        return this.makeString("{", ", ", "}");
    }

    public int size()
    {
        return this.occupiedWithData + (this.sentinelValues == null ? 0 : this.sentinelValues.size());
    }

    public boolean isEmpty()
    {
        return this.occupiedWithData == 0 && (this.sentinelValues == null || this.sentinelValues.size() == 0);
    }

    public boolean notEmpty()
    {
        return this.occupiedWithData != 0 || (this.sentinelValues != null && this.sentinelValues.size() != 0);
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            boolean first = true;

            if (this.sentinelValues != null)
            {
                if (this.sentinelValues.containsZeroKey)
                {
                    appendable.append(String.valueOf(EMPTY_KEY)).append("=").append(String.valueOf(this.sentinelValues.zeroValue));
                    first = false;
                }
                if (this.sentinelValues.containsOneKey)
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(REMOVED_KEY)).append("=").append(String.valueOf(this.sentinelValues.oneValue));
                    first = false;
                }
            }
            for (int i = 0; i < this.keys.length; i++)
            {
                long key = this.keys[i];
                if (isNonSentinel(key))
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(key)).append("=").append(String.valueOf(this.values[i]));
                    first = false;
                }
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public ByteIterator byteIterator()
    {
        return new InternalByteIterator();
    }

    public byte[] toArray()
    {
        byte[] array = new byte[this.size()];
        int index = 0;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                array[index] = this.sentinelValues.zeroValue;
                index++;
            }
            if (this.sentinelValues.containsOneKey)
            {
                array[index] = this.sentinelValues.oneValue;
                index++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                array[index] = this.values[i];
                index++;
            }
        }

        return array;
    }

    public boolean contains(byte value)
    {
        return this.containsValue(value);
    }

    public boolean containsAll(byte... source)
    {
        for (byte each : source)
        {
            if (!this.contains(each))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(ByteIterable source)
    {
        return source.allSatisfy(new BytePredicate()
        {
            public boolean accept(byte value)
            {
                return LongByteHashMap.this.contains(value);
            }
        });
    }

    public void forEach(ByteProcedure procedure)
    {
        this.forEachValue(procedure);
    }

    public MutableByteCollection select(BytePredicate predicate)
    {
        ByteArrayList result = new ByteArrayList();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }

        return result;
    }

    public MutableByteCollection reject(BytePredicate predicate)
    {
        ByteArrayList result = new ByteArrayList();
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                result.add(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                result.add(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public <V> MutableCollection<V> collect(ByteToObjectFunction<? extends V> function)
    {
        FastList<V> target = FastList.newList(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                target.add(function.valueOf(this.sentinelValues.zeroValue));
            }
            if (this.sentinelValues.containsOneKey)
            {
                target.add(function.valueOf(this.sentinelValues.oneValue));
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                target.add(function.valueOf(this.values[i]));
            }
        }
        return target;
    }

    public byte detectIfNone(BytePredicate predicate, byte value)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return this.values[i];
            }
        }
        return value;
    }

    public int count(BytePredicate predicate)
    {
        int count = 0;
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                count++;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                count++;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(BytePredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return true;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return true;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(BytePredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(BytePredicate predicate)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(this.sentinelValues.zeroValue))
            {
                return false;
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(this.sentinelValues.oneValue))
            {
                return false;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public <V> V injectInto(V injectedValue, ObjectByteToObjectFunction<? super V, ? extends V> function)
    {
        V result = injectedValue;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result = function.valueOf(result, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                result = function.valueOf(result, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result = function.valueOf(result, this.values[i]);
            }
        }

        return result;
    }

    public MutableByteList toList()
    {
        return ByteArrayList.newList(this);
    }

    public MutableByteSet toSet()
    {
        return ByteHashSet.newSet(this);
    }

    public MutableByteBag toBag()
    {
        return ByteHashBag.newBag(this);
    }

    public LazyByteIterable asLazy()
    {
        return new LazyByteIterableAdapter(this);
    }

    public void clear()
    {
        this.sentinelValues = null;
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;
        Arrays.fill(this.keys, EMPTY_KEY);
        Arrays.fill(this.values, EMPTY_VALUE);
    }

    public void put(long key, byte value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return;
        }

        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
            }
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return;
        }

        int index = this.probe(key);

        if (this.keys[index] == key)
        {
            // key already present in map
            this.values[index] = value;
            return;
        }

        if (this.keys[index] == REMOVED_KEY)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = key;
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
    }

    public void putAll(LongByteMap map)
    {
        map.forEachKeyValue(new LongByteProcedure()
        {
            public void value(long key, byte value)
            {
                LongByteHashMap.this.put(key, value);
            }
        });
    }

    public void removeKey(long key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return;
            }
            if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.containsZeroKey = false;
                this.sentinelValues.zeroValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.containsOneKey = false;
                this.sentinelValues.oneValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.keys[index] = REMOVED_KEY;
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }
        }
    }

    public void remove(long key)
    {
        this.removeKey(key);
    }

    public byte removeKeyIfAbsent(long key, byte value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return value;
            }
            byte oldValue = this.sentinelValues.zeroValue;
            if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.containsZeroKey = false;
                this.sentinelValues.zeroValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return oldValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return value;
            }
            byte oldValue = this.sentinelValues.oneValue;
            if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.containsOneKey = false;
                this.sentinelValues.oneValue = EMPTY_VALUE;
            }
            else
            {
                this.sentinelValues = null;
            }
            return oldValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.keys[index] = REMOVED_KEY;
            byte oldValue = this.values[index];
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }

            return oldValue;
        }
        return value;
    }

    public byte getIfAbsentPut(long key, byte value)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        if (this.keys[index] == REMOVED_KEY)
        {
            this.occupiedWithSentinels--;
        }
        this.keys[index] = key;
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public byte getIfAbsentPut(long key, ByteFunction0 function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                byte value = function.value();
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            byte value = function.value();
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                byte value = function.value();
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            byte value = function.value();
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        if (this.keys[index] == REMOVED_KEY)
        {
            this.occupiedWithSentinels--;
        }
        this.keys[index] = key;
        byte value = function.value();
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public <P> byte getIfAbsentPutWith(long key, ByteFunction<? super P> function, P parameter)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                byte value = function.byteValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            byte value = function.byteValueOf(parameter);
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                byte value = function.byteValueOf(parameter);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            byte value = function.byteValueOf(parameter);
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        if (this.keys[index] == REMOVED_KEY)
        {
            this.occupiedWithSentinels--;
        }
        this.keys[index] = key;
        byte value = function.byteValueOf(parameter);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public byte getIfAbsentPutWithKey(long key, LongToByteFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                byte value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = value;
                return value;
            }
            if (this.sentinelValues.containsZeroKey)
            {
                return this.sentinelValues.zeroValue;
            }
            byte value = function.valueOf(key);
            this.sentinelValues.containsZeroKey = true;
            this.sentinelValues.zeroValue = value;
            return value;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                byte value = function.valueOf(key);
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = value;
                return value;
            }
            if (this.sentinelValues.containsOneKey)
            {
                return this.sentinelValues.oneValue;
            }
            byte value = function.valueOf(key);
            this.sentinelValues.containsOneKey = true;
            this.sentinelValues.oneValue = value;
            return value;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        if (this.keys[index] == REMOVED_KEY)
        {
            this.occupiedWithSentinels--;
        }
        this.keys[index] = key;
        byte value = function.valueOf(key);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public byte updateValue(long key, byte initialValueIfAbsent, ByteToByteFunction function)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = function.valueOf(initialValueIfAbsent);
            }
            else if (this.sentinelValues.containsZeroKey)
            {
                this.sentinelValues.zeroValue = function.valueOf(this.sentinelValues.zeroValue);
            }
            else
            {
                this.sentinelValues.containsZeroKey = true;
                this.sentinelValues.zeroValue = function.valueOf(initialValueIfAbsent);
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null)
            {
                this.sentinelValues = new SentinelValues();
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = function.valueOf(initialValueIfAbsent);
            }
            else if (this.sentinelValues.containsOneKey)
            {
                this.sentinelValues.oneValue = function.valueOf(this.sentinelValues.oneValue);
            }
            else
            {
                this.sentinelValues.containsOneKey = true;
                this.sentinelValues.oneValue = function.valueOf(initialValueIfAbsent);
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            this.values[index] = function.valueOf(this.values[index]);
            return this.values[index];
        }
        if (this.keys[index] == REMOVED_KEY)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = key;
        byte value = function.valueOf(initialValueIfAbsent);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return value;
    }

    public LongByteHashMap withKeyValue(long key1, byte value1)
    {
        this.put(key1, value1);
        return this;
    }

    public LongByteHashMap withKeysValues(long key1, byte value1, long key2, byte value2)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        return this;
    }

    public LongByteHashMap withKeysValues(long key1, byte value1, long key2, byte value2, long key3, byte value3)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        return this;
    }

    public LongByteHashMap withKeysValues(long key1, byte value1, long key2, byte value2, long key3, byte value3, long key4, byte value4)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        this.put(key4, value4);
        return this;
    }

    public LongByteHashMap withoutKey(long key)
    {
        this.removeKey(key);
        return this;
    }

    public LongByteHashMap withoutAllKeys(LongIterable keys)
    {
        keys.forEach(new LongProcedure()
        {
            public void value(long key)
            {
                LongByteHashMap.this.removeKey(key);
            }
        });
        return this;
    }

    public MutableLongByteMap asUnmodifiable()
    {
        return new UnmodifiableLongByteMap(this);
    }

    public MutableLongByteMap asSynchronized()
    {
        return new SynchronizedLongByteMap(this);
    }

    public ImmutableLongByteMap toImmutable()
    {
        return LongByteMaps.immutable.ofAll(this);
    }

    public byte get(long key)
    {
        return this.getIfAbsent(key, EMPTY_VALUE);
    }

    public byte getIfAbsent(long key, byte ifAbsent)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                return ifAbsent;
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (this.keys[index] == key)
        {
            return this.values[index];
        }
        return ifAbsent;
    }

    public byte getOrThrow(long key)
    {
        if (isEmptyKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsZeroKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.zeroValue;
        }
        if (isRemovedKey(key))
        {
            if (this.sentinelValues == null || !this.sentinelValues.containsOneKey)
            {
                throw new IllegalStateException("Key " + key + " not present.");
            }
            return this.sentinelValues.oneValue;
        }
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]))
        {
            return this.values[index];
        }
        throw new IllegalStateException("Key " + key + " not present.");
    }

    public boolean containsKey(long key)
    {
        if (isEmptyKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsZeroKey;
        }
        if (isRemovedKey(key))
        {
            return this.sentinelValues != null && this.sentinelValues.containsOneKey;
        }
        return this.keys[this.probe(key)] == key;
    }

    public boolean containsValue(byte value)
    {
        if (this.sentinelValues != null && this.sentinelValues.containsValue(value))
        {
            return true;
        }
        for (int i = 0; i < this.values.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && this.values[i] == value)
            {
                return true;
            }
        }
        return false;
    }

    public void forEachValue(ByteProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.values[i]);
            }
        }
    }

    public void forEachKey(LongProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY);
            }
        }
        for (long key : this.keys)
        {
            if (isNonSentinel(key))
            {
                procedure.value(key);
            }
        }
    }

    public void forEachKeyValue(LongByteProcedure procedure)
    {
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                procedure.value(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                procedure.value(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.keys[i], this.values[i]);
            }
        }
    }

    public LazyLongIterable keysView()
    {
        return new KeysView();
    }

    public LongByteHashMap select(LongBytePredicate predicate)
    {
        LongByteHashMap result = new LongByteHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.keys[i], this.values[i]))
            {
                result.put(this.keys[i], this.values[i]);
            }
        }

        return result;
    }

    public LongByteHashMap reject(LongBytePredicate predicate)
    {
        LongByteHashMap result = new LongByteHashMap();

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY, this.sentinelValues.zeroValue))
            {
                result.put(EMPTY_KEY, this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY, this.sentinelValues.oneValue))
            {
                result.put(REMOVED_KEY, this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.keys[i], this.values[i]))
            {
                result.put(this.keys[i], this.values[i]);
            }
        }
        return result;
    }

    public long sum()
    {
        long result = 0L;

        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                result += this.sentinelValues.zeroValue;
            }
            if (this.sentinelValues.containsOneKey)
            {
                result += this.sentinelValues.oneValue;
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += this.values[i];
            }
        }

        return result;
    }

    public byte max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        ByteIterator iterator = this.byteIterator();
        byte max = iterator.next();
        while (iterator.hasNext())
        {
            byte value = iterator.next();
            if (max < value)
            {
                max = value;
            }
        }
        return max;
    }

    public byte maxIfEmpty(byte defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public byte min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        ByteIterator iterator = this.byteIterator();
        byte min = iterator.next();
        while (iterator.hasNext())
        {
            byte value = iterator.next();
            if (value < min)
            {
                min = value;
            }
        }
        return min;
    }

    public byte minIfEmpty(byte defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return (double) this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        byte[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            byte first = sortedArray[middleIndex];
            byte second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public byte[] toSortedArray()
    {
        byte[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public MutableByteList toSortedList()
    {
        return ByteArrayList.newList(this).sortThis();
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());
        if (this.sentinelValues != null)
        {
            if (this.sentinelValues.containsZeroKey)
            {
                out.writeLong(EMPTY_KEY);
                out.writeByte(this.sentinelValues.zeroValue);
            }
            if (this.sentinelValues.containsOneKey)
            {
                out.writeLong(REMOVED_KEY);
                out.writeByte(this.sentinelValues.oneValue);
            }
        }
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                out.writeLong(this.keys[i]);
                out.writeByte(this.values[i]);
            }
        }
    }

    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
    {
        int size = in.readInt();
        for (int i = 0; i < size; i++)
        {
            this.put(in.readLong(), in.readByte());
        }
    }

    /**
     * Rehashes every element in the set into a new backing table of the smallest possible size and eliminating removed sentinels.
     */
    public void compact()
    {
        this.rehash(this.smallestPowerOfTwoGreaterThan(this.size()));
    }

    private void rehash()
    {
        this.rehash(this.keys.length);
    }

    private void rehashAndGrow()
    {
        this.rehash(this.keys.length << 1);
    }

    private void rehash(int newCapacity)
    {
        int oldLength = this.keys.length;
        long[] old = this.keys;
        byte[] oldValues = this.values;
        this.allocateTable(newCapacity);
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;

        for (int i = 0; i < oldLength; i++)
        {
            if (isNonSentinel(old[i]))
            {
                this.put(old[i], oldValues[i]);
            }
        }
    }

    // exposed for testing
    int probe(long element)
    {
        int index = this.spread(element);
        long keyAtIndex = this.keys[index];

        if (keyAtIndex == element || keyAtIndex == EMPTY_KEY)
        {
            return index;
        }

        int removedIndex = keyAtIndex == REMOVED_KEY ? index : -1;
        int nextIndex = index;
        int probe = 17;

        // loop until an empty slot is reached
        while (true)
        {
            // Probe algorithm: 17*n*(n+1)/2 where n = number of collisions
            nextIndex += probe;
            probe += 17;
            nextIndex &= this.keys.length - 1;

            if (this.keys[nextIndex] == element)
            {
                return nextIndex;
            }
            if (this.keys[nextIndex] == REMOVED_KEY)
            {
                if (removedIndex == -1)
                {
                    removedIndex = nextIndex;
                }
            }
            else if (this.keys[nextIndex] == EMPTY_KEY)
            {
                return removedIndex == -1 ? nextIndex : removedIndex;
            }
        }
    }

    // exposed for testing
    int spread(long element)
    {
        long code = element;
        code = ~code + (code << 18);
        code = (code << 18) - code - 1;
        code ^= code >>> 31;
        code *= 21;
        code += (code << 2) + (code << 4);
        code ^= code >>> 11;
        code += code << 6;
        code ^= code >>> 22;
        return (int) code & (this.keys.length - 1);
    }

    private void allocateTable(int sizeToAllocate)
    {
        this.keys = new long[sizeToAllocate];
        this.values = new byte[sizeToAllocate];
    }

    private static boolean isEmptyKey(long key)
    {
        return key == EMPTY_KEY;
    }

    private static boolean isRemovedKey(long key)
    {
        return key == REMOVED_KEY;
    }

    private static boolean isNonSentinel(long key)
    {
        return !isEmptyKey(key) && !isRemovedKey(key);
    }

    private int maxOccupiedWithData()
    {
        int capacity = this.keys.length;
        // need at least one free slot for open addressing
        return Math.min(capacity - 1, capacity / OCCUPIED_DATA_RATIO);
    }

    private int maxOccupiedWithSentinels()
    {
        return this.keys.length / OCCUPIED_SENTINEL_RATIO;
    }

    private static final class SentinelValues
    {
        private boolean containsZeroKey;
        private boolean containsOneKey;
        private byte zeroValue;
        private byte oneValue;

        public int size()
        {
            return (this.containsZeroKey ? 1 : 0) + (this.containsOneKey ? 1 : 0);
        }

        public boolean containsValue(byte value)
        {
            boolean valueEqualsZeroValue = this.containsZeroKey && this.zeroValue == value;
            boolean valueEqualsOneValue = this.containsOneKey && this.oneValue == value;
            return valueEqualsZeroValue || valueEqualsOneValue;
        }
    }

    private class InternalByteIterator implements ByteIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < LongByteHashMap.this.size();
        }

        public byte next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (LongByteHashMap.this.containsKey(EMPTY_KEY))
                {
                    return LongByteHashMap.this.get(EMPTY_KEY);
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (LongByteHashMap.this.containsKey(REMOVED_KEY))
                {
                    return LongByteHashMap.this.get(REMOVED_KEY);
                }
            }

            long[] keys = LongByteHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            byte result = LongByteHashMap.this.values[this.position];
            this.position++;
            return result;
        }
    }

    private class KeysView implements LazyLongIterable
    {
        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public int size()
        {
            return LongByteHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return LongByteHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return LongByteHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (LongByteHashMap.this.sentinelValues != null)
                {
                    if (LongByteHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (LongByteHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (long key : LongByteHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public LongIterator longIterator()
        {
            return new KeySetIterator();
        }

        public boolean contains(long key)
        {
            return LongByteHashMap.this.containsKey(key);
        }

        public boolean containsAll(long... keys)
        {
            for (long key : keys)
            {
                if (!LongByteHashMap.this.containsKey(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(LongIterable source)
        {
            return source.allSatisfy(new LongPredicate()
            {
                public boolean accept(long key)
                {
                    return LongByteHashMap.this.containsKey(key);
                }
            });
        }

        public void forEach(LongProcedure procedure)
        {
            LongByteHashMap.this.forEachKey(procedure);
        }

        public int count(LongPredicate predicate)
        {
            int count = 0;
            if (LongByteHashMap.this.sentinelValues != null)
            {
                if (LongByteHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (LongByteHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (long key : LongByteHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(LongPredicate predicate)
        {
            if (LongByteHashMap.this.sentinelValues != null)
            {
                if (LongByteHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (LongByteHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (long key : LongByteHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(LongPredicate predicate)
        {
            if (LongByteHashMap.this.sentinelValues != null)
            {
                if (LongByteHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (LongByteHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (long key : LongByteHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(LongPredicate predicate)
        {
            if (LongByteHashMap.this.sentinelValues != null)
            {
                if (LongByteHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (LongByteHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (long key : LongByteHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public LazyLongIterable select(LongPredicate predicate)
        {
            return new SelectLongIterable(this, predicate);
        }

        public LazyLongIterable reject(LongPredicate predicate)
        {
            return new SelectLongIterable(this, LongPredicates.not(predicate));
        }

        public long detectIfNone(LongPredicate predicate, long value)
        {
            if (LongByteHashMap.this.sentinelValues != null)
            {
                if (LongByteHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (LongByteHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (long key : LongByteHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return value;
        }

        public <V> LazyIterable<V> collect(LongToObjectFunction<? extends V> function)
        {
            return new CollectLongToObjectIterable<V>(this, function);
        }

        public long sum()
        {
            long result = 0L;

            if (LongByteHashMap.this.sentinelValues != null)
            {
                if (LongByteHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += EMPTY_KEY;
                }
                if (LongByteHashMap.this.sentinelValues.containsOneKey)
                {
                    result += REMOVED_KEY;
                }
            }
            for (long key : LongByteHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result += key;
                }
            }

            return result;
        }

        public long max()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            LongIterator iterator = this.longIterator();
            long max = iterator.next();
            while (iterator.hasNext())
            {
                long value = iterator.next();
                if (max < value)
                {
                    max = value;
                }
            }
            return max;
        }

        public long min()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            LongIterator iterator = this.longIterator();
            long min = iterator.next();
            while (iterator.hasNext())
            {
                long value = iterator.next();
                if (value < min)
                {
                    min = value;
                }
            }
            return min;
        }

        public long minIfEmpty(long defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.min();
        }

        public long maxIfEmpty(long defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.max();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return (double) this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            long[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                long first = sortedArray[middleIndex];
                long second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public long[] toSortedArray()
        {
            long[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public long[] toArray()
        {
            int size = LongByteHashMap.this.size();
            final long[] result = new long[size];
            LongByteHashMap.this.forEachKey(new LongProcedure()
            {
                private int index;

                public void value(long each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public <T> T injectInto(T injectedValue, ObjectLongToObjectFunction<? super T, ? extends T> function)
        {
            T result = injectedValue;
            if (LongByteHashMap.this.sentinelValues != null)
            {
                if (LongByteHashMap.this.sentinelValues.containsZeroKey)
                {
                    result = function.valueOf(result, EMPTY_KEY);
                }
                if (LongByteHashMap.this.sentinelValues.containsOneKey)
                {
                    result = function.valueOf(result, REMOVED_KEY);
                }
            }
            for (int i = 0; i < LongByteHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(LongByteHashMap.this.keys[i]))
                {
                    result = function.valueOf(result, LongByteHashMap.this.keys[i]);
                }
            }
            return result;
        }

        public MutableLongList toList()
        {
            return LongArrayList.newList(this);
        }

        public MutableLongList toSortedList()
        {
            return LongArrayList.newList(this).sortThis();
        }

        public MutableLongSet toSet()
        {
            return LongHashSet.newSet(this);
        }

        public MutableLongBag toBag()
        {
            return LongHashBag.newBag(this);
        }

        public LazyLongIterable asLazy()
        {
            return this;
        }
    }

    private class KeySetIterator implements LongIterator
    {
        private int count;
        private int position;
        private boolean handledZero;
        private boolean handledOne;

        public boolean hasNext()
        {
            return this.count < LongByteHashMap.this.size();
        }

        public long next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            if (!this.handledZero)
            {
                this.handledZero = true;
                if (LongByteHashMap.this.containsKey(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
            }
            if (!this.handledOne)
            {
                this.handledOne = true;
                if (LongByteHashMap.this.containsKey(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }

            long[] keys = LongByteHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            long result = keys[this.position];
            this.position++;
            return result;
        }
    }

    public MutableLongSet keySet()
    {
        return new KeySet();
    }

    private class KeySet implements MutableLongSet
    {
        public LongIterator longIterator()
        {
            return new KeySetIterator();
        }

        public void forEach(LongProcedure procedure)
        {
            LongByteHashMap.this.forEachKey(procedure);
        }

        public int count(LongPredicate predicate)
        {
            int count = 0;
            if (LongByteHashMap.this.sentinelValues != null)
            {
                if (LongByteHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    count++;
                }
                if (LongByteHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    count++;
                }
            }
            for (long key : LongByteHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    count++;
                }
            }
            return count;
        }

        public boolean anySatisfy(LongPredicate predicate)
        {
            if (LongByteHashMap.this.sentinelValues != null)
            {
                if (LongByteHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return true;
                }
                if (LongByteHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return true;
                }
            }
            for (long key : LongByteHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return true;
                }
            }
            return false;
        }

        public boolean allSatisfy(LongPredicate predicate)
        {
            if (LongByteHashMap.this.sentinelValues != null)
            {
                if (LongByteHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (LongByteHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (long key : LongByteHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean noneSatisfy(LongPredicate predicate)
        {
            if (LongByteHashMap.this.sentinelValues != null)
            {
                if (LongByteHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return false;
                }
                if (LongByteHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return false;
                }
            }
            for (long key : LongByteHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean add(long element)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(long... source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(LongIterable source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(long key)
        {
            int oldSize = LongByteHashMap.this.size();
            LongByteHashMap.this.removeKey(key);
            return oldSize != LongByteHashMap.this.size();
        }

        public boolean removeAll(LongIterable source)
        {
            int oldSize = LongByteHashMap.this.size();
            LongIterator iterator = source.longIterator();
            while (iterator.hasNext())
            {
                LongByteHashMap.this.removeKey(iterator.next());
            }
            return oldSize != LongByteHashMap.this.size();
        }

        public boolean removeAll(long... source)
        {
            int oldSize = LongByteHashMap.this.size();
            for (long item : source)
            {
                LongByteHashMap.this.removeKey(item);
            }
            return oldSize != LongByteHashMap.this.size();
        }

        public void clear()
        {
            LongByteHashMap.this.clear();
        }

        public MutableLongSet select(LongPredicate predicate)
        {
            MutableLongSet result = new LongHashSet();
            if (LongByteHashMap.this.sentinelValues != null)
            {
                if (LongByteHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (LongByteHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (long key : LongByteHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableLongSet reject(LongPredicate predicate)
        {
            MutableLongSet result = new LongHashSet();
            if (LongByteHashMap.this.sentinelValues != null)
            {
                if (LongByteHashMap.this.sentinelValues.containsZeroKey && !predicate.accept(EMPTY_KEY))
                {
                    result.add(EMPTY_KEY);
                }
                if (LongByteHashMap.this.sentinelValues.containsOneKey && !predicate.accept(REMOVED_KEY))
                {
                    result.add(REMOVED_KEY);
                }
            }
            for (long key : LongByteHashMap.this.keys)
            {
                if (isNonSentinel(key) && !predicate.accept(key))
                {
                    result.add(key);
                }
            }
            return result;
        }

        public MutableLongSet with(long element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableLongSet without(long element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableLongSet withAll(LongIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableLongSet withoutAll(LongIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public long detectIfNone(LongPredicate predicate, long ifNone)
        {
            if (LongByteHashMap.this.sentinelValues != null)
            {
                if (LongByteHashMap.this.sentinelValues.containsZeroKey && predicate.accept(EMPTY_KEY))
                {
                    return EMPTY_KEY;
                }
                if (LongByteHashMap.this.sentinelValues.containsOneKey && predicate.accept(REMOVED_KEY))
                {
                    return REMOVED_KEY;
                }
            }
            for (long key : LongByteHashMap.this.keys)
            {
                if (isNonSentinel(key) && predicate.accept(key))
                {
                    return key;
                }
            }
            return ifNone;
        }

        public <V> MutableSet<V> collect(LongToObjectFunction<? extends V> function)
        {
            MutableSet<V> result = Sets.mutable.with();
            if (LongByteHashMap.this.sentinelValues != null)
            {
                if (LongByteHashMap.this.sentinelValues.containsZeroKey)
                {
                    result.add(function.valueOf(EMPTY_KEY));
                }
                if (LongByteHashMap.this.sentinelValues.containsOneKey)
                {
                    result.add(function.valueOf(REMOVED_KEY));
                }
            }
            for (long key : LongByteHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    result.add(function.valueOf(key));
                }
            }
            return result;
        }

        public MutableLongSet asUnmodifiable()
        {
            return UnmodifiableLongSet.of(this);
        }

        public MutableLongSet asSynchronized()
        {
            return SynchronizedLongSet.of(this);
        }

        public long sum()
        {
            long sum = 0L;
            if (LongByteHashMap.this.sentinelValues != null)
            {
                if (LongByteHashMap.this.sentinelValues.containsZeroKey)
                {
                    sum += EMPTY_KEY;
                }
                if (LongByteHashMap.this.sentinelValues.containsOneKey)
                {
                    sum += REMOVED_KEY;
                }
            }
            for (long key : LongByteHashMap.this.keys)
            {
                if (isNonSentinel(key))
                {
                    sum += key;
                }
            }
            return sum;
        }

        public long max()
        {
            if (LongByteHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            long max = 0;
            boolean isMaxSet = false;

            if (LongByteHashMap.this.sentinelValues != null)
            {
                if (LongByteHashMap.this.sentinelValues.containsZeroKey)
                {
                    max = EMPTY_KEY;
                    isMaxSet = true;
                }
                if (LongByteHashMap.this.sentinelValues.containsOneKey && (!isMaxSet || max < REMOVED_KEY))
                {
                    max = REMOVED_KEY;
                    isMaxSet = true;
                }
            }
            for (int i = 0; i < LongByteHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(LongByteHashMap.this.keys[i]) && (!isMaxSet || max < LongByteHashMap.this.keys[i]))
                {
                    max = LongByteHashMap.this.keys[i];
                    isMaxSet = true;
                }
            }
            return max;
        }

        public long maxIfEmpty(long defaultValue)
        {
            if (LongByteHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.max();
        }

        public long min()
        {
            if (LongByteHashMap.this.isEmpty())
            {
                throw new NoSuchElementException();
            }

            long min = 0;
            boolean isMinSet = false;

            if (LongByteHashMap.this.sentinelValues != null)
            {
                if (LongByteHashMap.this.sentinelValues.containsZeroKey)
                {
                    min = EMPTY_KEY;
                    isMinSet = true;
                }
                if (LongByteHashMap.this.sentinelValues.containsOneKey && (!isMinSet || REMOVED_KEY < min))
                {
                    min = REMOVED_KEY;
                    isMinSet = true;
                }
            }
            for (int i = 0; i < LongByteHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(LongByteHashMap.this.keys[i]) && (!isMinSet || LongByteHashMap.this.keys[i] < min))
                {
                    min = LongByteHashMap.this.keys[i];
                    isMinSet = true;
                }
            }
            return min;
        }

        public long minIfEmpty(long defaultValue)
        {
            if (LongByteHashMap.this.isEmpty())
            {
                return defaultValue;
            }

            return this.min();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return (double) this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            long[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                long first = sortedArray[middleIndex];
                long second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public long[] toSortedArray()
        {
            long[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public MutableLongList toSortedList()
        {
            return LongArrayList.newList(this).sortThis();
        }

        public long[] toArray()
        {
            int size = LongByteHashMap.this.size();
            final long[] result = new long[size];
            LongByteHashMap.this.forEachKey(new LongProcedure()
            {
                private int index;

                public void value(long each)
                {
                    result[this.index] = each;
                    this.index++;
                }
            });
            return result;
        }

        public boolean contains(long value)
        {
            return LongByteHashMap.this.containsKey(value);
        }

        public boolean containsAll(long... source)
        {
            for (long item : source)
            {
                if (!LongByteHashMap.this.containsKey(item))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(LongIterable source)
        {
            LongIterator iterator = source.longIterator();
            while (iterator.hasNext())
            {
                if (!LongByteHashMap.this.containsKey(iterator.next()))
                {
                    return false;
                }
            }
            return true;
        }

        public MutableLongList toList()
        {
            return LongArrayList.newList(this);
        }

        public MutableLongSet toSet()
        {
            return LongHashSet.newSet(this);
        }

        public MutableLongBag toBag()
        {
            return LongHashBag.newBag(this);
        }

        public LazyLongIterable asLazy()
        {
            return new LazyLongIterableAdapter(this);
        }

        public <T> T injectInto(T injectedValue, ObjectLongToObjectFunction<? super T, ? extends T> function)
        {
            T result = injectedValue;
            if (LongByteHashMap.this.sentinelValues != null)
            {
                if (LongByteHashMap.this.sentinelValues.containsZeroKey)
                {
                    result = function.valueOf(result, EMPTY_KEY);
                }
                if (LongByteHashMap.this.sentinelValues.containsOneKey)
                {
                    result = function.valueOf(result, REMOVED_KEY);
                }
            }
            for (int i = 0; i < LongByteHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(LongByteHashMap.this.keys[i]))
                {
                    result = function.valueOf(result, LongByteHashMap.this.keys[i]);
                }
            }
            return result;
        }

        public LongSet freeze()
        {
            throw new UnsupportedOperationException();
        }

        public ImmutableLongSet toImmutable()
        {
            return LongSets.immutable.withAll(this);
        }

        public int size()
        {
            return LongByteHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return LongByteHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return LongByteHashMap.this.notEmpty();
        }

        @Override
        public boolean equals(Object obj)
        {
            if (this == obj)
            {
                return true;
            }

            if (!(obj instanceof LongSet))
            {
                return false;
            }

            LongSet other = (LongSet) obj;
            return this.size() == other.size() && this.containsAll(other.toArray());
        }

        @Override
        public int hashCode()
        {
            int result = 0;

            if (LongByteHashMap.this.sentinelValues != null)
            {
                if (LongByteHashMap.this.sentinelValues.containsZeroKey)
                {
                    result += (int) (EMPTY_KEY ^ EMPTY_KEY >>> 32);
                }
                if (LongByteHashMap.this.sentinelValues.containsOneKey)
                {
                    result += (int) (REMOVED_KEY ^ REMOVED_KEY >>> 32);
                }
            }
            for (int i = 0; i < LongByteHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(LongByteHashMap.this.keys[i]))
                {
                    result += (int) (LongByteHashMap.this.keys[i] ^ LongByteHashMap.this.keys[i] >>> 32);
                }
            }

            return result;
        }

        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);
                boolean first = true;
                if (LongByteHashMap.this.sentinelValues != null)
                {
                    if (LongByteHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(EMPTY_KEY));
                        first = false;
                    }
                    if (LongByteHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(REMOVED_KEY));
                        first = false;
                    }
                }
                for (long key : LongByteHashMap.this.keys)
                {
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(key));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }
    }

    public MutableByteCollection values()
    {
        return new ValuesCollection();
    }

    private class ValuesCollection implements MutableByteCollection
    {
        public void clear()
        {
            LongByteHashMap.this.clear();
        }

        public MutableByteCollection select(BytePredicate predicate)
        {
            return LongByteHashMap.this.select(predicate);
        }

        public MutableByteCollection reject(BytePredicate predicate)
        {
            return LongByteHashMap.this.reject(predicate);
        }

        public byte detectIfNone(BytePredicate predicate, byte ifNone)
        {
            return LongByteHashMap.this.detectIfNone(predicate, ifNone);
        }

        public <V> MutableCollection<V> collect(ByteToObjectFunction<? extends V> function)
        {
            return LongByteHashMap.this.collect(function);
        }

        public <T> T injectInto(T injectedValue, ObjectByteToObjectFunction<? super T, ? extends T> function)
        {
            return LongByteHashMap.this.injectInto(injectedValue, function);
        }

        public long sum()
        {
            return LongByteHashMap.this.sum();
        }

        public byte max()
        {
            return LongByteHashMap.this.max();
        }

        public byte maxIfEmpty(byte defaultValue)
        {
            return LongByteHashMap.this.maxIfEmpty(defaultValue);
        }

        public byte min()
        {
            return LongByteHashMap.this.min();
        }

        public byte minIfEmpty(byte defaultValue)
        {
            return LongByteHashMap.this.minIfEmpty(defaultValue);
        }

        public double average()
        {
            return LongByteHashMap.this.average();
        }

        public double median()
        {
            return LongByteHashMap.this.median();
        }

        public byte[] toSortedArray()
        {
            return LongByteHashMap.this.toSortedArray();
        }

        public MutableByteList toSortedList()
        {
            return LongByteHashMap.this.toSortedList();
        }

        public MutableByteCollection with(byte element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableByteCollection without(byte element)
        {
            throw new UnsupportedOperationException();
        }

        public MutableByteCollection withAll(ByteIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableByteCollection withoutAll(ByteIterable elements)
        {
            throw new UnsupportedOperationException();
        }

        public MutableByteCollection asUnmodifiable()
        {
            return UnmodifiableByteCollection.of(this);
        }

        public MutableByteCollection asSynchronized()
        {
            return SynchronizedByteCollection.of(this);
        }

        public ImmutableByteCollection toImmutable()
        {
            return ByteLists.immutable.withAll(this);
        }

        public boolean contains(byte value)
        {
            return LongByteHashMap.this.containsValue(value);
        }

        public boolean containsAll(byte... source)
        {
            return LongByteHashMap.this.containsAll(source);
        }

        public boolean containsAll(ByteIterable source)
        {
            return LongByteHashMap.this.containsAll(source);
        }

        public MutableByteList toList()
        {
            return LongByteHashMap.this.toList();
        }

        public MutableByteSet toSet()
        {
            return LongByteHashMap.this.toSet();
        }

        public MutableByteBag toBag()
        {
            return LongByteHashMap.this.toBag();
        }

        public LazyByteIterable asLazy()
        {
            return new LazyByteIterableAdapter(this);
        }

        public boolean isEmpty()
        {
            return LongByteHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return LongByteHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);

                boolean first = true;

                if (LongByteHashMap.this.sentinelValues != null)
                {
                    if (LongByteHashMap.this.sentinelValues.containsZeroKey)
                    {
                        appendable.append(String.valueOf(LongByteHashMap.this.sentinelValues.zeroValue));
                        first = false;
                    }
                    if (LongByteHashMap.this.sentinelValues.containsOneKey)
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(LongByteHashMap.this.sentinelValues.oneValue));
                        first = false;
                    }
                }
                for (int i = 0; i < LongByteHashMap.this.keys.length; i++)
                {
                    long key = LongByteHashMap.this.keys[i];
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(LongByteHashMap.this.values[i]));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public ByteIterator byteIterator()
        {
            return LongByteHashMap.this.byteIterator();
        }

        public void forEach(ByteProcedure procedure)
        {
            LongByteHashMap.this.forEach(procedure);
        }

        public int count(BytePredicate predicate)
        {
            return LongByteHashMap.this.count(predicate);
        }

        public boolean anySatisfy(BytePredicate predicate)
        {
            return LongByteHashMap.this.anySatisfy(predicate);
        }

        public boolean allSatisfy(BytePredicate predicate)
        {
            return LongByteHashMap.this.allSatisfy(predicate);
        }

        public boolean noneSatisfy(BytePredicate predicate)
        {
            return LongByteHashMap.this.noneSatisfy(predicate);
        }

        public boolean add(byte element)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(byte... source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(ByteIterable source)
        {
            throw new UnsupportedOperationException();
        }

        public boolean remove(byte item)
        {
            int oldSize = LongByteHashMap.this.size();

            if (LongByteHashMap.this.sentinelValues != null)
            {
                if (LongByteHashMap.this.sentinelValues.containsZeroKey && item == LongByteHashMap.this.sentinelValues.zeroValue)
                {
                    LongByteHashMap.this.removeKey(EMPTY_KEY);
                }
                if (LongByteHashMap.this.sentinelValues.containsOneKey && item == LongByteHashMap.this.sentinelValues.oneValue)
                {
                    LongByteHashMap.this.removeKey(REMOVED_KEY);
                }
            }
            for (int i = 0; i < LongByteHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(LongByteHashMap.this.keys[i]) && item == LongByteHashMap.this.values[i])
                {
                    LongByteHashMap.this.removeKey(LongByteHashMap.this.keys[i]);
                }
            }
            return oldSize != LongByteHashMap.this.size();
        }

        public boolean removeAll(ByteIterable source)
        {
            int oldSize = LongByteHashMap.this.size();

            ByteIterator iterator = source.byteIterator();
            while (iterator.hasNext())
            {
                this.remove(iterator.next());
            }
            return oldSize != LongByteHashMap.this.size();
        }

        public boolean removeAll(byte... source)
        {
            int oldSize = LongByteHashMap.this.size();

            for (byte item : source)
            {
                this.remove(item);
            }
            return oldSize != LongByteHashMap.this.size();
        }

        public int size()
        {
            return LongByteHashMap.this.size();
        }

        public byte[] toArray()
        {
            return LongByteHashMap.this.toArray();
        }
    }
}
